# Copyright (c) 2021 Emanuele Bellocchia
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.


# Imports
import unittest
from bip_utils import (
    Bip44BitcoinCashMainNet, Bip44BitcoinCashTestNet, Bip44LitecoinMainNet, Bip44LitecoinTestNet,
    Bip44, Bip44Coins, Bip49Coins, Bip84Coins
)
from tests.bip.bip44_base.test_bip44_base import Bip44BaseTestHelper

# Results generated with: https://iancoleman.io/bip39
# Or with coin-specific/multi-coin wallets (e.g. TronLink, Cosmostation, Trust Wallet, Math Wallet)
# There are some differences from the website and the specs I found for Litecoin testnet (extended keys prefixes) so,
# in that case, the keys were generated by this library after begin tested for the correct addresses
TEST_VECT = [
    #
    # Main nets
    #

    # Algorand
    {
        "coin": Bip44Coins.ALGORAND,
        "names": ("Algorand", "ALGO"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K4GYBne6aGFZLHUSCPzQ3amKxfzz6kHpqDLBHk58dvPW7twcRFJeo1oQAvNeiYpZEeUkWCW3YgdvXHUcQQU6xjh8jBo4Y8xG",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6Cury1Tm445cVt5je2szhiiAhfgJW1ke9hNjNbuMPE7MZ7ipvjhfp7N1wM39u6jc7GhsTDqpZc3hZM4716FWfvNTMFQStzE1yf7jQWZUsKP",
            "ex_priv": "xprv9yvWZVvsDgXKHQ1GY1LzLamS9dqp6Z2nnUT8aDVjptaNgKPgPCPRGK3Y69Nb1TcLHsvh7iAA9HiRdJxgdYr6B8m14p7vWzUrYTB8acAjZMk",
        },
        "chain_ext": {
            "ex_pub": "xpub6EFjDRgeg9872k9ahTSsiTKDoxaL7DaVjuqkMmSFqYncfCS3ZXTBENkWyQJdrHJXKJbEo3xdbvtjApfnczBbHhYtPEGUGtZxs1ULDFpgm4e",
            "ex_priv": "xprvA1GNov9kqmZopG57bRusMKNVFvjqhkreNgv9ZP2eHDFdnQ6u1z8vgaS38CeRMJLCWASNR3AGWhLAaKnyD1j4XSxsZ3HLr3kkzA2A15SAdMW",
        },
        "addresses": [
            "EP2D7TV7IAFANZHK3B6QLKB53N5UTD7RARVXZTWCPCRQQBKYVGM2XIMT2Q",
            "4TY6NBYEJMUOI5WPNUG7YRM63GQNVZA47336IICDLIARPF7GFUMSUJ2CUU",
            "YJBVNZYARZIVME3DWJLXWMJ22YWMANMTHGPLO4O7A2FDWP6662SER5SWLA",
            "4FNYH2QEESPLDGZJGNCOLRD7EZCIKWO3BHUPXTCU54JRJKCUN35FTQCTFQ",
            "NLMHFPS37XEHFWDPUUN3MHWVPYEBRUPFQWZVHIMRKPHSW5UPUYLJCVZN5I",
        ],
    },
    # Avax C-Chain
    {
        "coin": Bip44Coins.AVAX_C_CHAIN,
        "names": ("Avax C-Chain", "AVAX"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6DCoCpSuQZB2jawqnGMEPS63ePKWkwWPH4TU45Q7LPXWuNd8TMtVxRrgjtEshuqpK3mdhaWHPFsBngh5GFZaM6si3yZdUsT8ddYM3PwnATt",
            "ex_priv": "xprv9zDSoJv1aBcjX6sNgEpE2J9K6MV2MUnXuqXsFgzVn3zY2aHyupaFQdYCtdCbNMkvcTdx9FeN49sgXw6mjrhrFLRSzJVnRYPfSCCgjeg4GxY",
        },
        "chain_ext": {
            "ex_pub": "xpub6EF8jXqFeFEW5bwMU7RpQtHkzE4KJxcqJtvkCjJumzW8CPpacXkb92ek4WzLQXjL93HycJwTPUAcuNxCqFPKKU5m5Z2Vq4nCyh5CyPeBFFr",
            "ex_priv": "xprvA1FnL2JMosgCs7rtN5tp3kM2SCDpuVtywg19QLuJDey9KbVS4zSLbELGDFdugxESvYxCU1wEJhDmMXvHKwmpCLd8QSoUJCHmVHCGvpFPxiw",
        },
        "addresses": [
            "0x9858EfFD232B4033E47d90003D41EC34EcaEda94",
            "0x6Fac4D18c912343BF86fa7049364Dd4E424Ab9C0",
            "0xb6716976A3ebe8D39aCEB04372f22Ff8e6802D7A",
            "0xF3f50213C1d2e255e4B2bAD430F8A38EEF8D718E",
            "0x51cA8ff9f1C0a99f88E86B8112eA3237F55374cA",
        ],
    },
    # Avax P-Chain
    {
        "coin": Bip44Coins.AVAX_P_CHAIN,
        "names": ("Avax P-Chain", "AVAX"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6BqVigHfL2TNNs8HyeEHn4JFFyTw1vL8kC5ZhzBhrhDbQ3FhgakpivT97Cd7oVCJiAwiqWu313vKMZMwCghXgSVDnYR3FrYzTz24yY3nFHR",
            "ex_priv": "xprv9xr9KAkmVeu5AP3pschHQvMWhwdScTcHNy9xubn6JMgcXEvZ93SaB88fFwyXpUc6VhWP1Pxteh7UkRfaz5HrLREFNMx81c7aeo9yMUyv3WX",
        },
        "chain_ext": {
            "ex_pub": "xpub6EoeHmAgXUU3wHGwVX5JSBhT8RjKF2qVkBHjwa8FtXfPmPsD62QbZqmcWARs46oU2M3DiRqnJDek7CmQtL2WaKMBxJkLvhL5SBYu4MQJ5kd",
            "ex_priv": "xprvA1pHtFdnh6ukioCUPVYJ53kiaPtpqa7eNxN99BieLC8QtbY4YV6M23T8esWd9Bnnq9Q6ENVGWvSFpgZaDYSpzUa3m8vHhLo7A1NuSkzkMo9",
        },
        "addresses": [
            "P-avax1p9575chzhvcwvmvzaqh7yeld76r3af0ha56phl",
            "P-avax1saceyycp6klllavjmt5xd9dxzk7mffzp6fzwtu",
            "P-avax1y0gg2ymufvsvvnfxtzsuxdcnlfq6g76g06crer",
            "P-avax1lwl7qfz6rx0kxnce3n0j24gyppm3c7kxkww490",
            "P-avax1kcl5ywvftgtdlqp33gh69xem8h04en0ahramxy",
        ],
    },
    # Avax X-Chain
    {
        "coin": Bip44Coins.AVAX_X_CHAIN,
        "names": ("Avax X-Chain", "AVAX"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6BqVigHfL2TNNs8HyeEHn4JFFyTw1vL8kC5ZhzBhrhDbQ3FhgakpivT97Cd7oVCJiAwiqWu313vKMZMwCghXgSVDnYR3FrYzTz24yY3nFHR",
            "ex_priv": "xprv9xr9KAkmVeu5AP3pschHQvMWhwdScTcHNy9xubn6JMgcXEvZ93SaB88fFwyXpUc6VhWP1Pxteh7UkRfaz5HrLREFNMx81c7aeo9yMUyv3WX",
        },
        "chain_ext": {
            "ex_pub": "xpub6EoeHmAgXUU3wHGwVX5JSBhT8RjKF2qVkBHjwa8FtXfPmPsD62QbZqmcWARs46oU2M3DiRqnJDek7CmQtL2WaKMBxJkLvhL5SBYu4MQJ5kd",
            "ex_priv": "xprvA1pHtFdnh6ukioCUPVYJ53kiaPtpqa7eNxN99BieLC8QtbY4YV6M23T8esWd9Bnnq9Q6ENVGWvSFpgZaDYSpzUa3m8vHhLo7A1NuSkzkMo9",
        },
        "addresses": [
            "X-avax1p9575chzhvcwvmvzaqh7yeld76r3af0ha56phl",
            "X-avax1saceyycp6klllavjmt5xd9dxzk7mffzp6fzwtu",
            "X-avax1y0gg2ymufvsvvnfxtzsuxdcnlfq6g76g06crer",
            "X-avax1lwl7qfz6rx0kxnce3n0j24gyppm3c7kxkww490",
            "X-avax1kcl5ywvftgtdlqp33gh69xem8h04en0ahramxy",
        ],
    },
    # Band Protocol
    {
        "coin": Bip44Coins.BAND_PROTOCOL,
        "names": ("Band Protocol", "BAND"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6BiXRoxTDK73KtKp3XCjv8WRBRqq57TJezk1JJLW92ZJ8B7JmYzrT2Vmh73dg3WfQ8JrRPgGVuLAAH6XGL5mJmCD1k8cScva3tMBZMbYenk",
            "ex_priv": "xprv9xjB2JRZNwYk7QFLwVfjYzZgdQ1LfejTHmpQVuvtah2KFNnAE1gbuEBHqqbD2AbnkrMefKehvkCLeQHoKmjFcqZXug1EcNfxqqRr79VQxvP",
        },
        "chain_ext": {
            "ex_pub": "xpub6E8ivQVjyUx49UrrCjzsG5zUK7DPAy6n7SwR96CFEQAvJjbwZeKFakcoHvnmhjNYMqPTr9BJCMXjkH5dKpYV2UFEdiWtffwaEjPUT46popz",
            "ex_priv": "xprvA19NWtxr97PkvznP6iTrtx3jm5NtmWNvkE1pLhndg4dwRwGo27112xJKSgH1mR3b5YEEX9B9hxQx9RhVxv7q4WPLGJ4aAfdF2yvrra4A2T1",
        },
        "addresses": [
            "band1vh8tr8ddf7g0gfep23t46yllmlvtqfjknjdd38",
            "band1c2pramju9wtdz2jqulfalku58pq2ecgxcqj0f8",
            "band1tzt3hf6vges3wqhl3tuhm70r89tqk80ckndy5a",
            "band1zjxdfh8954dkehatc06jajh5v6jqlxtsc5pm7w",
            "band1n2eculyv945phq5y7cj4l0a6tkg2px79stvxpd",
        ],
    },
    # Binance Chain
    {
        "coin": Bip44Coins.BINANCE_CHAIN,
        "names": ("Binance Chain", "BNB"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6CRX5uDQjw6gU7QUQ8v8nKA99TvDxi3ZGM1f72LgkjsgKN7Ewi7ZL8krAiAzyuUvARRpLN5jh6YLfZLQ3TWu5oJYh1Wfd8Vz3rMJnh6bXj6",
            "ex_priv": "xprv9ySAgPgWuZYPFdL1J7P8RBDQbS5jZFKhu864Jdw5CQLhSZn6QAoJnLSNKTPGiXhPQuq6qqdqQhmBMo8B95xfh2CpkBpNXyqWUJH6jPqYbSZ",
        },
        "chain_ext": {
            "ex_pub": "xpub6FHVUGAXC7qrQ9akZqAJZXL998HcnkcnsxFf7vZBR6hHGyk4R3Ruc2G4xZGk1iZ9oudNmF1jKKXcowtufcuymASMB1YvPXqchUzgCoqPZLL",
            "ex_priv": "xprvA2J94kddMkHZBfWHTodJCPPQb6T8PHtwWjL4KY9ZrmAJQBQusW7f4Dwb7HQH5s4cHpMDCSXpD9Ar31fdvmC8gDcUovS3ZdSxHr3PZLejToP",
        },
        "addresses": [
            "bnb1rxhz5vdv4fvdjye8gxqvfv0yvg20jtlwf4f38d",
            "bnb13ajdzjhcczjg7mh60mc0d4dsqhj2q0xs22czn8",
            "bnb1qw8m2q8rhjvpq29drvn3n8e3eujx4l4phs27ac",
            "bnb182uz9jph8ran264scmpe529n4rupsvsf03y2xl",
            "bnb1jpcfwe6x9q3q5t0dfjcfz052x8q94vlygyva4r",
        ],
    },
    # Binance Smart Chain
    {
        "coin": Bip44Coins.BINANCE_SMART_CHAIN,
        "names": ("Binance Smart Chain", "BNB"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6DCoCpSuQZB2jawqnGMEPS63ePKWkwWPH4TU45Q7LPXWuNd8TMtVxRrgjtEshuqpK3mdhaWHPFsBngh5GFZaM6si3yZdUsT8ddYM3PwnATt",
            "ex_priv": "xprv9zDSoJv1aBcjX6sNgEpE2J9K6MV2MUnXuqXsFgzVn3zY2aHyupaFQdYCtdCbNMkvcTdx9FeN49sgXw6mjrhrFLRSzJVnRYPfSCCgjeg4GxY",
        },
        "chain_ext": {
            "ex_pub": "xpub6EF8jXqFeFEW5bwMU7RpQtHkzE4KJxcqJtvkCjJumzW8CPpacXkb92ek4WzLQXjL93HycJwTPUAcuNxCqFPKKU5m5Z2Vq4nCyh5CyPeBFFr",
            "ex_priv": "xprvA1FnL2JMosgCs7rtN5tp3kM2SCDpuVtywg19QLuJDey9KbVS4zSLbELGDFdugxESvYxCU1wEJhDmMXvHKwmpCLd8QSoUJCHmVHCGvpFPxiw",
        },
        "addresses": [
            "0x9858EfFD232B4033E47d90003D41EC34EcaEda94",
            "0x6Fac4D18c912343BF86fa7049364Dd4E424Ab9C0",
            "0xb6716976A3ebe8D39aCEB04372f22Ff8e6802D7A",
            "0xF3f50213C1d2e255e4B2bAD430F8A38EEF8D718E",
            "0x51cA8ff9f1C0a99f88E86B8112eA3237F55374cA",
        ],
    },
    # Bitcoin
    {
        "coin": Bip44Coins.BITCOIN,
        "names": ("Bitcoin", "BTC"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
        "wif_master": "5HzxC8XHHAtoC5jVvScY8Tr99Ud9MwFdF2pJKYsMTUknJZEurYr",
        "account": {
            "ex_pub": "xpub6BosfCnifzxcFwrSzQiqu2DBVTshkCXacvNsWGYJVVhhawA7d4R5WSWGFNbi8Aw6ZRc1brxMyWMzG3DSSSSoekkudhUd9yLb6qx39T9nMdj",
            "ex_priv": "xprv9xpXFhFpqdQK3TmytPBqXtGSwS3DLjojFhTGht8gwAAii8py5X6pxeBnQ6ehJiyJ6nDjWGJfZ95WxByFXVkDxHXrqu53WCRGypk2ttuqncb",
        },
        "chain_ext": {
            "ex_pub": "xpub6ELHKXNimKbxMCytPh7EdC2QXx46T9qLDJWGnTraz1H9kMMFdcduoU69wh9cxP12wDxqAAfbaESWGYt5rREsX1J8iR2TEunvzvddduAPYcY",
            "ex_priv": "xprvA1Lvv1qpvx3f8iuRHfaEG45fyvDc3h7Ur5afz5SyRfkAsZ2765KfFfmg6Q9oEJDgf4UdYHphzzJybLykZfznUMKL2KNUU8pLRQgstN5kmFe",
        },
        "addresses": [
            "1LqBGSKuX5yYUonjxT5qGfpUsXKYYWeabA",
            "1Ak8PffB2meyfYnbXZR9EGfLfFZVpzJvQP",
            "1MNF5RSaabFwcbtJirJwKnDytsXXEsVsNb",
            "1MVGa13XFvvpKGZdX389iU8b3qwtmAyrsJ",
            "1Gka4JdwhLxRwXaC6oLNH4YuEogeeSwqW7",
        ],
    },
    # Bitcoin Cash
    {
        "coin": Bip44Coins.BITCOIN_CASH,
        "names": ("Bitcoin Cash", "BCH"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
        "wif_master": "5HzxC8XHHAtoC5jVvScY8Tr99Ud9MwFdF2pJKYsMTUknJZEurYr",
        "account": {
            "ex_pub": "xpub6ByHsPNSQXTWZ7PLESMY2FufyYWtLXagSUpMQq7Un96SiThZH2iJB1X7pwviH1WtKVeDP6K8d6xxFzzoaFzF3s8BKCZx8oEDdDkNnp4owAZ",
            "ex_priv": "xprv9xywTsqYa9uDLdJs8QpXf7xwRWgPw4rq5FtkcShsDoZTqfNQjVQ3dDCdyedXX3FqB18U8e8PfVMeFqkhzPGseKVMDjGe5rPdiUXMxy7BQNJ",
        },
        "chain_ext": {
            "ex_pub": "xpub6F2iaK2JUPcgrZ6RTGH6t8VybLPu1XzfrHsDsaKvK6NfULznU6i6aw6ZoefDW2DpNruSLw73RwQg46qvpqB3eryeJJ2tkFCF4Z6gbr8Pjja",
            "ex_priv": "xprvA23NAoVQe24Pe51xMEk6WzZF3JZQc5GpV4wd5BvJkkqgbYfdvZPr38n5xR8cpzWJ6AjAnLRYVcqLWMsCHghvCdQTtcZm9HWTStmvYiT7BTC",
        },
        "addresses_legacy": {
            "cls": Bip44BitcoinCashMainNet,
            "addresses": [
                "1mW6fDEMjKrDHvLvoEsaeLxSCzZBf3Bfg",
                "18Cp2ivkLHyJwHMm9NzDRBh6Gi7m4MC2we",
                "15Ax9BJRJ4TABF85UsPpz9QvuBpiJhCfsw",
                "1H53u5dU1axLgun9VxUjQZMBMxW65DazhB",
                "18rTBsgLJosVgZodepiQ1wFoGFhhMK3iwL",
            ]
        },
        "addresses": [
            "bitcoincash:qqyx49mu0kkn9ftfj6hje6g2wfer34yfnq5tahq3q6",
            "bitcoincash:qp8sfdhgjlq68hlzka9lcsxtcnvuvnd0xqxugfzzc5",
            "bitcoincash:qqkuy34ntrye9a2h4xpdstcu4aq5wfrwscjtaphenr",
            "bitcoincash:qzcyvxr0e23d408u62ulf6cnspc0k4dyduy8kh77nc",
            "bitcoincash:qptzx8m39zjuuyvrf86s3kywuledfht2jcty8we6gv",
        ],
    },
    # BitcoinSV
    {
        "coin": Bip44Coins.BITCOIN_SV,
        "names": ("BitcoinSV", "BSV"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
        "wif_master": "5HzxC8XHHAtoC5jVvScY8Tr99Ud9MwFdF2pJKYsMTUknJZEurYr",
        "account": {
            "ex_pub": "xpub6CdMDgU2hzWyeZ852LWqp5AfDz3ty2cRfi4jEw9BT8aNYugMQvVykQsKLARZdbqKKp7yTviJdL1N9saYLmJNKD1rwVAwLTmU8r8qKeoyG4R",
            "ex_priv": "xprv9ydzpAw8scxgS53bvJyqSwDvfxDQZZtaJV98SYjZto3Pg7MCsPBjCcYqUtnWPRNayEXUcSYZDvXux545bHZwda7YUWvReJiRkx38VXathgK",
        },
        "chain_ext": {
            "ex_pub": "xpub6F4QRDXQLHsQatQpTvCAz7tiRZvKUj74BpUXYL4DneDnCGxtdGo44BUWQtGy2wAotFE6t6cB3kDEWbnxqaEdVdg3bb3uLRECKkBnnK1pdzQ",
            "ex_priv": "xprvA2541hzWVvK7NQLMMtfAcywysY5q5GPCpbYvjwecEJgoKUdk5jUoWPA2ZcJDkGmD3Fg2cVpmafjqJsVrdPPKq5cQ7cVyz2yQZrg4ceRnTXo",
        },
        "addresses": [
            "1K6LZdwpKT5XkEZo2T2kW197aMXYbYMc4f",
            "1DhquSu6ky8QQnf88b1d3tRYeUkMLASZg9",
            "155Vurs4bMMu5BemtZ6cVPhryGWef4VxZu",
            "1MgmeSFC4F5L8idUUW7fSL7QFfwbxJaW8B",
            "1D4eJS94heKPtAaU5jcfecMEk5i1vzEvAo",
        ],
    },
    # Cosmos
    {
        "coin": Bip44Coins.COSMOS,
        "names": ("Cosmos", "ATOM"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6DGzViq8bmgMLYdVZ3xnLVEdKwzBnGdzzJZ4suG8kVb9TTLAbrwv8YdKBb8FWKdBNinaHKmBv7JpQvqBYx4rxch7WnHzNFzSVrMf8hQepTP",
            "ex_priv": "xprv9zHe6DJEmQ8484Z2T2RmyMHtmv9hNov9d5dU5WrXCA4Aaf124KdfakJqLGnVYwvBvMP4ttqm7QL5vuPTepiJiEJREiJnhswKmj6zLttwCMu",
        },
        "chain_ext": {
            "ex_pub": "xpub6EbeQ8h8S7mEGxYTn34gKedwvsmytMW61v51RAc8e6VzgMMLFkkNMN6sYtqvtZQ8TjBKswbxUTzxzj3nZfwr4pzFB7rySMaBUUCJnCuxu33",
            "ex_priv": "xprvA1cHzdAEbkCw4UTzg1XfxWhDNqwVUtnEeh9QcnCX5ky1oZ2BiDS7oZnPhdgFNqWwGtB3BNebior1tS26wAtbQzCTe5Pty1BhkcwT8QfWQSP",
        },
        "addresses": [
            "cosmos19rl4cm2hmr8afy4kldpxz3fka4jguq0auqdal4",
            "cosmos1jrkmdcwgq94uaamx6zax2luewlhf7u4kucx3kz",
            "cosmos1kng7tv83qesgvv2ze7hxlw4urfrjk8vqqnpqdt",
            "cosmos1zuvk68xw4y9swp06796rx8zarjvvkrt606nxtl",
            "cosmos12ecyuddg4wlnhztusgju5zxxjgznkz7fsf57kq",
        ],
    },
    # Dash
    {
        "coin": Bip44Coins.DASH,
        "names": ("Dash", "DASH"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
        "wif_master": "7qjXbkdi3WJ1SRu6pVnUcqQzk9RimTJB3cCUrw9V1HGFoqaB948",
        "account": {
            "ex_pub": "xpub6CYEjsU6zPM3sADS2ubu2aZeGxCm3C5KabkCpo4rkNbXGAH9M7rRUJ4E5CKiyUddmRzrSCopPzisTBrXkfCD4o577XKM9mzyZtP1Xdbizyk",
            "ex_priv": "xprv9yYtLMwDA1nkeg8xvt4tfScuivNGdjMUDNpc2QfFC34YPMwzoaYAvVjkDvJ1APVeGMJnbJ6gPZMq4G7UfExoP5PpAS3UCF9utjeWa5eXq23",
        },
        "chain_ext": {
            "ex_pub": "xpub6FKQUVtZJMKhDeyeLuVTXT9UzGYy1hLn6rBSEdDbuCbpRsTytXimeU1kS2ai17SuC29cx43Q7RtmL9SGG22PqMdb36EGWKbUS3meW9sWdEm",
            "ex_priv": "xprvA2L44zMfTymQ1AuBEsxTAKCkSEiUcEcvjdFqSEozLs4qZ58qLzQX6fhGanTTqMrwQevXCq3NUSw4xncudnC4s6htZkGBzJWMyG7LFgy4Kvs",
        },
        "addresses": [
            "XoJA8qE3N2Y3jMLEtZ3vcN42qseZ8LvFf5",
            "XbctnEsgWTn5j1co3emZynemxSFPqkLRKZ",
            "XdD2biTJ3saZtcR6ravwJ9bvmkvmDq49Xg",
            "XkBrNhE8srfb8BbeTRSU4dxWsjjedra4Xn",
            "Xe8n8PZNgngjbMCFEA9unH26TmEWPPjm6a",
        ],
    },
    # Dogecoin
    {
        "coin": Bip44Coins.DOGECOIN,
        "names": ("Dogecoin", "DOGE"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "dgpv51eADS3spNJh98bWAfYnAW8K1gMy86HKmH1dpyT8kLsUKBqssT3jsLLFWyK4zbruL51UjejFDzrFzBcwjjA57rSv6D2978QigKG4xbCfJV6",
        "wif_master": "6JKHV5zEqwBbEhAf7qEWk5qNcu6gs4XtrCwXe3WFH8xR7BcSCbr",
        "account": {
            "ex_pub": "dgub8rUhDtD3YFGZTUphBfpBbzvFxSMKQXYLzg87Me2ta78r2SdVLmypBUkkxrrn9RTnchsyiJSkHZyLWxD13ibBiXtuFWktBoDaGaZjQUBLNLs",
            "ex_priv": "dgpv57bftCH9z6cEAdAY9SCDV9NfVsygaQWdi5LuCXdumz5qUPWnw1S3YBM7PdHXMvA8oSGS6Pbes1xEHMd5Zi2qHVK45y5FKKXzBXsZcTtYmX5",
        },
        "chain_ext": {
            "ex_pub": "dgub8toY7HgJc8TXA6fsb32FeTVrnqYMsBghLkw1Bu1XeDVJRun2YmnQTJUcNFPfcNs6Kf14DJo1gfKMFucNbdwQweXU1qknNHmTQfLKQLQTaS1",
            "ex_priv": "dgpv59vWmbkR3yoBsF1iYoQHXbxGLHAj34ez4A9o2ncYr6SHsrfL91Edp14xo29tx1a3cZA5M3o3fbr6g3ZViHrHkgSVE3WNJ23jgESJtkkL7hs",
        },
        "addresses": [
            "DBus3bamQjgJULBJtYXpEzDWQRwF5iwxgC",
            "DAcDAtJRztxBHyA6D6h8du1HguyTR43Mas",
            "D8K3KyDQ9FXeC3ADCuWW7cnWC7RvjHjV8H",
            "D6RRdXkUbb3pazkYGXAXwbJY5iC8Tyqwzh",
            "DTdrvUHbk5oMyi62tM7LqrjAcXfqB7eaad",
        ],
    },
    # Elrond
    {
        "coin": Bip44Coins.ELROND,
        "names": ("Elrond eGold", "eGLD"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K4GYBne6aGFZLHUSCPzQ3amKxfzz6kHpqDLBHk58dvPW7twcRFJeo1oQAvNeiYpZEeUkWCW3YgdvXHUcQQU6xjh8jBo4Y8xG",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6CMUiCFDobygrAj2LiiiBuMMmrzW2pAEPtn73C7QAsLSPMZCWBqjceGoRHXj4msYnuTbi2a2qPviifLL5fmpsS46BMpRbmADwBFmDUD5EZr",
            "ex_priv": "xprv9yN8JgiKyERPdgeZEhBhpmQdDqA1dMSP2frWEohncXoTWZE3xeXV4qxKa6s4owr4xL8Cb69LMCapEcnw7TZbuvg6YNbBn4nptjERCKdoZwq",
        },
        "chain_ext": {
            "ex_pub": "xpub6FBJfKWjsnDV6eDQZtzqRppmZd2cZxEXwzVNqVSba66Z7NvvQoZ85A92A3TgGmPw9a3apuTUc42bc38PfjiyfS9kCta68oVhRyM4mDtJMZD",
            "ex_priv": "xprvA2BxFoyr3QfBtA8wTsTq4gt31bC8AVWgamZn372z1kZaEabmsGEsXMpYJspWB4ZyMGjMHTMAzcrap1T71LKMX6aJeGLJ6tiigN7NuD9vK8V",
        },
        "addresses": [
            "erd1sqhjrtmsn5yjk6w85099p8v0ly0g8z9pxeqe5dvu5rlf2n7vq3vqytny9g",
            "erd1xkrttq324elvla4kk83r6wns35cjyqw7vg5tmdfn7qmrc2drd7qswlwt6z",
            "erd19assuvyq236dmr4th5wjy59c75cn98zjusd3m74kv5y9ylerl0esenuv3m",
            "erd1hhwwsj0837t6dkjy53waxtgp3aqws863rrvx6lj2v9t3jfxmhq5qgp0dj0",
            "erd1yvj6es98ecj0axw399a8juekwvc4a3f4rvxpy8vudfvy4v6p088qexz52j",
        ],
    },
    # EOS
    {
        "coin": Bip44Coins.EOS,
        "names": ("EOS", "EOS"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6DEKAPvkKccksoXcd1DjLJJpWW8bz4hD1CQZJxzFFKDUzSwhEzLqSsCqZf9HvDYHCADj5uDkh2RkDMUHAhn7UmNaENmNbf5bnFyomFFCG1Q",
            "ex_priv": "xprv9zExktPrVF4TfKT9WygiyAN5xUJ7abyMdyUxWaadgygW7ecYhT2au4tMiMmA9DiPxcx3NTjx1YFmbwUxb4Nd6bm15vUjSH7YaUvMtJNj1ss",
        },
        "chain_ext": {
            "ex_pub": "xpub6ErJ445aK8TLm6hMLGXZoUqEHM89YHdkApQsDUvyxaPs7PuxXGc7cG9mayvwU7EGd68KKuZTJHpg6RoUk6esWXh8V7CmUPizLVQmUtRzAs7",
            "ex_priv": "xprvA1rweYYgUku3YcctEEzZSLtVjKHf8putobVGR6XNQErtEbaoyjHs4TqHjjD45EYb5eJ1nVyy6jfN4m478SrZcdvpemVrMkeahRzLq5LTnyy",
        },
        "addresses": [
            "EOS6zpSNY1YoLxNt2VsvJjoDfBueU6xC1M1ERJw1UoekL1NHn8KNA",
            "EOS62cPUiWnLqbUjiBMxbEU4pm4Hp5X3RGk4KMTadvZNygjX72yHW",
            "EOS5HW5gdesVezdB74ruENxfLkJU1xYntXxPg9tCbdF7ZkeRU6TKR",
            "EOS7m3j93jyawa2MGDaLe9s5trUFWB7jabHKF111S8L24wJRLUi4t",
            "EOS6ugvESPgGtaxJKsUf3cBmGVTJzGc1fvjhmSFyHheGAUqj3Tn7Y",
        ],
    },
    # Ethereum
    {
        "coin": Bip44Coins.ETHEREUM,
        "names": ("Ethereum", "ETH"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6DCoCpSuQZB2jawqnGMEPS63ePKWkwWPH4TU45Q7LPXWuNd8TMtVxRrgjtEshuqpK3mdhaWHPFsBngh5GFZaM6si3yZdUsT8ddYM3PwnATt",
            "ex_priv": "xprv9zDSoJv1aBcjX6sNgEpE2J9K6MV2MUnXuqXsFgzVn3zY2aHyupaFQdYCtdCbNMkvcTdx9FeN49sgXw6mjrhrFLRSzJVnRYPfSCCgjeg4GxY",
        },
        "chain_ext": {
            "ex_pub": "xpub6EF8jXqFeFEW5bwMU7RpQtHkzE4KJxcqJtvkCjJumzW8CPpacXkb92ek4WzLQXjL93HycJwTPUAcuNxCqFPKKU5m5Z2Vq4nCyh5CyPeBFFr",
            "ex_priv": "xprvA1FnL2JMosgCs7rtN5tp3kM2SCDpuVtywg19QLuJDey9KbVS4zSLbELGDFdugxESvYxCU1wEJhDmMXvHKwmpCLd8QSoUJCHmVHCGvpFPxiw",
        },
        "addresses": [
            "0x9858EfFD232B4033E47d90003D41EC34EcaEda94",
            "0x6Fac4D18c912343BF86fa7049364Dd4E424Ab9C0",
            "0xb6716976A3ebe8D39aCEB04372f22Ff8e6802D7A",
            "0xF3f50213C1d2e255e4B2bAD430F8A38EEF8D718E",
            "0x51cA8ff9f1C0a99f88E86B8112eA3237F55374cA",
        ],
    },
    # Ethereum Classic
    {
        "coin": Bip44Coins.ETHEREUM_CLASSIC,
        "names": ("Ethereum Classic", "ETC"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6CqTNrUVV95XgHEP2LVe42cdpWKh2jeA4GnyXrQ219wJJVPAFajkbuFEdzf6xfGgdJWrhKuK5p8v47vxVQHFGU85J1J8Di95Z6hFJx8GhYz",
            "ex_priv": "xprv9yr6yLwbemXETo9uvJxdgtfuGUVCdGvJh3sNjTzQSpQKRh41i3RW46vkni1casJdVAqZ7xe2GQ5gMX8z7v3dPodSe1dzkTBZJdmuRoWt6XH",
        },
        "chain_ext": {
            "ex_pub": "xpub6EE7YMPj76jViZjqa4LfaBX6ZdaVs6QJpwF29otjUUVAyYs7J1Doq8m2mXGFtz3M1TLBaFvh2dcWsczUTLkvH7gZcZkuN9BjbF3yZMDnJnc",
            "ex_priv": "xprvA1Em8qrqGjBCW5fNU2ofD3aN1bk1TdgTTiKRMRV7v8xC6kXxkTuZHLSYvDUht8aTHFUY16Ea9ES8SRpmht6VSAe3X8ViDVYhjqbJymS8Lk3",
        },
        "addresses": [
            "0xFA22515E43658ce56A7682B801e9B5456f511420",
            "0xAee144cB8098B17BCD98D82802fDf32E8ca518cF",
            "0x6A9F90A1241E1960AD1A0cCb2400C9d235eA0846",
            "0x4Fb914Ca25C64595B28aa6dD78321E7b885a8eC7",
            "0x327b2ac3B8623eBdB3eEB39B150f960b6d3ec1a9",
        ],
    },
    # Fantom Opera
    {
        "coin": Bip44Coins.FANTOM_OPERA,
        "names": ("Fantom Opera", "FTM"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6DCoCpSuQZB2jawqnGMEPS63ePKWkwWPH4TU45Q7LPXWuNd8TMtVxRrgjtEshuqpK3mdhaWHPFsBngh5GFZaM6si3yZdUsT8ddYM3PwnATt",
            "ex_priv": "xprv9zDSoJv1aBcjX6sNgEpE2J9K6MV2MUnXuqXsFgzVn3zY2aHyupaFQdYCtdCbNMkvcTdx9FeN49sgXw6mjrhrFLRSzJVnRYPfSCCgjeg4GxY",
        },
        "chain_ext": {
            "ex_pub": "xpub6EF8jXqFeFEW5bwMU7RpQtHkzE4KJxcqJtvkCjJumzW8CPpacXkb92ek4WzLQXjL93HycJwTPUAcuNxCqFPKKU5m5Z2Vq4nCyh5CyPeBFFr",
            "ex_priv": "xprvA1FnL2JMosgCs7rtN5tp3kM2SCDpuVtywg19QLuJDey9KbVS4zSLbELGDFdugxESvYxCU1wEJhDmMXvHKwmpCLd8QSoUJCHmVHCGvpFPxiw",
        },
        "addresses": [
            "0x9858EfFD232B4033E47d90003D41EC34EcaEda94",
            "0x6Fac4D18c912343BF86fa7049364Dd4E424Ab9C0",
            "0xb6716976A3ebe8D39aCEB04372f22Ff8e6802D7A",
            "0xF3f50213C1d2e255e4B2bAD430F8A38EEF8D718E",
            "0x51cA8ff9f1C0a99f88E86B8112eA3237F55374cA",
        ],
    },
    # Filecoin
    {
        "coin": Bip44Coins.FILECOIN,
        "names": ("Filecoin", "FIL"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6D2RAta9bdnVzbRSCqZHMeiEiVd19V8g3sP92jHFLrfNDjGgJ5Rfej7j4Bc3sYNUbgfgY1kjTJkKhycEitDjUAjXyvUqx5TBFyQTzsQHhZE",
            "ex_priv": "xprv9z34mP3FmGECn7Ly6p2GzWmWATnWk2QpgeTYELsdnX8PLvwXkY7R6voFCwN1xTbY1cj4ztJJHjJXAP1DwYZDqSkh6tAkZNcZcMK7xoFxhnU",
        },
        "chain_ext": {
            "ex_pub": "xpub6ECTNqNWBXsWkCBwNhY181bmW7RdN6xuXJsyfvJBEq6AqBsukX1E1C8iQfBjpjQVHdG1qHFxSuJnKAPoQjytRKGhuoss6LK6fiThLmjXgFd",
            "ex_priv": "xprvA1D6yKqcMAKDXi7UGfzzksf2x5b8xeF4A5xNsXtZgVZBxPYmCygyTPpEZNJt3VgdB9Nc4yxTFjm26CoyPheQhsXXzyE73SjsJ1MLq84cXWk",
        },
        "addresses": [
            "f1qode47ievxlxzk6z2viuovedabmn3tq6t57uqhq",
            "f12nzdrhfh6caurft7gwy6d3uazvgy3lhl7rfzvpq",
            "f1ulkj6hjjssczzoyjrnw2dsu7xsw3hd2nsh2vf5i",
            "f1c6kiiaqgtkxtp2sgirvfmtasoeyq5kuiszksjna",
            "f1w2bkkhrdempg3ab45cnl22t4yfsdw2sspl7p5ra",
        ],
    },
    # Harmony One (Atom address)
    {
        "coin": Bip44Coins.HARMONY_ONE_ATOM,
        "names": ("Harmony One", "ONE"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6D1uaphVsP3GnzyuMuuo1p6q9ff6pvJEtEfXkjm9jTS65fwwxGZ7UfKny5qUn7RwunUbKtjQDat1vJqUyKgJkT6n342Wq95pKxeNaifKsGt",
            "ex_priv": "xprv9z2ZBKAc31UyaWuSFtNnegA6bdpcRTaPX1jvxMMYB7u7CscoQjErvs1K7ncAv2KCEDWSZQPscYNZ6YX9ZquxAb28VAznP7BV9dcUxg6XXPs",
        },
        "chain_ext": {
            "ex_pub": "xpub6ENtqxYx6UsgxtGBD4Z5JYSrwTViyCJNFT8FSZxwM5a3uk1ghfhW9kRsEniAhiiCUnCR4ZWwnyDayFzABYqZ8nB5FdKk4Q6HNjf2nwcnJM3",
            "ex_priv": "xprvA1PYST24G7KPkQBi7324wQW8PRfEZjaWtECeeBZKnk352wgYA8PFbx7PPXAwcoVAPiv1dTmdr2x7DZoHjWsHuSxZ4N3vi8ieqqjcbaYKEoa",
        },
        "addresses": [
            "one1q6gkzcap0uruuu8r6sldxuu47pd4ww9w9t7tg6",
            "one13c4p4dzqjtcu6wvkvhqkpesagng5tfsl625gqm",
            "one1lyt6xhyymyd22qwp3vrq0dq03wanktqsereqtt",
            "one1ynsd3aq3v4cz9twqttp3d4hwzadqdxm85s64v3",
            "one1gk9tzen4ltq2lw208m62pwkcsnk73uwklq7mgd",
        ],
    },
    # Harmony One (Ethereum address)
    {
        "coin": Bip44Coins.HARMONY_ONE_ETH,
        "names": ("Harmony One", "ONE"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6D1uaphVsP3GnzyuMuuo1p6q9ff6pvJEtEfXkjm9jTS65fwwxGZ7UfKny5qUn7RwunUbKtjQDat1vJqUyKgJkT6n342Wq95pKxeNaifKsGt",
            "ex_priv": "xprv9z2ZBKAc31UyaWuSFtNnegA6bdpcRTaPX1jvxMMYB7u7CscoQjErvs1K7ncAv2KCEDWSZQPscYNZ6YX9ZquxAb28VAznP7BV9dcUxg6XXPs",
        },
        "chain_ext": {
            "ex_pub": "xpub6ENtqxYx6UsgxtGBD4Z5JYSrwTViyCJNFT8FSZxwM5a3uk1ghfhW9kRsEniAhiiCUnCR4ZWwnyDayFzABYqZ8nB5FdKk4Q6HNjf2nwcnJM3",
            "ex_priv": "xprvA1PYST24G7KPkQBi7324wQW8PRfEZjaWtECeeBZKnk352wgYA8PFbx7PPXAwcoVAPiv1dTmdr2x7DZoHjWsHuSxZ4N3vi8ieqqjcbaYKEoa",
        },
        "addresses": [
            "0x06916163A17F07ce70e3d43Ed37395f05B5738aE",
            "0x8E2A1aB44092F1CD399665c160e61D44D145A61F",
            "0xF917A35c84D91AA501C18b0607B40f8bBB3B2C10",
            "0x24e0D8F411657022Adc05Ac316D6ee175A069b67",
            "0x458Ab16675FAC0aFb94f3EF4a0BAd884ede8F1D6",
        ],
    },
    # Harmony One (Metamask address)
    {
        "coin": Bip44Coins.HARMONY_ONE_METAMASK,
        "names": ("Harmony One", "ONE"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6DCoCpSuQZB2jawqnGMEPS63ePKWkwWPH4TU45Q7LPXWuNd8TMtVxRrgjtEshuqpK3mdhaWHPFsBngh5GFZaM6si3yZdUsT8ddYM3PwnATt",
            "ex_priv": "xprv9zDSoJv1aBcjX6sNgEpE2J9K6MV2MUnXuqXsFgzVn3zY2aHyupaFQdYCtdCbNMkvcTdx9FeN49sgXw6mjrhrFLRSzJVnRYPfSCCgjeg4GxY",
        },
        "chain_ext": {
            "ex_pub": "xpub6EF8jXqFeFEW5bwMU7RpQtHkzE4KJxcqJtvkCjJumzW8CPpacXkb92ek4WzLQXjL93HycJwTPUAcuNxCqFPKKU5m5Z2Vq4nCyh5CyPeBFFr",
            "ex_priv": "xprvA1FnL2JMosgCs7rtN5tp3kM2SCDpuVtywg19QLuJDey9KbVS4zSLbELGDFdugxESvYxCU1wEJhDmMXvHKwmpCLd8QSoUJCHmVHCGvpFPxiw",
        },
        "addresses": [
            "0x9858EfFD232B4033E47d90003D41EC34EcaEda94",
            "0x6Fac4D18c912343BF86fa7049364Dd4E424Ab9C0",
            "0xb6716976A3ebe8D39aCEB04372f22Ff8e6802D7A",
            "0xF3f50213C1d2e255e4B2bAD430F8A38EEF8D718E",
            "0x51cA8ff9f1C0a99f88E86B8112eA3237F55374cA",
        ],
    },
    # Huobi Chain
    {
        "coin": Bip44Coins.HUOBI_CHAIN,
        "names": ("Huobi Token", "HT"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6DCoCpSuQZB2jawqnGMEPS63ePKWkwWPH4TU45Q7LPXWuNd8TMtVxRrgjtEshuqpK3mdhaWHPFsBngh5GFZaM6si3yZdUsT8ddYM3PwnATt",
            "ex_priv": "xprv9zDSoJv1aBcjX6sNgEpE2J9K6MV2MUnXuqXsFgzVn3zY2aHyupaFQdYCtdCbNMkvcTdx9FeN49sgXw6mjrhrFLRSzJVnRYPfSCCgjeg4GxY",
        },
        "chain_ext": {
            "ex_pub": "xpub6EF8jXqFeFEW5bwMU7RpQtHkzE4KJxcqJtvkCjJumzW8CPpacXkb92ek4WzLQXjL93HycJwTPUAcuNxCqFPKKU5m5Z2Vq4nCyh5CyPeBFFr",
            "ex_priv": "xprvA1FnL2JMosgCs7rtN5tp3kM2SCDpuVtywg19QLuJDey9KbVS4zSLbELGDFdugxESvYxCU1wEJhDmMXvHKwmpCLd8QSoUJCHmVHCGvpFPxiw",
        },
        "addresses": [
            "0x9858EfFD232B4033E47d90003D41EC34EcaEda94",
            "0x6Fac4D18c912343BF86fa7049364Dd4E424Ab9C0",
            "0xb6716976A3ebe8D39aCEB04372f22Ff8e6802D7A",
            "0xF3f50213C1d2e255e4B2bAD430F8A38EEF8D718E",
            "0x51cA8ff9f1C0a99f88E86B8112eA3237F55374cA",
        ],
    },
    # IRIS Network
    {
        "coin": Bip44Coins.IRIS_NET,
        "names": ("IRIS Network", "IRIS"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6DGzViq8bmgMLYdVZ3xnLVEdKwzBnGdzzJZ4suG8kVb9TTLAbrwv8YdKBb8FWKdBNinaHKmBv7JpQvqBYx4rxch7WnHzNFzSVrMf8hQepTP",
            "ex_priv": "xprv9zHe6DJEmQ8484Z2T2RmyMHtmv9hNov9d5dU5WrXCA4Aaf124KdfakJqLGnVYwvBvMP4ttqm7QL5vuPTepiJiEJREiJnhswKmj6zLttwCMu",
        },
        "chain_ext": {
            "ex_pub": "xpub6EbeQ8h8S7mEGxYTn34gKedwvsmytMW61v51RAc8e6VzgMMLFkkNMN6sYtqvtZQ8TjBKswbxUTzxzj3nZfwr4pzFB7rySMaBUUCJnCuxu33",
            "ex_priv": "xprvA1cHzdAEbkCw4UTzg1XfxWhDNqwVUtnEeh9QcnCX5ky1oZ2BiDS7oZnPhdgFNqWwGtB3BNebior1tS26wAtbQzCTe5Pty1BhkcwT8QfWQSP",
        },
        "addresses": [
            "iaa19rl4cm2hmr8afy4kldpxz3fka4jguq0afzdvay",
            "iaa1jrkmdcwgq94uaamx6zax2luewlhf7u4kf6xq5n",
            "iaa1kng7tv83qesgvv2ze7hxlw4urfrjk8vq43p306",
            "iaa1zuvk68xw4y9swp06796rx8zarjvvkrt66cnhfw",
            "iaa12ecyuddg4wlnhztusgju5zxxjgznkz7f9t5053",
        ],
    },
    # Kava
    {
        "coin": Bip44Coins.KAVA,
        "names": ("Kava", "KAVA"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6BiXRoxTDK73KtKp3XCjv8WRBRqq57TJezk1JJLW92ZJ8B7JmYzrT2Vmh73dg3WfQ8JrRPgGVuLAAH6XGL5mJmCD1k8cScva3tMBZMbYenk",
            "ex_priv": "xprv9xjB2JRZNwYk7QFLwVfjYzZgdQ1LfejTHmpQVuvtah2KFNnAE1gbuEBHqqbD2AbnkrMefKehvkCLeQHoKmjFcqZXug1EcNfxqqRr79VQxvP",
        },
        "chain_ext": {
            "ex_pub": "xpub6E8ivQVjyUx49UrrCjzsG5zUK7DPAy6n7SwR96CFEQAvJjbwZeKFakcoHvnmhjNYMqPTr9BJCMXjkH5dKpYV2UFEdiWtffwaEjPUT46popz",
            "ex_priv": "xprvA19NWtxr97PkvznP6iTrtx3jm5NtmWNvkE1pLhndg4dwRwGo27112xJKSgH1mR3b5YEEX9B9hxQx9RhVxv7q4WPLGJ4aAfdF2yvrra4A2T1",
        },
        "addresses": [
            "kava1vh8tr8ddf7g0gfep23t46yllmlvtqfjkkgejlg",
            "kava1c2pramju9wtdz2jqulfalku58pq2ecgxa6xs8g",
            "kava1tzt3hf6vges3wqhl3tuhm70r89tqk80cnfem6j",
            "kava1zjxdfh8954dkehatc06jajh5v6jqlxtsaw4ysp",
            "kava1n2eculyv945phq5y7cj4l0a6tkg2px7943ce0z",
        ],
    },
    # Kusama (ed25519 SLIP-0010)
    {
        "coin": Bip44Coins.KUSAMA_ED25519_SLIP,
        "names": ("Kusama", "KSM"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K4GYBne6aGFZLHUSCPzQ3amKxfzz6kHpqDLBHk58dvPW7twcRFJeo1oQAvNeiYpZEeUkWCW3YgdvXHUcQQU6xjh8jBo4Y8xG",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6DDG9a79WtgKgGAhxAewwumqyz7MWrXDkvTS6Vuyop1h8PwqY91SFhsBGdcFFsFeCVvgSgq9DvYCTKfqveQet5np5h926BEfgyGkZpydKCe",
            "ex_priv": "xprv9zDuk4aFgX82Tn6Er97wamq7RxGs7PoNPhXqJ7WNFUUiFbcgzbhBhuYhRRTWbK7hsqXkGMgsJcFVet7rmiPKK38nwamzouaNPuj5hLjHrge",
        },
        "chain_ext": {
            "ex_pub": "xpub6E6cq4GXH8UdeeBPfdcXfLchJXr3eyGLkFPTiryKSkv2N8GPyPM48FPzxmhuPtYnjkFWH6yJAAv2Mf3v7DporUctKm7vEzjRMCF9rNdkcoR",
            "ex_priv": "xprvA17GRYjdSkvLSA6vZc5XJCfxkW1ZFWYVP2TrvUZhtRP3VKwFRr2oaT5X7ZW2x3Giih7M8aGpyhGh8ncNmqapwsStrjFeZyVVNqU14Ssop2t",
        },
        "addresses": [
            "FoTxsgYKH4AUngJAJNsqgmK85RzCc6cerkrsN18wiFfwBrn",
            "ETSjrrKL2gkbZYJZQZoer8efTgKnKW2nv7iUY2eLqaqe7JF",
            "CbdpcF5Q9MNGsSffiGivMWEb6754vqkpiLeQSxFVwxZ1G8S",
            "J13cKG6JCiZoWGyEyfhJoy8j56SSuDTEtqbkj5Bfxi7be3N",
            "DETciVX4dFXUDoUFM4oEShZ6c1biS3qSH3f7eRqAoAhoxE7",
        ],
    },
    # Litecoin
    {
        "coin": Bip44Coins.LITECOIN,
        "names": ("Litecoin", "LTC"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
        "ex_master_cls": Bip44LitecoinMainNet,
        "ex_master_alt": "Ltpv71G8qDifUiNetExQFUsQdYezsBDUaB814BhGZsphfZBrT3zqfeXTed3NqdFEoARcM1uJVNKbqmXwJscuMY753LZUtc36GiKqV5anTrnqWfr",
        "wif_master": "6uJgfG4pBbMffTdMSGQVurdK6xBcZjhf1iDU2jtPAw5PzRdhx9m",
        "account": {
            "ex_pub": "xpub6BnJJjq783EdyBeQPA9P9ao9DTS3fUqyKG5NJDcrCiwwxEkesGoHN94LZRGE7rz1jgcvmmp8j55BNx573KFq1WBwKiemzkdfNKffKx6Mvku",
            "ex_priv": "xprv9xnwuEJDHfgLkhZwH8cNnSrQfRbZG287x39mVqDEePQy5SRWKjV2pLjri8x93UfBSQdsGPeJFi8qXKexJUNj5eU3gEKfHUzo6EmuK6CSgEq",
            "ex_pub_alt": "Ltub2YDQmP391UYeDYvLye9P1SuNJFkcRGN7SYHM8JMxaDnegcPTXHJ2BnYmvHnFnGPGKu2WMuCga6iZV3SDxDMGrRyMcrYEfSPhrpS1EPkC43E",
            "ex_priv_alt": "Ltpv7735AbcbmL1gbgDWj2ezvs59rh4RM1oTN2BKTKbfe3146FCPCNFbBBSWfuV9vCJNMXD9LuHpQnqVWpn2hbMhikqPdoGqbS3ptdPoNWEvvgR",
        },
        "chain_ext": {
            "ex_pub": "xpub6F8vdBLe2hjJ6hn9Mh7gH4dGQrUvWy8TSoNWPF1Kz1sbcmzdDQDTRZKoWGdxDyF4K2bDnKYmzb3oLWiMGnghLxwv3qywBcDJy2PLuX86BML",
            "ex_priv": "xprvA29aDfokCLAztDhgFfafuvgXrpeS7WQc5aSuarbiRgLcjyfUfruCsm1KezybgSj2V3g4rxmfJ3bKyLxtmRk8rLtUgLu8Ta8j3fdLG5n9RSV",
            "ex_pub_alt": "Ltub2ba35pYfv93JM545xB7g8vjVVeoVGkeba5aVDKkSMWiJM9dRsQiCFCpEs99ytNeJuEzoNSwKqchBSc5UBgn9BtjLLysPrHyMTX9gp7D46dD",
            "ex_priv_alt": "Ltpv7APhV388fzWLjCMFhZdJ4LuH467JCW5wVZUTYLz9RKvhknSMYVfmEbhycmWcZANDQAFLwURBT8Hyxr5yAYj7VTFpdurJmXBkr4FEKVQPVMv",
        },
        "addresses_depr": {
            "cls": Bip44LitecoinMainNet,
            "addresses": [
                "1AHSLc3DxZnkrkkXxUz9uDhAc5QZqxhR2B",
                "1KMwfvFubh2KDzP8BE78QBnyZUkYsFq3gL",
                "1CqbYXEoZXVzD5nTgqDKQ9mnkXMLAzUxvA",
                "1NNeaasnHwtsYNEhgc4Ghp8hqKgzrCHvNA",
                "1Q9GgPdMcruaUjRjQuhDeqTyhJ36ZPjE32",
            ],
        },
        "addresses": [
            "LUWPbpM43E2p7ZSh8cyTBEkvpHmr3cB8Ez",
            "Ldatw8ZjgMGNUo5HMN6RgCrjmh7q494Si3",
            "LX4YojYdeBk3TtUcryCcgAqYxjicKfK7AD",
            "LgbbqoBcNc8voAvrrk3ZyqCU3Y4H24aauc",
            "LiNDwbwBhX9djY7tb3gWvrXjuWQNerLjnP",
        ],
    },
    # Monero (ed25519 SLIP-0010)
    {
        "coin": Bip44Coins.MONERO_ED25519_SLIP,
        "names": ("Monero", "XMR"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K4GYBne6aGFZLHUSCPzQ3amKxfzz6kHpqDLBHk58dvPW7twcRFJeo1oQAvNeiYpZEeUkWCW3YgdvXHUcQQU6xjh8jBo4Y8xG",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6CCZuih1gfVKndnKAYqdsJPUYkDBmpP4M28UmEJP9j121M6c82dcvPddRhESuNfwrVF545JHzysHeNeSmqfC9A33w8TsnghkWwEADUq1n9p",
            "ex_priv": "xprv9yDDWDA7rHw2a9hr4XJdWASjziNhNMfCyoCsxqtmbPU38YmTaVKNNbK9aWnR9DznrjqPwy9bQVNJdG6Vw772wHqP8Fs4imvyuXEu3icrEvV",
        },
        "chain_ext": {
            "ex_pub": "xpub6F8c8xNKr6W8NBjeC1eu4E1gYQEJpQY4ZX3HLdqXmuqPJ7jeqgKtRawN7sAdi8fEL1yBXYQ2Lvpoa4gacZ2x6X3o1ZEavuh3Mr8UdtJ1fW6",
            "ex_priv": "xprvA29FjSqS1iwq9hfB5z7th64wzNPpQwpDCJ7gYFRvDaJQRKQWJ91dsnctGhEhoYFmgxwtoY4nUWHBbjiZLc3ptcqdAr9vyYBKcAvq2PSuxsA",
        },
        "addresses": [
            "48wpY5S7Ksq18DtvSaomPqEkFHueWCXcK7Cj2iiKRDXP65BWToZKj1ZgVzhG5Ngusy53b8q5xYwYaXepz4WXPTQ5ChJAe9n",
            "43kepnmL2XnCe9NaMKNx9WSSbwVaDXsQGLpuTzppH5cXcSvLvckQ12tffpyht12F79aJhWU4mAmEmDQCyh6Pccid4A5d1d2",
            "4551j15NFTQJkVKBHZdFQaMz1nDZHtTPZ683SHmXa4NUjn1bty9zfXfUTqsmByvZ7K6rSXPwXkTaEAK54tb3rvYC2KLVaL8",
            "46vXBjvxdbM78LRncwPpsLbFR3Qnyf8kUEsu2MSf899eaU9pTKkEiknhCsXCp3u3zfECmMBqYL87uEvnoA4M6hX9QywR61A",
            "48X4rKu5aeKFvLqrv3KmRV3XK7k9CdekAWvduncUoUXnGXEmjtESoG6aVtDHTkUU4pBN6WKbVjhrk5SVjCChH1iqSoWLs9W",
        ],
    },
    # Monero (secp256k1)
    {
        "coin": Bip44Coins.MONERO_SECP256K1,
        "names": ("Monero", "XMR"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6CfKT2U2oQBrL46KxcvSqN68vocGJYBaCr9eErhXNjaBnWG8GDSjtsZCkkxfqTrJyRvefgAN1mUwZ6jTbdEZJZJVc19zvgk2N3FfoJ7xt79",
            "ex_priv": "xprv9yfy3Ww8y2dZ7a1rrbPSUE9QNmmmu5TiqdE3SUHupQ3Cuhvyig8VM5EiuSsqmH2chuL5ftifzieRdRPpgGm69UzGJnJKW1RTcJC1S2AN8KX",
        },
        "chain_ext": {
            "ex_pub": "xpub6ESWbwNkHZkTnFEHEmoqK9NwNw3sJRgdP8dgDAnVdwrr198h2x3cc4qE6JX9i6SX4T1k5n7YxpGNmXBCdKHF2Af5arpBwdqzeZk3CtAudSe",
            "ex_priv": "xprvA1TACRqrTCCAZm9p8kGpx1SCpuDNtxxn1ui5QnNt5cKs8LoYVQjN4GWkF2LimJRvk9U4zDh62CoJiw73MAZ7rfTycXxLnH7m5kZrsD88tyS",
        },
        "addresses": [
            "49vDbkSo7eve3J41sBdjvjaBUyz8qHohsQcGtRf63qEUTMBvmA45fpp5pSacMdSg7A3b71RejLzB8EkGbfjp5PELVF2N4Zn",
            "43zMkXAq58qSDPMgrQUmoJ95nZrBC8fxXfLFxzx3VNwUeVBDFAKFthSNYb6SdHunWJGGpDupzi8u4d26oVvLXyGw8wwC8A4",
            "43saSi9Dks9YV8YyacmZQQJdMBKY9nbEsRvaK3roJB4LPgJUNviVCoH1WBYLCuoxBsFJjsrPf5h4GHCdhVVVbWCJQdscKeN",
            "4ATV7ushExTh5jmdJcAUEo8WS7BcpZp93WG4mGebJgfscE4WCizMCFWSad27gXmAyJGTXdwnyPQ2URHuo9UrTuV4SxFoLtA",
            "45owUkS3J6B5mYs6knc7ZBBG4fCiiX672BgrK2AA75zEAMryFmKnz7s63eHA96fo8aVAUigtmRj7Z4V9bK4fkAab23QGme3",
        ],
    },
    # Nano
    {
        "coin": Bip44Coins.NANO,
        "names": ("Nano", "NANO"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K4GYBne6aGFZLHUSCPzQ3amKxfzz6kHpqDLBHk58dvPW7twcRFJeo1oQAvNeiYpZEeUkWCW3YgdvXHUcQQU6xjh8jBo4Y8xG",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6CVEifkbQX1M5DRh2gyTJPhE8t4hzVXMZNMP6XVekwSuguCr3ffdGYPPPpaRq6r97iiRS83dfiuikDAiUtVCFVWTThAv1YgbgP6pVCnxwc7",
            "ex_priv": "xprv9yVtKADha9T3rjMDvfSSwFkVarEDb2oWC9RnJ963Cbuvp6shW8MNik4uYdjH4WSG66sie1Hg2HFE5cqMWeomHvDJqhwyrTs3YeGapg4Bwzc",
        },
        "chain_ext": {
            "ex_pub": "xpub6E7WSJXXab9xqim1bu6HRr1bLKUmCUdGahzn8xfvgWXQnBwqDkGBPjv8coMNAo6nVqMahnjeTHMysXq7z2wS858yByLV7E4ocmEM5oxSbcs",
            "ex_priv": "xprvA18A2nzdkDbfdEgYVsZH4i4rnHeGo1uRDV5BLaGK8AzRuPcggCwvqwbemc5LFQVik6mEULt6i1mnDuVbXYnXC89NisRMDdFuZgoHdXd9bzt",
        },
        "addresses": [
            "nano_1dnnkaocjmyjhd8msu31rj6s1mjtb75yukydsh6rqeub6d9wocqi85koohbe",
            "nano_359bi3ub8kxaj3y18699nmap1juacuseuet9cw9ra3qdqxcbjkwyhsb1zpyw",
            "nano_1ipeqhnp66f8oohm1f7ykkzora9hgxrokqrp5pq4htjmsj1ents3osa38seb",
            "nano_3qdzy47tph81tnzijpdb1es19koahszyoben65hqsronxi5813oh1ed6rp13",
            "nano_1bk6bt8qp4tepperhtmw3bc8uef9rumzqz6113rrkt1er3qkr6kgbrxzr393",
        ],
    },
    # NEO
    {
        "coin": Bip44Coins.NEO,
        "names": ("NEO", "NEO"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K43eeN7bb2E9YrDJsa6fYcbWK67iu4N397GNE1hoJP4SaeLz1g4QzPEokQD7XmzWszVPDB7zDmyrtwrHFPYxHasf2GmhCJE5",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6C14EL78ogJs6aQEpzMdPa6oUdo7dUFdScW7ckC6MXxUAN7zDiEP6pGQphtcjS2cv4Nusp4i9CVJPmtuGmp1RKN3pCvUPWkDFcoMHCHERTA",
            "ex_priv": "xprv9y1hppaEyJkZt6Kmixpd2SA4vbxdE1Xn5PaWpMnUoCRVHZnqgAv8Z1wvyRpyPivKfJ2CtiT5mts6j4agm3SmdZA8ASftK1nPMuhzn4nWTMq",
        },
        "chain_ext": {
            "ex_pub": "xpub6EUcJ14d2zGWuzv59LbpBkM3Qe23rk6fAgnQpDgLgfT9zYXxCwcFHpWgRaDQPeCaqE7AMf9uAyrDRmj4ymcd19EbVxQrPhDz3SfjNnspxvf",
            "ex_priv": "xprvA1VFtVXjCciDhWqc3K4opcQJrcBZTHNooTrp1qGj8KvB7kCofQHzk2CCaKVEfZdby56TRFWWdYjkgbhDyut9HdF66Ah2sucDTmKW6UYLFQN",
        },
        "addresses": [
            "AJHeWQn2qKKqD4nBE82etebgT3GEM9HDRH",
            "AYrG8CHdxTMiWiKDT1cZWvLh6WENaQZsRg",
            "AW2ZSp5jCFoTAawyQHiDX5jzBHXzw8ydjZ",
            "Aae1tmeie6ExyAG8RD3mTLYKfk6yx6SSmq",
            "ARgD7zaeX578LqTyHagyx4W8CgbWfcUeCk",
        ],
    },
    # OKEx Chain (Atom address)
    {
        "coin": Bip44Coins.OKEX_CHAIN_ATOM,
        "names": ("OKExChain", "OKT"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6DCoCpSuQZB2jawqnGMEPS63ePKWkwWPH4TU45Q7LPXWuNd8TMtVxRrgjtEshuqpK3mdhaWHPFsBngh5GFZaM6si3yZdUsT8ddYM3PwnATt",
            "ex_priv": "xprv9zDSoJv1aBcjX6sNgEpE2J9K6MV2MUnXuqXsFgzVn3zY2aHyupaFQdYCtdCbNMkvcTdx9FeN49sgXw6mjrhrFLRSzJVnRYPfSCCgjeg4GxY",
        },
        "chain_ext": {
            "ex_pub": "xpub6EF8jXqFeFEW5bwMU7RpQtHkzE4KJxcqJtvkCjJumzW8CPpacXkb92ek4WzLQXjL93HycJwTPUAcuNxCqFPKKU5m5Z2Vq4nCyh5CyPeBFFr",
            "ex_priv": "xprvA1FnL2JMosgCs7rtN5tp3kM2SCDpuVtywg19QLuJDey9KbVS4zSLbELGDFdugxESvYxCU1wEJhDmMXvHKwmpCLd8QSoUJCHmVHCGvpFPxiw",
        },
        "addresses": [
            "ex1npvwllfr9dqr8erajqqr6s0vxnk2ak55peywkd",
            "ex1d7ky6xxfzg6rh7r05uzfxexafepy4wwqr4rxvg",
            "ex1keckja4ra05d8xkwkpph9u30lrngqtt6wvs5kj",
            "ex1706syy7p6t39te9jht2rp79r3mhc6uvw9kqult",
            "ex1289gl703cz5elz8gdwq3963jxl64xax27sqa7g",
        ],
    },
    # OKEx Chain (Atom old address)
    {
        "coin": Bip44Coins.OKEX_CHAIN_ATOM_OLD,
        "names": ("OKExChain", "OKT"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6CfUe4mAn2nxb94zjkWk9KgY4rZWQqkyZEZApyGMknJJgsW65C9zDq9Tc5xpp37qfxD5ftbY3dXjuaKSYvvFFAh9Etj1ypiXfhqngwBHN8e",
            "ex_priv": "xprv9yg8EZEGwfEfNezXdiyjnBjoWpj21P38C1da2arkCSmKp5AwXeqjg2pyko2SJvgKnE1yWRCV54Es6R3upWtdF8XJ9yvGLDdwvnRXP4JSfsk",
        },
        "chain_ext": {
            "ex_pub": "xpub6EwpPxZGbUGJGMH2V2xWyjx7mDFVSUrydV2yDthHa1EMxLqf6rbKduFNnFSAeVKZS2TF1bJgdZqw79uSQSni8qpkQC7ucmsm79VepRMo42q",
            "ex_priv": "xprvA1xTzT2Nm6i13sCZP1RWcc1PDBR13298GG7NRWHg1fhP5YWWZKH566vtvxagAGngxHxTPDkomeEGKLFPYCWajY78Hk27kMkrvR12KrhJsqj",
        },
        "addresses": [
            "ex19dhmgh5vxg9cq5f85l88jm6h4r4uklkgpamn38",
            "ex19qm8st4v3wut8ddenj9gh98xkhylngq36pnshq",
            "ex14uqe6p2fqgxvf029gjhn4l6re97sl5auqw9f2e",
            "ex13tgz70jdd92fmgkt9ctsd04q347t3r20sc6usw",
            "ex1fu97k478nepxkkeqy0786g4vseh8ksffg6dq6c",
        ],
    },
    # OKEx Chain (Ethereum address)
    {
        "coin": Bip44Coins.OKEX_CHAIN_ETH,
        "names": ("OKExChain", "OKT"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6DCoCpSuQZB2jawqnGMEPS63ePKWkwWPH4TU45Q7LPXWuNd8TMtVxRrgjtEshuqpK3mdhaWHPFsBngh5GFZaM6si3yZdUsT8ddYM3PwnATt",
            "ex_priv": "xprv9zDSoJv1aBcjX6sNgEpE2J9K6MV2MUnXuqXsFgzVn3zY2aHyupaFQdYCtdCbNMkvcTdx9FeN49sgXw6mjrhrFLRSzJVnRYPfSCCgjeg4GxY",
        },
        "chain_ext": {
            "ex_pub": "xpub6EF8jXqFeFEW5bwMU7RpQtHkzE4KJxcqJtvkCjJumzW8CPpacXkb92ek4WzLQXjL93HycJwTPUAcuNxCqFPKKU5m5Z2Vq4nCyh5CyPeBFFr",
            "ex_priv": "xprvA1FnL2JMosgCs7rtN5tp3kM2SCDpuVtywg19QLuJDey9KbVS4zSLbELGDFdugxESvYxCU1wEJhDmMXvHKwmpCLd8QSoUJCHmVHCGvpFPxiw",
        },
        "addresses": [
            "0x9858EfFD232B4033E47d90003D41EC34EcaEda94",
            "0x6Fac4D18c912343BF86fa7049364Dd4E424Ab9C0",
            "0xb6716976A3ebe8D39aCEB04372f22Ff8e6802D7A",
            "0xF3f50213C1d2e255e4B2bAD430F8A38EEF8D718E",
            "0x51cA8ff9f1C0a99f88E86B8112eA3237F55374cA",
        ],
    },
    # Ontology
    {
        "coin": Bip44Coins.ONTOLOGY,
        "names": ("Ontology", "ONT"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K43eeN7bb2E9YrDJsa6fYcbWK67iu4N397GNE1hoJP4SaeLz1g4QzPEokQD7XmzWszVPDB7zDmyrtwrHFPYxHasf2GmhCJE5",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6C2rQpEFfwXdrCNcRJwxzp7zQp7FtW8aKUwxU4ZaxiByHUzqNf5HbjASdaEM3YkG5MdxsA2deZDitWFwPd3HACTSKpdua9zwqb4LYb2szdn",
            "ex_priv": "xprv9y3W1JhMqZyLdiJ9KHQxdgBFrnGmV3QixG2Mfg9yQNezQgfgq7m33vqxnKCyJhmfB645HjhBhyHZGhoWC2LsySFBqPELTtYXYNH4DpjMvEW",
        },
        "chain_ext": {
            "ex_pub": "xpub6FM7NzhV3rE5P9RirKW8gkjQBBMyp1SfAGm9L7nMA8LD96CTXgnz5eB3iwgUmrkbTGivWqnXJfc5UAA2D9M1Vk1P2xj6yTQE8ENfLBWYHCa",
            "ex_priv": "xprvA2MkyVAbDUfnAfMFkHy8Kcnfd9XVQYioo3qYXjNjbnoEGHsJz9UjXqrZsfugaizUqBzPxEsrTHUibiXQzQ7mTzw5ubpNeyToZBfvJQZ2GVc",
        },
        "addresses": [
            "AZMgu1jizeCvwKx89qtKq3Rd92iPXHJQ9W",
            "AHWLr5UTGb1V3EUYNBTMAUAyzADQmyfDD9",
            "AHr4ACYEGD5uPmNFkqnJrvh1uq8vG23XE4",
            "ALxAERzk6hqA9oCjemC6Mi4idEWrakMQCx",
            "AWn55fzAcAKfDxkgzqLov3a5KieZXyzDJW",
        ],
    },
    # Polkadot (ed25519 SLIP-0010)
    {
        "coin": Bip44Coins.POLKADOT_ED25519_SLIP,
        "names": ("Polkadot", "DOT"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K4GYBne6aGFZLHUSCPzQ3amKxfzz6kHpqDLBHk58dvPW7twcRFJeo1oQAvNeiYpZEeUkWCW3YgdvXHUcQQU6xjh8jBo4Y8xG",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6DDG9a79WtgKgGAhxAewwumqyz7MWrXDkvTS6Vuyop1h8PwqY91SFhsBGdcFFsFeCVvgSgq9DvYCTKfqveQet5np5h926BEfgyGkZpydKCe",
            "ex_priv": "xprv9zDuk4aFgX82Tn6Er97wamq7RxGs7PoNPhXqJ7WNFUUiFbcgzbhBhuYhRRTWbK7hsqXkGMgsJcFVet7rmiPKK38nwamzouaNPuj5hLjHrge",
        },
        "chain_ext": {
            "ex_pub": "xpub6E6cq4GXH8UdeeBPfdcXfLchJXr3eyGLkFPTiryKSkv2N8GPyPM48FPzxmhuPtYnjkFWH6yJAAv2Mf3v7DporUctKm7vEzjRMCF9rNdkcoR",
            "ex_priv": "xprvA17GRYjdSkvLSA6vZc5XJCfxkW1ZFWYVP2TrvUZhtRP3VKwFRr2oaT5X7ZW2x3Giih7M8aGpyhGh8ncNmqapwsStrjFeZyVVNqU14Ssop2t",
        },
        "addresses": [
            "14E9StbjYhJiAfsNMEcq5tETq79Q6EqaGyebdziY214hNWDH",
            "12t8DsmWZSwJHSjNkLoku3boNVPjfxEzR31TFAk3R8Ps5JD8",
            "12KJdAGdZbuxkdjreWgAYyPJ7pUxZaiSqEPB5feaEmaSr1v",
            "16Rj6LBHXcy7VPU3RuueZ1SHS6orLXxQs1jLXMnakFX9326p",
            "1f96jQiJ3W5A6zYSHJkUeAhodj1c4no4PwPtH9EF5yjFJsf",
        ],
    },
    # Polygon
    {
        "coin": Bip44Coins.POLYGON,
        "names": ("Polygon", "MATIC"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6DCoCpSuQZB2jawqnGMEPS63ePKWkwWPH4TU45Q7LPXWuNd8TMtVxRrgjtEshuqpK3mdhaWHPFsBngh5GFZaM6si3yZdUsT8ddYM3PwnATt",
            "ex_priv": "xprv9zDSoJv1aBcjX6sNgEpE2J9K6MV2MUnXuqXsFgzVn3zY2aHyupaFQdYCtdCbNMkvcTdx9FeN49sgXw6mjrhrFLRSzJVnRYPfSCCgjeg4GxY",
        },
        "chain_ext": {
            "ex_pub": "xpub6EF8jXqFeFEW5bwMU7RpQtHkzE4KJxcqJtvkCjJumzW8CPpacXkb92ek4WzLQXjL93HycJwTPUAcuNxCqFPKKU5m5Z2Vq4nCyh5CyPeBFFr",
            "ex_priv": "xprvA1FnL2JMosgCs7rtN5tp3kM2SCDpuVtywg19QLuJDey9KbVS4zSLbELGDFdugxESvYxCU1wEJhDmMXvHKwmpCLd8QSoUJCHmVHCGvpFPxiw",
        },
        "addresses": [
            "0x9858EfFD232B4033E47d90003D41EC34EcaEda94",
            "0x6Fac4D18c912343BF86fa7049364Dd4E424Ab9C0",
            "0xb6716976A3ebe8D39aCEB04372f22Ff8e6802D7A",
            "0xF3f50213C1d2e255e4B2bAD430F8A38EEF8D718E",
            "0x51cA8ff9f1C0a99f88E86B8112eA3237F55374cA",
        ],
    },
    # Ripple
    {
        "coin": Bip44Coins.RIPPLE,
        "names": ("Ripple", "XRP"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6CFKyZTfzj3cyeRLUDKwQQ5s1tqTTdVgywKMVkrB2i1taGFbhazkxDzWVsfBHZpv7rg6qpDBGYR5oA8iazEfa44CdQkkknPFHJ7YCzncCS9",
            "ex_priv": "xprv9yFya3vnAMVKmALsNBnw3G98Trzy4AmqciPkhNSZUNUuhTvTA3gWQRg2ecJFS3PDLsfYFgwDW1UukaapjTDUENfiwg22ryd4mWiph8Faw3p",
        },
        "chain_ext": {
            "ex_pub": "xpub6F119CdfnfeSVeXbJG7h88TG6SjjtvFYBup7HszDLmaCCDgbZ4n1tP6J6R8PikUqtiqgdJSXXB79V3SxVA4LkygGDkesJ4pW5bq3dQU7cbm",
            "ex_priv": "xprvA21ejh6mxJ69HAT8CEagkzWXYQuFVTXgpgtWVVabnS3DKRMT1XTmLampF9DgbLwYuqZAZCHMokjt2rCCMnScQLZWmHdnS11XfySXtiA5ygo",
        },
        "addresses": [
            "rHsMGQEkVNJmpGWs8XUBoTBiAAbwxZN5v3",
            "r3AgF9mMBFtaLhKcg96weMhbbEFLZ3mx17",
            "r4Sh61HP7nxB6mQxXSSeN2DCkG3sTrzb2c",
            "rwT7dzQuZim2SdY1jGFGwpre4bh6xpr31a",
            "rPdQVkTzpZ7ToRqTRBRrUKPoCTty7n3UVj",
        ],
    },
    # Solana
    {
        "coin": Bip44Coins.SOLANA,
        "names": ("Solana", "SOL"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K4GYBne6aGFZLHUSCPzQ3amKxfzz6kHpqDLBHk58dvPW7twcRFJeo1oQAvNeiYpZEeUkWCW3YgdvXHUcQQU6xjh8jBo4Y8xG",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6BxGMZDzacMx38jcsWVxXtqgrU85CgRKN2NWXvaJhnCBPEUq5YcTLZreVRkm3uuhC3LiQk7WWp2ursTkxkMRfKnSvweLRtzdfbyUkCxTNgx",
            "ex_priv": "xprv9xxux3h6kEoepef9mUxxAktxJSHaoDhTzoSujYAh9SfCWS9gY1JCnmYAeEdZ93ooDXdzAnq5D4nPPMHArDrm7CYVzw2X2WfZ8gwMg34X1dY",
        },
        "chain_ext": {
            "ex_pub": "xpub6FQ5DsXbWT8Dp2HAaU6aD5fuY318tkgsGHW3oHDkm7msw4W2PYjQFMrEFpgAtFzNKFt9AmR7N4Yam9pgbK4hzF5XvsoGakWjnqg8X18Rtxr",
            "ex_priv": "xprvA2QipMzhg5ZvbYChUSZZqwjAz1AeVHy1u4aSztp9CnEu4GAsr1R9hZXkQc8i6m2LZzuUmd8YzMgunJfRrmkfJhGYuxuGU9huHYLWiqz4b78",
        },
        "addresses": [
            "B9sVeu4rJU12oUrUtzjc6BSNuEXdfvurZkdcaTVkP2LY",
            "634j9U9kjxbM8TmPzNCRQhjeENowxtAYC86Pwy2eGcje",
            "CjrzauKUn7WuiudgSkh4wrLuMkG6H1BhQnVdKMozganK",
            "DHMbxQoRFocMyhJXcUn88Rp2FwoXmxHXuaA32dYb3ATZ",
            "5VCDybrepFCgPprUHUo9MByZpS7zsTkrBwQryKC8zakF",
        ],
    },
    # Stellar
    {
        "coin": Bip44Coins.STELLAR,
        "names": ("Stellar", "XLM"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K4GYBne6aGFZLHUSCPzQ3amKxfzz6kHpqDLBHk58dvPW7twcRFJeo1oQAvNeiYpZEeUkWCW3YgdvXHUcQQU6xjh8jBo4Y8xG",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6C8gKgafAtV8z4p53yvq2LEyzhgNX82JNiiPu4bxuGaRAb7H835cEcciVWKaVHuhksQGaamT4s9nz8XCWirjzYpjXDWTw8isKpnD2xx9zjg",
            "ex_priv": "xprv9y9KvB3mLWvqmajbwxPpfCJFSfqt7fJT1Vno6gCMLw3SHnn8aVmMgpJEeK5V8QFuHWuPnSYZKerHY3bHi9RcNdRsrzCpFoUbbvUjLTikZc8",
        },
        "chain_ext": {
            "ex_pub": "xpub6E5zfoYGfkexdFr7qpk2oV5r1DpFrwu9RKs61gVVY3qKTfWKNedf968ZfMgko5oARGqLVDWYPyPPv7dWucazWQk5noZE2oQQojoTxmnScuU",
            "ex_priv": "xprvA16eGJ1NqP6fQmmejoD2SM97TBymTVBJ46wVDJ5syiJLasBAq7KQbHp5pAygXTDa9Dhv8ZpDMAdG6PPdr4qPy2JBx2mzqs336inRkXyk225",
        },
        "addresses": [
            "GBAECME5H27NLRFYZLFF6SAF5NSFRPOWP72PNKGSRUWRMTGSAOCD3AYN",
            "GCRKKQCMMYF433G7N3BVZIJ5EOF5XVUETH2JQHLAXY2BPXNTKOQLM5G4",
            "GB5MODTP6KD7W3GPWCNYFZV5TGML5VPJEQPPOFQVQARTTSZHDAQ64ZNO",
            "GAPNUERN7U4RSH252QTI2BHFUFEVPN7GLFSJZ6UZ3SQW4FFO33QTWDKS",
            "GDRRNY7FDCTOXXZPAOOBUU6TFAPFKAA57XOVAURWRQBYZQZDWQMNU457",
        ],
    },
    # Terra
    {
        "coin": Bip44Coins.TERRA,
        "names": ("Terra", "LUNA"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6CpXk1ari53YGLf8YFmh7bcUgmM1H6G7nKxodjVHL4nzSuxPhKq57biM4cgYkAbop78wxSvwPY9YYQa7Jdv2ZPSngzHVGq8VXq1cLVXiCVt",
            "ex_priv": "xprv9yqBLW3xshVF3rafSEEgkTfk8jWWsdYGR73CqM5fmjG1a7dF9nWpZoPsDMjRoRw8HQzKGPNhEsue92MT1fLeR3MbT1TBYqs7SakFHJHsQmo",
        },
        "chain_ext": {
            "ex_pub": "xpub6EWWxQVjGS5KGKojMyTNUWH6BkPWZ3FzgpAxFSkocwzBfcALcj1bTgGh9ojBGnj7rqQKgkMf7soVE9F3tHWfHekWkyT6y8Z6v3SPUQA9GzN",
            "ex_priv": "xprvA1XAYtxqS4X23qjGFwvN7NLMdiZ29aY9KbFMT4MC4cTCnoqC5BhLusxDJYdHw2whwczZMz7bEMvGkjQ5HRtqUbU24eiGCp9GS3esXe6JAzr",
        },
        "addresses": [
            "terra1amdttz2937a3dytmxmkany53pp6ma6dy4vsllv",
            "terra1lpywxkky22stctm9yzn5quwkr9fnxu3atqk2ax",
            "terra1was3u69zgthug7duuen2mulfrvvjrq3rasw7dr",
            "terra1sra8dye8kd4zsx9qcwea4eyg6aecypdkwws072",
            "terra16exv3gcch2vucqpaxgxfne93thweqhamqpdyt8",
        ],
    },
    # Tezos
    {
        "coin": Bip44Coins.TEZOS,
        "names": ("Tezos", "XTZ"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K4GYBne6aGFZLHUSCPzQ3amKxfzz6kHpqDLBHk58dvPW7twcRFJeo1oQAvNeiYpZEeUkWCW3YgdvXHUcQQU6xjh8jBo4Y8xG",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6D6eabhdYuAPkmJeFh3ihdPBrEGEGXhPGWsf8JeY25iVn6a8K6VaibiSJjuFBAL4mSUJPtVtnYCB6ZcxEjorhTNAaBApg4UF8CuS5Z813Ae",
            "ex_priv": "xprv9z7JB6AjiXc6YHEB9fWiLVSTJCRjs4yXuHx4KvEvTkBWuJEymZBLAoPxTYcVKBdUXr2b6HsVDFurRN9n77xPaDrpaZJyqe7NXF866zivAmk",
        },
        "chain_ext": {
            "ex_pub": "xpub6EqHb2yAUGePWabtZ6gQYoxXqXbxYfA2LkDsLxSW1fvni16fRivuK8YH8gJnyyNjR194fht1N4b4xDdpU1aUXxBLeW1djeWyvuKupaysqhj",
            "ex_priv": "xprvA1qwBXSGdu66J6XRT59QBg1oHVmU9CSAyXJGYa2tTLPoqCmWtBcemLDoHWFYcCp1vQ3kQTpXPZvJUpGjE2AKM6d1XeGQscJikKKRcBUtYnm",
        },
        "addresses": [
            "tz1QSN6cBu6kbssDE5TSsTpftvKzptd1sgRY",
            "tz1Rofn8Nh7Z99ef2M9MgVhq2mLuKzRATtbz",
            "tz1SL7ssKJb66kobc1ZT8c5PYUX42y3UDhuR",
            "tz1VeusEZR3qouTAsfmudLYSNx5o1gKc7ZG7",
            "tz1U2QVtbgZfbFanzogVd8wmEzMDvyR1SkV7",
        ],
    },
    # Theta Network
    {
        "coin": Bip44Coins.THETA,
        "names": ("Theta Network", "THETA"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6CiroqWJYjzbiVf8QNrgrHy1XpeZxqksSt7YFRDUrH5ybHha2iRXPNjJpYu5Bi8imeW4GEuQCEmwExNH5L8CnvgQgprKPJ9NvB41mqPqcUB",
            "ex_priv": "xprv9yjWQKyQiNSJW1afJMKgVA2Gynp5ZP325fBwT2osHwYziVNRVB7GqaQpyK4PTpcbxETZHMVLGQUoMvFJ4CoAbgreqDVyVpkkTuBSKmVVAiQ",
        },
        "chain_ext": {
            "ex_pub": "xpub6EnX6Yw6wxEme5wtArCPg6jXuSfGAo8w985nr9HmQAZkErJEoe7AtytWYYcmXf5244k4bcaQwwgBJ7DSJa4RVb77NwpcuGu3LmTK4HeycXz",
            "ex_priv": "xprvA1oAh3QD7agURbsR4pfPJxnoMQpmmLR5muAC3kt9qq2mN3y6G6nvMBa2hHuJjqyFbeDM5csgRaqGpSkFGgs82uQ9sezFZsqm55P1BKszpsX",
        },
        "addresses": [
            "0xB16f8D941d7D281aDCEf3e950aBF2147D7AC2D33",
            "0xf2142cb79CEEC9aD39C15151CdEa90f0b853EA1C",
            "0x17C6899cbD8628a614E69CeF96E6CB2Ea47A5Ace",
            "0x92554055C248b7e155326ad827C8D5A8780322fF",
            "0x02f9C86C8dA95C6748d063AA00c6DCD581311C06",
        ],
    },
    # Tron
    {
        "coin": Bip44Coins.TRON,
        "names": ("Tron", "TRX"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6D1AabNHCupeiLM65ZR9UStMhJ1vCpyV4XbZdyhMZBiJXALQtmn9p42VTQckoHVn8WNqS7dqnJokZHAHcHGoaQgmv8D45oNUKx6DZMNZBCd",
            "ex_priv": "xprv9z1pB5qPNYGMVrGcyXt97Jwd9GBRoNFdhJfxqbHjzrBKeN1GMETuGFi1c73SQkP8kkKz5MVoMtLRcsDWggUcaPF32AXN4qNsNWBoJbaHcQ7",
        },
        "chain_ext": {
            "ex_pub": "xpub6EuK4CZWW5urEHdwAVDdDw327danAtccFcrXYvgf1DHrPXRwErt36xStQ2PNhn4hpwzPbzJ8pJVpewgChRnSs59q5Ay61GCfQZKUe71gbLq",
            "ex_priv": "xprvA1uxeh2cfiMZ1oZU4Tgcro6HZbkHmRtktPvvkYH3SsksWj6nhKZnZA8QYjdXx8LzM6wuMTE3LcVoAiANjLSRwbh24GWxdztn6xvjBuctSrk",
        },
        "addresses": [
            "TUEZSdKsoDHQMeZwihtdoBiN46zxhGWYdH",
            "TSeJkUh4Qv67VNFwY8LaAxERygNdy6NQZK",
            "TYJPRrdB5APNeRs4R7fYZSwW3TcrTKw2gx",
            "TRhVWK5XEDkQBDevcdCWW7RW51aRncty4W",
            "TT2X2yyubp7qpAWYYNE5JQWBtoZ7ikQFsY",
        ],
    },
    # VeChain
    {
        "coin": Bip44Coins.VECHAIN,
        "names": ("VeChain", "VET"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6DFQXWMJmxXgQrCwUboTCdbWRujz3FU7VGkCC7W2Pe2JNGkwGTonKAS73u6yg8jGiPvbuHBMDgwyF2Vb4rPExFA5WxigX5xDwiSy8ySjars",
            "ex_priv": "xprv9zG47zpQwayPCN8UNaGSqVemssuVdnkG83pbPj6QqJVKVURnivVXmN7dCcwHmqsqPG4pwHbaXXpUWrnvroytRA7zd8izYBrd82YhCGNExbV",
        },
        "chain_ext": {
            "ex_pub": "xpub6EV2C29DE2vifLG7VfDz3SrnW9ZJCGFXyi8sB4VaTDicJThQSm2LCftd54R7PgaHdr9izo7GnVH1FKj678DU3vrimvZRJuiBDj4dTBDXuLK",
            "ex_priv": "xprvA1VfnWcKPfNRSrBePdgygJv3x7ionoXgcVDGNg5xttBdRfNFuDi5esa9DjmgqFNwNUTCxyHA47gPNL7jBHQ5Kz4w7tPKb8WMMu6vv6BXveV",
        },
        "addresses": [
            "0xE5d846748409dF0b23D5bd47CEB14afc1CBbbcB3",
            "0xDb62FE39887b1b3b80a05e16596150CcDB12199e",
            "0xE1695De2C14Bd840D111EE45Dc4d9b7D9Baddf3D",
            "0x98CEa2F92172db44e062C7482e7539cb38B1B233",
            "0x076988846613FBf7713ca25453b97b97219f5c05",
        ],
    },
    # Zcash
    {
        "coin": Bip44Coins.ZCASH,
        "names": ("Zcash", "ZEC"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
        "wif_master": "5HzxC8XHHAtoC5jVvScY8Tr99Ud9MwFdF2pJKYsMTUknJZEurYr",
        "account": {
            "ex_pub": "xpub6CLX6bXDznyoBCRaHYmzfApq4dLc34UydL1gRHQTG55pNwbKSC2GvXjTQ4VS3n6P24fRd14uKz7P92xJQ3MWdRzUxGkqiftZf3riboiJLJs",
            "ex_priv": "xprv9yMAh5zLARRVxiM7BXEzJ2t6WbW7dbm8G765ctzqhjYqW9GAtei2NjQyYmDsoVoWxTdfY5D1uDAm58bcTb35GHTxKRCVzpv42SfuxTfPTCm",
        },
        "chain_ext": {
            "ex_pub": "xpub6F5Ve7yRXMnZD5xLJ22HNsL9z65Erc2Z4qc8ZzGi6xSBLYRZmvkeT9b9gTJxzratf6FvPFJgSR52CfEDNx8RXZ7BohAMdwSAsBoHEYKUXWF",
            "ex_priv": "xprvA269EcSXgzEFzbssBzVH1jPRS4EkT9JhhcgXmbs6YcuCTk6REPSPuMGfqAea9CXzni5Xd22edwYpmCcUnxGjVvgGRRmDccfE8fB25HB9hjf",
        },
        "addresses": [
            "t1XVXWCvpMgBvUaed4XDqWtgQgJSu1Ghz7F",
            "t1aQ2b1XszNVo15BguYLbQGqETBL9QZA8Jq",
            "t1gLHX58QpcZr2SfaMhc5XbxabE455KZhoA",
            "t1M4FWL2VYmWy5qEuPUHrhEEcxsdkZ79xYn",
            "t1QAU8388aa3G9DXnbGSverQoEoz3WeGEiy",
        ],
    },
    # Zilliqa
    {
        "coin": Bip44Coins.ZILLIQA,
        "names": ("Zilliqa", "ZIL"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
        "wif_master": "",
        "account": {
            "ex_pub": "xpub6DTx9qhc1XhEHMqwk1LyA5igoAAySsdKGpGFGPFPsDEKqbSecx6TkzuD1ZQd6Kp1QTvFwkAAR1oNYqYhon3NARKVzBSgZxVywPs4Pqx65ER",
            "ex_priv": "xprv9zUbkLAiBA8w4smUdyoxnwmxF8LV3QuTubLeTzqnJshLxo7W5QnDDCajAJkMZZaay7p2ci7z1SpqsMD3KGWdnZ2MU3hkNfiUxCaUPc67qtg",
        },
        "chain_ext": {
            "ex_pub": "xpub6E6wnYzenDvuyJvv5ht28SgPgDuQDwq6C8AwxiiFJvdvafuEXUEjnv3VFPPzn4p5xazvTAiW2H25tLj9LnpimpHMkp7HuVPwbTPcR5A1wm4",
            "ex_priv": "xprvA17bP3TkwrNckprSygM1mJjf8C4upV7EpuFMALJdkb6whsa5yvvVF7j1Q7EFBy83rKTEYByip8DLpu6NWPsSeMSpv8LaPsXzJqWAEwuRVTU",
        },
        "addresses": [
            "zil1y8cv5w9u3l4nz4vxga3a8uuafyu0xn387npzy7",
            "zil15xvgngyagn08hswpyazgcfe2sz385n3zda0tcl",
            "zil16yjqdug74ygdgt6n6dd57k2l4hegfh24yknj3k",
            "zil1xydwg7m86fyg92gavfgxy0xmvcskcv4nx5hgsy",
            "zil1u4czr4q7vgv3xuzdul2nx5zl7m8u3w5d5a28ka",
        ],
    },

    #
    # Test nets
    #

    # Bitcoin test net
    {
        "coin": Bip44Coins.BITCOIN_TESTNET,
        "names": ("Bitcoin TestNet", "BTC"),
        "is_testnet": True,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "tprv8ZgxMBicQKsPe5YMU9gHen4Ez3ApihUfykaqUorj9t6FDqy3nP6eoXiAo2ssvpAjoLroQxHqr3R5nE3a5dU3DHTjTgJDd7zrbniJr6nrCzd",
        "wif_master": "91mamsLpsPxwA9EnYnWT14Q6o8yrX6npaygFQBDroDVq5dZG3q3",
        "account": {
            "ex_pub": "tpubDC5FSnBiZDMmhiuCmWAYsLwgLYrrT9rAqvTySfuCCrgsWz8wxMXUS9Tb9iVMvcRbvFcAHGkMD5Kx8koh4GquNGNTfohfk7pgjhaPCdXpoba",
            "ex_priv": "tprv8fPDJN9UQqg6pFsQsrVxTwHZmXLvHpfGGcsCA9rtnatUgVtBKxhtFeqiyaYKSWydunKpjhvgJf6PwTwgirwuCbFq8YKgpQiaVJf3JCrNmkR",
        },
        "chain_ext": {
            "ex_pub": "tpubDEQBfiy13hMZzGT4NWqNnaSWwVqYQ58kuu2pDYjkrf8F6DLKAprm8c65Pyh7PrzodXHtJuEXFu5yf6JbvYaL8rz7v28zapwbuzZzr7z4UvR",
            "ex_priv": "tprv8hi9XJvkuKfu6oRGUsAnPAnQNUKcEjwrLbS2w2hTSPKrFj5YYS3Ax7UDDrZZHd4PSnPLW5whNxAXTW5bBrSNiSD1LUeg9n4j5sdGRJsZZwP",
        },
        "addresses": [
            "mkpZhYtJu2r87Js3pDiWJDmPte2NRZ8bJV",
            "mzpbWabUQm1w8ijuJnAof5eiSTep27deVH",
            "mnTkxhNkgx7TsZrEdRcPti564yQTzynGJp",
            "mpW3iVi2Td1vqDK8Nfie29ddZXf9spmZkX",
            "n2BMo5arHDyAK2CM8c56eoEd18uEkKnRLC",
        ],
    },
    # Bitcoin Cash test net
    {
        "coin": Bip44Coins.BITCOIN_CASH_TESTNET,
        "names": ("Bitcoin Cash TestNet", "BCH"),
        "is_testnet": True,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "tprv8ZgxMBicQKsPe5YMU9gHen4Ez3ApihUfykaqUorj9t6FDqy3nP6eoXiAo2ssvpAjoLroQxHqr3R5nE3a5dU3DHTjTgJDd7zrbniJr6nrCzd",
        "wif_master": "91mamsLpsPxwA9EnYnWT14Q6o8yrX6npaygFQBDroDVq5dZG3q3",
        "account": {
            "ex_pub": "tpubDC5FSnBiZDMmhiuCmWAYsLwgLYrrT9rAqvTySfuCCrgsWz8wxMXUS9Tb9iVMvcRbvFcAHGkMD5Kx8koh4GquNGNTfohfk7pgjhaPCdXpoba",
            "ex_priv": "tprv8fPDJN9UQqg6pFsQsrVxTwHZmXLvHpfGGcsCA9rtnatUgVtBKxhtFeqiyaYKSWydunKpjhvgJf6PwTwgirwuCbFq8YKgpQiaVJf3JCrNmkR",
        },
        "chain_ext": {
            "ex_pub": "tpubDEQBfiy13hMZzGT4NWqNnaSWwVqYQ58kuu2pDYjkrf8F6DLKAprm8c65Pyh7PrzodXHtJuEXFu5yf6JbvYaL8rz7v28zapwbuzZzr7z4UvR",
            "ex_priv": "tprv8hi9XJvkuKfu6oRGUsAnPAnQNUKcEjwrLbS2w2hTSPKrFj5YYS3Ax7UDDrZZHd4PSnPLW5whNxAXTW5bBrSNiSD1LUeg9n4j5sdGRJsZZwP",
        },
        "addresses_legacy": {
            "cls": Bip44BitcoinCashTestNet,
            "addresses": [
                "mkpZhYtJu2r87Js3pDiWJDmPte2NRZ8bJV",
                "mzpbWabUQm1w8ijuJnAof5eiSTep27deVH",
                "mnTkxhNkgx7TsZrEdRcPti564yQTzynGJp",
                "mpW3iVi2Td1vqDK8Nfie29ddZXf9spmZkX",
                "n2BMo5arHDyAK2CM8c56eoEd18uEkKnRLC",
            ]
        },
        "addresses": [
            "bchtest:qqaz6s295ncfs53m86qj0uw6sl8u2kuw0ymst35fx4",
            "bchtest:qrfuppcw3cf6nmpjpufgpzy3y74ptfxq5yxdy864k4",
            "bchtest:qpxzu2ljsp5sgk5wkcnusw0lyvrtwwa6xq7ysu5t02",
            "bchtest:qp3g60wa822p8xp6padds48mrs8c0ywd2g7nk9ff45",
            "bchtest:qr32turulady0gcxhah9mgezqcczjunl6q0d90hm4d",
        ],
    },
    # BitcoinSV
    {
        "coin": Bip44Coins.BITCOIN_SV_TESTNET,
        "names": ("BitcoinSV TestNet", "BSV"),
        "is_testnet": True,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "tprv8ZgxMBicQKsPe5YMU9gHen4Ez3ApihUfykaqUorj9t6FDqy3nP6eoXiAo2ssvpAjoLroQxHqr3R5nE3a5dU3DHTjTgJDd7zrbniJr6nrCzd",
        "wif_master": "91mamsLpsPxwA9EnYnWT14Q6o8yrX6npaygFQBDroDVq5dZG3q3",
        "account": {
            "ex_pub": "tpubDC5FSnBiZDMmhiuCmWAYsLwgLYrrT9rAqvTySfuCCrgsWz8wxMXUS9Tb9iVMvcRbvFcAHGkMD5Kx8koh4GquNGNTfohfk7pgjhaPCdXpoba",
            "ex_priv": "tprv8fPDJN9UQqg6pFsQsrVxTwHZmXLvHpfGGcsCA9rtnatUgVtBKxhtFeqiyaYKSWydunKpjhvgJf6PwTwgirwuCbFq8YKgpQiaVJf3JCrNmkR",
        },
        "chain_ext": {
            "ex_pub": "tpubDEQBfiy13hMZzGT4NWqNnaSWwVqYQ58kuu2pDYjkrf8F6DLKAprm8c65Pyh7PrzodXHtJuEXFu5yf6JbvYaL8rz7v28zapwbuzZzr7z4UvR",
            "ex_priv": "tprv8hi9XJvkuKfu6oRGUsAnPAnQNUKcEjwrLbS2w2hTSPKrFj5YYS3Ax7UDDrZZHd4PSnPLW5whNxAXTW5bBrSNiSD1LUeg9n4j5sdGRJsZZwP",
        },
        "addresses": [
            "mkpZhYtJu2r87Js3pDiWJDmPte2NRZ8bJV",
            "mzpbWabUQm1w8ijuJnAof5eiSTep27deVH",
            "mnTkxhNkgx7TsZrEdRcPti564yQTzynGJp",
            "mpW3iVi2Td1vqDK8Nfie29ddZXf9spmZkX",
            "n2BMo5arHDyAK2CM8c56eoEd18uEkKnRLC",
        ],
    },
    # Dash test net
    {
        "coin": Bip44Coins.DASH_TESTNET,
        "names": ("Dash TestNet", "DASH"),
        "is_testnet": True,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "tprv8ZgxMBicQKsPe5YMU9gHen4Ez3ApihUfykaqUorj9t6FDqy3nP6eoXiAo2ssvpAjoLroQxHqr3R5nE3a5dU3DHTjTgJDd7zrbniJr6nrCzd",
        "wif_master": "91mamsLpsPxwA9EnYnWT14Q6o8yrX6npaygFQBDroDVq5dZG3q3",
        "account": {
            "ex_pub": "tpubDC5FSnBiZDMmhiuCmWAYsLwgLYrrT9rAqvTySfuCCrgsWz8wxMXUS9Tb9iVMvcRbvFcAHGkMD5Kx8koh4GquNGNTfohfk7pgjhaPCdXpoba",
            "ex_priv": "tprv8fPDJN9UQqg6pFsQsrVxTwHZmXLvHpfGGcsCA9rtnatUgVtBKxhtFeqiyaYKSWydunKpjhvgJf6PwTwgirwuCbFq8YKgpQiaVJf3JCrNmkR",
        },
        "chain_ext": {
            "ex_pub": "tpubDEQBfiy13hMZzGT4NWqNnaSWwVqYQ58kuu2pDYjkrf8F6DLKAprm8c65Pyh7PrzodXHtJuEXFu5yf6JbvYaL8rz7v28zapwbuzZzr7z4UvR",
            "ex_priv": "tprv8hi9XJvkuKfu6oRGUsAnPAnQNUKcEjwrLbS2w2hTSPKrFj5YYS3Ax7UDDrZZHd4PSnPLW5whNxAXTW5bBrSNiSD1LUeg9n4j5sdGRJsZZwP",
        },
        "addresses": [
            "yRd4FhXfVGHXpsuZXPNkMrfD9GVj46pnjt",
            "yfd64jEpzzTLrHnR1wq3iiYXh68AiU8mcw",
            "yTGFWr27HBYsb8tkLbGdxLxuKbspamdbnG",
            "yVJYGeMP3rTLYnMe5qNt5nXSpA8WUTcz49",
            "ygyrMEECsTQa2bErqmjLiS8SFmNbQ4ynNF",
        ],
    },
    # Dogecoin test net
    {
        "coin": Bip44Coins.DOGECOIN_TESTNET,
        "names": ("Dogecoin TestNet", "DOGE"),
        "is_testnet": True,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "tgpv1aRS3XcGkbKXDbWwtRDZFYqUaCsEVXgkHt9m5mEjaWGGA18gaU1qZatwWCmjT66o2CmSNJmXkAvG29sYFXrz11WEaLwVrckr5LMkUrVeQmp",
        "wif_master": "95f58LEtaFKPRnQCMwkSuhbedNDHpWUaAFpTtUbenXWc4oSGq5N",
        "account": {
            "ex_pub": "tgub5QziLPy2KFnZgdWALMfCBZcPpFQxtmu3sMDKvDdt1L7WAiAWXpAqf1S9FzNEEa7ipDM6kEy4o9mrVGshjxMKMoq383HELMHguMNGgRryJwt",
            "ex_priv": "tgpv1g7gzi38m78EPmr1J83E4i4oMh3L4esLakS7m7EuDD4Vcf3p83d51i2VgkSAxnuh8eETh4QNCnbaBPmetmLqzKJLFCxy3uUZxrJUw2K7GUp",
        },
        "chain_ext": {
            "ex_pub": "tgub5TKeZLkJojnMyB41wNL26o7ERCPeqhBdwKnAh6USf8YsjwMskHW8MU4dWFZyhpgvXV2pmsTEqyXt1cNccE5k8QShNFiZB4Qc5eMtKv6s9mt",
            "ex_priv": "tgpv1iSdDepRFb82gKPru8i3ywZdxe221a9veizxXz5Ts1VsBtFBLWxMiAeyw2TQotzSfeHyTSRPH5fhhRuZMkqKWAFWT9HxPGpiZRGi3zY83Mf",
        },
        "addresses": [
            "nZVmfmUtKPmskB9Ds4P9GUJy9eYFqPKHqH",
            "noVoUoC3q7wgmb25McqSdLCHhUAhSREkT9",
            "nb8xvuyL7K3DWS8QgGH2rxcfKyvMQ1wiYe",
            "ndBFgiJbsywgU5bJRWPGzQBCpYB3M1DZCU",
            "nprZmJBRhatuwtUXBSjjd3nCG9R8DDm3y3",
        ],
    },
    # Litecoin test net
    {
        "coin": Bip44Coins.LITECOIN_TESTNET,
        "names": ("Litecoin TestNet", "LTC"),
        "is_testnet": True,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "ttpv96BtqegdxXceQk8r9KuoG5yiMACLxANu9hh98NpMwpzcCa8XfrJ7uwnRBMzsE5n9y2exs7VQBBdHNiJ66BrDUWE28WoexgbFVRkRc2abBR9",
        "wif_master": "91mamsLpsPxwA9EnYnWT14Q6o8yrX6npaygFQBDroDVq5dZG3q3",
        "account": {
            "ex_pub": "ttub4d4VPcY3DBxKCoAjoeDr9q3FaD4dbY89X65XUhapWiaSFbEjYLwnNg2EcHgEVSEALaEVdZnprREdcWMPJxqFkvN89FcPRFBueauxVCvFpUt",
            "ex_priv": "ttpv9Bt9nq7Vy3RMavTuZ2jU5FD38eNSXHZVSZyVoipXaXnqfE3fDRuMN4uyMufJjnb45U7zBs8EdoJbXxCCjRL5Tp27oNq89yJyNwhA48Rqv9P",
        },
        "chain_ext": {
            "ex_pub": "ttub4fPRcZKKhfx7VLibQetg54Y6BA3KYTQjb4eNFaRPAX1oppS6kpH558eirYsyxgoN3qvDfCGzuEzf8qrJBEZgXWynPU3iFxJppsua8eJixXb",
            "ex_priv": "ttpv9ED61mtnTXR9sU1mA3QHzUhsjbM8UCr5WYYLabf6ELEDETF2RuEe4XYTcBgYatfocUBVxF9Fi6Nj3zL7CQpYyeyJ1KA7VLf7yWfPB8Cjufb",
        },
        "addresses_depr": {
            "cls": Bip44LitecoinTestNet,
            "addresses": [
                "mkpZhYtJu2r87Js3pDiWJDmPte2NRZ8bJV",
                "mzpbWabUQm1w8ijuJnAof5eiSTep27deVH",
                "mnTkxhNkgx7TsZrEdRcPti564yQTzynGJp",
                "mpW3iVi2Td1vqDK8Nfie29ddZXf9spmZkX",
                "n2BMo5arHDyAK2CM8c56eoEd18uEkKnRLC",
            ],
        },
        "addresses": [
            "mkpZhYtJu2r87Js3pDiWJDmPte2NRZ8bJV",
            "mzpbWabUQm1w8ijuJnAof5eiSTep27deVH",
            "mnTkxhNkgx7TsZrEdRcPti564yQTzynGJp",
            "mpW3iVi2Td1vqDK8Nfie29ddZXf9spmZkX",
            "n2BMo5arHDyAK2CM8c56eoEd18uEkKnRLC",
        ],
    },
    # Zcash test net
    {
        "coin": Bip44Coins.ZCASH_TESTNET,
        "names": ("Zcash TestNet", "ZEC"),
        "is_testnet": True,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "tprv8ZgxMBicQKsPe5YMU9gHen4Ez3ApihUfykaqUorj9t6FDqy3nP6eoXiAo2ssvpAjoLroQxHqr3R5nE3a5dU3DHTjTgJDd7zrbniJr6nrCzd",
        "wif_master": "91mamsLpsPxwA9EnYnWT14Q6o8yrX6npaygFQBDroDVq5dZG3q3",
        "account": {
            "ex_pub": "tpubDC5FSnBiZDMmhiuCmWAYsLwgLYrrT9rAqvTySfuCCrgsWz8wxMXUS9Tb9iVMvcRbvFcAHGkMD5Kx8koh4GquNGNTfohfk7pgjhaPCdXpoba",
            "ex_priv": "tprv8fPDJN9UQqg6pFsQsrVxTwHZmXLvHpfGGcsCA9rtnatUgVtBKxhtFeqiyaYKSWydunKpjhvgJf6PwTwgirwuCbFq8YKgpQiaVJf3JCrNmkR",
        },
        "chain_ext": {
            "ex_pub": "tpubDEQBfiy13hMZzGT4NWqNnaSWwVqYQ58kuu2pDYjkrf8F6DLKAprm8c65Pyh7PrzodXHtJuEXFu5yf6JbvYaL8rz7v28zapwbuzZzr7z4UvR",
            "ex_priv": "tprv8hi9XJvkuKfu6oRGUsAnPAnQNUKcEjwrLbS2w2hTSPKrFj5YYS3Ax7UDDrZZHd4PSnPLW5whNxAXTW5bBrSNiSD1LUeg9n4j5sdGRJsZZwP",
        },
        "addresses": [
            "tmF1xjfhsSzhy55dmhorzTnKjtHhZmPKzts",
            "tmV1zYhR2xisn6VWdCNKHpeD4S7L1U1nPH6",
            "tmGf9zpCKEuyJqLcxX1kt4GdS4d5fMoLkyj",
            "tmJhSkcXb1asmnz5rGFs8BiByZBLMGRW1SK",
            "tmWNkqCQQqBq1Gny52CDapMnxznaS86xw7D",
        ],
    },
]

# Tests for default path derivation
TEST_VECT_DEFAULT_PATH = [
    #
    # Main nets
    #

    # Algorand
    {
        "coin": Bip44Coins.ALGORAND,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "EP2D7TV7IAFANZHK3B6QLKB53N5UTD7RARVXZTWCPCRQQBKYVGM2XIMT2Q",
    },
    # Avax C-Chain
    {
        "coin": Bip44Coins.AVAX_C_CHAIN,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "0x9858EfFD232B4033E47d90003D41EC34EcaEda94",
    },
    # Avax P-Chain
    {
        "coin": Bip44Coins.AVAX_P_CHAIN,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "P-avax1p9575chzhvcwvmvzaqh7yeld76r3af0ha56phl",
    },
    # Avax X-Chain
    {
        "coin": Bip44Coins.AVAX_X_CHAIN,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "X-avax1p9575chzhvcwvmvzaqh7yeld76r3af0ha56phl",
    },
    # Band Protocol
    {
        "coin": Bip44Coins.BAND_PROTOCOL,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "band1vh8tr8ddf7g0gfep23t46yllmlvtqfjknjdd38",
    },
    # Binance Chain
    {
        "coin": Bip44Coins.BINANCE_CHAIN,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "bnb1rxhz5vdv4fvdjye8gxqvfv0yvg20jtlwf4f38d",
    },
    # Binance Smart Chain
    {
        "coin": Bip44Coins.BINANCE_SMART_CHAIN,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "0x9858EfFD232B4033E47d90003D41EC34EcaEda94",
    },
    # Bitcoin
    {
        "coin": Bip44Coins.BITCOIN,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "1LqBGSKuX5yYUonjxT5qGfpUsXKYYWeabA",
    },
    # Bitcoin Cash
    {
        "coin": Bip44Coins.BITCOIN_CASH,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "bitcoincash:qqyx49mu0kkn9ftfj6hje6g2wfer34yfnq5tahq3q6",
    },
    # BitcoinSV
    {
        "coin": Bip44Coins.BITCOIN_SV,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "1K6LZdwpKT5XkEZo2T2kW197aMXYbYMc4f",
    },
    # Cosmos
    {
        "coin": Bip44Coins.COSMOS,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "cosmos19rl4cm2hmr8afy4kldpxz3fka4jguq0auqdal4",
    },
    # Dash
    {
        "coin": Bip44Coins.DASH,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "XoJA8qE3N2Y3jMLEtZ3vcN42qseZ8LvFf5",
    },
    # Dogecoin
    {
        "coin": Bip44Coins.DOGECOIN,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "DBus3bamQjgJULBJtYXpEzDWQRwF5iwxgC",
    },
    # Elrond
    {
        "coin": Bip44Coins.ELROND,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "erd1sqhjrtmsn5yjk6w85099p8v0ly0g8z9pxeqe5dvu5rlf2n7vq3vqytny9g",
    },
    # EOS
    {
        "coin": Bip44Coins.EOS,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "EOS6zpSNY1YoLxNt2VsvJjoDfBueU6xC1M1ERJw1UoekL1NHn8KNA",
    },
    # Ethereum
    {
        "coin": Bip44Coins.ETHEREUM,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "0x9858EfFD232B4033E47d90003D41EC34EcaEda94",
    },
    # Ethereum Classic
    {
        "coin": Bip44Coins.ETHEREUM_CLASSIC,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "0xFA22515E43658ce56A7682B801e9B5456f511420",
    },
    # Fantom Opera
    {
        "coin": Bip44Coins.FANTOM_OPERA,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "0x9858EfFD232B4033E47d90003D41EC34EcaEda94",
    },
    # Filecoin
    {
        "coin": Bip44Coins.FILECOIN,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "f1qode47ievxlxzk6z2viuovedabmn3tq6t57uqhq",
    },
    # Harmony One (Atom address)
    {
        "coin": Bip44Coins.HARMONY_ONE_ATOM,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "one1q6gkzcap0uruuu8r6sldxuu47pd4ww9w9t7tg6",
    },
    # Harmony One (Ethereum address)
    {
        "coin": Bip44Coins.HARMONY_ONE_ETH,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "0x06916163A17F07ce70e3d43Ed37395f05B5738aE",
    },
    # Harmony One (Metamask address)
    {
        "coin": Bip44Coins.HARMONY_ONE_METAMASK,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "0x9858EfFD232B4033E47d90003D41EC34EcaEda94",
    },
    # Huobi Chain
    {
        "coin": Bip44Coins.HUOBI_CHAIN,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "0x9858EfFD232B4033E47d90003D41EC34EcaEda94",
    },
    # IRIS Network
    {
        "coin": Bip44Coins.IRIS_NET,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "iaa19rl4cm2hmr8afy4kldpxz3fka4jguq0afzdvay",
    },
    # Kava
    {
        "coin": Bip44Coins.KAVA,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "kava1vh8tr8ddf7g0gfep23t46yllmlvtqfjkkgejlg",
    },
    # Kusama (ed25519 SLIP-0010)
    {
        "coin": Bip44Coins.KUSAMA_ED25519_SLIP,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "FoTxsgYKH4AUngJAJNsqgmK85RzCc6cerkrsN18wiFfwBrn",
    },
    # Litecoin
    {
        "coin": Bip44Coins.LITECOIN,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "LUWPbpM43E2p7ZSh8cyTBEkvpHmr3cB8Ez",
    },
    # Monero (ed25519 SLIP-0010)
    {
        "coin": Bip44Coins.MONERO_ED25519_SLIP,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "48wpY5S7Ksq18DtvSaomPqEkFHueWCXcK7Cj2iiKRDXP65BWToZKj1ZgVzhG5Ngusy53b8q5xYwYaXepz4WXPTQ5ChJAe9n",
    },
    # Monero (secp256k1)
    {
        "coin": Bip44Coins.MONERO_SECP256K1,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "49vDbkSo7eve3J41sBdjvjaBUyz8qHohsQcGtRf63qEUTMBvmA45fpp5pSacMdSg7A3b71RejLzB8EkGbfjp5PELVF2N4Zn",
    },
    # Nano
    {
        "coin": Bip44Coins.NANO,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "nano_1p6hocygi1pzjidi3hho3wn85qiw3ykapg7khu9b45dwf7momgqoytn1c1jz",
    },
    # NEO
    {
        "coin": Bip44Coins.NEO,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "AJHeWQn2qKKqD4nBE82etebgT3GEM9HDRH",
    },
    # OKEx Chain (Atom address)
    {
        "coin": Bip44Coins.OKEX_CHAIN_ATOM,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "ex1npvwllfr9dqr8erajqqr6s0vxnk2ak55peywkd",
    },
    # OKEx Chain (Atom old address)
    {
        "coin": Bip44Coins.OKEX_CHAIN_ATOM_OLD,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "ex19dhmgh5vxg9cq5f85l88jm6h4r4uklkgpamn38",
    },
    # OKEx Chain (Ethereum address)
    {
        "coin": Bip44Coins.OKEX_CHAIN_ETH,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "0x9858EfFD232B4033E47d90003D41EC34EcaEda94",
    },
    # Ontology
    {
        "coin": Bip44Coins.ONTOLOGY,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "AZMgu1jizeCvwKx89qtKq3Rd92iPXHJQ9W",
    },
    # Polkadot (ed25519 SLIP-0010)usdc
    {
        "coin": Bip44Coins.POLKADOT_ED25519_SLIP,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "14E9StbjYhJiAfsNMEcq5tETq79Q6EqaGyebdziY214hNWDH",
    },
    # Polygon
    {
        "coin": Bip44Coins.POLYGON,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "0x9858EfFD232B4033E47d90003D41EC34EcaEda94",
    },
    # Ripple
    {
        "coin": Bip44Coins.RIPPLE,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "rHsMGQEkVNJmpGWs8XUBoTBiAAbwxZN5v3",
    },
    # Solana
    {
        "coin": Bip44Coins.SOLANA,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "GjJyeC1r2RgkuoCWMyPYkCWSGSGLcz266EaAkLA27AhL",
    },
    # Stellar
    {
        "coin": Bip44Coins.STELLAR,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "GB3JDWCQJCWMJ3IILWIGDTQJJC5567PGVEVXSCVPEQOTDN64VJBDQBYX",
    },
    # Terra
    {
        "coin": Bip44Coins.TERRA,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "terra1amdttz2937a3dytmxmkany53pp6ma6dy4vsllv",
    },
    # Tezos
    {
        "coin": Bip44Coins.TEZOS,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "tz1VQA4RP4fLjEEMW2FR4pE9kAg5abb5h5GL",
    },
    # Theta Network
    {
        "coin": Bip44Coins.THETA,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "0xB16f8D941d7D281aDCEf3e950aBF2147D7AC2D33",
    },
    # Tron
    {
        "coin": Bip44Coins.TRON,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "TUEZSdKsoDHQMeZwihtdoBiN46zxhGWYdH",
    },
    # VeChain
    {
        "coin": Bip44Coins.VECHAIN,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "0xE5d846748409dF0b23D5bd47CEB14afc1CBbbcB3",
    },
    # Zcash
    {
        "coin": Bip44Coins.ZCASH,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "t1XVXWCvpMgBvUaed4XDqWtgQgJSu1Ghz7F",
    },
    # Zilliqa
    {
        "coin": Bip44Coins.ZILLIQA,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "zil1y8cv5w9u3l4nz4vxga3a8uuafyu0xn387npzy7",
    },

    #
    # Test nets
    #

    # Bitcoin test net
    {
        "coin": Bip44Coins.BITCOIN_TESTNET,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "mkpZhYtJu2r87Js3pDiWJDmPte2NRZ8bJV",
    },
    # Bitcoin Cash test net
    {
        "coin": Bip44Coins.BITCOIN_CASH_TESTNET,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "bchtest:qqaz6s295ncfs53m86qj0uw6sl8u2kuw0ymst35fx4",
    },
    # BitcoinSV
    {
        "coin": Bip44Coins.BITCOIN_SV_TESTNET,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "mkpZhYtJu2r87Js3pDiWJDmPte2NRZ8bJV",
    },
    # Dash test net
    {
        "coin": Bip44Coins.DASH_TESTNET,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "yRd4FhXfVGHXpsuZXPNkMrfD9GVj46pnjt",
    },
    # Dogecoin test net
    {
        "coin": Bip44Coins.DOGECOIN_TESTNET,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "nZVmfmUtKPmskB9Ds4P9GUJy9eYFqPKHqH",
    },
    # Litecoin test net
    {
        "coin": Bip44Coins.LITECOIN_TESTNET,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "mkpZhYtJu2r87Js3pDiWJDmPte2NRZ8bJV",
    },
    # Zcash test net
    {
        "coin": Bip44Coins.ZCASH_TESTNET,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "tmF1xjfhsSzhy55dmhorzTnKjtHhZmPKzts",
    },
]

# Tests for different key formats
TEST_VECT_KEY_FORMATS = {
    "coin": Bip44Coins.BITCOIN,
    "seed": "5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
    "ex_priv": "xprv9s21ZrQH143K3GJpoapnV8SFfukcVBSfeCficPSGfubmSFDxo1kuHnLisriDvSnRRuL2Qrg5ggqHKNVpxR86QEC8w35uxmGoggxtQTPvfUu",
    "raw_priv": "1837c1be8e2995ec11cda2b066151be2cfb48adf9e47b151d46adab3a21cdf67",
    "ex_pub": "xpub661MyMwAqRbcFkPHucMnrGNzDwb6teAX1RbKQmqtEF8kK3Z7LZ59qafCjB9eCRLiTVG3uxBxgKvRgbubRhqSKXnGGb1aoaqLrpMBDrVxga8",
    "raw_compr_pub": "03d902f35f560e0470c63313c7369168d9d7df2d49bf295fd9fb7cb109ccee0494",
    "raw_uncompr_pub": "04d902f35f560e0470c63313c7369168d9d7df2d49bf295fd9fb7cb109ccee04947d000a1345d3845dd83b4c5814f876c918305b598f066c958fad972bf59f2ec7",
}

# Tests for extended keys with valid and invalid depths
TEST_VECT_EX_KEY_DEPTHS = {
    # Private key with depth 5 is fine
    "ex_priv_5": "xprvA47LQAPDXEkr9wwtUHNta4qWqiobTGKppzVy5JZVCszLdGTTQbuRxRMCJGqSdBg91M13Z2RTP2BKU5yDD41WFwZ7yavhhbCEs7cTnyvkxJV",
    # Private key with depth 6 shall raise an exception
    "ex_priv_6": "xprvA69uJSR3uVgvYFM5AFabGnMuAvtdLrbm84CwEieMBbk5Kjk9ZGeYPF4AWuJ9EPBzC8pLn117Y6TFqgNKZ6EVKmoDxT4EjT1BaG3RhWL6wdF",
    # Public key with depth 2 shall raise an exception
    "ex_pub_2": "xpub6AmukNpN4yyLgyzSysjU6JqqoYA1mVUvtinHYdBGPDppatJXHxT8CcDsmBo9n3yLBgrcw9z62ygt1siT9xai4UaJ2w4FPmY6kPCF96YN2cF",
    # Public key with depth 3 shall raise an exception
    "ex_pub_3": "xpub6BosfCnifzxcFwrSzQiqu2DBVTshkCXacvNsWGYJVVhhawA7d4R5WSWGFNbi8Aw6ZRc1brxMyWMzG3DSSSSoekkudhUd9yLb6qx39T9nMdj",
    # Public key with depth 5 is fine
    "ex_pub_5": "xpub6Fbrwk4KhC8qnFVXTcR3wRsqiTGkedcSSZKyTqKaxXjFN6rZv3UJYZ4mQtjNYY3gCa181iCHSBWyWst2PFiXBKgLpFVSdcyLbHyAahin8pd",
    # Public key with depth 6 shall raise an exception
    "ex_pub_6": "xpub6JtuhUVosPSgpBQFZS9oy6oorydcmXS66Kr2TmURvm8uu5wWBXRmRziMT85N4epgkVtwgxpt5FnduVJFi1ARiUcSELWhnZwp9Ge1icYFvhj",
}

# Seed for generic tests that need it
TEST_SEED = b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4"


#
# Tests
#
class Bip44Tests(unittest.TestCase):
    # Test specification name
    def test_spec_name(self):
        self.assertEqual(Bip44.SpecName(), "BIP-0044")

    # Run all tests in test vector using FromSeed for construction
    def test_from_seed(self):
        Bip44BaseTestHelper.test_from_seed(self, Bip44, TEST_VECT)

    # Run all tests in test vector using FromExtendedKey for construction
    def test_from_ex_key(self):
        Bip44BaseTestHelper.test_from_ex_key(self, Bip44, TEST_VECT)

    # Run all tests in test vector using FromPrivateKey for construction
    def test_from_priv_key(self):
        Bip44BaseTestHelper.test_from_priv_key(self, Bip44, TEST_VECT)

    # Test default path derivation
    def test_default_path_derivation(self):
        Bip44BaseTestHelper.test_default_path_derivation(self, Bip44, TEST_VECT_DEFAULT_PATH)

    # Test for IsLevel method
    def test_is_level(self):
        Bip44BaseTestHelper.test_is_level(self, Bip44, Bip44Coins, TEST_SEED)

    # Test different key formats
    def test_key_formats(self):
        Bip44BaseTestHelper.test_key_formats(self, Bip44, TEST_VECT_KEY_FORMATS)

    # Test construction from extended keys with valid and invalid depths
    def test_from_ex_key_depth(self):
        Bip44BaseTestHelper.test_from_ex_key_depth(self, Bip44, Bip44Coins, TEST_VECT_EX_KEY_DEPTHS)

    # Test type error during construction
    def test_type_error(self):
        Bip44BaseTestHelper.test_type_error(self, Bip44, [Bip49Coins, Bip84Coins])

    # Test invalid path derivations
    def test_invalid_derivations(self):
        Bip44BaseTestHelper.test_invalid_derivations(self, Bip44, Bip44Coins, TEST_SEED)
