#  Copyright (c) 2019 JD Williams
#
#  This file is part of Firefly, a Python SOA framework built by JD Williams. Firefly is free software; you can
#  redistribute it and/or modify it under the terms of the GNU General Public License as published by the
#  Free Software Foundation; either version 3 of the License, or (at your option) any later version.
#
#  Firefly is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
#  Public License for more details. You should have received a copy of the GNU Lesser General Public
#  License along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
#  You should have received a copy of the GNU General Public License along with Firefly. If not, see
#  <http://www.gnu.org/licenses/>.

"""gypd output module

This module produces gyp input as its output.  Output files are given the
.gypd extension to avoid overwriting the .gyp files that they are generated
from.  Internal references to .gyp files (such as those found in
"dependencies" sections) are not adjusted to point to .gypd files instead;
unlike other paths, which are relative to the .gyp or .gypd file, such paths
are relative to the directory from which gyp was run to create the .gypd file.

This generator module is intended to be a sample and a debugging aid, hence
the "d" for "debug" in .gypd.  It is useful to inspect the results of the
various merges, expansions, and conditional evaluations performed by gyp
and to see a representation of what would be fed to a generator module.

It's not advisable to rename .gypd files produced by this module to .gyp,
because they will have all merges, expansions, and evaluations already
performed and the relevant constructs not present in the output; paths to
dependencies may be wrong; and various sections that do not belong in .gyp
files such as such as "included_files" and "*_excluded" will be present.
Output will also be stripped of comments.  This is not intended to be a
general-purpose gyp pretty-printer; for that, you probably just want to
run "pprint.pprint(eval(open('source.gyp').read()))", which will still strip
comments but won't do all of the other things done to this module's output.

The specific formatting of the output generated by this module is subject
to change.
"""


import gyp.common
import errno
import os
import pprint


# These variables should just be spit back out as variable references.
_generator_identity_variables = [
  'CONFIGURATION_NAME',
  'EXECUTABLE_PREFIX',
  'EXECUTABLE_SUFFIX',
  'INTERMEDIATE_DIR',
  'LIB_DIR',
  'PRODUCT_DIR',
  'RULE_INPUT_ROOT',
  'RULE_INPUT_DIRNAME',
  'RULE_INPUT_EXT',
  'RULE_INPUT_NAME',
  'RULE_INPUT_PATH',
  'SHARED_INTERMEDIATE_DIR',
  'SHARED_LIB_DIR',
  'SHARED_LIB_PREFIX',
  'SHARED_LIB_SUFFIX',
  'STATIC_LIB_PREFIX',
  'STATIC_LIB_SUFFIX',
]

# gypd doesn't define a default value for OS like many other generator
# modules.  Specify "-D OS=whatever" on the command line to provide a value.
generator_default_variables = {
}

# gypd supports multiple toolsets
generator_supports_multiple_toolsets = True

# TODO(mark): This always uses <, which isn't right.  The input module should
# notify the generator to tell it which phase it is operating in, and this
# module should use < for the early phase and then switch to > for the late
# phase.  Bonus points for carrying @ back into the output too.
for v in _generator_identity_variables:
  generator_default_variables[v] = '<(%s)' % v


def GenerateOutput(target_list, target_dicts, data, params):
  output_files = {}
  for qualified_target in target_list:
    [input_file, target] = \
        gyp.common.ParseQualifiedTarget(qualified_target)[0:2]

    if input_file[-4:] != '.gyp':
      continue
    input_file_stem = input_file[:-4]
    output_file = input_file_stem + params['options'].suffix + '.gypd'

    if not output_file in output_files:
      output_files[output_file] = input_file

  for output_file, input_file in output_files.iteritems():
    output = open(output_file, 'w')
    pprint.pprint(data[input_file], output)
    output.close()
