#  Copyright (c) 2019 JD Williams
#
#  This file is part of Firefly, a Python SOA framework built by JD Williams. Firefly is free software; you can
#  redistribute it and/or modify it under the terms of the GNU General Public License as published by the
#  Free Software Foundation; either version 3 of the License, or (at your option) any later version.
#
#  Firefly is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
#  Public License for more details. You should have received a copy of the GNU Lesser General Public
#  License along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
#  You should have received a copy of the GNU General Public License along with Firefly. If not, see
#  <http://www.gnu.org/licenses/>.

from __future__ import annotations

import inspect
import keyword
import typing
from datetime import datetime

import firefly.domain as ffd

# __pragma__('skip')
from dataclasses import is_dataclass, fields
from abc import ABC
# __pragma__('noskip')
# __pragma__ ('ecom')
from dateparser import parse

"""?
from firefly.presentation.web.polyfills import is_dataclass, fields
?"""
# __pragma__ ('noecom')


keyword_list = keyword.kwlist
keyword_list.append('id')


def _fix_keywords(params: dict):
    if not isinstance(params, dict):
        return params

    for k, v in params.copy().items():
        if k in keyword_list:
            params[f'{k}_'] = v
    return params


class Void:
    pass


void = Void()


def _handle_type_hint(params: typing.Any, t: type, key: str = None, required: bool = False):
    ret = {}

    origin = ffd.get_origin(t)
    args = ffd.get_args(t)

    if origin is typing.List:
        if key not in params:
            if required:
                raise ffd.MissingArgument()
            return []

        if ffd.is_type_hint(args[0]):
            if key is not None:
                ret[key] = list(map(lambda a: _handle_type_hint(a, args[0]), params[key]))
            else:
                ret = list(map(lambda a: _handle_type_hint(a, args[0]), params[key]))
        elif inspect.isclass(args[0]) and issubclass(args[0], ffd.ValueObject):
            try:
                if key is not None:
                    ret[key] = list(map(lambda a: _build_value_object(a, args[0], required), params[key]))
                else:
                    ret = list(map(lambda a: _build_value_object(a, args[0], required), params[key]))
            except TypeError:
                return
        else:
            ret[key] = params[key]

    elif origin is typing.Dict:
        if key not in params:
            if required:
                raise ffd.MissingArgument()
            return {}

        if ffd.is_type_hint(args[1]):
            ret[key] = {k: _handle_type_hint(v, args[1]) for k, v in params[key].items()}
        elif inspect.isclass(args[1]) and issubclass(args[1], ffd.ValueObject):
            ret[key] = {k: _build_value_object(v, args[1], required) for k, v in params[key].items()}
        else:
            ret[key] = params[key]

    elif origin is typing.Union:
        for arg in args:
            r = _handle_type_hint(params, arg, key, required)
            if r is not void:
                if key is not None:
                    ret[key] = r
                else:
                    ret = r
                break

    else:
        if inspect.isclass(t) and issubclass(t, ffd.ValueObject):
            return _build_value_object(params, t, required)

        try:
            if key is not None:
                if t(params[key]) == params[key]:
                    return params[key]
                if key in params and params[key] is None:
                    return None
            elif key is None and t(params) == params:
                return params
        except (TypeError, KeyError):
            pass

    if ret == {}:
        return void

    return ret


def _build_value_object(obj, type_, required):
    try:
        if isinstance(obj, type_):
            return obj
    except TypeError:
        pass

    try:
        e = _generate_model(obj, type_)
        if e is False and required is True:
            raise ffd.MissingArgument()
        return e
    except ffd.MissingArgument:
        if required is False:
            return
        raise


def _generate_model(args: dict, model_type: type, strict: bool = False):
    subclasses = model_type.__subclasses__()
    if len(subclasses):
        for subclass in subclasses:
            try:
                return _generate_model(args, subclass, strict=True)
            except RuntimeError:
                continue

    entity_args = build_argument_list(args, model_type)
    if strict:
        for k in args.keys():
            if k not in entity_args:
                raise RuntimeError()

    return model_type(**entity_args)


def build_argument_list(params: dict, obj: typing.Union[typing.Callable, type], strict: bool = True):
    args = {}
    field_dict = {}
    is_dc = False
    params = _fix_keywords(params)

    if is_dataclass(obj):
        is_dc = True
        field_dict = {}
        # noinspection PyDataclass
        for field_ in fields(obj):
            field_dict[field_.name] = field_
        sig = inspect.signature(obj.__init__)
        types = typing.get_type_hints(obj)
    elif isinstance(obj, ffd.MetaAware):
        sig = inspect.signature(obj.__call__)
        types = typing.get_type_hints(obj.__call__)
    elif isinstance(obj, type):
        sig = inspect.signature(obj.__init__)
        types = typing.get_type_hints(obj.__init__)
    else:
        sig = inspect.signature(obj)
        try:
            types = typing.get_type_hints(obj)
        except NameError:
            types = obj.__annotations__

    for param in sig.parameters.values():
        if param.kind == inspect.Parameter.VAR_KEYWORD:
            return params

    for name, param in sig.parameters.items():
        if name == 'self' or param.kind in (inspect.Parameter.VAR_POSITIONAL, inspect.Parameter.VAR_KEYWORD):
            continue

        required = False
        if is_dc:
            required = field_dict[name].metadata.get('required', False) is True
            try:
                d = field_dict[name].default_factory()
                if not isinstance(d, ffd.Empty):
                    required = False
            except (AttributeError, TypeError):
                pass
        elif param.default is not None:
            required = True

        type_ = types[name] if name in types else None
        if type_ is datetime and name in params and isinstance(params[name], str):
            params[name] = parse(params[name]).replace(tzinfo=None)

        if isinstance(type_, type) and issubclass(type_, ffd.ValueObject):
            if name in params and isinstance(params[name], type_):
                args[name] = params[name]
                continue

            try:
                nested = False
                if name in params and isinstance(params[name], dict):
                    e = _generate_model(params[name], type_)
                    nested = True
                else:
                    e = _generate_model(params, type_)
            except ffd.MissingArgument:
                if required is False:
                    continue
                raise
            # TODO use factories where appropriate
            args[name] = e
            if nested:
                del params[name]
            else:
                for key in e.to_dict().keys():
                    if (not hasattr(type_, 'id_name') or key != type_.id_name()) and key in params:
                        del params[key]
                        if key in args:
                            del args[key]

        elif ffd.is_type_hint(type_):
            parameter_args = _handle_type_hint(params, type_, key=name, required=required)
            if parameter_args:
                args.update(parameter_args)

        elif isinstance(params, dict) and name in params:
            args[name] = params[name]
        elif name.endswith('_') and name.rstrip('_') in params:
            args[name] = params[name.rstrip('_')]
        elif required is True and strict:
            raise ffd.MissingArgument(f'Argument: {name} is required')

    return args
