"""This module contains a Tomek_Land_em_coupling cardiac cell model

The module was modified from a file that was autogenerated from a gotran ode file
"""
from __future__ import division

from collections import OrderedDict
from typing import Dict

import dolfin
import ufl
from cbcbeat.cellmodels import CardiacCellModel
from dolfin import as_vector
from dolfin import Constant

from ... import utils
from .em_model import EMCoupling

logger = utils.getLogger(__name__)


def Max(a, b):
    return (a + b + abs(a - b)) / Constant(2.0)


def Min(a, b):
    return (a + b - abs(a - b)) / Constant(2.0)


def vs_functions_to_dict(vs):
    state_names = TorLandFull.default_initial_conditions().keys()
    return {
        name: utils.sub_function(vs, index) for index, name in enumerate(state_names)
    }


class TorLandFull(CardiacCellModel):
    def __init__(
        self,
        coupling: EMCoupling,
        params=None,
        init_conditions=None,
    ):
        """
        Create cardiac cell model

        *Arguments*
         params (dict, :py:class:`dolfin.Mesh`, optional)
           optional model parameters
         init_conditions (dict, :py:class:`dolfin.Mesh`, optional)
           optional initial conditions
        """
        logger.debug("Initialize Tor Land model")

        super().__init__(params, init_conditions)
        self.lmbda = coupling.lmbda_ep
        self.Zetas = coupling.Zetas_ep
        self.Zetaw = coupling.Zetaw_ep

    @staticmethod
    def default_parameters(disease_state: str = "healthy") -> Dict[str, float]:
        """Set-up and return default parameters.

        Parameters
        ----------
        disease_state : str, optional
            String with "hf" or "healthy", by default "healthy".
            If "hf", then parameters representing heart failure
            will be used.

        Returns
        -------
        OrderedDict
            Dictionary with default values
        """
        params = OrderedDict(
            [
                ("ICaL_fractionSS", 0.80),
                ("INaCa_fractionSS", 0.35),
                ("scale_ICaL", 1.018),
                ("scale_IK1", 1.414),
                ("scale_IKr", 1.119),
                ("scale_IKs", 1.648),
                ("scale_INaL", 2.274),
                ("celltype", 0),
                ("cao", 1.8),
                ("ko", 5.0),
                ("nao", 140.0),
                ("F", 96485.0),
                ("R", 8314.0),
                ("T", 310.0),
                ("L", 0.01),
                ("rad", 0.0011),
                ("Ahf", 0.99),
                ("GNa", 11.7802),
                ("thL", 200.0),
                ("Gto", 0.16),
                ("delta_epi", 1.0),
                ("Aff", 0.6),
                ("Kmn", 0.002),
                ("k2n", 500.0),
                ("tjca", 75.0),
                ("zca", 2.0),
                ("bt", 4.75),
                ("Beta0", 2.3),
                ("Beta1", -2.4),
                ("Tot_A", 25),
                ("Tref", 120),
                ("Trpn50", 0.35),
                ("calib", 1),
                ("cat50_ref", 0.805),
                ("emcoupling", 1),
                ("etal", 200),
                ("etas", 20),
                ("gammas", 0.0085),
                ("gammaw", 0.615),
                ("isacs", 0),
                ("ktrpn", 0.1),
                ("ku", 0.04),
                ("kuw", 0.182),
                ("kws", 0.012),
                ("mode", 1),
                ("ntm", 2.4),
                ("ntrpn", 2.0),
                ("p_a", 2.1),
                ("p_b", 9.1),
                ("p_k", 7.0),
                ("phi", 2.23),
                ("rs", 0.25),
                ("rw", 0.5),
                ("CaMKo", 0.05),
                ("KmCaM", 0.0015),
                ("KmCaMK", 0.15),
                ("aCaMK", 0.05),
                ("bCaMK", 0.00068),
                ("PKNa", 0.01833),
                ("Gncx", 0.0034),
                ("KmCaAct", 0.00015),
                ("kasymm", 12.5),
                ("kcaoff", 5000.0),
                ("kcaon", 1500000.0),
                ("kna1", 15.0),
                ("kna2", 5.0),
                ("kna3", 88.12),
                ("qca", 0.167),
                ("qna", 0.5224),
                ("wca", 60000.0),
                ("wna", 60000.0),
                ("wnaca", 5000.0),
                ("H", 1e-07),
                ("Khp", 1.698e-07),
                ("Kki", 0.5),
                ("Kko", 0.3582),
                ("Kmgatp", 1.698e-07),
                ("Knai0", 9.073),
                ("Knao0", 27.78),
                ("Knap", 224.0),
                ("Kxkur", 292.0),
                ("MgADP", 0.05),
                ("MgATP", 9.8),
                ("Pnak", 15.4509),
                ("delta", -0.155),
                ("eP", 4.2),
                ("k1m", 182.4),
                ("k1p", 949.5),
                ("k2m", 39.4),
                ("k2p", 687.2),
                ("k3m", 79300.0),
                ("k3p", 1899.0),
                ("k4m", 40.0),
                ("k4p", 639.0),
                ("zk", 1.0),
                ("GKb", 0.0189),
                ("PNab", 1.9239e-09),
                ("PCab", 5.9194e-8),
                ("GpCa", 0.0005),
                ("Esac_ns", -10.0),
                ("Gsac_k", 1.097904761904762),
                ("Gsac_ns", 0.006),
                ("lambda_max", 1.1),
                ("amp", -80.0),
                ("duration", 0.5),
                ("BSLmax", 1.124),
                ("BSRmax", 0.047),
                ("KmBSL", 0.0087),
                ("KmBSR", 0.00087),
                ("cmdnmax", 0.05),
                ("csqnmax", 10.0),
                ("kmcmdn", 0.00238),
                ("kmcsqn", 0.8),
                ("kmtrpn", 0.0005),
                ("trpnmax", 0.07),
                # Drug factors
                ("scale_drug_INa", 1.0),
                ("scale_drug_INaL", 1.0),
                ("scale_drug_Ito", 1.0),
                ("scale_drug_ICaL", 1.0),
                ("scale_drug_IKr", 1.0),
                ("scale_drug_IKs", 1.0),
                ("scale_drug_IK1", 1.0),
                ("scale_drug_IKb", 1.0),
                ("scale_drug_INab", 1.0),
                ("scale_drug_ICab", 1.0),
                ("scale_drug_IpCa", 1.0),
                ("scale_drug_Isacns", 1.0),
                ("scale_drug_Isack", 1.0),
                ("scale_drug_kws", 1.0),
                ("scale_drug_kuw", 1.0),
                ("scale_drug_kwu", 1.0),
                # Population factors
                ("scale_popu_GNa", 1.0),
                ("scale_popu_GCaL", 1.0),
                ("scale_popu_Gto", 1.0),
                ("scale_popu_GNaL", 1.0),
                ("scale_popu_GKr", 1.0),
                ("scale_popu_GKs", 1.0),
                ("scale_popu_GK1", 1.0),
                ("scale_popu_GbK", 1.0),
                ("scale_popu_KNCX", 1.0),
                ("scale_popu_KNaK", 1.0),
                ("scale_popu_GbNa", 1.0),
                ("scale_popu_GbCa", 1.0),
                ("scale_popu_KpCa", 1.0),
                ("scale_popu_KRyR", 1.0),
                ("scale_popu_KSERCA", 1.0),
                ("scale_popu_Kleak", 1.0),
                ("scale_popu_nTm", 1.0),
                ("scale_popu_CaT50ref", 1.0),
                ("scale_popu_Tref", 1.0),
                ("scale_popu_kuw", 1.0),
                ("scale_popu_kws", 1.0),
                ("scale_popu_kTRPN", 1.0),
                ("scale_popu_nTRPN", 1.0),
                ("scale_popu_ku", 1.0),
                ("scale_popu_TRPN50", 1.0),
                ("scale_popu_rw", 1.0),
                ("scale_popu_rs", 1.0),
                # HF scaling factors
                ("HF_scaling_CaMKa", 1.0),
                ("HF_scaling_Jrel_inf", 1.0),
                ("HF_scaling_Jleak", 1.0),
                ("HF_scaling_Jup", 1.0),
                ("HF_scaling_GNaL", 1.0),
                ("HF_scaling_GK1", 1.0),
                ("HF_scaling_thL", 1.0),
                ("HF_scaling_Gto", 1.0),
                ("HF_scaling_Gncx", 1.0),
                ("HF_scaling_Pnak", 1.0),
                ("HF_scaling_cat50_ref", 1.0),
            ],
        )

        if disease_state.lower() == "hf":
            logger.info("Update scaling parameters for heart failure model")
            params["HF_scaling_CaMKa"] = 1.50
            params["HF_scaling_Jrel_inf"] = pow(0.8, 8.0)
            params["HF_scaling_Jleak"] = 1.3
            params["HF_scaling_Jup"] = 0.45
            params["HF_scaling_GNaL"] = 1.3
            params["HF_scaling_GK1"] = 0.68
            params["HF_scaling_thL"] = 1.8
            params["HF_scaling_Gto"] = 0.4
            params["HF_scaling_Gncx"] = 1.6
            params["HF_scaling_Pnak"] = 0.7
            params["HF_scaling_cat50_ref"] = 0.6

        return params

    @staticmethod
    def default_initial_conditions():
        "Set-up and return default initial conditions."
        ic = OrderedDict(
            [
                ("v", -88.869),
                ("CaMKt", 0.011075),
                ("m", 7.8766e-4),
                ("h", 0.83066),
                ("j", 0.83047),
                ("hp", 0.67410),
                ("jp", 0.83009),
                ("mL", 1.5967e-4),
                ("hL", 0.52826),
                ("hLp", 0.28878),
                ("a", 9.4425e-4),
                ("iF", 0.99962),
                ("iS", 0.59368),
                ("ap", 4.8111e-4),
                ("iFp", 0.99962),
                ("iSp", 0.65409),
                ("d", 0.0),
                ("ff", 1.0),
                ("fs", 0.93897),
                ("fcaf", 1.0),
                ("fcas", 0.99990),
                ("jca", 0.99998),
                ("ffp", 1.0),
                ("fcafp", 1.0),
                ("nca", 4.9209e-4),
                ("nca_i", 8.3371e-04),
                ("kr_c0", 0.99807),
                ("kr_c1", 8.4475e-4),
                ("kr_c2", 6.9817e-4),
                ("kr_o", 3.7040e-4),
                ("kr_i", 1.3024e-5),
                ("xs1", 0.24716),
                ("xs2", 1.7502e-4),
                ("Jrelnp", 0.0),
                ("Jrelp", 0.0),
                ("ki", 1.4241e2),
                ("kss", 1.4241e2),
                ("cass", 6.5042e-5),
                ("cansr", 1.5304),
                ("cajsr", 1.5280),
                ("XS", 0),
                ("XW", 0),
                ("CaTrpn", 0),
                ("TmB", 1),
                ("Cd", 0),
                ("cai", 7.4554e-5),
                ("nai", 12.100),
                ("nass", 12.100),
            ],
        )
        return ic

    def _I(self, v, s, time):
        """
        Original gotran transmembrane current dV/dt
        """
        time = time if time else Constant(0.0)
        logger.debug("Evaluate transmembrane current")
        # Assign states
        assert len(s) == 47
        (
            CaMKt,
            m,
            h,
            j,
            hp,
            jp,
            mL,
            hL,
            hLp,
            a,
            iF,
            iS,
            ap,
            iFp,
            iSp,
            d,
            ff,
            fs,
            fcaf,
            fcas,
            jca,
            ffp,
            fcafp,
            nca,
            nca_i,
            kr_c0,
            kr_c1,
            kr_c2,
            kr_o,
            kr_i,
            xs1,
            xs2,
            Jrelnp,
            Jrelp,
            ki,
            kss,
            cass,
            cansr,
            cajsr,
            XS,
            XW,
            CaTrpn,
            TmB,
            Cd,
            cai,
            nai,
            nass,
        ) = s

        # Assign parameters
        ICaL_fractionSS = self._parameters["ICaL_fractionSS"]
        INaCa_fractionSS = self._parameters["INaCa_fractionSS"]
        scale_ICaL = self._parameters["scale_ICaL"]
        scale_IK1 = self._parameters["scale_IK1"]
        scale_IKr = self._parameters["scale_IKr"]
        scale_IKs = self._parameters["scale_IKs"]
        scale_INaL = self._parameters["scale_INaL"]
        cao = self._parameters["cao"]
        ko = self._parameters["ko"]
        nao = self._parameters["nao"]
        F = self._parameters["F"]
        R = self._parameters["R"]
        T = self._parameters["T"]
        CaMKo = self._parameters["CaMKo"]
        KmCaM = self._parameters["KmCaM"]
        KmCaMK = self._parameters["KmCaMK"]
        PKNa = self._parameters["PKNa"]
        GNa = self._parameters["GNa"]
        Gto = self._parameters["Gto"]
        Aff = self._parameters["Aff"]
        zca = self._parameters["zca"]
        Gncx = self._parameters["Gncx"]
        KmCaAct = self._parameters["KmCaAct"]
        kasymm = self._parameters["kasymm"]
        kcaoff = self._parameters["kcaoff"]
        kcaon = self._parameters["kcaon"]
        kna1 = self._parameters["kna1"]
        kna2 = self._parameters["kna2"]
        kna3 = self._parameters["kna3"]
        qca = self._parameters["qca"]
        qna = self._parameters["qna"]
        wca = self._parameters["wca"]
        wna = self._parameters["wna"]
        wnaca = self._parameters["wnaca"]
        H = self._parameters["H"]
        Khp = self._parameters["Khp"]
        Kki = self._parameters["Kki"]
        Kko = self._parameters["Kko"]
        Kmgatp = self._parameters["Kmgatp"]
        Knai0 = self._parameters["Knai0"]
        Knao0 = self._parameters["Knao0"]
        Knap = self._parameters["Knap"]
        Kxkur = self._parameters["Kxkur"]
        MgADP = self._parameters["MgADP"]
        MgATP = self._parameters["MgATP"]
        Pnak = self._parameters["Pnak"]
        delta = self._parameters["delta"]
        eP = self._parameters["eP"]
        k1m = self._parameters["k1m"]
        k1p = self._parameters["k1p"]
        k2m = self._parameters["k2m"]
        k2p = self._parameters["k2p"]
        k3m = self._parameters["k3m"]
        k3p = self._parameters["k3p"]
        k4m = self._parameters["k4m"]
        k4p = self._parameters["k4p"]
        zk = self._parameters["zk"]
        GKb = self._parameters["GKb"]
        PNab = self._parameters["PNab"]
        PCab = self._parameters["PCab"]
        GpCa = self._parameters["GpCa"]
        # Drug factor
        scale_drug_INa = self._parameters["scale_drug_INa"]
        scale_drug_INaL = self._parameters["scale_drug_INaL"]
        scale_drug_Ito = self._parameters["scale_drug_Ito"]
        scale_drug_ICaL = self._parameters["scale_drug_ICaL"]
        scale_drug_IKr = self._parameters["scale_drug_IKr"]
        scale_drug_IKs = self._parameters["scale_drug_IKs"]
        scale_drug_IK1 = self._parameters["scale_drug_IK1"]
        scale_drug_IKb = self._parameters["scale_drug_IKb"]
        scale_drug_INab = self._parameters["scale_drug_INab"]
        scale_drug_ICab = self._parameters["scale_drug_ICab"]
        scale_drug_IpCa = self._parameters["scale_drug_IpCa"]
        scale_drug_Isacns = self._parameters["scale_drug_Isacns"]
        scale_drug_Isack = self._parameters["scale_drug_Isack"]
        # Population factors
        scale_popu_GNa = self._parameters["scale_popu_GNa"]
        scale_popu_GCaL = self._parameters["scale_popu_GCaL"]
        scale_popu_Gto = self._parameters["scale_popu_Gto"]
        scale_popu_GNaL = self._parameters["scale_popu_GNaL"]
        scale_popu_GKr = self._parameters["scale_popu_GKr"]
        scale_popu_GKs = self._parameters["scale_popu_GKs"]
        scale_popu_GK1 = self._parameters["scale_popu_GK1"]
        scale_popu_GbK = self._parameters["scale_popu_GbK"]
        scale_popu_KNCX = self._parameters["scale_popu_KNCX"]
        scale_popu_KNaK = self._parameters["scale_popu_KNaK"]
        scale_popu_GbNa = self._parameters["scale_popu_GbNa"]
        scale_popu_GbCa = self._parameters["scale_popu_GbCa"]
        scale_popu_KpCa = self._parameters["scale_popu_KpCa"]
        # Systolic Heart Failure (HF with preserved ejection fraction)
        HF_scaling_CaMKa = self._parameters["HF_scaling_CaMKa"]
        HF_scaling_GNaL = self._parameters["HF_scaling_GNaL"]
        HF_scaling_GK1 = self._parameters["HF_scaling_GK1"]
        HF_scaling_Gto = self._parameters["HF_scaling_Gto"]
        HF_scaling_Gncx = self._parameters["HF_scaling_Gncx"]
        HF_scaling_Pnak = self._parameters["HF_scaling_Pnak"]

        # Init return args
        current = [ufl.zero()] * 1

        # Expressions for the CaMKt component
        CaMKb = CaMKo * (1.0 - CaMKt) / (1.0 + KmCaM / cass)
        CaMKa = (CaMKb + CaMKt) * HF_scaling_CaMKa

        # Expressions for the reversal potentials component
        ENa = R * T * ufl.ln(nao / nai) / F
        EK = R * T * ufl.ln(ko / ki) / F
        EKs = R * T * ufl.ln((ko + PKNa * nao) / (PKNa * nai + ki)) / F
        vffrt = (F * F) * v / (R * T)
        vfrt = F * v / (R * T)

        # Expressions for the I_Na component
        fINap = 1.0 / (1.0 + KmCaMK / CaMKa)
        INa = (
            GNa
            * scale_drug_INa
            * scale_popu_GNa
            * ufl.elem_pow(m, 3.0)
            * (-ENa + v)
            * ((1.0 - fINap) * h * j + fINap * hp * jp)
        )

        # Expressions for the INaL component
        GNaL = 0.0075 * scale_INaL * scale_drug_INaL * scale_popu_GNaL * HF_scaling_GNaL
        fINaLp = 1.0 / (1.0 + KmCaMK / CaMKa)
        INaL = (-ENa + v) * ((1.0 - fINaLp) * hL + fINaLp * hLp) * GNaL * mL

        # Expressions for the Ito component
        AiF = 1.0 / (1.0 + 0.24348537187522867 * ufl.exp(0.006613756613756614 * v))
        AiS = 1.0 - AiF
        i = AiF * iF + AiS * iS
        ip = AiF * iFp + AiS * iSp
        fItop = 1.0 / (1.0 + KmCaMK / CaMKa)
        Ito = (
            Gto
            * scale_drug_Ito
            * scale_popu_Gto
            * HF_scaling_Gto
            * (-EK + v)
            * ((1.0 - fItop) * a * i + ap * fItop * ip)
        )

        # Expressions for the ICaL ICaNa ICaK component
        Afs = 1.0 - Aff
        f = Aff * ff + Afs * fs
        Afcaf = 0.3 + 0.6 / (1.0 + 0.36787944117144233 * ufl.exp(0.1 * v))
        Afcas = 1.0 - Afcaf
        fca = Afcaf * fcaf + Afcas * fcas
        fp = Aff * ffp + Afs * fs
        fcap = Afcaf * fcafp + Afcas * fcas

        clo = 150
        cli = 24
        Io = 0.5 * (nao + ko + clo + 4.0 * cao) / 1000.0
        Ii = 0.5 * (nass + kss + cli + 4.0 * cass) / 1000.0
        dielConstant = 74
        constA = 1.82 * 1000000.0 * ufl.elem_pow(dielConstant * T, -1.5)

        gamma_cai = ufl.exp(
            -constA * 4.0 * (ufl.sqrt(Ii) / (1.0 + ufl.sqrt(Ii)) - 0.3 * Ii),
        )
        gamma_cao = ufl.exp(
            -constA * 4.0 * (ufl.sqrt(Io) / (1.0 + ufl.sqrt(Io)) - 0.3 * Io),
        )
        gamma_nai = ufl.exp(
            -constA * 1.0 * (ufl.sqrt(Ii) / (1.0 + ufl.sqrt(Ii)) - 0.3 * Ii),
        )
        gamma_nao = ufl.exp(
            -constA * 1.0 * (ufl.sqrt(Io) / (1.0 + ufl.sqrt(Io)) - 0.3 * Io),
        )
        gamma_ki = ufl.exp(
            -constA * 1.0 * (ufl.sqrt(Ii) / (1.0 + ufl.sqrt(Ii)) - 0.3 * Ii),
        )
        gamma_kao = ufl.exp(
            -constA * 1.0 * (ufl.sqrt(Io) / (1.0 + ufl.sqrt(Io)) - 0.3 * Io),
        )

        PhiCaL_ss = (
            4.0
            * vffrt
            * (gamma_cai * cass * ufl.exp(2.0 * vfrt) - gamma_cao * cao)
            / (-1.0 + ufl.exp(2.0 * vfrt))
        )
        PhiCaNa_ss = (
            1.0
            * vffrt
            * (gamma_nai * nass * ufl.exp(1.0 * vfrt) - gamma_nao * nao)
            / (-1.0 + ufl.exp(1.0 * vfrt))
        )
        PhiCaK_ss = (
            1.0
            * vffrt
            * (gamma_ki * kss * ufl.exp(1.0 * vfrt) - gamma_kao * ko)
            / (-1.0 + ufl.exp(1.0 * vfrt))
        )
        PhiCaL_i = (
            4.0
            * vffrt
            * (gamma_cai * cai * ufl.exp(2.0 * vfrt) - gamma_cao * cao)
            / (-1.0 + ufl.exp(2.0 * vfrt))
        )
        PhiCaNa_i = (
            1.0
            * vffrt
            * (gamma_nai * nai * ufl.exp(1.0 * vfrt) - gamma_nao * nao)
            / (-1.0 + ufl.exp(1.0 * vfrt))
        )
        PhiCaK_i = (
            1.0
            * vffrt
            * (gamma_ki * ki * ufl.exp(1.0 * vfrt) - gamma_kao * ko)
            / (-1.0 + ufl.exp(1.0 * vfrt))
        )
        PCa = 0.0001 * scale_ICaL * scale_drug_ICaL * scale_popu_GCaL
        PCap = 1.1 * PCa
        PCaNa = 0.00125 * PCa
        PCaK = 0.0003574 * PCa
        PCaNap = 0.00125 * PCap
        PCaKp = 0.0003574 * PCap
        fICaLp = 1.0 / (1.0 + KmCaMK / CaMKa)

        ICaL_tot_ss = (1.0 - fICaLp) * (
            (1.0 - nca) * f + fca * jca * nca
        ) * PCa * PhiCaL_ss * d + (
            (1.0 - nca) * fp + fcap * jca * nca
        ) * PCap * PhiCaL_ss * d * fICaLp
        ICaNa_tot_ss = (1.0 - fICaLp) * (
            (1.0 - nca) * f + fca * jca * nca
        ) * PCaNa * PhiCaNa_ss * d + (
            (1.0 - nca) * fp + fcap * jca * nca
        ) * PCaNap * PhiCaNa_ss * d * fICaLp
        ICaK_tot_ss = (1.0 - fICaLp) * (
            (1.0 - nca) * f + fca * jca * nca
        ) * PCaK * PhiCaK_ss * d + (
            (1.0 - nca) * fp + fcap * jca * nca
        ) * PCaKp * PhiCaK_ss * d * fICaLp

        ICaL_tot_i = (1.0 - fICaLp) * (
            (1.0 - nca) * f + fca * jca * nca
        ) * PCa * PhiCaL_i * d + (
            (1.0 - nca) * fp + fcap * jca * nca
        ) * PCap * PhiCaL_i * d * fICaLp
        ICaNa_tot_i = (1.0 - fICaLp) * (
            (1.0 - nca) * f + fca * jca * nca
        ) * PCaNa * PhiCaNa_i * d + (
            (1.0 - nca) * fp + fcap * jca * nca
        ) * PCaNap * PhiCaNa_i * d * fICaLp
        ICaK_tot_i = (1.0 - fICaLp) * (
            (1.0 - nca) * f + fca * jca * nca
        ) * PCaK * PhiCaK_i * d + (
            (1.0 - nca) * fp + fcap * jca * nca
        ) * PCaKp * PhiCaK_i * d * fICaLp

        ICaL_ss = ICaL_tot_ss * ICaL_fractionSS
        ICaNa_ss = ICaNa_tot_ss * ICaL_fractionSS
        ICaK_ss = ICaK_tot_ss * ICaL_fractionSS
        ICaL_i = ICaL_tot_i * (1.0 - ICaL_fractionSS)
        ICaNa_i = ICaNa_tot_i * (1.0 - ICaL_fractionSS)
        ICaK_i = ICaK_tot_i * (1.0 - ICaL_fractionSS)

        ICaL = ICaL_ss + ICaL_i
        ICaNa = ICaNa_ss + ICaNa_i
        ICaK = ICaK_ss + ICaK_i

        # Expressions for the IKr component
        GKr = 0.0321 * scale_IKr * scale_drug_IKr * scale_popu_GKr
        IKr = GKr * ufl.sqrt(ko / 5.0) * kr_o * (-EK + v)

        # Expressions for the IKs component
        KsCa = 1.0 + 0.6 / (1.0 + 6.481821026062645e-07 * ufl.elem_pow(1.0 / cai, 1.4))
        GKs = 0.0011 * scale_IKs * scale_drug_IKs * scale_popu_GKs
        IKs = (-EKs + v) * GKs * KsCa * xs1 * xs2

        # Expressions for the IK1 components
        aK1 = 4.094 / (1.0 + ufl.exp(0.1217 * (v - EK - 49.934)))
        bK1 = (
            15.72 * ufl.exp(0.0674 * (v - EK - 3.257))
            + ufl.exp(0.0618 * (v - EK - 594.31))
        ) / (1.0 + ufl.exp(-0.1629 * (v - EK + 14.207)))
        K1ss = aK1 / (aK1 + bK1)
        GK1 = 0.6992 * scale_IK1 * scale_drug_IK1 * scale_popu_GK1 * HF_scaling_GK1
        IK1 = ufl.sqrt(ko) * (-EK + v) * GK1 * GK1 * K1ss

        # Expressions for the INaCa_i component
        hca = ufl.exp(F * qca * v / (R * T))
        hna = ufl.exp(F * qna * v / (R * T))
        h1_i = 1 + (1 + hna) * nai / kna3
        h2_i = hna * nai / (kna3 * h1_i)
        h3_i = 1.0 / h1_i
        h4_i = 1.0 + (1 + nai / kna2) * nai / kna1
        h5_i = (nai * nai) / (kna1 * kna2 * h4_i)
        h6_i = 1.0 / h4_i
        h7_i = 1.0 + nao * (1.0 + 1.0 / hna) / kna3
        h8_i = nao / (kna3 * h7_i * hna)
        h9_i = 1.0 / h7_i
        h10_i = 1.0 + kasymm + nao * (1.0 + nao / kna2) / kna1
        h11_i = (nao * nao) / (kna1 * kna2 * h10_i)
        h12_i = 1.0 / h10_i
        k1_i = cao * kcaon * h12_i
        k2_i = kcaoff
        k3p_i = wca * h9_i
        k3pp_i = wnaca * h8_i
        k3_i = k3p_i + k3pp_i
        k4p_i = wca * h3_i / hca
        k4pp_i = wnaca * h2_i
        k4_i = k4p_i + k4pp_i
        k5_i = kcaoff
        k6_i = kcaon * cai * h6_i
        k7_i = wna * h2_i * h5_i
        k8_i = wna * h11_i * h8_i
        x1_i = (k2_i + k3_i) * k5_i * k7_i + (k6_i + k7_i) * k2_i * k4_i
        x2_i = (k1_i + k8_i) * k4_i * k6_i + (k4_i + k5_i) * k1_i * k7_i
        x3_i = (k2_i + k3_i) * k6_i * k8_i + (k6_i + k7_i) * k1_i * k3_i
        x4_i = (k1_i + k8_i) * k3_i * k5_i + (k4_i + k5_i) * k2_i * k8_i
        E1_i = x1_i / (x1_i + x2_i + x3_i + x4_i)
        E2_i = x2_i / (x1_i + x2_i + x3_i + x4_i)
        E3_i = x3_i / (x1_i + x2_i + x3_i + x4_i)
        E4_i = x4_i / (x1_i + x2_i + x3_i + x4_i)
        allo_i = 1.0 / (1.0 + ufl.elem_pow(KmCaAct / cai, 2.0))
        zna = 1.0
        JncxNa_i = E3_i * k4pp_i - E2_i * k3pp_i + 3.0 * E4_i * k7_i - 3.0 * E1_i * k8_i
        JncxCa_i = E2_i * k2_i - E1_i * k1_i
        INaCa_i = (
            Gncx
            * (1.0 - INaCa_fractionSS)
            * scale_popu_KNCX
            * HF_scaling_Gncx
            * (zca * JncxCa_i + zna * JncxNa_i)
            * allo_i
        )

        # Expressions for the INaCa_ss component
        h1 = 1.0 + (1.0 + hna) * nass / kna3
        h2 = hna * nass / (kna3 * h1)
        h3 = 1.0 / h1
        h4 = 1.0 + (1 + nass / kna2) * nass / kna1
        h5 = (nass * nass) / (kna1 * kna2 * h4)
        h6 = 1.0 / h4
        h7 = 1.0 + nao * (1.0 + 1.0 / hna) / kna3
        h8 = nao / (kna3 * h7 * hna)
        h9 = 1.0 / h7
        h10 = 1.0 + kasymm + nao * (1 + nao / kna2) / kna1
        h11 = (nao * nao) / (kna1 * kna2 * h10)
        h12 = 1.0 / h10
        k1 = cao * kcaon * h12
        k2 = kcaoff
        k3p_ss = wca * h9
        k3pp = wnaca * h8
        k3 = k3p_ss + k3pp
        k4p_ss = wca * h3 / hca
        k4pp = wnaca * h2
        k4 = k4p_ss + k4pp
        k5 = kcaoff
        k6 = kcaon * cass * h6
        k7 = wna * h2 * h5
        k8 = wna * h11 * h8
        x1_ss = (k2 + k3) * k5 * k7 + (k6 + k7) * k2 * k4
        x2_ss = (k1 + k8) * k4 * k6 + (k4 + k5) * k1 * k7
        x3_ss = (k2 + k3) * k6 * k8 + (k6 + k7) * k1 * k3
        x4_ss = (k1 + k8) * k3 * k5 + (k4 + k5) * k2 * k8
        E1_ss = x1_ss / (x1_ss + x2_ss + x3_ss + x4_ss)
        E2_ss = x2_ss / (x1_ss + x2_ss + x3_ss + x4_ss)
        E3_ss = x3_ss / (x1_ss + x2_ss + x3_ss + x4_ss)
        E4_ss = x4_ss / (x1_ss + x2_ss + x3_ss + x4_ss)
        allo_ss = 1.0 / (1.0 + ufl.elem_pow(KmCaAct / cass, 2.0))
        JncxNa_ss = E3_ss * k4pp - E2_ss * k3pp + 3.0 * E4_ss * k7 - 3.0 * E1_ss * k8
        JncxCa_ss = E2_ss * k2 - E1_ss * k1
        INaCa_ss = (
            Gncx
            * INaCa_fractionSS
            * scale_popu_KNCX
            * HF_scaling_Gncx
            * (zca * JncxCa_ss + zna * JncxNa_ss)
            * allo_ss
        )

        # Expressions for the INaK component
        Knai = Knai0 * ufl.exp(0.3333333333333333 * F * delta * v / (R * T))
        Knao = Knao0 * ufl.exp(0.3333333333333333 * F * (1.0 - delta) * v / (R * T))
        P = eP / (1.0 + H / Khp + nai / Knap + ki / Kxkur)
        a1 = (
            k1p
            * ufl.elem_pow(nai / Knai, 3.0)
            / (
                -1.0
                + ufl.elem_pow(1.0 + ki / Kki, 2.0)
                + ufl.elem_pow(1.0 + nai / Knai, 3.0)
            )
        )
        b1 = MgADP * k1m
        a2 = k2p
        b2 = (
            k2m
            * ufl.elem_pow(nao / Knao, 3.0)
            / (
                -1.0
                + ufl.elem_pow(1.0 + ko / Kko, 2.0)
                + ufl.elem_pow(1.0 + nao / Knao, 3.0)
            )
        )
        a3 = (
            k3p
            * ufl.elem_pow(ko / Kko, 2.0)
            / (
                -1.0
                + ufl.elem_pow(1.0 + ko / Kko, 2.0)
                + ufl.elem_pow(1.0 + nao / Knao, 3.0)
            )
        )
        b3 = H * k3m * P / (1.0 + MgATP / Kmgatp)
        a4 = MgATP * k4p / (Kmgatp * (1.0 + MgATP / Kmgatp))
        b4 = (
            k4m
            * ufl.elem_pow(ki / Kki, 2.0)
            / (
                -1.0
                + ufl.elem_pow(1.0 + ki / Kki, 2.0)
                + ufl.elem_pow(1.0 + nai / Knai, 3.0)
            )
        )
        x1 = a1 * a2 * a4 + a1 * a2 * b3 + a2 * b3 * b4 + b2 * b3 * b4
        x2 = a1 * a2 * a3 + a2 * a3 * b4 + a3 * b1 * b4 + b1 * b2 * b4
        x3 = a2 * a3 * a4 + a3 * a4 * b1 + a4 * b1 * b2 + b1 * b2 * b3
        x4 = a1 * a3 * a4 + a1 * a4 * b2 + a1 * b2 * b3 + b2 * b3 * b4
        E1 = x1 / (x1 + x2 + x3 + x4)
        E2 = x2 / (x1 + x2 + x3 + x4)
        E3 = x3 / (x1 + x2 + x3 + x4)
        E4 = x4 / (x1 + x2 + x3 + x4)
        JnakNa = 3.0 * E1 * a3 - 3.0 * E2 * b3
        JnakK = 2.0 * E4 * b1 - 2.0 * E3 * a1
        INaK = Pnak * scale_popu_KNaK * HF_scaling_Pnak * (zk * JnakK + zna * JnakNa)

        # Expressions for the IKb component
        xkb = 1.0 / (1.0 + 2.202363450949239 * ufl.exp(-0.05452562704471101 * v))
        IKb = GKb * scale_popu_GbK * scale_drug_IKb * (-EK + v) * xkb

        # Expressions for the INab component
        INab = (
            PNab
            * scale_drug_INab
            * scale_popu_GbNa
            * (-nao + ufl.exp(vfrt) * nai)
            * vffrt
            / (-1.0 + ufl.exp(vfrt))
        )

        # Expressions for the ICab component
        gammaCaoMyo = gamma_cao
        gammaCaiMyo = gamma_cai

        ICab = (
            4.0
            * PCab
            * scale_drug_ICab
            * scale_popu_GbCa
            * (cai * gammaCaiMyo * ufl.exp(2.0 * vfrt) - cao * gammaCaoMyo)
            * vffrt
            / (-1.0 + ufl.exp(2.0 * vfrt))
        )

        # Expressions for the ICaCl component (calcium activated chloride current)
        ecl = R * T * ufl.ln(cli / clo) / F
        Fjunc = 1.0
        Fsl = 1.0 - Fjunc
        GClCa = 0.2843
        KdClCa = 0.1

        I_ClCa_junc = Fjunc * GClCa / (1.0 + KdClCa / cass) * (v - ecl)
        I_ClCa_sl = Fsl * GClCa / (1.0 + KdClCa / cai) * (v - ecl)
        I_ClCa = I_ClCa_junc + I_ClCa_sl

        # Expressions for the IClb component
        GClB = 0.00198
        I_Clbk = GClB * (v - ecl)

        # Expressions for the IpCa component
        IpCa = GpCa * scale_drug_IpCa * scale_popu_KpCa * cai / (0.0005 + cai)

        # Expressions for the Isac (Pueyo)--> ns + k component
        Gsac_ns = 0.006 * scale_drug_Isacns
        Esac_ns = -10.0
        lambda_max = 1.1
        Gsac_k = (0.2882 * 800.0 / 210.0) * scale_drug_Isack  # Pueyo endo

        Isac_P_ns = ufl.conditional(
            ufl.lt(self.lmbda, 1.0),
            0.0,
            Gsac_ns * ((self.lmbda - 1.0) / (lambda_max - 1.0)) * (v - Esac_ns),
        )
        Isac_P_k = ufl.conditional(
            ufl.lt(self.lmbda, 1.0),
            0.0,
            Gsac_k
            * ((self.lmbda - 1.0) / (lambda_max - 1.0))
            * (1.0 / (1.0 + ufl.exp((19.05 - v) / (29.98)))),
        )

        # Expressions for the Istim component
        Istim = 0  # amp*(ufl.le(time, duration))

        # Expressions for the membrane potential component
        current[0] = (
            -Isac_P_k
            - Isac_P_ns
            - ICaK
            - ICaL
            - ICaNa
            - ICab
            - IK1
            - IKb
            - IKr
            - IKs
            - INa
            - INaCa_i
            - INaCa_ss
            - INaK
            - INaL
            - INab
            - IpCa
            - Istim
            - Ito
            - I_ClCa
            - I_Clbk
        )

        # Return results
        return current[0]

    def I(self, v, s, time=None):  # noqa: E741, E743
        """
        Transmembrane current

           I = -dV/dt

        """
        return -self._I(v, s, time)

    def F(self, v, s, time=None):
        """
        Right hand side for ODE system
        """
        logger.debug("Evaluate RHS")
        time = time if time else Constant(0.0)

        # Assign states
        assert len(s) == 47
        (
            CaMKt,
            m,
            h,
            j,
            hp,
            jp,
            mL,
            hL,
            hLp,
            a,
            iF,
            iS,
            ap,
            iFp,
            iSp,
            d,
            ff,
            fs,
            fcaf,
            fcas,
            jca,
            ffp,
            fcafp,
            nca,
            nca_i,
            kr_c0,
            kr_c1,
            kr_c2,
            kr_o,
            kr_i,
            xs1,
            xs2,
            Jrelnp,
            Jrelp,
            ki,
            kss,
            cass,
            cansr,
            cajsr,
            XS,
            XW,
            CaTrpn,
            TmB,
            Cd,
            cai,
            nai,
            nass,
        ) = s

        # Assign parameters
        ICaL_fractionSS = self._parameters["ICaL_fractionSS"]
        INaCa_fractionSS = self._parameters["INaCa_fractionSS"]
        scale_ICaL = self._parameters["scale_ICaL"]
        scale_IK1 = self._parameters["scale_IK1"]
        scale_IKr = self._parameters["scale_IKr"]
        scale_IKs = self._parameters["scale_IKs"]
        scale_INaL = self._parameters["scale_INaL"]
        cao = self._parameters["cao"]
        ko = self._parameters["ko"]
        nao = self._parameters["nao"]
        F = self._parameters["F"]
        R = self._parameters["R"]
        T = self._parameters["T"]
        L = self._parameters["L"]
        rad = self._parameters["rad"]
        CaMKo = self._parameters["CaMKo"]
        KmCaM = self._parameters["KmCaM"]
        KmCaMK = self._parameters["KmCaMK"]
        aCaMK = self._parameters["aCaMK"]
        bCaMK = self._parameters["bCaMK"]
        PKNa = self._parameters["PKNa"]
        # Ahf = self._parameters["Ahf"]
        GNa = self._parameters["GNa"]
        thL = self._parameters["thL"]
        Gto = self._parameters["Gto"]
        delta_epi = self._parameters["delta_epi"]
        Aff = self._parameters["Aff"]
        Kmn = self._parameters["Kmn"]
        k2n = self._parameters["k2n"]
        tjca = self._parameters["tjca"]
        zca = self._parameters["zca"]
        Gncx = self._parameters["Gncx"]
        KmCaAct = self._parameters["KmCaAct"]
        kasymm = self._parameters["kasymm"]
        kcaoff = self._parameters["kcaoff"]
        kcaon = self._parameters["kcaon"]
        kna1 = self._parameters["kna1"]
        kna2 = self._parameters["kna2"]
        kna3 = self._parameters["kna3"]
        qca = self._parameters["qca"]
        qna = self._parameters["qna"]
        wca = self._parameters["wca"]
        wna = self._parameters["wna"]
        wnaca = self._parameters["wnaca"]
        H = self._parameters["H"]
        Khp = self._parameters["Khp"]
        Kki = self._parameters["Kki"]
        Kko = self._parameters["Kko"]
        Kmgatp = self._parameters["Kmgatp"]
        Knai0 = self._parameters["Knai0"]
        Knao0 = self._parameters["Knao0"]
        Knap = self._parameters["Knap"]
        Kxkur = self._parameters["Kxkur"]
        MgADP = self._parameters["MgADP"]
        MgATP = self._parameters["MgATP"]
        Pnak = self._parameters["Pnak"]
        delta = self._parameters["delta"]
        eP = self._parameters["eP"]
        k1m = self._parameters["k1m"]
        k1p = self._parameters["k1p"]
        k2m = self._parameters["k2m"]
        k2p = self._parameters["k2p"]
        k3m = self._parameters["k3m"]
        k3p = self._parameters["k3p"]
        k4m = self._parameters["k4m"]
        k4p = self._parameters["k4p"]
        zk = self._parameters["zk"]
        GKb = self._parameters["GKb"]
        PNab = self._parameters["PNab"]
        PCab = self._parameters["PCab"]
        GpCa = self._parameters["GpCa"]
        bt = self._parameters["bt"]
        BSLmax = self._parameters["BSLmax"]
        BSRmax = self._parameters["BSRmax"]
        KmBSL = self._parameters["KmBSL"]
        KmBSR = self._parameters["KmBSR"]
        cmdnmax = self._parameters["cmdnmax"]
        csqnmax = self._parameters["csqnmax"]
        kmcmdn = self._parameters["kmcmdn"]
        kmcsqn = self._parameters["kmcsqn"]
        trpnmax = self._parameters["trpnmax"]
        Beta1 = self._parameters["Beta1"]

        Trpn50 = self._parameters["Trpn50"]
        cat50_ref = self._parameters["cat50_ref"]

        etal = self._parameters["etal"]
        etas = self._parameters["etas"]
        gammas = self._parameters["gammas"]
        gammaw = self._parameters["gammaw"]
        ktrpn = self._parameters["ktrpn"]
        ku = self._parameters["ku"]
        kuw = self._parameters["kuw"]
        kws = self._parameters["kws"]
        # lmbda = self._parameters["lmbda"]
        ntm = self._parameters["ntm"]
        ntrpn = self._parameters["ntrpn"]
        p_k = self._parameters["p_k"]

        rs = self._parameters["rs"]
        rw = self._parameters["rw"]

        # Drug factor
        scale_drug_INa = self._parameters["scale_drug_INa"]
        scale_drug_INaL = self._parameters["scale_drug_INaL"]
        scale_drug_Ito = self._parameters["scale_drug_Ito"]
        scale_drug_ICaL = self._parameters["scale_drug_ICaL"]
        scale_drug_IKr = self._parameters["scale_drug_IKr"]
        scale_drug_IKs = self._parameters["scale_drug_IKs"]
        scale_drug_IK1 = self._parameters["scale_drug_IK1"]
        scale_drug_IKb = self._parameters["scale_drug_IKb"]
        scale_drug_INab = self._parameters["scale_drug_INab"]
        scale_drug_ICab = self._parameters["scale_drug_ICab"]
        scale_drug_IpCa = self._parameters["scale_drug_IpCa"]
        scale_drug_Isacns = self._parameters["scale_drug_Isacns"]
        scale_drug_Isack = self._parameters["scale_drug_Isack"]
        scale_drug_kws = self._parameters["scale_drug_kws"]
        scale_drug_kuw = self._parameters["scale_drug_kuw"]
        scale_drug_kwu = self._parameters["scale_drug_kwu"]

        # Population factors
        scale_popu_GNa = self._parameters["scale_popu_GNa"]
        scale_popu_GCaL = self._parameters["scale_popu_GCaL"]
        scale_popu_Gto = self._parameters["scale_popu_Gto"]
        scale_popu_GNaL = self._parameters["scale_popu_GNaL"]
        scale_popu_GKr = self._parameters["scale_popu_GKr"]
        scale_popu_GKs = self._parameters["scale_popu_GKs"]
        scale_popu_GK1 = self._parameters["scale_popu_GK1"]
        scale_popu_GbK = self._parameters["scale_popu_GbK"]
        scale_popu_KNCX = self._parameters["scale_popu_KNCX"]
        scale_popu_KNaK = self._parameters["scale_popu_KNaK"]
        scale_popu_GbNa = self._parameters["scale_popu_GbNa"]
        scale_popu_GbCa = self._parameters["scale_popu_GbCa"]
        scale_popu_KpCa = self._parameters["scale_popu_KpCa"]
        scale_popu_KRyR = self._parameters["scale_popu_KRyR"]
        scale_popu_KSERCA = self._parameters["scale_popu_KSERCA"]
        scale_popu_Kleak = self._parameters["scale_popu_Kleak"]
        scale_popu_nTm = self._parameters["scale_popu_nTm"]
        scale_popu_CaT50ref = self._parameters["scale_popu_CaT50ref"]
        scale_popu_kuw = self._parameters["scale_popu_kuw"]
        scale_popu_kws = self._parameters["scale_popu_kws"]
        scale_popu_kTRPN = self._parameters["scale_popu_kTRPN"]
        scale_popu_nTRPN = self._parameters["scale_popu_nTRPN"]
        scale_popu_ku = self._parameters["scale_popu_ku"]
        scale_popu_TRPN50 = self._parameters["scale_popu_TRPN50"]
        scale_popu_rw = self._parameters["scale_popu_rw"]
        scale_popu_rs = self._parameters["scale_popu_rs"]

        # Systolic Heart Failure (HF with preserved ejection fraction)
        HF_scaling_CaMKa = self._parameters["HF_scaling_CaMKa"]
        HF_scaling_Jrel_inf = self._parameters["HF_scaling_Jrel_inf"]
        HF_scaling_Jleak = self._parameters["HF_scaling_Jleak"]
        HF_scaling_Jup = self._parameters["HF_scaling_Jup"]
        HF_scaling_GNaL = self._parameters["HF_scaling_GNaL"]
        HF_scaling_GK1 = self._parameters["HF_scaling_GK1"]
        HF_scaling_thL = self._parameters["HF_scaling_thL"]
        HF_scaling_Gto = self._parameters["HF_scaling_Gto"]
        HF_scaling_Gncx = self._parameters["HF_scaling_Gncx"]
        HF_scaling_Pnak = self._parameters["HF_scaling_Pnak"]
        HF_scaling_cat50_ref = self._parameters["HF_scaling_cat50_ref"]

        # Init return args
        F_expressions = [dolfin.Constant(0.0)] * 47

        # Expressions for the cell geometry component
        vcell = 3140.0 * L * (rad * rad)
        Ageo = 6.28 * (rad * rad) + 6.28 * L * rad
        Acap = 2 * Ageo
        vmyo = 0.68 * vcell
        vnsr = 0.0552 * vcell
        vjsr = 0.0048 * vcell
        vss = 0.02 * vcell

        # Expressions for the CaMKt component
        CaMKb = CaMKo * (1.0 - CaMKt) / (1.0 + KmCaM / cass)
        CaMKa = (CaMKb + CaMKt) * HF_scaling_CaMKa
        F_expressions[0] = -bCaMK * CaMKt + aCaMK * (CaMKb + CaMKt) * CaMKb

        # Expressions for the reversal potentials component
        ENa = R * T * ufl.ln(nao / nai) / F
        EK = R * T * ufl.ln(ko / ki) / F
        EKs = R * T * ufl.ln((ko + PKNa * nao) / (PKNa * nai + ki)) / F
        vffrt = (F * F) * v / (R * T)
        vfrt = F * v / (R * T)

        # Expressions for the I_Na component
        mss = 1.0 / (ufl.elem_pow(1.0 + ufl.exp(-1.0 * ((v + 56.86) / 9.03)), 2.0))
        taum = 0.1292 * ufl.exp(
            -1.0 * ufl.elem_pow((v + 45.79) / 15.54, 2.0),
        ) + 0.06487 * ufl.exp(-1.0 * ufl.elem_pow((v - 4.823) / 51.12, 2.0))
        F_expressions[1] = (mss - m) / taum

        ah = ufl.conditional(
            ufl.lt(v, -40.0),
            0.057 * ufl.exp(-1.0 * ((v + 80.0) / 6.8)),
            0.0,
        )
        bh = ufl.conditional(
            ufl.lt(v, -40.0),
            2.7 * ufl.exp(0.079 * v) + 3.1e5 * ufl.exp(0.3485 * v),
            0.77 / (0.13 * (1.0 + ufl.exp(-1.0 * ((v + 10.66) / 11.1)))),
        )
        tauh = 1.0 / (ah + bh)
        hss = 1.0 / (ufl.elem_pow(1.0 + ufl.exp((v + 71.55) / 7.43), 2.0))
        F_expressions[2] = (hss - h) / tauh

        # thf = 1.0 / (
        #    1.183856958289087e-05 * ufl.exp(-0.15910898965791567 * v)
        #    + 6.305549185817275 * ufl.exp(0.0493339911198816 * v)
        # )
        # ths = 1.0 / (
        #    0.005164670235381792 * ufl.exp(-0.035650623885918005 * v)
        #    + 0.36987619372096325 * ufl.exp(0.017649135192375574 * v)
        # )
        # Ahs = 1.0 - Ahf
        # F_expressions[2] = (-hf + hss) / thf
        # F_expressions[3] = (-hs + hss) / ths
        # h = Ahf * hf + Ahs * hs

        jss = hss
        aj = ufl.conditional(
            ufl.lt(v, -40.0),
            (
                (-25428 * ufl.exp(0.2444 * v) - 6.948e-6 * ufl.exp(-0.04391 * v))
                * (v + 37.78)
            )
            / (1.0 + ufl.exp(0.311 * (v + 79.23))),
            0.0,
        )
        bj = ufl.conditional(
            ufl.lt(v, -40.0),
            (0.02424 * ufl.exp(-0.01052 * v)) / (1.0 + ufl.exp(-0.1378 * (v + 40.14))),
            (0.6 * ufl.exp(0.057 * v)) / (1.0 + ufl.exp(-0.1 * (v + 32.0))),
        )
        tauj = 1.0 / (aj + bj)
        F_expressions[3] = (jss - j) / tauj

        hssp = 1.0 / ufl.elem_pow(1.0 + ufl.exp((v + 71.55 + 6.0) / 7.43), 2.0)
        # thsp = 3.0 * ths
        F_expressions[4] = (hssp - hp) / tauh
        # hp = Ahf * hf + Ahs * hsp
        taujp = 1.46 * tauj
        F_expressions[5] = (jss - jp) / taujp
        fINap = 1.0 / (1.0 + KmCaMK / CaMKa)
        INa = (
            GNa
            * scale_drug_INa
            * scale_popu_GNa
            * ufl.elem_pow(m, 3.0)
            * (-ENa + v)
            * ((1.0 - fINap) * h * j + fINap * hp * jp)
        )

        # Expressions for the INaL component
        mLss = 1.0 / (1.0 + ufl.exp((-1.0 * (v + 42.85)) / 5.264))
        tmL = 0.1292 * ufl.exp(
            -1.0 * ufl.elem_pow((v + 45.79) / 15.54, 2.0),
        ) + 0.06487 * ufl.exp(-1.0 * ufl.elem_pow((v - 4.823) / 51.12, 2.0))
        F_expressions[6] = (-mL + mLss) / tmL
        hLss = 1.0 / (1.0 + 120578.15595522427 * ufl.exp(0.13354700854700854 * v))
        F_expressions[7] = (-hL + hLss) / (thL * HF_scaling_thL)
        hLssp = 1.0 / (1.0 + 275969.2903869871 * ufl.exp(0.13354700854700854 * v))
        thLp = 3.0 * thL * HF_scaling_thL
        F_expressions[8] = (-hLp + hLssp) / thLp
        GNaL = 0.0279 * scale_INaL * scale_drug_INaL * scale_popu_GNaL * HF_scaling_GNaL
        fINaLp = 1.0 / (1.0 + KmCaMK / CaMKa)
        INaL = (-ENa + v) * ((1.0 - fINaLp) * hL + fINaLp * hLp) * GNaL * mL

        # Expressions for the Ito component
        ass = 1.0 / (1.0 + 2.6316508161673635 * ufl.exp(-0.06747638326585695 * v))
        ta = 1.0515 / (
            1.0 / (1.2089 + 2.2621017070578837 * ufl.exp(-0.03403513787634354 * v))
            + 3.5 / (1.0 + 30.069572727397507 * ufl.exp(0.03403513787634354 * v))
        )
        F_expressions[9] = (-a + ass) / ta
        iss = 1.0 / (1.0 + 2194.970764538301 * ufl.exp(0.17510068289266328 * v))
        tiF = 4.562 + delta_epi / (
            0.14468698421272827 * ufl.exp(-0.01 * v)
            + 1.6300896349780942 * ufl.exp(0.06027727546714889 * v)
        )
        tiS = 23.62 + delta_epi / (
            0.00027617763953377436 * ufl.exp(-0.01693480101608806 * v)
            + 0.024208962804604526 * ufl.exp(0.12377769525931426 * v)
        )
        AiF = 1.0 / (1.0 + 0.24348537187522867 * ufl.exp(0.006613756613756614 * v))
        AiS = 1.0 - AiF
        F_expressions[10] = (-iF + iss) / tiF
        F_expressions[11] = (-iS + iss) / tiS
        i = AiF * iF + AiS * iS
        assp = 1.0 / (1.0 + 5.167428462230666 * ufl.exp(-0.06747638326585695 * v))
        F_expressions[12] = (-ap + assp) / ta
        dti_develop = 1.354 + 0.0001 / (
            2.6591269045230603e-05 * ufl.exp(0.06293266205160478 * v)
            + 4.5541779737128264e24 * ufl.exp(-4.642525533890436 * v)
        )
        dti_recover = 1.0 - 0.5 / (1.0 + 33.11545195869231 * ufl.exp(0.05 * v))
        tiFp = dti_develop * dti_recover * tiF
        tiSp = dti_develop * dti_recover * tiS
        F_expressions[13] = (-iFp + iss) / tiFp
        F_expressions[14] = (-iSp + iss) / tiSp
        ip = AiF * iFp + AiS * iSp
        fItop = 1.0 / (1.0 + KmCaMK / CaMKa)
        Ito = (
            Gto
            * scale_drug_Ito
            * scale_popu_Gto
            * HF_scaling_Gto
            * (-EK + v)
            * ((1.0 - fItop) * a * i + ap * fItop * ip)
        )

        # Expressions for the ICaL ICaNa ICaK component
        dss = ufl.conditional(
            ufl.gt(v, 31.4978),
            1.0,
            1.0763 * ufl.exp(-1.007 * ufl.exp(-0.0829 * v)),
        )
        td = 0.6 + 1.0 / (ufl.exp(-0.05 * (v + 6.0)) + ufl.exp(0.09 * (v + 14.0)))
        F_expressions[15] = (-d + dss) / td
        fss = 1.0 / (1.0 + 199.86038496778565 * ufl.exp(0.27056277056277056 * v))
        tff = 7.0 + 1.0 / (
            0.03325075244518792 * ufl.exp(0.1 * v)
            + 0.0006090087745647571 * ufl.exp(-0.1 * v)
        )
        tfs = 1000.0 + 1.0 / (
            1.0027667890106652e-05 * ufl.exp(-0.25 * v)
            + 8.053415618124885e-05 * ufl.exp(0.16666666666666666 * v)
        )
        Afs = 1.0 - Aff
        F_expressions[16] = (-ff + fss) / tff
        F_expressions[17] = (-fs + fss) / tfs
        f = Aff * ff + Afs * fs
        fcass = fss
        tfcaf = 7.0 + 1.0 / (
            0.0708317980974062 * ufl.exp(-0.14285714285714285 * v)
            + 0.02258872488031037 * ufl.exp(0.14285714285714285 * v)
        )
        tfcas = 100.0 + 1.0 / (
            0.00012 * ufl.exp(0.14285714285714285 * v)
            + 0.00012 * ufl.exp(-0.3333333333333333 * v)
        )
        Afcaf = 0.3 + 0.6 / (1.0 + 0.36787944117144233 * ufl.exp(0.1 * v))
        Afcas = 1.0 - Afcaf
        F_expressions[18] = (-fcaf + fcass) / tfcaf
        F_expressions[19] = (-fcas + fcass) / tfcas
        fca = Afcaf * fcaf + Afcas * fcas
        F_expressions[20] = (-jca + fcass) / tjca
        tffp = 2.5 * tff
        F_expressions[21] = (-ffp + fss) / tffp
        fp = Aff * ffp + Afs * fs
        tfcafp = 2.5 * tfcaf
        F_expressions[22] = (-fcafp + fcass) / tfcafp
        fcap = Afcaf * fcafp + Afcas * fcas
        km2n = 1.0 * jca
        anca = 1.0 / (ufl.elem_pow(1.0 + Kmn / cass, 4.0) + k2n / km2n)
        F_expressions[23] = k2n * anca - km2n * nca

        anca_i = 1.0 / (ufl.elem_pow(1.0 + Kmn / cai, 4.0) + k2n / km2n)
        F_expressions[24] = anca_i * k2n - nca_i * km2n

        clo = 150
        cli = 24
        Io = 0.5 * (nao + ko + clo + 4.0 * cao) / 1000.0
        Ii = 0.5 * (nass + kss + cli + 4.0 * cass) / 1000.0
        dielConstant = 74
        constA = 1.82 * 1000000.0 * ufl.elem_pow(dielConstant * T, -1.5)

        gamma_cai = ufl.exp(
            -constA * 4.0 * (ufl.sqrt(Ii) / (1.0 + ufl.sqrt(Ii)) - 0.3 * Ii),
        )
        gamma_cao = ufl.exp(
            -constA * 4.0 * (ufl.sqrt(Io) / (1.0 + ufl.sqrt(Io)) - 0.3 * Io),
        )
        gamma_nai = ufl.exp(
            -constA * 1.0 * (ufl.sqrt(Ii) / (1.0 + ufl.sqrt(Ii)) - 0.3 * Ii),
        )
        gamma_nao = ufl.exp(
            -constA * 1.0 * (ufl.sqrt(Io) / (1.0 + ufl.sqrt(Io)) - 0.3 * Io),
        )
        gamma_ki = ufl.exp(
            -constA * 1.0 * (ufl.sqrt(Ii) / (1.0 + ufl.sqrt(Ii)) - 0.3 * Ii),
        )
        gamma_kao = ufl.exp(
            -constA * 1.0 * (ufl.sqrt(Io) / (1.0 + ufl.sqrt(Io)) - 0.3 * Io),
        )

        gammaCaoMyo = gamma_cao
        gammaCaiMyo = gamma_cai

        PhiCaL_ss = (
            4.0
            * vffrt
            * (gamma_cai * cass * ufl.exp(2.0 * vfrt) - gamma_cao * cao)
            / (-1.0 + ufl.exp(2.0 * vfrt))
        )
        PhiCaNa_ss = (
            1.0
            * vffrt
            * (gamma_nai * nass * ufl.exp(1.0 * vfrt) - gamma_nao * nao)
            / (-1.0 + ufl.exp(1.0 * vfrt))
        )
        PhiCaK_ss = (
            1.0
            * vffrt
            * (gamma_ki * kss * ufl.exp(1.0 * vfrt) - gamma_kao * ko)
            / (-1.0 + ufl.exp(1.0 * vfrt))
        )
        PhiCaL_i = (
            4.0
            * vffrt
            * (gamma_cai * cai * ufl.exp(2.0 * vfrt) - gamma_cao * cao)
            / (-1.0 + ufl.exp(2.0 * vfrt))
        )
        PhiCaNa_i = (
            1.0
            * vffrt
            * (gamma_nai * nai * ufl.exp(1.0 * vfrt) - gamma_nao * nao)
            / (-1.0 + ufl.exp(1.0 * vfrt))
        )
        PhiCaK_i = (
            1.0
            * vffrt
            * (gamma_ki * ki * ufl.exp(1.0 * vfrt) - gamma_kao * ko)
            / (-1.0 + ufl.exp(1.0 * vfrt))
        )

        PCa = 0.000083757 * scale_ICaL * scale_drug_ICaL * scale_popu_GCaL
        PCap = 1.1 * PCa
        PCaNa = 0.00125 * PCa
        PCaK = 0.0003574 * PCa
        PCaNap = 0.00125 * PCap
        PCaKp = 0.0003574 * PCap
        fICaLp = 1.0 / (1.0 + KmCaMK / CaMKa)

        ICaL_tot_ss = (1.0 - fICaLp) * (
            (1.0 - nca) * f + fca * jca * nca
        ) * PCa * PhiCaL_ss * d + (
            (1.0 - nca) * fp + fcap * jca * nca
        ) * PCap * PhiCaL_ss * d * fICaLp
        ICaNa_tot_ss = (1.0 - fICaLp) * (
            (1.0 - nca) * f + fca * jca * nca
        ) * PCaNa * PhiCaNa_ss * d + (
            (1.0 - nca) * fp + fcap * jca * nca
        ) * PCaNap * PhiCaNa_ss * d * fICaLp
        ICaK_tot_ss = (1.0 - fICaLp) * (
            (1.0 - nca) * f + fca * jca * nca
        ) * PCaK * PhiCaK_ss * d + (
            (1.0 - nca) * fp + fcap * jca * nca
        ) * PCaKp * PhiCaK_ss * d * fICaLp

        ICaL_tot_i = (1.0 - fICaLp) * (
            (1.0 - nca) * f + fca * jca * nca
        ) * PCa * PhiCaL_i * d + (
            (1.0 - nca) * fp + fcap * jca * nca
        ) * PCap * PhiCaL_i * d * fICaLp
        ICaNa_tot_i = (1.0 - fICaLp) * (
            (1.0 - nca) * f + fca * jca * nca
        ) * PCaNa * PhiCaNa_i * d + (
            (1.0 - nca) * fp + fcap * jca * nca
        ) * PCaNap * PhiCaNa_i * d * fICaLp
        ICaK_tot_i = (1.0 - fICaLp) * (
            (1.0 - nca) * f + fca * jca * nca
        ) * PCaK * PhiCaK_i * d + (
            (1.0 - nca) * fp + fcap * jca * nca
        ) * PCaKp * PhiCaK_i * d * fICaLp

        ICaL_ss = ICaL_tot_ss * ICaL_fractionSS
        ICaNa_ss = ICaNa_tot_ss * ICaL_fractionSS
        ICaK_ss = ICaK_tot_ss * ICaL_fractionSS
        ICaL_i = ICaL_tot_i * (1.0 - ICaL_fractionSS)
        ICaNa_i = ICaNa_tot_i * (1.0 - ICaL_fractionSS)
        ICaK_i = ICaK_tot_i * (1.0 - ICaL_fractionSS)

        ICaL = ICaL_ss + ICaL_i
        # ICaNa = ICaNa_ss + ICaNa_i
        # ICaK = ICaK_ss + ICaK_i

        # Expressions for the IKr component
        # _c0: from c0 to c1; _c1: from c1 to c2; _c2o: from c2 to o;
        # _oi: from open to i; _c2i: from c2 to i
        alpha_c0 = 0.1161 * ufl.exp(0.2990 * vfrt)
        beta_c0 = 0.2442 * ufl.exp(-1.604 * vfrt)
        alpha_c1 = 1.25 * 0.1235
        beta_c1 = 0.1911
        alpha_c2o = 0.0578 * ufl.exp(0.9710 * vfrt)
        beta_c2o = 0.349e-3 * ufl.exp(-1.062 * vfrt)
        alpha_oi = 0.2533 * ufl.exp(0.5953 * vfrt)
        beta_oi = 1.25 * 0.0522 * ufl.exp(-0.8209 * vfrt)
        alpha_c2i = 0.52e-4 * ufl.exp(1.525 * vfrt)
        beta_c2i = (beta_c2o * beta_oi * alpha_c2i) / (alpha_c2o * alpha_oi)

        F_expressions[25] = kr_c1 * beta_c0 - kr_c0 * alpha_c0
        F_expressions[26] = (
            kr_c0 * alpha_c0 + kr_c2 * beta_c1 - kr_c1 * (beta_c0 + alpha_c1)
        )
        F_expressions[27] = (
            kr_c1 * alpha_c1
            + kr_o * beta_c2o
            + kr_i * beta_c2i
            - kr_c2 * (beta_c1 + alpha_c2o + alpha_c2i)
        )
        F_expressions[28] = (
            kr_c2 * alpha_c2o + kr_i * beta_oi - kr_o * (beta_c2o + alpha_oi)
        )
        F_expressions[29] = (
            kr_c2 * alpha_c2i + kr_o * alpha_oi - kr_i * (beta_c2i + beta_oi)
        )

        # xrss = 1.0 / (1.0 + 0.29287308872377504 * ufl.exp(-0.14729709824716453 * v))
        # txrf = 12.98 + 1.0 / (
        #    0.0001020239312894894 * ufl.exp(0.25846471956577927 * v)
        #    + 0.00042992960891929087 * ufl.exp(-0.04906771344455348 * v)
        # )
        # txrs = 1.865 + 1.0 / (
        #    0.0005922420036809394 * ufl.exp(0.13596193065941536 * v)
        #    + 3.549966111802463e-05 * ufl.exp(-0.03855050115651503 * v)
        # )
        # Axrf = 1.0 / (1.0 + 4.197299094734718 * ufl.exp(0.02617115938236064 * v))
        # Axrs = 1.0 - Axrf
        # F_expressions[25] = (-xrf + xrss) / txrf
        # F_expressions[26] = (-xrs + xrss) / txrs
        # xr = Axrf * xrf + Axrs * xrs
        # rkr = 1.0 / (
        #    (1.0 + 2.0820090840784555 * ufl.exp(0.013333333333333334 * v))
        #    * (1.0 + 0.7165313105737893 * ufl.exp(0.03333333333333333 * v))
        # )

        GKr = 0.0321 * scale_IKr * scale_drug_IKr * scale_popu_GKr
        IKr = GKr * ufl.sqrt(ko / 5.0) * kr_o * (-EK + v)

        # Expressions for the IKs component
        xs1ss = 1.0 / (1.0 + 0.27288596035656526 * ufl.exp(-0.11195700850873264 * v))
        txs1 = 817.3 + 1.0 / (
            0.003504067763074858 * ufl.exp(0.056179775280898875 * v)
            + 0.0005184809083581659 * ufl.exp(-0.004347826086956522 * v)
        )
        F_expressions[30] = (-xs1 + xs1ss) / txs1
        xs2ss = xs1ss
        txs2 = 1.0 / (
            0.0022561357010639103 * ufl.exp(-0.03225806451612903 * v)
            + 0.0008208499862389881 * ufl.exp(0.05 * v)
        )
        F_expressions[31] = (-xs2 + xs2ss) / txs2
        KsCa = 1.0 + 0.6 / (1.0 + 6.481821026062645e-07 * ufl.elem_pow(1.0 / cai, 1.4))
        GKs = 0.0011 * scale_IKs * scale_drug_IKs * scale_popu_GKs
        IKs = (-EKs + v) * GKs * KsCa * xs1 * xs2

        # Expressions for the IK1 component
        aK1 = 4.094 / (1.0 + ufl.exp(0.1217 * (v - EK - 49.934)))
        bK1 = (
            15.72 * ufl.exp(0.0674 * (v - EK - 3.257))
            + ufl.exp(0.0618 * (v - EK - 594.31))
        ) / (1.0 + ufl.exp(-0.1629 * (v - EK + 14.207)))
        K1ss = aK1 / (aK1 + bK1)

        # xk1ss = 1.0 / (
        #    1.0 + ufl.exp((-144.59 - v - 2.5538 * ko) / (3.8115 + 1.5692 * ko))
        # )
        # txk1 = 122.2 / (
        #    0.0019352007631390235 * ufl.exp(-0.049115913555992145 * v)
        #    + 30.43364757524903 * ufl.exp(0.014423770373575654 * v)
        # )
        # F_expressions[] = (-xk1 + xk1ss) / txk1
        # rk1 = 1.0 / (
        #    1.0
        #    + 69220.6322106767
        #    * ufl.exp(0.10534077741493732 * v - 0.27388602127883704 * ko)
        # )

        GK1 = 0.6992 * scale_IK1 * scale_drug_IK1 * scale_popu_GK1 * HF_scaling_GK1
        IK1 = ufl.sqrt(ko / 5.0) * (-EK + v) * GK1 * K1ss

        # Expressions for the INaCa_i component
        hca = ufl.exp(F * qca * v / (R * T))
        hna = ufl.exp(F * qna * v / (R * T))
        h1_i = 1.0 + (1.0 + hna) * nai / kna3
        h2_i = hna * nai / (kna3 * h1_i)
        h3_i = 1.0 / h1_i
        h4_i = 1.0 + (1 + nai / kna2) * nai / kna1
        h5_i = (nai * nai) / (kna1 * kna2 * h4_i)
        h6_i = 1.0 / h4_i
        h7_i = 1.0 + nao * (1.0 + 1.0 / hna) / kna3
        h8_i = nao / (kna3 * h7_i * hna)
        h9_i = 1.0 / h7_i
        h10_i = 1.0 + kasymm + nao * (1.0 + nao / kna2) / kna1
        h11_i = (nao * nao) / (kna1 * kna2 * h10_i)
        h12_i = 1.0 / h10_i
        k1_i = cao * kcaon * h12_i
        k2_i = kcaoff
        k3p_i = wca * h9_i
        k3pp_i = wnaca * h8_i
        k3_i = k3p_i + k3pp_i
        k4p_i = wca * h3_i / hca
        k4pp_i = wnaca * h2_i
        k4_i = k4p_i + k4pp_i
        k5_i = kcaoff
        k6_i = kcaon * cai * h6_i
        k7_i = wna * h2_i * h5_i
        k8_i = wna * h11_i * h8_i
        x1_i = (k2_i + k3_i) * k5_i * k7_i + (k6_i + k7_i) * k2_i * k4_i
        x2_i = (k1_i + k8_i) * k4_i * k6_i + (k4_i + k5_i) * k1_i * k7_i
        x3_i = (k2_i + k3_i) * k6_i * k8_i + (k6_i + k7_i) * k1_i * k3_i
        x4_i = (k1_i + k8_i) * k3_i * k5_i + (k4_i + k5_i) * k2_i * k8_i
        E1_i = x1_i / (x1_i + x2_i + x3_i + x4_i)
        E2_i = x2_i / (x1_i + x2_i + x3_i + x4_i)
        E3_i = x3_i / (x1_i + x2_i + x3_i + x4_i)
        E4_i = x4_i / (x1_i + x2_i + x3_i + x4_i)
        allo_i = 1.0 / (1.0 + ufl.elem_pow(KmCaAct / cai, 2.0))
        zna = 1.0
        JncxNa_i = E3_i * k4pp_i - E2_i * k3pp_i + 3.0 * E4_i * k7_i - 3.0 * E1_i * k8_i
        JncxCa_i = E2_i * k2_i - E1_i * k1_i
        INaCa_i = (
            Gncx
            * (1.0 - INaCa_fractionSS)
            * scale_popu_KNCX
            * HF_scaling_Gncx
            * (zca * JncxCa_i + zna * JncxNa_i)
            * allo_i
        )

        # Expressions for the INaCa_ss component
        h1 = 1.0 + (1.0 + hna) * nass / kna3
        h2 = hna * nass / (kna3 * h1)
        h3 = 1.0 / h1
        h4 = 1.0 + (1 + nass / kna2) * nass / kna1
        h5 = (nass * nass) / (kna1 * kna2 * h4)
        h6 = 1.0 / h4
        h7 = 1.0 + nao * (1.0 + 1.0 / hna) / kna3
        h8 = nao / (kna3 * h7 * hna)
        h9 = 1.0 / h7
        h10 = 1.0 + kasymm + nao * (1 + nao / kna2) / kna1
        h11 = (nao * nao) / (kna1 * kna2 * h10)
        h12 = 1.0 / h10
        k1 = cao * kcaon * h12
        k2 = kcaoff
        k3p_ss = wca * h9
        k3pp = wnaca * h8
        k3 = k3p_ss + k3pp
        k4p_ss = wca * h3 / hca
        k4pp = wnaca * h2
        k4 = k4p_ss + k4pp
        k5 = kcaoff
        k6 = kcaon * cass * h6
        k7 = wna * h2 * h5
        k8 = wna * h11 * h8
        x1_ss = (k2 + k3) * k5 * k7 + (k6 + k7) * k2 * k4
        x2_ss = (k1 + k8) * k4 * k6 + (k4 + k5) * k1 * k7
        x3_ss = (k2 + k3) * k6 * k8 + (k6 + k7) * k1 * k3
        x4_ss = (k1 + k8) * k3 * k5 + (k4 + k5) * k2 * k8
        E1_ss = x1_ss / (x1_ss + x2_ss + x3_ss + x4_ss)
        E2_ss = x2_ss / (x1_ss + x2_ss + x3_ss + x4_ss)
        E3_ss = x3_ss / (x1_ss + x2_ss + x3_ss + x4_ss)
        E4_ss = x4_ss / (x1_ss + x2_ss + x3_ss + x4_ss)
        allo_ss = 1.0 / (1.0 + ufl.elem_pow(KmCaAct / cass, 2.0))
        JncxNa_ss = E3_ss * k4pp - E2_ss * k3pp + 3.0 * E4_ss * k7 - 3.0 * E1_ss * k8
        JncxCa_ss = E2_ss * k2 - E1_ss * k1
        INaCa_ss = (
            Gncx
            * INaCa_fractionSS
            * scale_popu_KNCX
            * HF_scaling_Gncx
            * (zca * JncxCa_ss + zna * JncxNa_ss)
            * allo_ss
        )

        # Expressions for the INaK component
        Knai = Knai0 * ufl.exp(0.3333333333333333 * F * delta * v / (R * T))
        Knao = Knao0 * ufl.exp(0.3333333333333333 * F * (1.0 - delta) * v / (R * T))
        P = eP / (1.0 + H / Khp + nai / Knap + ki / Kxkur)
        a1 = (
            k1p
            * ufl.elem_pow(nai / Knai, 3.0)
            / (
                -1.0
                + ufl.elem_pow(1.0 + ki / Kki, 2.0)
                + ufl.elem_pow(1.0 + nai / Knai, 3.0)
            )
        )
        b1 = MgADP * k1m
        a2 = k2p
        b2 = (
            k2m
            * ufl.elem_pow(nao / Knao, 3.0)
            / (
                -1.0
                + ufl.elem_pow(1.0 + ko / Kko, 2.0)
                + ufl.elem_pow(1.0 + nao / Knao, 3.0)
            )
        )
        a3 = (
            k3p
            * ufl.elem_pow(ko / Kko, 2.0)
            / (
                -1.0
                + ufl.elem_pow(1.0 + ko / Kko, 2.0)
                + ufl.elem_pow(1.0 + nao / Knao, 3.0)
            )
        )
        b3 = H * k3m * P / (1.0 + MgATP / Kmgatp)
        a4 = MgATP * k4p / (Kmgatp * (1.0 + MgATP / Kmgatp))
        b4 = (
            k4m
            * ufl.elem_pow(ki / Kki, 2.0)
            / (
                -1.0
                + ufl.elem_pow(1.0 + ki / Kki, 2.0)
                + ufl.elem_pow(1.0 + nai / Knai, 3.0)
            )
        )
        x1 = a1 * a2 * a4 + a1 * a2 * b3 + a2 * b3 * b4 + b2 * b3 * b4
        x2 = a1 * a2 * a3 + a2 * a3 * b4 + a3 * b1 * b4 + b1 * b2 * b4
        x3 = a2 * a3 * a4 + a3 * a4 * b1 + a4 * b1 * b2 + b1 * b2 * b3
        x4 = a1 * a3 * a4 + a1 * a4 * b2 + a1 * b2 * b3 + b2 * b3 * b4
        E1 = x1 / (x1 + x2 + x3 + x4)
        E2 = x2 / (x1 + x2 + x3 + x4)
        E3 = x3 / (x1 + x2 + x3 + x4)
        E4 = x4 / (x1 + x2 + x3 + x4)
        JnakNa = 3.0 * E1 * a3 - 3.0 * E2 * b3
        JnakK = 2.0 * E4 * b1 - 2.0 * E3 * a1
        INaK = Pnak * scale_popu_KNaK * HF_scaling_Pnak * (zk * JnakK + zna * JnakNa)

        # Expressions for the IKb component
        xkb = 1.0 / (1.0 + ufl.exp(-(v - 10.8968) / 23.9871))
        IKb = GKb * scale_popu_GbK * scale_drug_IKb * (-EK + v) * xkb

        # Expressions for the INab component
        INab = (
            PNab
            * scale_drug_INab
            * scale_popu_GbNa
            * (-nao + ufl.exp(vfrt) * nai)
            * vffrt
            / (-1.0 + ufl.exp(vfrt))
        )

        # Expressions for the ICab component
        ICab = (
            4.0
            * PCab
            * scale_drug_ICab
            * scale_popu_GbCa
            * (cai * gammaCaiMyo * ufl.exp(2.0 * vfrt) - cao * gammaCaoMyo)
            * vffrt
            / (-1.0 + ufl.exp(2.0 * vfrt))
        )

        # Expressions for the ICaCl component (calcium activated chloride current)
        # ecl = R * T * ufl.ln(cli / clo) / F
        # Fjunc = 1.0
        # Fsl = 1.0 - Fjunc
        # GClCa = 0.2843
        # KdClCa = 0.1

        # I_ClCa_junc = Fjunc * GClCa / (1.0 + KdClCa / cass) * (v - ecl)
        # I_ClCa_sl = Fsl * GClCa / (1.0 + KdClCa / cai) * (v - ecl)
        # I_ClCa = I_ClCa_junc + I_ClCa_sl

        # Expressions for the IClb component
        # GClB = 0.00198
        # I_Clbk = GClB * (v - ecl)

        # Expressions for the IpCa component
        IpCa = GpCa * scale_drug_IpCa * scale_popu_KpCa * cai / (0.0005 + cai)

        # Expressions for the Isac (Pueyo)--> ns + k component
        Gsac_ns = 0.006 * scale_drug_Isacns
        Esac_ns = -10.0
        lambda_max = 1.1
        Gsac_k = (0.2882 * 800.0 / 210.0) * scale_drug_Isack  # Pueyo endo
        Isac_P_ns = ufl.conditional(
            ufl.lt(self.lmbda, 1.0),
            0.0,
            Gsac_ns * ((self.lmbda - 1.0) / (lambda_max - 1.0)) * (v - Esac_ns),
        )
        Isac_P_k = ufl.conditional(
            ufl.lt(self.lmbda, 1.0),
            0.0,
            Gsac_k
            * ((self.lmbda - 1.0) / (lambda_max - 1.0))
            * (1.0 / (1.0 + ufl.exp((19.05 - v) / (29.98)))),
        )

        # Expressions for the Istim component
        Istim = 0  # amp*(ufl.le(time, duration))

        # Expressions for the diffusion fluxes component
        JdiffNa = 0.5 * nass - 0.5 * nai
        JdiffK = 0.5 * kss - 0.5 * ki
        Jdiff = 5.0 * cass - 5.0 * cai

        # Expressions for the ryanodine receptor component
        a_rel = 0.5 * bt
        jsrMidpoint = 1.7
        Jrel_inf = (
            -ICaL
            * a_rel
            / (1.0 + HF_scaling_Jrel_inf * ufl.elem_pow(jsrMidpoint / cajsr, 8.0))
        )
        tau_rel_tmp = bt / (1.0 + 0.0123 / cajsr)
        tau_rel = ufl.conditional(ufl.lt(tau_rel_tmp, 0.001), 0.001, tau_rel_tmp)
        F_expressions[32] = (-Jrelnp + Jrel_inf) / tau_rel
        btp = 1.25 * bt
        a_relp = 0.5 * btp
        Jrel_infp = (
            -ICaL
            * a_relp
            / (1.0 + HF_scaling_Jrel_inf * ufl.elem_pow(jsrMidpoint / cajsr, 8.0))
        )
        tau_relp_tmp = btp / (1.0 + 0.0123 / cajsr)
        tau_relp = ufl.conditional(ufl.lt(tau_relp_tmp, 0.001), 0.001, tau_relp_tmp)
        F_expressions[33] = (-Jrelp + Jrel_infp) / tau_relp
        fJrelp = 1.0 / (1.0 + KmCaMK / CaMKa)
        Jrel = 1.5378 * ((1.0 - fJrelp) * Jrelnp + Jrelp * fJrelp) * scale_popu_KRyR

        # Expressions for the calcium buffers component
        Jupnp = 0.005425 * cai / (0.00092 + cai)
        Jupp = 2.75 * 0.005425 * cai / (0.00092 - 0.00017 + cai)
        fJupp = 1.0 / (1.0 + KmCaMK / CaMKa)
        Jleak = 0.0048825 * cansr * scale_popu_Kleak * HF_scaling_Jleak / 15.0
        Jup = (
            -Jleak
            + ((1.0 - fJupp) * Jupnp + Jupp * fJupp)
            * scale_popu_KSERCA
            * HF_scaling_Jup
        )
        Jtr = (cansr - cajsr) / 60.0

        # Expressions for the intracellular concentrations component
        F_expressions[45] = JdiffNa * vss / vmyo + (
            -INa - INaL - INab - ICaNa_i - Isac_P_ns / 3.0 - 3.0 * INaCa_i - 3.0 * INaK
        ) * Acap / (F * vmyo)
        F_expressions[46] = -JdiffNa + (-ICaNa_ss - 3.0 * INaCa_ss) * Acap / (F * vss)
        F_expressions[34] = JdiffK * vss / vmyo + (
            -ICaK_i
            - Isac_P_k
            - IK1
            - IKb
            - IKr
            - IKs
            - Istim
            - Ito
            - Isac_P_ns / 3.0
            + 2.0 * INaK
        ) * Acap / (F * vmyo)
        F_expressions[35] = -JdiffK - Acap * ICaK_ss / (F * vss)
        Bcass = 1.0 / (
            1.0
            + BSLmax * KmBSL * ufl.elem_pow(KmBSL + cass, -2.0)
            + BSRmax * KmBSR * ufl.elem_pow(KmBSR + cass, -2.0)
        )
        F_expressions[36] = (
            -Jdiff
            + Jrel * vjsr / vss
            + 0.5 * (-ICaL_ss + 2.0 * INaCa_ss) * Acap / (F * vss)
        ) * Bcass
        F_expressions[37] = -Jtr * vjsr / vnsr + Jup
        Bcajsr = 1.0 / (1.0 + csqnmax * kmcsqn * ufl.elem_pow(kmcsqn + cajsr, -2.0))
        F_expressions[38] = (-Jrel + Jtr) * Bcajsr

        # Expressions for the mechanics component
        kwu = -kws * scale_popu_kws + (kuw * scale_popu_kuw) * (
            -1.0 + 1.0 / (rw * scale_popu_rw)
        )
        ksu = (
            kws
            * scale_popu_kws
            * rw
            * scale_popu_rw
            * (-1.0 + 1.0 / (rs * scale_popu_rs))
        )

        lambda_min12 = ufl.conditional(ufl.lt(self.lmbda, 1.2), self.lmbda, 1.2)
        XS = ufl.conditional(ufl.lt(XS, 0.0), 0.0, XS)
        XW = ufl.conditional(ufl.lt(XW, 0.0), 0.0, XW)

        XU = 1.0 - TmB - XS - XW
        gammawu = gammaw * abs(self.Zetaw)
        # gammasu = gammas*ufl.conditional(ufl.gt(Zetas*(ufl.gt(Zetas, 0)), (-1 -\
        #     Zetas)*(ufl.lt(Zetas, -1))), Zetas*(ufl.gt(Zetas, 0)), (-1 -\
        #     Zetas)*(ufl.lt(Zetas, -1)))
        zetas1 = self.Zetas * ufl.conditional(ufl.gt(self.Zetas, 0.0), 1.0, 0.0)
        zetas2 = (-1.0 - self.Zetas) * ufl.conditional(
            ufl.lt(self.Zetas, -1.0),
            1.0,
            0.0,
        )
        gammasu = gammas * Max(zetas1, zetas2)

        F_expressions[39] = (
            kws * scale_drug_kws * scale_popu_kws * XW - XS * gammasu - XS * ksu
        )
        F_expressions[40] = (
            kuw * scale_drug_kuw * scale_popu_kuw * XU
            - kws * scale_drug_kws * scale_popu_kws * XW
            - XW * gammawu
            - XW * kwu * scale_drug_kwu
        )
        cat50 = (
            cat50_ref * scale_popu_CaT50ref + Beta1 * (-1.0 + lambda_min12)
        ) * HF_scaling_cat50_ref
        CaTrpn = ufl.conditional(ufl.lt(CaTrpn, 0.0), 0.0, CaTrpn)
        F_expressions[41] = (
            ktrpn
            * scale_popu_kTRPN
            * (
                -CaTrpn
                + ufl.elem_pow(1000.0 * cai / cat50, ntrpn * scale_popu_nTRPN)
                * (1.0 - CaTrpn)
            )
        )
        kb = (
            ku
            * scale_popu_ku
            * ufl.elem_pow(Trpn50 * scale_popu_TRPN50, (ntm * scale_popu_nTm))
            / (
                1.0
                - (rs * scale_popu_rs)
                - rw * scale_popu_rw * (1.0 - (rs * scale_popu_rs))
            )
        )

        F_expressions[42] = (
            ufl.conditional(
                ufl.lt(ufl.elem_pow(CaTrpn, -(ntm * scale_popu_nTm) / 2.0), 100.0),
                ufl.elem_pow(CaTrpn, -(ntm * scale_popu_nTm) / 2.0),
                100.0,
            )
            * XU
            * kb
            - ku
            * scale_popu_ku
            * ufl.elem_pow(CaTrpn, (ntm * scale_popu_nTm) / 2.0)
            * TmB
        )

        C = -1.0 + lambda_min12
        dCd = -Cd + C
        eta = ufl.conditional(ufl.lt(dCd, 0.0), etas, etal)
        F_expressions[43] = p_k * (-Cd + C) / eta
        Bcai = 1.0 / (1.0 + cmdnmax * kmcmdn * ufl.elem_pow(kmcmdn + cai, -2.0))
        J_TRPN = trpnmax * F_expressions[41]
        F_expressions[44] = (
            -J_TRPN
            + Jdiff * vss / vmyo
            - Jup * vnsr / vmyo
            + 0.5 * (-ICab - IpCa - Isac_P_ns / 3.0 + 2.0 * INaCa_i) * Acap / (F * vmyo)
        ) * Bcai

        # Return results
        return as_vector(F_expressions)

    def num_states(self):
        return 47

    def __str__(self):
        return "Tomek_Land_em_coupling cardiac cell model"
