# -*- coding: UTF-8 -*-
#   Copyright 2012-2020 Fumail Project
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
#
#

######################
# new get_tld functions with support for subsubdomains
######################

def tld_tree_path(tldlist, tree, path=None):
    """
    walk list tldlist through tld tree and return a list of all tld candidates found in tree.
    candidate list is a list of tuples
        first tuple item contains the tld part,
        second tuple item a boolean: True if this is a leaf, False if intermediate node
    """
    if path is None:
        path = []
    
    if len(tldlist) == 0:  # list is finished
        return path
    
    if tldlist[0] in tree:
        node = tree[tldlist[0]]
        path.append((tldlist[0], node[0]))
        return tld_tree_path(tldlist[1:], node[1], path)
    
    return path


def tld_tree_update(d, u):
    """add tld tree u into tree d"""
    for k, v in u.items():
        kbranch = d.get(k, (False, {}))
        leaf = v[0] or kbranch[0]
        r = tld_tree_update(kbranch[1], v[1])
        d[k] = (leaf, r)
    return d


def tld_list_to_tree(tldlist):
    """translate a list into a tree path"""
    d = {}
    if len(tldlist) == 0:
        return d
    else:
        is_leaf = len(tldlist)==1
        d[tldlist[0]] = (is_leaf, tld_list_to_tree(tldlist[1:]))
        return d



#######################
# old get_tld functions with subsubdomain detection issues
#######################

import collections

def dict_path(l, node, path=None):
    """walk list l through dict node and return a list of all nodes found up until a leaf node"""
    if path is None:
        path = []
    
    if len(l) == 0:  # list is finished
        return path
    
    if not isinstance(node, collections.Mapping):  # leafnode
        if l[0] == node:
            path.append(node)
        return path
    
    if l[0] in node:
        path.append(l[0])
        return dict_path(l[1:], node[l[0]], path)
    
    return path


def dict_update(d, u):
    """add dict u into d changing leafnodes to dicts where necessary"""
    if not isinstance(d, collections.Mapping):
        d = {d: {}}
    for k, v in u.items():
        if isinstance(v, collections.Mapping):
            r = dict_update(d.get(k, {}), v)
            d[k] = r
        else:
            d[k] = u[k]
    return d


def dict_topdown_iterator(d, path=None, skiptop=False):
    """walk through a dict and list all possible paths. Optional path prepends additional path elements
    if skiptop is True, skips the current level and drills down to the next level
    """

    allpaths = []
    if path is None:
        path = []
    currentlevel = sorted(d.keys())
    if not skiptop:
        for node in currentlevel:
            allpaths.append(path[:] + [node, ])

    # drill down
    for node in currentlevel:
        if isinstance(d[node], collections.Mapping):
            for result in dict_topdown_iterator(d[node], path=path + [node, ], skiptop=False):
                allpaths.append(result)

    for path in sorted(allpaths, key=len):
        yield path


def list_to_dict(l):
    """translate a list into a tree path"""
    d = {}
    if len(l) == 0:
        return d
    else:
        d[l[0]] = list_to_dict(l[1:])
        return d


#######################
# unicode lookalike cleanup functions
# originally taken from https://github.com/caffeinewriter/phishing_catcher/tree/unicode-confusables
#######################

confusables = {
    '\u2460': '1',
    '\u2780': '1',
    '\U0001D7D0': '2',
    '\U0001D7DA': '2',
    '\U0001D7EE': '2',
    '\U0001D7F8': '2',
    '\uA75A': '2',
    '\u01A7': '2',
    '\u03E8': '2',
    '\uA644': '2',
    '\u14BF': '2',
    '\uA6EF': '2',
    '\u2461': '2',
    '\u2781': '2',
    '\u01BB': '2',
    '\U0001F103': '2.',
    '\u2489': '2.',
    '\U0001D206': '3',
    '\U0001D7D1': '3',
    '\U0001D7DB': '3',
    '\U0001D7E5': '3',
    '\U0001D7EF': '3',
    '\U0001D7F9': '3',
    '\U00016F3B': '3',
    '\U000118CA': '3',
    '\uA7AB': '3',
    '\u021C': '3',
    '\u01B7': '3',
    '\uA76A': '3',
    '\u2CCC': '3',
    '\u0417': '3',
    '\u04E0': '3',
    '\u0AE9': '3',
    '\u2462': '3',
    '\u0498': '3',
    '\U0001F104': '3.',
    '\u248A': '3.',
    '\U0001D7D2': '4',
    '\U0001D7DC': '4',
    '\U0001D7E6': '4',
    '\U0001D7F0': '4',
    '\U0001D7FA': '4',
    '\U000118AF': '4',
    '\u13CE': '4',
    '\u2463': '4',
    '\u2783': '4',
    '\u1530': '4',
    '\U0001F105': '4.',
    '\u248B': '4.',
    '\U0001D7D3': '5',
    '\U0001D7DD': '5',
    '\U0001D7E7': '5',
    '\U0001D7F1': '5',
    '\U0001D7FB': '5',
    '\U000118BB': '5',
    '\u01BC': '5',
    '\u2464': '5',
    '\u2784': '5',
    '\U0001F106': '5.',
    '\u248C': '5.',
    '\U0001D7D4': '6',
    '\U0001D7DE': '6',
    '\U0001D7E8': '6',
    '\U0001D7F2': '6',
    '\U0001D7FC': '6',
    '\U000118D5': '6',
    '\u2CD2': '6',
    '\u0431': '6',
    '\u13EE': '6',
    '\u2465': '6',
    '\u2785': '6',
    '\U0001F107': '6.',
    '\u248D': '6.',
    '\U0001D212': '7',
    '\U0001D7D5': '7',
    '\U0001D7DF': '7',
    '\U0001D7E9': '7',
    '\U0001D7F3': '7',
    '\U0001D7FD': '7',
    '\U000104D2': '7',
    '\U000118C6': '7',
    '\u2466': '7',
    '\u2786': '7',
    '\U0001F108': '7.',
    '\u248E': '7.',
    '\U0001E8CB': '8',
    '\U0001D7D6': '8',
    '\U0001D7E0': '8',
    '\U0001D7EA': '8',
    '\U0001D7F4': '8',
    '\U0001D7FE': '8',
    '\U0001031A': '8',
    '\u0B03': '8',
    '\u09EA': '8',
    '\u0A6A': '8',
    '\u0223': '8',
    '\u0222': '8',
    '\u2467': '8',
    '\u2787': '8',
    '\U0001F109': '8.',
    '\u248F': '8.',
    '\U0001D7D7': '9',
    '\U0001D7E1': '9',
    '\U0001D7E4': '9',
    '\U0001D7EB': '9',
    '\U0001D7F5': '9',
    '\U0001D7FF': '9',
    '\U000118CC': '9',
    '\U000118AC': '9',
    '\U000118D6': '9',
    '\u0A67': '9',
    '\u0B68': '9',
    '\u09ED': '9',
    '\u0D6D': '9',
    '\uA76E': '9',
    '\u2CCA': '9',
    '\u0967': '9',
    '\u06F9': '9',
    '\u2468': '9',
    '\u2788': '9',
    '\U0001F10A': '9.',
    '\u2490': '9.',
    '\U0001D41A': 'a',
    '\U0001D44E': 'a',
    '\U0001D482': 'a',
    '\U0001D4B6': 'a',
    '\U0001D4EA': 'a',
    '\U0001D51E': 'a',
    '\U0001D552': 'a',
    '\U0001D586': 'a',
    '\U0001D5BA': 'a',
    '\U0001D5EE': 'a',
    '\U0001D622': 'a',
    '\U0001D656': 'a',
    '\U0001D68A': 'a',
    '\U0001D6C2': 'a',
    '\U0001D6FC': 'a',
    '\U0001D736': 'a',
    '\U0001D770': 'a',
    '\U0001D7AA': 'a',
    '\U0001D400': 'a',
    '\U0001D434': 'a',
    '\U0001D468': 'a',
    '\U0001D49C': 'a',
    '\U0001D4D0': 'a',
    '\U0001D504': 'a',
    '\U0001D538': 'a',
    '\U0001D56C': 'a',
    '\U0001D5A0': 'a',
    '\U0001D5D4': 'a',
    '\U0001D608': 'a',
    '\U0001D63C': 'a',
    '\U0001D670': 'a',
    '\U0001D6A8': 'a',
    '\U0001D6E2': 'a',
    '\U0001D71C': 'a',
    '\U0001D756': 'a',
    '\U0001D790': 'a',
    '\u237A': 'a',
    '\uFF41': 'a',
    '\u0251': 'a',
    '\u03B1': 'a',
    '\u0430': 'a',
    '\u2DF6': 'a',
    '\uFF21': 'a',
    '\u0391': 'a',
    '\u0410': 'a',
    '\u13AA': 'a',
    '\u15C5': 'a',
    '\uA4EE': 'a',
    '\u2376': 'a',
    '\u01CE': 'a',
    '\u0103': 'a',
    '\u01CD': 'a',
    '\u0102': 'a',
    '\u0227': 'a',
    '\u00E5': 'a',
    '\u0226': 'a',
    '\u00C5': 'a',
    '\u1E9A': 'a',
    '\u1EA3': 'a',
    '\uAB7A': 'a',
    '\u1D00': 'a',
    '\uA733': 'aa',
    '\uA732': 'aa',
    '\u00E6': 'ae',
    '\u04D5': 'ae',
    '\u00C6': 'ae',
    '\u04D4': 'ae',
    '\uA735': 'ao',
    '\uA734': 'ao',
    '\U0001F707': 'ar',
    '\uA737': 'a',
    '\uA736': 'a',
    '\uA738': 'av',
    '\uA739': 'av',
    '\uA73A': 'av',
    '\uA73B': 'av',
    '\uA73D': 'ay',
    '\uA73C': 'ay',
    '\U0001D41B': 'b',
    '\U0001D44F': 'b',
    '\U0001D483': 'b',
    '\U0001D4B7': 'b',
    '\U0001D4EB': 'b',
    '\U0001D51F': 'b',
    '\U0001D553': 'b',
    '\U0001D587': 'b',
    '\U0001D5BB': 'b',
    '\U0001D5EF': 'b',
    '\U0001D623': 'b',
    '\U0001D657': 'b',
    '\U0001D68B': 'b',
    '\U0001D401': 'b',
    '\U0001D435': 'b',
    '\U0001D469': 'b',
    '\U0001D4D1': 'b',
    '\U0001D505': 'b',
    '\U0001D539': 'b',
    '\U0001D56D': 'b',
    '\U0001D5A1': 'b',
    '\U0001D5D5': 'b',
    '\U0001D609': 'b',
    '\U0001D63D': 'b',
    '\U0001D671': 'b',
    '\U0001D6A9': 'b',
    '\U0001D6E3': 'b',
    '\U0001D71D': 'b',
    '\U0001D757': 'b',
    '\U0001D791': 'b',
    '\U00010282': 'b',
    '\U000102A1': 'b',
    '\U00010301': 'b',
    '\U0001D6C3': 'b',
    '\U0001D6FD': 'b',
    '\U0001D737': 'b',
    '\U0001D771': 'b',
    '\U0001D7AB': 'b',
    '\u0184': 'b',
    '\u042C': 'b',
    '\u13CF': 'b',
    '\u15AF': 'b',
    '\uFF22': 'b',
    '\u212C': 'b',
    '\uA7B4': 'b',
    '\u0392': 'b',
    '\u0412': 'b',
    '\u13F4': 'b',
    '\u15F7': 'b',
    '\uA4D0': 'b',
    '\u0253': 'b',
    '\u0183': 'b',
    '\u0182': 'b',
    '\u0411': 'b',
    '\u0180': 'b',
    '\u048D': 'b',
    '\u048C': 'b',
    '\u0463': 'b',
    '\u0462': 'b',
    '\u0432': 'b',
    '\u13FC': 'b',
    '\u0299': 'b',
    '\uA7B5': 'b',
    '\u03B2': 'b',
    '\u03D0': 'b',
    '\u13F0': 'b',
    '\u00DF': 'b',
    '\u042B': 'bl',
    '\U0001D41C': 'c',
    '\U0001D450': 'c',
    '\U0001D484': 'c',
    '\U0001D4B8': 'c',
    '\U0001D4EC': 'c',
    '\U0001D520': 'c',
    '\U0001D554': 'c',
    '\U0001D588': 'c',
    '\U0001D5BC': 'c',
    '\U0001D5F0': 'c',
    '\U0001D624': 'c',
    '\U0001D658': 'c',
    '\U0001D68C': 'c',
    '\U0001043D': 'c',
    '\U0001F74C': 'c',
    '\U000118F2': 'c',
    '\U000118E9': 'c',
    '\U0001D402': 'c',
    '\U0001D436': 'c',
    '\U0001D46A': 'c',
    '\U0001D49E': 'c',
    '\U0001D4D2': 'c',
    '\U0001D56E': 'c',
    '\U0001D5A2': 'c',
    '\U0001D5D6': 'c',
    '\U0001D60A': 'c',
    '\U0001D63E': 'c',
    '\U0001D672': 'c',
    '\U000102A2': 'c',
    '\U00010302': 'c',
    '\U00010415': 'c',
    '\U0001051C': 'c',
    '\uFF43': 'c',
    '\u217D': 'c',
    '\u1D04': 'c',
    '\u03F2': 'c',
    '\u2CA5': 'c',
    '\u0441': 'c',
    '\uABAF': 'c',
    '\u2DED': 'c',
    '\uFF23': 'c',
    '\u216D': 'c',
    '\u2102': 'c',
    '\u212D': 'c',
    '\u03F9': 'c',
    '\u2CA4': 'c',
    '\u0421': 'c',
    '\u13DF': 'c',
    '\uA4DA': 'c',
    '\u00A2': 'c',
    '\u023C': 'c',
    '\u20A1': 'c',
    '\u00E7': 'c',
    '\u04AB': 'c',
    '\u00C7': 'c',
    '\u04AA': 'c',
    '\u0187': 'c',
    '\U0001D41D': 'd',
    '\U0001D451': 'd',
    '\U0001D485': 'd',
    '\U0001D4B9': 'd',
    '\U0001D4ED': 'd',
    '\U0001D521': 'd',
    '\U0001D555': 'd',
    '\U0001D589': 'd',
    '\U0001D5BD': 'd',
    '\U0001D5F1': 'd',
    '\U0001D625': 'd',
    '\U0001D659': 'd',
    '\U0001D68D': 'd',
    '\U0001D403': 'd',
    '\U0001D437': 'd',
    '\U0001D46B': 'd',
    '\U0001D49F': 'd',
    '\U0001D4D3': 'd',
    '\U0001D507': 'd',
    '\U0001D53B': 'd',
    '\U0001D56F': 'd',
    '\U0001D5A3': 'd',
    '\U0001D5D7': 'd',
    '\U0001D60B': 'd',
    '\U0001D63F': 'd',
    '\U0001D673': 'd',
    '\u217E': 'd',
    '\u2146': 'd',
    '\u0501': 'd',
    '\u13E7': 'd',
    '\u146F': 'd',
    '\uA4D2': 'd',
    '\u216E': 'd',
    '\u2145': 'd',
    '\u13A0': 'd',
    '\u15DE': 'd',
    '\u15EA': 'd',
    '\uA4D3': 'd',
    '\u0257': 'd',
    '\u0256': 'd',
    '\u018C': 'd',
    '\u0111': 'd',
    '\u0110': 'd',
    '\u00D0': 'd',
    '\u0189': 'd',
    '\u20AB': 'd',
    '\u147B': 'd',
    '\u1487': 'd',
    '\uAB70': 'd',
    '\U0001D41E': 'e',
    '\U0001D452': 'e',
    '\U0001D486': 'e',
    '\U0001D4EE': 'e',
    '\U0001D522': 'e',
    '\U0001D556': 'e',
    '\U0001D58A': 'e',
    '\U0001D5BE': 'e',
    '\U0001D5F2': 'e',
    '\U0001D626': 'e',
    '\U0001D65A': 'e',
    '\U0001D68E': 'e',
    '\U0001D404': 'e',
    '\U0001D438': 'e',
    '\U0001D46C': 'e',
    '\U0001D4D4': 'e',
    '\U0001D508': 'e',
    '\U0001D53C': 'e',
    '\U0001D570': 'e',
    '\U0001D5A4': 'e',
    '\U0001D5D8': 'e',
    '\U0001D60C': 'e',
    '\U0001D640': 'e',
    '\U0001D674': 'e',
    '\U0001D6AC': 'e',
    '\U0001D6E6': 'e',
    '\U0001D720': 'e',
    '\U0001D75A': 'e',
    '\U0001D794': 'e',
    '\U000118A6': 'e',
    '\U000118AE': 'e',
    '\U00010286': 'e',
    '\U0001D221': 'e',
    '\u212E': 'e',
    '\uFF45': 'e',
    '\u212F': 'e',
    '\u2147': 'e',
    '\uAB32': 'e',
    '\u0435': 'e',
    '\u04BD': 'e',
    '\u2DF7': 'e',
    '\u22FF': 'e',
    '\uFF25': 'e',
    '\u2130': 'e',
    '\u0395': 'e',
    '\u0415': 'e',
    '\u2D39': 'e',
    '\u13AC': 'e',
    '\uA4F0': 'e',
    '\u011B': 'e',
    '\u011A': 'e',
    '\u0247': 'e',
    '\u0246': 'e',
    '\u04BF': 'e',
    '\uAB7C': 'e',
    '\u1D07': 'e',
    '\u0259': 'e',
    '\u01DD': 'e',
    '\u04D9': 'e',
    '\u2107': 'e',
    '\u0510': 'e',
    '\u13CB': 'e',
    '\U0001D41F': 'f',
    '\U0001D453': 'f',
    '\U0001D487': 'f',
    '\U0001D4BB': 'f',
    '\U0001D4EF': 'f',
    '\U0001D523': 'f',
    '\U0001D557': 'f',
    '\U0001D58B': 'f',
    '\U0001D5BF': 'f',
    '\U0001D5F3': 'f',
    '\U0001D627': 'f',
    '\U0001D65B': 'f',
    '\U0001D68F': 'f',
    '\U0001D213': 'f',
    '\U0001D405': 'f',
    '\U0001D439': 'f',
    '\U0001D46D': 'f',
    '\U0001D4D5': 'f',
    '\U0001D509': 'f',
    '\U0001D53D': 'f',
    '\U0001D571': 'f',
    '\U0001D5A5': 'f',
    '\U0001D5D9': 'f',
    '\U0001D60D': 'f',
    '\U0001D641': 'f',
    '\U0001D675': 'f',
    '\U0001D7CA': 'f',
    '\U000118C2': 'f',
    '\U000118A2': 'f',
    '\U00010287': 'f',
    '\U000102A5': 'f',
    '\U00010525': 'f',
    '\uAB35': 'f',
    '\uA799': 'f',
    '\u017F': 'f',
    '\u1E9D': 'f',
    '\u0584': 'f',
    '\u2131': 'f',
    '\uA798': 'f',
    '\u03DC': 'f',
    '\u15B4': 'f',
    '\uA4DD': 'f',
    '\u0192': 'f',
    '\u0191': 'f',
    '\u1D6E': 'f',
    '\uFB00': 'ff',
    '\uFB03': 'ffi',
    '\uFB04': 'ffl',
    '\uFB01': 'fi',
    '\uFB02': 'fl',
    '\u02A9': 'fn',
    '\U0001D420': 'g',
    '\U0001D454': 'g',
    '\U0001D488': 'g',
    '\U0001D4F0': 'g',
    '\U0001D524': 'g',
    '\U0001D558': 'g',
    '\U0001D58C': 'g',
    '\U0001D5C0': 'g',
    '\U0001D5F4': 'g',
    '\U0001D628': 'g',
    '\U0001D65C': 'g',
    '\U0001D690': 'g',
    '\U0001D406': 'g',
    '\U0001D43A': 'g',
    '\U0001D46E': 'g',
    '\U0001D4A2': 'g',
    '\U0001D4D6': 'g',
    '\U0001D50A': 'g',
    '\U0001D53E': 'g',
    '\U0001D572': 'g',
    '\U0001D5A6': 'g',
    '\U0001D5DA': 'g',
    '\U0001D60E': 'g',
    '\U0001D642': 'g',
    '\U0001D676': 'g',
    '\uFF47': 'g',
    '\u210A': 'g',
    '\u0261': 'g',
    '\u1D83': 'g',
    '\u018D': 'g',
    '\u0581': 'g',
    '\u050C': 'g',
    '\u13C0': 'g',
    '\u13F3': 'g',
    '\uA4D6': 'g',
    '\u1DA2': 'g',
    '\u1D4D': 'g',
    '\u0260': 'g',
    '\u01E7': 'g',
    '\u011F': 'g',
    '\u01E6': 'g',
    '\u011E': 'g',
    '\u01F5': 'g',
    '\u0123': 'g',
    '\u01E5': 'g',
    '\u01E4': 'g',
    '\u0193': 'g',
    '\u050D': 'g',
    '\uAB90': 'g',
    '\u13FB': 'g',
    '\U0001D421': 'h',
    '\U0001D489': 'h',
    '\U0001D4BD': 'h',
    '\U0001D4F1': 'h',
    '\U0001D525': 'h',
    '\U0001D559': 'h',
    '\U0001D58D': 'h',
    '\U0001D5C1': 'h',
    '\U0001D5F5': 'h',
    '\U0001D629': 'h',
    '\U0001D65D': 'h',
    '\U0001D691': 'h',
    '\U0001D407': 'h',
    '\U0001D43B': 'h',
    '\U0001D46F': 'h',
    '\U0001D4D7': 'h',
    '\U0001D573': 'h',
    '\U0001D5A7': 'h',
    '\U0001D5DB': 'h',
    '\U0001D60F': 'h',
    '\U0001D643': 'h',
    '\U0001D677': 'h',
    '\U0001D6AE': 'h',
    '\U0001D6E8': 'h',
    '\U0001D722': 'h',
    '\U0001D75C': 'h',
    '\U0001D796': 'h',
    '\U000102CF': 'h',
    '\U00010199': 'h',
    '\uFF48': 'h',
    '\u210E': 'h',
    '\u04BB': 'h',
    '\u0570': 'h',
    '\u13C2': 'h',
    '\uFF28': 'h',
    '\u210B': 'h',
    '\u210C': 'h',
    '\u210D': 'h',
    '\u0397': 'h',
    '\u2C8E': 'h',
    '\u041D': 'h',
    '\u13BB': 'h',
    '\u157C': 'h',
    '\uA4E7': 'h',
    '\u1D78': 'h',
    '\u1D34': 'h',
    '\u0266': 'h',
    '\uA695': 'h',
    '\u13F2': 'h',
    '\u2C67': 'h',
    '\u04A2': 'h',
    '\u0127': 'h',
    '\u210F': 'h',
    '\u045B': 'h',
    '\u0126': 'h',
    '\u04C9': 'h',
    '\u04C7': 'h',
    '\u043D': 'h',
    '\u029C': 'h',
    '\uAB8B': 'h',
    '\u04A3': 'h',
    '\u04CA': 'h',
    '\u04C8': 'h',
    '\U0001D422': 'i',
    '\U0001D456': 'i',
    '\U0001D48A': 'i',
    '\U0001D4BE': 'i',
    '\U0001D4F2': 'i',
    '\U0001D526': 'i',
    '\U0001D55A': 'i',
    '\U0001D58E': 'i',
    '\U0001D5C2': 'i',
    '\U0001D5F6': 'i',
    '\U0001D62A': 'i',
    '\U0001D65E': 'i',
    '\U0001D692': 'i',
    '\U0001D6A4': 'i',
    '\U0001D6CA': 'i',
    '\U0001D704': 'i',
    '\U0001D73E': 'i',
    '\U0001D778': 'i',
    '\U0001D7B2': 'i',
    '\U000118C3': 'i',
    '\u02DB': 'i',
    '\u2373': 'i',
    '\uFF49': 'i',
    '\u2170': 'i',
    '\u2139': 'i',
    '\u2148': 'i',
    '\u0131': 'i',
    '\u026A': 'i',
    '\u0269': 'i',
    '\u03B9': 'i',
    '\u1FBE': 'i',
    '\u037A': 'i',
    '\u0456': 'i',
    '\uA647': 'i',
    '\u04CF': 'i',
    '\uAB75': 'i',
    '\u13A5': 'i',
    '\u24DB': 'i',
    '\u2378': 'i',
    '\u01D0': 'i',
    '\u01CF': 'i',
    '\u0268': 'i',
    '\u1D7B': 'i',
    '\u1D7C': 'i',
    '\u2171': 'ii',
    '\u2172': 'iii',
    '\u0133': 'ij',
    '\u2173': 'iv',
    '\u2178': 'ix',
    '\U0001D423': 'j',
    '\U0001D457': 'j',
    '\U0001D48B': 'j',
    '\U0001D4BF': 'j',
    '\U0001D4F3': 'j',
    '\U0001D527': 'j',
    '\U0001D55B': 'j',
    '\U0001D58F': 'j',
    '\U0001D5C3': 'j',
    '\U0001D5F7': 'j',
    '\U0001D62B': 'j',
    '\U0001D65F': 'j',
    '\U0001D693': 'j',
    '\U0001D409': 'j',
    '\U0001D43D': 'j',
    '\U0001D471': 'j',
    '\U0001D4A5': 'j',
    '\U0001D4D9': 'j',
    '\U0001D50D': 'j',
    '\U0001D541': 'j',
    '\U0001D575': 'j',
    '\U0001D5A9': 'j',
    '\U0001D5DD': 'j',
    '\U0001D611': 'j',
    '\U0001D645': 'j',
    '\U0001D679': 'j',
    '\U0001D6A5': 'j',
    '\uFF4A': 'j',
    '\u2149': 'j',
    '\u03F3': 'j',
    '\u0458': 'j',
    '\uFF2A': 'j',
    '\uA7B2': 'j',
    '\u037F': 'j',
    '\u0408': 'j',
    '\u13AB': 'j',
    '\u148D': 'j',
    '\uA4D9': 'j',
    '\u0249': 'j',
    '\u0248': 'j',
    '\u1499': 'j',
    '\u0575': 'j',
    '\uAB7B': 'j',
    '\u1D0A': 'j',
    '\U0001D424': 'k',
    '\U0001D458': 'k',
    '\U0001D48C': 'k',
    '\U0001D4C0': 'k',
    '\U0001D4F4': 'k',
    '\U0001D528': 'k',
    '\U0001D55C': 'k',
    '\U0001D590': 'k',
    '\U0001D5C4': 'k',
    '\U0001D5F8': 'k',
    '\U0001D62C': 'k',
    '\U0001D660': 'k',
    '\U0001D694': 'k',
    '\U0001D40A': 'k',
    '\U0001D43E': 'k',
    '\U0001D472': 'k',
    '\U0001D4A6': 'k',
    '\U0001D4DA': 'k',
    '\U0001D50E': 'k',
    '\U0001D542': 'k',
    '\U0001D576': 'k',
    '\U0001D5AA': 'k',
    '\U0001D5DE': 'k',
    '\U0001D612': 'k',
    '\U0001D646': 'k',
    '\U0001D67A': 'k',
    '\U0001D6B1': 'k',
    '\U0001D6EB': 'k',
    '\U0001D725': 'k',
    '\U0001D75F': 'k',
    '\U0001D799': 'k',
    '\U0001D6CB': 'k',
    '\U0001D6DE': 'k',
    '\U0001D705': 'k',
    '\U0001D718': 'k',
    '\U0001D73F': 'k',
    '\U0001D752': 'k',
    '\U0001D779': 'k',
    '\U0001D78C': 'k',
    '\U0001D7B3': 'k',
    '\U0001D7C6': 'k',
    '\u212A': 'k',
    '\uFF2B': 'k',
    '\u039A': 'k',
    '\u2C94': 'k',
    '\u041A': 'k',
    '\u13E6': 'k',
    '\u16D5': 'k',
    '\uA4D7': 'k',
    '\u0199': 'k',
    '\u2C69': 'k',
    '\u049A': 'k',
    '\u20AD': 'k',
    '\uA740': 'k',
    '\u049E': 'k',
    '\u0198': 'k',
    '\u1D0B': 'k',
    '\u0138': 'k',
    '\u03BA': 'k',
    '\u03F0': 'k',
    '\u2C95': 'k',
    '\u043A': 'k',
    '\uABB6': 'k',
    '\u049B': 'k',
    '\u049F': 'k',
    '\U00010320': 'l',
    '\U0001E8C7': 'l',
    '\U0001D7CF': 'l',
    '\U0001D7D9': 'l',
    '\U0001D7E3': 'l',
    '\U0001D7ED': 'l',
    '\U0001D7F7': 'l',
    '\U0001D408': 'l',
    '\U0001D43C': 'l',
    '\U0001D470': 'l',
    '\U0001D4D8': 'l',
    '\U0001D540': 'l',
    '\U0001D574': 'l',
    '\U0001D5A8': 'l',
    '\U0001D5DC': 'l',
    '\U0001D610': 'l',
    '\U0001D644': 'l',
    '\U0001D678': 'l',
    '\U0001D425': 'l',
    '\U0001D459': 'l',
    '\U0001D48D': 'l',
    '\U0001D4C1': 'l',
    '\U0001D4F5': 'l',
    '\U0001D529': 'l',
    '\U0001D55D': 'l',
    '\U0001D591': 'l',
    '\U0001D5C5': 'l',
    '\U0001D5F9': 'l',
    '\U0001D62D': 'l',
    '\U0001D661': 'l',
    '\U0001D695': 'l',
    '\U0001D6B0': 'l',
    '\U0001D6EA': 'l',
    '\U0001D724': 'l',
    '\U0001D75E': 'l',
    '\U0001D798': 'l',
    '\U0001EE00': 'l',
    '\U0001EE80': 'l',
    '\U00016F28': 'l',
    '\U0001028A': 'l',
    '\U00010309': 'l',
    '\U0001D22A': 'l',
    '\U0001D40B': 'l',
    '\U0001D43F': 'l',
    '\U0001D473': 'l',
    '\U0001D4DB': 'l',
    '\U0001D50F': 'l',
    '\U0001D543': 'l',
    '\U0001D577': 'l',
    '\U0001D5AB': 'l',
    '\U0001D5DF': 'l',
    '\U0001D613': 'l',
    '\U0001D647': 'l',
    '\U0001D67B': 'l',
    '\U00016F16': 'l',
    '\U000118A3': 'l',
    '\U000118B2': 'l',
    '\U0001041B': 'l',
    '\U00010526': 'l',
    '\U00010443': 'l',
    '\u05C0': 'l',
    '\u007C': 'l',
    '\u2223': 'l',
    '\u23FD': 'l',
    '\uFFE8': 'l',
    '\u0031': 'l',
    '\u0661': 'l',
    '\u06F1': 'l',
    '\u0049': 'l',
    '\uFF29': 'l',
    '\u2160': 'l',
    '\u2110': 'l',
    '\u2111': 'l',
    '\u0196': 'l',
    '\uFF4C': 'l',
    '\u217C': 'l',
    '\u2113': 'l',
    '\u01C0': 'l',
    '\u0399': 'l',
    '\u2C92': 'l',
    '\u0406': 'l',
    '\u04C0': 'l',
    '\u05D5': 'l',
    '\u05DF': 'l',
    '\u0627': 'l',
    '\uFE8E': 'l',
    '\uFE8D': 'l',
    '\u07CA': 'l',
    '\u2D4F': 'l',
    '\u16C1': 'l',
    '\uA4F2': 'l',
    '\u216C': 'l',
    '\u2112': 'l',
    '\u2CD0': 'l',
    '\u13DE': 'l',
    '\u14AA': 'l',
    '\uA4E1': 'l',
    '\uFD3C': 'l',
    '\uFD3D': 'l',
    '\u0142': 'l',
    '\u0141': 'l',
    '\u026D': 'l',
    '\u0197': 'l',
    '\u019A': 'l',
    '\u026B': 'l',
    '\u0625': 'l',
    '\uFE88': 'l',
    '\uFE87': 'l',
    '\u0673': 'l',
    '\u0140': 'l',
    '\u013F': 'l',
    '\u14B7': 'l',
    '\u0623': 'l',
    '\uFE84': 'l',
    '\uFE83': 'l',
    '\u0672': 'l',
    '\u0675': 'l',
    '\u2CD1': 'l',
    '\uABAE': 'l',
    '\U0001F102': 'l.',
    '\u2488': 'l.',
    '\u01C9': 'lj',
    '\u0132': 'lj',
    '\u01C8': 'lj',
    '\u01C7': 'lj',
    '\u2016': 'll',
    '\u2225': 'll',
    '\u2161': 'll',
    '\u01C1': 'll',
    '\u05F0': 'll',
    '\u2162': 'lll',
    '\u02AA': 'ls',
    '\u20B6': 'lt',
    '\u2163': 'lv',
    '\u2168': 'lx',
    '\u02AB': 'lz',
    '\U0001D40C': 'm',
    '\U0001D440': 'm',
    '\U0001D474': 'm',
    '\U0001D4DC': 'm',
    '\U0001D510': 'm',
    '\U0001D544': 'm',
    '\U0001D578': 'm',
    '\U0001D5AC': 'm',
    '\U0001D5E0': 'm',
    '\U0001D614': 'm',
    '\U0001D648': 'm',
    '\U0001D67C': 'm',
    '\U0001D6B3': 'm',
    '\U0001D6ED': 'm',
    '\U0001D727': 'm',
    '\U0001D761': 'm',
    '\U0001D79B': 'm',
    '\U000102B0': 'm',
    '\U00010311': 'm',
    '\uFF2D': 'm',
    '\u216F': 'm',
    '\u2133': 'm',
    '\u039C': 'm',
    '\u03FA': 'm',
    '\u2C98': 'm',
    '\u041C': 'm',
    '\u13B7': 'm',
    '\u15F0': 'm',
    '\u16D6': 'm',
    '\uA4DF': 'm',
    '\u04CD': 'm',
    '\u2DE8': 'm',
    '\u1DDF': 'm',
    '\u1E43': 'm',
    '\U0001F76B': 'mb',
    '\U0001D427': 'n',
    '\U0001D45B': 'n',
    '\U0001D48F': 'n',
    '\U0001D4C3': 'n',
    '\U0001D4F7': 'n',
    '\U0001D52B': 'n',
    '\U0001D55F': 'n',
    '\U0001D593': 'n',
    '\U0001D5C7': 'n',
    '\U0001D5FB': 'n',
    '\U0001D62F': 'n',
    '\U0001D663': 'n',
    '\U0001D697': 'n',
    '\U0001D40D': 'n',
    '\U0001D441': 'n',
    '\U0001D475': 'n',
    '\U0001D4A9': 'n',
    '\U0001D4DD': 'n',
    '\U0001D511': 'n',
    '\U0001D579': 'n',
    '\U0001D5AD': 'n',
    '\U0001D5E1': 'n',
    '\U0001D615': 'n',
    '\U0001D649': 'n',
    '\U0001D67D': 'n',
    '\U0001D6B4': 'n',
    '\U0001D6EE': 'n',
    '\U0001D728': 'n',
    '\U0001D762': 'n',
    '\U0001D79C': 'n',
    '\U00010513': 'n',
    '\U0001018E': 'n',
    '\U0001D6C8': 'n',
    '\U0001D702': 'n',
    '\U0001D73C': 'n',
    '\U0001D776': 'n',
    '\U0001D7B0': 'n',
    '\U0001044D': 'n',
    '\u0578': 'n',
    '\u057C': 'n',
    '\uFF2E': 'n',
    '\u2115': 'n',
    '\u039D': 'n',
    '\u2C9A': 'n',
    '\uA4E0': 'n',
    '\u0273': 'n',
    '\u019E': 'n',
    '\u03B7': 'n',
    '\u019D': 'n',
    '\u1D70': 'n',
    '\u0146': 'n',
    '\u0272': 'n',
    '\u01CC': 'nj',
    '\u01CB': 'nj',
    '\u01CA': 'nj',
    '\u2116': 'no',
    '\U0001D428': 'o',
    '\U0001D45C': 'o',
    '\U0001D490': 'o',
    '\U0001D4F8': 'o',
    '\U0001D52C': 'o',
    '\U0001D560': 'o',
    '\U0001D594': 'o',
    '\U0001D5C8': 'o',
    '\U0001D5FC': 'o',
    '\U0001D630': 'o',
    '\U0001D664': 'o',
    '\U0001D698': 'o',
    '\U0001D6D0': 'o',
    '\U0001D70A': 'o',
    '\U0001D744': 'o',
    '\U0001D77E': 'o',
    '\U0001D7B8': 'o',
    '\U0001D6D4': 'o',
    '\U0001D70E': 'o',
    '\U0001D748': 'o',
    '\U0001D782': 'o',
    '\U0001D7BC': 'o',
    '\U0001EE24': 'o',
    '\U0001EE64': 'o',
    '\U0001EE84': 'o',
    '\U000104EA': 'o',
    '\U000118C8': 'o',
    '\U000118D7': 'o',
    '\U0001042C': 'o',
    '\U000114D0': 'o',
    '\U000118E0': 'o',
    '\U0001D7CE': 'o',
    '\U0001D7D8': 'o',
    '\U0001D7E2': 'o',
    '\U0001D7EC': 'o',
    '\U0001D7F6': 'o',
    '\U0001D40E': 'o',
    '\U0001D442': 'o',
    '\U0001D476': 'o',
    '\U0001D4AA': 'o',
    '\U0001D4DE': 'o',
    '\U0001D512': 'o',
    '\U0001D546': 'o',
    '\U0001D57A': 'o',
    '\U0001D5AE': 'o',
    '\U0001D5E2': 'o',
    '\U0001D616': 'o',
    '\U0001D64A': 'o',
    '\U0001D67E': 'o',
    '\U0001D6B6': 'o',
    '\U0001D6F0': 'o',
    '\U0001D72A': 'o',
    '\U0001D764': 'o',
    '\U0001D79E': 'o',
    '\U000104C2': 'o',
    '\U000118B5': 'o',
    '\U00010292': 'o',
    '\U000102AB': 'o',
    '\U00010404': 'o',
    '\U00010516': 'o',
    '\U0001D21A': 'o',
    '\U0001F714': 'o',
    '\U0001D6C9': 'o',
    '\U0001D6DD': 'o',
    '\U0001D703': 'o',
    '\U0001D717': 'o',
    '\U0001D73D': 'o',
    '\U0001D751': 'o',
    '\U0001D777': 'o',
    '\U0001D78B': 'o',
    '\U0001D7B1': 'o',
    '\U0001D7C5': 'o',
    '\U0001D6AF': 'o',
    '\U0001D6B9': 'o',
    '\U0001D6E9': 'o',
    '\U0001D6F3': 'o',
    '\U0001D723': 'o',
    '\U0001D72D': 'o',
    '\U0001D75D': 'o',
    '\U0001D767': 'o',
    '\U0001D797': 'o',
    '\U0001D7A1': 'o',
    '\u0C02': 'o',
    '\u0C82': 'o',
    '\u0D02': 'o',
    '\u0D82': 'o',
    '\u0966': 'o',
    '\u0A66': 'o',
    '\u0AE6': 'o',
    '\u0BE6': 'o',
    '\u0C66': 'o',
    '\u0CE6': 'o',
    '\u0D66': 'o',
    '\u0E50': 'o',
    '\u0ED0': 'o',
    '\u1040': 'o',
    '\u0665': 'o',
    '\u06F5': 'o',
    '\uFF4F': 'o',
    '\u2134': 'o',
    '\u1D0F': 'o',
    '\u1D11': 'o',
    '\uAB3D': 'o',
    '\u03BF': 'o',
    '\u03C3': 'o',
    '\u2C9F': 'o',
    '\u043E': 'o',
    '\u10FF': 'o',
    '\u0585': 'o',
    '\u05E1': 'o',
    '\u0647': 'o',
    '\uFEEB': 'o',
    '\uFEEC': 'o',
    '\uFEEA': 'o',
    '\uFEE9': 'o',
    '\u06BE': 'o',
    '\uFBAC': 'o',
    '\uFBAD': 'o',
    '\uFBAB': 'o',
    '\uFBAA': 'o',
    '\u06C1': 'o',
    '\uFBA8': 'o',
    '\uFBA9': 'o',
    '\uFBA7': 'o',
    '\uFBA6': 'o',
    '\u06D5': 'o',
    '\u0D20': 'o',
    '\u101D': 'o',
    '\u07C0': 'o',
    '\u09E6': 'o',
    '\u0B66': 'o',
    '\u3007': 'o',
    '\uFF2F': 'o',
    '\u039F': 'o',
    '\u2C9E': 'o',
    '\u041E': 'o',
    '\u0555': 'o',
    '\u2D54': 'o',
    '\u12D0': 'o',
    '\u0B20': 'o',
    '\uA4F3': 'o',
    '\u2070': 'o',
    '\u00BA': 'o',
    '\u1D52': 'o',
    '\u01D2': 'o',
    '\u014F': 'o',
    '\u01D1': 'o',
    '\u014E': 'o',
    '\u06FF': 'o',
    '\u00F8': 'o',
    '\uAB3E': 'o',
    '\u00D8': 'o',
    '\u2D41': 'o',
    '\u01FE': 'o',
    '\u0275': 'o',
    '\uA74B': 'o',
    '\u04E9': 'o',
    '\u0473': 'o',
    '\uAB8E': 'o',
    '\uABBB': 'o',
    '\u2296': 'o',
    '\u229D': 'o',
    '\u236C': 'o',
    '\u019F': 'o',
    '\uA74A': 'o',
    '\u03B8': 'o',
    '\u03D1': 'o',
    '\u0398': 'o',
    '\u03F4': 'o',
    '\u04E8': 'o',
    '\u0472': 'o',
    '\u2D31': 'o',
    '\u13BE': 'o',
    '\u13EB': 'o',
    '\uAB74': 'o',
    '\uFCD9': 'o',
    '\u01A1': 'o',
    '\u01A0': 'o',
    '\u13A4': 'o',
    '\U0001F101': 'o.',
    '\U0001F100': 'o.',
    '\u0153': 'oe',
    '\u0152': 'oe',
    '\u0276': 'oe',
    '\u221E': 'oo',
    '\uA74F': 'oo',
    '\uA699': 'oo',
    '\uA74E': 'oo',
    '\uA698': 'oo',
    '\u1010': 'oo',
    '\U0001D429': 'p',
    '\U0001D45D': 'p',
    '\U0001D491': 'p',
    '\U0001D4C5': 'p',
    '\U0001D4F9': 'p',
    '\U0001D52D': 'p',
    '\U0001D561': 'p',
    '\U0001D595': 'p',
    '\U0001D5C9': 'p',
    '\U0001D5FD': 'p',
    '\U0001D631': 'p',
    '\U0001D665': 'p',
    '\U0001D699': 'p',
    '\U0001D6D2': 'p',
    '\U0001D6E0': 'p',
    '\U0001D70C': 'p',
    '\U0001D71A': 'p',
    '\U0001D746': 'p',
    '\U0001D754': 'p',
    '\U0001D780': 'p',
    '\U0001D78E': 'p',
    '\U0001D7BA': 'p',
    '\U0001D7C8': 'p',
    '\U0001D40F': 'p',
    '\U0001D443': 'p',
    '\U0001D477': 'p',
    '\U0001D4AB': 'p',
    '\U0001D4DF': 'p',
    '\U0001D513': 'p',
    '\U0001D57B': 'p',
    '\U0001D5AF': 'p',
    '\U0001D5E3': 'p',
    '\U0001D617': 'p',
    '\U0001D64B': 'p',
    '\U0001D67F': 'p',
    '\U0001D6B8': 'p',
    '\U0001D6F2': 'p',
    '\U0001D72C': 'p',
    '\U0001D766': 'p',
    '\U0001D7A0': 'p',
    '\U00010295': 'p',
    '\u2374': 'p',
    '\uFF50': 'p',
    '\u03C1': 'p',
    '\u03F1': 'p',
    '\u2CA3': 'p',
    '\u0440': 'p',
    '\uFF30': 'p',
    '\u2119': 'p',
    '\u03A1': 'p',
    '\u2CA2': 'p',
    '\u0420': 'p',
    '\u13E2': 'p',
    '\u146D': 'p',
    '\uA4D1': 'p',
    '\u01A5': 'p',
    '\u1D7D': 'p',
    '\u1477': 'p',
    '\u1486': 'p',
    '\u1D29': 'p',
    '\uABB2': 'p',
    '\U0001D42A': 'q',
    '\U0001D45E': 'q',
    '\U0001D492': 'q',
    '\U0001D4C6': 'q',
    '\U0001D4FA': 'q',
    '\U0001D52E': 'q',
    '\U0001D562': 'q',
    '\U0001D596': 'q',
    '\U0001D5CA': 'q',
    '\U0001D5FE': 'q',
    '\U0001D632': 'q',
    '\U0001D666': 'q',
    '\U0001D69A': 'q',
    '\U0001D410': 'q',
    '\U0001D444': 'q',
    '\U0001D478': 'q',
    '\U0001D4AC': 'q',
    '\U0001D4E0': 'q',
    '\U0001D514': 'q',
    '\U0001D57C': 'q',
    '\U0001D5B0': 'q',
    '\U0001D5E4': 'q',
    '\U0001D618': 'q',
    '\U0001D64C': 'q',
    '\U0001D680': 'q',
    '\u051B': 'q',
    '\u0563': 'q',
    '\u0566': 'q',
    '\u211A': 'q',
    '\u2D55': 'q',
    '\u02A0': 'q',
    '\u1D90': 'q',
    '\u024B': 'q',
    '\U0001D42B': 'r',
    '\U0001D45F': 'r',
    '\U0001D493': 'r',
    '\U0001D4C7': 'r',
    '\U0001D4FB': 'r',
    '\U0001D52F': 'r',
    '\U0001D563': 'r',
    '\U0001D597': 'r',
    '\U0001D5CB': 'r',
    '\U0001D5FF': 'r',
    '\U0001D633': 'r',
    '\U0001D667': 'r',
    '\U0001D69B': 'r',
    '\U0001D216': 'r',
    '\U0001D411': 'r',
    '\U0001D445': 'r',
    '\U0001D479': 'r',
    '\U0001D4E1': 'r',
    '\U0001D57D': 'r',
    '\U0001D5B1': 'r',
    '\U0001D5E5': 'r',
    '\U0001D619': 'r',
    '\U0001D64D': 'r',
    '\U0001D681': 'r',
    '\U000104B4': 'r',
    '\uAB47': 'r',
    '\uAB48': 'r',
    '\u1D26': 'r',
    '\u2C85': 'r',
    '\u0433': 'r',
    '\uAB81': 'r',
    '\u211B': 'r',
    '\u211C': 'r',
    '\u211D': 'r',
    '\u01A6': 'r',
    '\u13A1': 'r',
    '\u13D2': 'r',
    '\u1587': 'r',
    '\uA4E3': 'r',
    '\u027D': 'r',
    '\u027C': 'r',
    '\u024D': 'r',
    '\u0493': 'r',
    '\u1D72': 'r',
    '\u0491': 'r',
    '\uAB71': 'r',
    '\u0280': 'r',
    '\uABA2': 'r',
    '\u1D73': 'r',
    '\U000118E3': 'rn',
    '\U0001D426': 'rn',
    '\U0001D45A': 'rn',
    '\U0001D48E': 'rn',
    '\U0001D4C2': 'rn',
    '\U0001D4F6': 'rn',
    '\U0001D52A': 'rn',
    '\U0001D55E': 'rn',
    '\U0001D592': 'rn',
    '\U0001D5C6': 'rn',
    '\U0001D5FA': 'rn',
    '\U0001D62E': 'rn',
    '\U0001D662': 'rn',
    '\U0001D696': 'rn',
    '\U00011700': 'rn',
    '\u217F': 'rn',
    '\u20A5': 'rn',
    '\u0271': 'rn',
    '\u1D6F': 'rn',
    '\U0001D42C': 's',
    '\U0001D460': 's',
    '\U0001D494': 's',
    '\U0001D4C8': 's',
    '\U0001D4FC': 's',
    '\U0001D530': 's',
    '\U0001D564': 's',
    '\U0001D598': 's',
    '\U0001D5CC': 's',
    '\U0001D600': 's',
    '\U0001D634': 's',
    '\U0001D668': 's',
    '\U0001D69C': 's',
    '\U000118C1': 's',
    '\U00010448': 's',
    '\U0001D412': 's',
    '\U0001D446': 's',
    '\U0001D47A': 's',
    '\U0001D4AE': 's',
    '\U0001D4E2': 's',
    '\U0001D516': 's',
    '\U0001D54A': 's',
    '\U0001D57E': 's',
    '\U0001D5B2': 's',
    '\U0001D5E6': 's',
    '\U0001D61A': 's',
    '\U0001D64E': 's',
    '\U0001D682': 's',
    '\U00016F3A': 's',
    '\U00010296': 's',
    '\U00010420': 's',
    '\uFF53': 's',
    '\uA731': 's',
    '\u01BD': 's',
    '\u0455': 's',
    '\uABAA': 's',
    '\uFF33': 's',
    '\u0405': 's',
    '\u054F': 's',
    '\u13D5': 's',
    '\u13DA': 's',
    '\uA4E2': 's',
    '\u0282': 's',
    '\u1D74': 's',
    '\U0001F75C': 'sss',
    '\uFB06': 'st',
    '\U0001D42D': 't',
    '\U0001D461': 't',
    '\U0001D495': 't',
    '\U0001D4C9': 't',
    '\U0001D4FD': 't',
    '\U0001D531': 't',
    '\U0001D565': 't',
    '\U0001D599': 't',
    '\U0001D5CD': 't',
    '\U0001D601': 't',
    '\U0001D635': 't',
    '\U0001D669': 't',
    '\U0001D69D': 't',
    '\U0001F768': 't',
    '\U0001D413': 't',
    '\U0001D447': 't',
    '\U0001D47B': 't',
    '\U0001D4AF': 't',
    '\U0001D4E3': 't',
    '\U0001D517': 't',
    '\U0001D54B': 't',
    '\U0001D57F': 't',
    '\U0001D5B3': 't',
    '\U0001D5E7': 't',
    '\U0001D61B': 't',
    '\U0001D64F': 't',
    '\U0001D683': 't',
    '\U0001D6BB': 't',
    '\U0001D6F5': 't',
    '\U0001D72F': 't',
    '\U0001D769': 't',
    '\U0001D7A3': 't',
    '\U00016F0A': 't',
    '\U000118BC': 't',
    '\U00010297': 't',
    '\U000102B1': 't',
    '\U00010315': 't',
    '\U0001D6D5': 't',
    '\U0001D70F': 't',
    '\U0001D749': 't',
    '\U0001D783': 't',
    '\U0001D7BD': 't',
    '\u22A4': 't',
    '\u27D9': 't',
    '\uFF34': 't',
    '\u03A4': 't',
    '\u2CA6': 't',
    '\u0422': 't',
    '\u13A2': 't',
    '\uA4D4': 't',
    '\u2361': 't',
    '\u023E': 't',
    '\u021A': 't',
    '\u0162': 't',
    '\u01AE': 't',
    '\u04AC': 't',
    '\u20AE': 't',
    '\u0167': 't',
    '\u0166': 't',
    '\u1D75': 't',
    '\U0001D42E': '',
    '\U0001D462': '',
    '\U0001D496': '',
    '\U0001D4CA': '',
    '\U0001D4FE': '',
    '\U0001D532': '',
    '\U0001D566': '',
    '\U0001D59A': '',
    '\U0001D5CE': '',
    '\U0001D602': '',
    '\U0001D636': '',
    '\U0001D66A': '',
    '\U0001D69E': '',
    '\U0001D6D6': '',
    '\U0001D710': '',
    '\U0001D74A': '',
    '\U0001D784': '',
    '\U0001D7BE': '',
    '\U000104F6': '',
    '\U000118D8': '',
    '\U0001D414': '',
    '\U0001D448': '',
    '\U0001D47C': '',
    '\U0001D4B0': '',
    '\U0001D4E4': '',
    '\U0001D518': '',
    '\U0001D54C': '',
    '\U0001D580': '',
    '\U0001D5B4': '',
    '\U0001D5E8': '',
    '\U0001D61C': '',
    '\U0001D650': '',
    '\U0001D684': '',
    '\U000104CE': '',
    '\U00016F42': '',
    '\U000118B8': '',
    '\uA79F': '',
    '\u1D1C': '',
    '\uAB4E': '',
    '\uAB52': '',
    '\u028B': '',
    '\u03C5': '',
    '\u057D': '',
    '\u222A': '',
    '\u22C3': '',
    '\u054D': '',
    '\u1200': '',
    '\u144C': '',
    '\uA4F4': '',
    '\u01D4': '',
    '\u01D3': '',
    '\u1D7E': '',
    '\uAB9C': '',
    '\u0244': '',
    '\u13CC': '',
    '\u1458': '',
    '\u1467': '',
    '\u2127': '',
    '\u162E': '',
    '\u1634': '',
    '\u01B1': '',
    '\u1D7F': '',
    '\u1D6B': 'ue',
    '\uAB63': 'uo',
    '\U0001D42F': 'v',
    '\U0001D463': 'v',
    '\U0001D497': 'v',
    '\U0001D4CB': 'v',
    '\U0001D4FF': 'v',
    '\U0001D533': 'v',
    '\U0001D567': 'v',
    '\U0001D59B': 'v',
    '\U0001D5CF': 'v',
    '\U0001D603': 'v',
    '\U0001D637': 'v',
    '\U0001D66B': 'v',
    '\U0001D69F': 'v',
    '\U0001D6CE': 'v',
    '\U0001D708': 'v',
    '\U0001D742': 'v',
    '\U0001D77C': 'v',
    '\U0001D7B6': 'v',
    '\U00011706': 'v',
    '\U000118C0': 'v',
    '\U0001D20D': 'v',
    '\U0001D415': 'v',
    '\U0001D449': 'v',
    '\U0001D47D': 'v',
    '\U0001D4B1': 'v',
    '\U0001D4E5': 'v',
    '\U0001D519': 'v',
    '\U0001D54D': 'v',
    '\U0001D581': 'v',
    '\U0001D5B5': 'v',
    '\U0001D5E9': 'v',
    '\U0001D61D': 'v',
    '\U0001D651': 'v',
    '\U0001D685': 'v',
    '\U00016F08': 'v',
    '\U000118A0': 'v',
    '\U0001051D': 'v',
    '\U00010197': 'v',
    '\U0001F708': 'v',
    '\u2228': 'v',
    '\u22C1': 'v',
    '\uFF56': 'v',
    '\u2174': 'v',
    '\u1D20': 'v',
    '\u03BD': 'v',
    '\u0475': 'v',
    '\u05D8': 'v',
    '\uABA9': 'v',
    '\u0667': 'v',
    '\u06F7': 'v',
    '\u2164': 'v',
    '\u0474': 'v',
    '\u2D38': 'v',
    '\u13D9': 'v',
    '\u142F': 'v',
    '\uA6DF': 'v',
    '\uA4E6': 'v',
    '\u143B': 'v',
    '\U0001F76C': 'vb',
    '\u2175': 'vi',
    '\u2176': 'vii',
    '\u2177': 'viii',
    '\u2165': 'vl',
    '\u2166': 'vll',
    '\u2167': 'vlll',
    '\U0001D430': 'w',
    '\U0001D464': 'w',
    '\U0001D498': 'w',
    '\U0001D4CC': 'w',
    '\U0001D500': 'w',
    '\U0001D534': 'w',
    '\U0001D568': 'w',
    '\U0001D59C': 'w',
    '\U0001D5D0': 'w',
    '\U0001D604': 'w',
    '\U0001D638': 'w',
    '\U0001D66C': 'w',
    '\U0001D6A0': 'w',
    '\U0001170A': 'w',
    '\U0001170E': 'w',
    '\U0001170F': 'w',
    '\U000118EF': 'w',
    '\U000118E6': 'w',
    '\U0001D416': 'w',
    '\U0001D44A': 'w',
    '\U0001D47E': 'w',
    '\U0001D4B2': 'w',
    '\U0001D4E6': 'w',
    '\U0001D51A': 'w',
    '\U0001D54E': 'w',
    '\U0001D582': 'w',
    '\U0001D5B6': 'w',
    '\U0001D5EA': 'w',
    '\U0001D61E': 'w',
    '\U0001D652': 'w',
    '\U0001D686': 'w',
    '\U000114C5': 'w',
    '\u026F': 'w',
    '\u1D21': 'w',
    '\u0461': 'w',
    '\u051D': 'w',
    '\u0561': 'w',
    '\uAB83': 'w',
    '\u051C': 'w',
    '\u13B3': 'w',
    '\u13D4': 'w',
    '\uA4EA': 'w',
    '\u047D': 'w',
    '\u20A9': 'w',
    '\uA761': 'w',
    '\U0001D431': 'x',
    '\U0001D465': 'x',
    '\U0001D499': 'x',
    '\U0001D4CD': 'x',
    '\U0001D501': 'x',
    '\U0001D535': 'x',
    '\U0001D569': 'x',
    '\U0001D59D': 'x',
    '\U0001D5D1': 'x',
    '\U0001D605': 'x',
    '\U0001D639': 'x',
    '\U0001D66D': 'x',
    '\U0001D6A1': 'x',
    '\U00010322': 'x',
    '\U000118EC': 'x',
    '\U0001D417': 'x',
    '\U0001D44B': 'x',
    '\U0001D47F': 'x',
    '\U0001D4B3': 'x',
    '\U0001D4E7': 'x',
    '\U0001D51B': 'x',
    '\U0001D54F': 'x',
    '\U0001D583': 'x',
    '\U0001D5B7': 'x',
    '\U0001D5EB': 'x',
    '\U0001D61F': 'x',
    '\U0001D653': 'x',
    '\U0001D687': 'x',
    '\U0001D6BE': 'x',
    '\U0001D6F8': 'x',
    '\U0001D732': 'x',
    '\U0001D76C': 'x',
    '\U0001D7A6': 'x',
    '\U00010290': 'x',
    '\U000102B4': 'x',
    '\U00010317': 'x',
    '\U00010527': 'x',
    '\U00010196': 'x',
    '\u166E': 'x',
    '\u00D7': 'x',
    '\u292B': 'x',
    '\u292C': 'x',
    '\u2A2F': 'x',
    '\uFF58': 'x',
    '\u2179': 'x',
    '\u0445': 'x',
    '\u1541': 'x',
    '\u157D': 'x',
    '\u2DEF': 'x',
    '\u036F': 'x',
    '\u166D': 'x',
    '\u2573': 'x',
    '\uFF38': 'x',
    '\u2169': 'x',
    '\uA7B3': 'x',
    '\u03A7': 'x',
    '\u2CAC': 'x',
    '\u0425': 'x',
    '\u2D5D': 'x',
    '\u16B7': 'x',
    '\uA4EB': 'x',
    '\u2A30': 'x',
    '\u04B2': 'x',
    '\u217A': 'xi',
    '\u217B': 'xii',
    '\u216A': 'xl',
    '\u216B': 'xll',
    '\U0001D432': 'y',
    '\U0001D466': 'y',
    '\U0001D49A': 'y',
    '\U0001D4CE': 'y',
    '\U0001D502': 'y',
    '\U0001D536': 'y',
    '\U0001D56A': 'y',
    '\U0001D59E': 'y',
    '\U0001D5D2': 'y',
    '\U0001D606': 'y',
    '\U0001D63A': 'y',
    '\U0001D66E': 'y',
    '\U0001D6A2': 'y',
    '\U0001D6C4': 'y',
    '\U0001D6FE': 'y',
    '\U0001D738': 'y',
    '\U0001D772': 'y',
    '\U0001D7AC': 'y',
    '\U000118DC': 'y',
    '\U0001D418': 'y',
    '\U0001D44C': 'y',
    '\U0001D480': 'y',
    '\U0001D4B4': 'y',
    '\U0001D4E8': 'y',
    '\U0001D51C': 'y',
    '\U0001D550': 'y',
    '\U0001D584': 'y',
    '\U0001D5B8': 'y',
    '\U0001D5EC': 'y',
    '\U0001D620': 'y',
    '\U0001D654': 'y',
    '\U0001D688': 'y',
    '\U0001D6BC': 'y',
    '\U0001D6F6': 'y',
    '\U0001D730': 'y',
    '\U0001D76A': 'y',
    '\U0001D7A4': 'y',
    '\U00016F43': 'y',
    '\U000118A4': 'y',
    '\U000102B2': 'y',
    '\u0263': 'y',
    '\u1D8C': 'y',
    '\uFF59': 'y',
    '\u028F': 'y',
    '\u1EFF': 'y',
    '\uAB5A': 'y',
    '\u03B3': 'y',
    '\u213D': 'y',
    '\u0443': 'y',
    '\u04AF': 'y',
    '\u10E7': 'y',
    '\uFF39': 'y',
    '\u03A5': 'y',
    '\u03D2': 'y',
    '\u2CA8': 'y',
    '\u0423': 'y',
    '\u04AE': 'y',
    '\u13A9': 'y',
    '\u13BD': 'y',
    '\uA4EC': 'y',
    '\u01B4': 'y',
    '\u024F': 'y',
    '\u04B1': 'y',
    '\u00A5': 'y',
    '\u024E': 'y',
    '\u04B0': 'y',
    '\U0001D433': 'z',
    '\U0001D467': 'z',
    '\U0001D49B': 'z',
    '\U0001D4CF': 'z',
    '\U0001D503': 'z',
    '\U0001D537': 'z',
    '\U0001D56B': 'z',
    '\U0001D59F': 'z',
    '\U0001D5D3': 'z',
    '\U0001D607': 'z',
    '\U0001D63B': 'z',
    '\U0001D66F': 'z',
    '\U0001D6A3': 'z',
    '\U000118C4': 'z',
    '\U000102F5': 'z',
    '\U000118E5': 'z',
    '\U0001D419': 'z',
    '\U0001D44D': 'z',
    '\U0001D481': 'z',
    '\U0001D4B5': 'z',
    '\U0001D4E9': 'z',
    '\U0001D585': 'z',
    '\U0001D5B9': 'z',
    '\U0001D5ED': 'z',
    '\U0001D621': 'z',
    '\U0001D655': 'z',
    '\U0001D689': 'z',
    '\U0001D6AD': 'z',
    '\U0001D6E7': 'z',
    '\U0001D721': 'z',
    '\U0001D75B': 'z',
    '\U0001D795': 'z',
    '\U000118A9': 'z',
    '\u1D22': 'z',
    '\uAB93': 'z',
    '\uFF3A': 'z',
    '\u2124': 'z',
    '\u2128': 'z',
    '\u0396': 'z',
    '\u13C3': 'z',
    '\uA4DC': 'z',
    '\u0290': 'z',
    '\u01B6': 'z',
    '\u01B5': 'z',
    '\u0225': 'z',
    '\u0224': 'z',
    '\u1D76': 'z',
    '\u2010': '-',
    '\u2011': '-',
    '\u2012': '-',
    '\u2013': '-',
    '\uFE58': '-',
    '\u06D4': '-',
    '\u2043': '-',
    '\u02D7': '-',
    '\u2212': '-',
    '\u2796': '-',
    '\u2CBA': '-'
}


def unconfuse(value):
    try:
        string = ''
        for i in range(len(value)):
            string += confusables.get(value[i], value[i])
    except UnicodeError:
        string = value
    return string


def is_ascii(value):
    asciionly = True
    try:
        # only unconfuse if non-ascii characters are found
        if hasattr(value, 'decode'):
            value.decode('ascii')
        if hasattr(value, 'encode'):
            value.encode('ascii')
    except (UnicodeDecodeError, UnicodeEncodeError):
        asciionly = False
    return asciionly


