/**
 * Implementation of the orbital elements Neural Network
 * for the Cambioni et al. (2019) model.
 *
 * Copyright (c) 2019 Arizona Board of Regents
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * @file
 * @author Alexandre Emsenhuber
 * @author Saverio Cambioni
 */

#include <math.h>

#include "orbital_hnr.h"

/**
 * Neural network predicting the relative orbit of the two largest remnants for Hit-and-run collisions.
 *
 * @param X predictors. Array of 4 values:
 *          - log10( mass of the target / Earth mass )
 *          - ratio mass projectile / mass target
 *          - impact angle [degree]
 *          - ration impact velocity / escape velocity
 *
 * @param Y output. Array of 3 values:
 *          - E: specific orbital energy of the second remnant; scaled to the kinetic energy at the mutual escape velocity
 *          - b: impact parameter
 *          - delta omega: shift of the pericentre [rad] in the direction of motion
 */
void orbital_hnr( const double X[4], double Y[3] ) {
	static const double bi[ 4 ] = { -0.993071593533487, 0.421131639722861, 56.4461893764434, 2.05981524249423 };
	static const double ai[ 4 ] = { 1.38570465236717, 4.05371898028927, 0.0480247338292496, 1.05551091905284 };

	double v1[ 4 ];

	static const double a1[ 4 ][ 50 ] = {
		{ -0.0027061582690720942, 0.031137184615957002, 0.0020390703987558712, 0.015175695397017502, 0.081042219519364384, -0.10714015360163014, -0.063767149660569816, -0.019025099833497164, -0.040243580810943841, -0.038781412917220745, 0.018621508824334932, -0.13556078343919417, -0.0066157604471441837, 0.01739778118387552, -0.0040616543341302344, -0.020707009840123591, 0.0018715168796718949, -0.014862494827099792, 0.055357446871101806, -0.0083224152154333843, -0.012238357896120374, 0.013908621391893788, 0.012822597411728663, -0.015069257003365229, -0.048470234449970222, 0.2353248090711802, -0.045149403703562158, -0.024490892142435852, -0.057910668077622622, 0.011985850068546449, -0.027414891341553886, -0.013766287780923397, -0.010837904468099057, 0.1083246281531747, 0.06576627101548585, -0.036028428823958895, 0.048535133993103025, -0.12958792076934536, 0.039716822911441345, -0.058452400447076377, 0.007369920397830197, -0.080303379636972538, 0.015296772594337184, -0.019513981359445574, 0.072996897875807112, -0.079256432726445711, 0.071646722961807546, 0.035099854144594524, 0.02038464795002368, 0.013579829285189287 },
		{ 0.0039651739968073179, 0.021195937341609298, 0.004174727916935881, -0.075256491614245011, 0.072224322292913284, 0.041014671371140232, 0.078017429169209535, -0.10065037186307743, -0.040323572100502136, 0.10134146692766249, 0.01647774455478054, -0.017978393491584775, 0.036576727354274856, -0.073002458169402012, -0.014656624271151043, -0.028838996379079409, 0.029316971361692229, -0.10804091241575288, 0.027685757744644265, -0.019490155025792578, 0.012178046425368766, 0.0099856632368875257, 0.051745867574737495, -0.062642791273315457, -0.092412651794261558, -0.094670511829825041, -0.06745798907173764, -0.023542594353021716, -0.037988167147562778, 0.01186726127580537, -0.0892600464459177, 0.013410433510013505, 0.12543246454444348, 0.11597728450816741, -0.034523893473478916, 0.022390732357038545, 0.0323347158840256, 0.0854727499590982, 0.026150549782661941, -0.0147949336679527, 0.12069783238143302, -0.020613276003575355, 0.025564883158426321, -0.00247529445623122, 0.11761352638988355, 0.019034019287013624, 0.01460606829085326, -0.076576495433923225, 0.020935651781985982, 0.0177335833490868 },
		{ 0.10597595358526569, -0.038465629043312154, 0.062328391064758487, -0.076443514668778717, -0.016605005898046653, -0.52332157086705866, 0.089174125782190042, -0.060238658036555918, 0.23613935871375563, 0.13158974073005222, 0.12935941322695671, -0.20535158804662362, -0.18751045651319379, -0.40786719292612056, 0.085551368672105527, 0.048052807881673751, -0.14108504920045942, -0.25480239107202995, 0.10367522688453835, -0.10602416934680783, 0.17968838857836794, 0.39030923228263203, 0.14159333735922075, -0.020772631782322883, 0.025997099088773078, 0.3348867692385345, -0.050831475139305748, 0.10249467078161323, -0.17017561703616516, 0.13530177088865272, -0.1904192849832658, -0.10090401647283793, -0.0064473382486891989, 0.23769749172470409, -0.16415376337659301, -0.095175933349746625, -0.14575760124799661, 0.18900364110673737, -0.24934579868623241, 0.23576469526367055, 0.095718669437051929, 0.19726239680646454, -0.050608813274325495, -0.038479728762542444, 0.092656999871271156, 0.20399912117723321, -0.10471651962129433, -0.13198119990236898, 0.10265839443289716, -0.047353705329025539 },
		{ -0.164486211142471, -0.097944132938435166, 0.012628230836967276, 0.15592331374700766, -0.074610591165832726, -0.17339092468164727, 0.16911868054100426, -0.17898561683484637, -0.26291144681441564, -0.080947630748273089, 0.12845386878343668, -0.063564704679154621, 0.26517641317652513, -0.11322555182497031, -0.15901708304316026, 0.082499618880079617, 0.087419439948208874, 0.016178468184188284, 0.085295558275087446, 0.17905477850986795, -0.24333793763751146, 0.1746853735343683, -0.183763966370477, -0.039678716032535574, 0.077892798714868747, 0.19659795919590856, 0.13785592224411466, 0.09040758698133998, 0.22574247501588535, -0.12917520202750993, 0.15020701314396093, -0.019451868904418235, 0.13681220437592953, -0.026225409985623373, 0.033982087627934879, 0.077684712829060892, -0.080667091811249467, 0.15454909753619031, -0.095358429797797781, 0.077457744823488175, 0.015268386923507378, 0.056921655415501426, -0.22317227819308108, 0.11247324895860765, -0.03345228784295879, -0.046765061223084665, -0.04325230821845432, 0.021249950627368633, 0.14507495890643016, 0.1595397173943264 }
	};
	static const double b1[ 50 ] = { 0.26981523315636835, 0.21153443436905514, -0.15187139856956314, 0.16826813749517772, -0.1019796369440379, -0.42636455668608531, -0.20107493486527758, 0.26941551971453359, -0.24162781341062725, 0.23981785320643467, -0.15215289419030226, 0.011120295483527184, 0.079873007119526981, -0.36433107526355979, 0.035411787394939047, 0.069163278464226513, -0.16921542973594964, -0.29601747520071148, 0.00081580306942439845, 0.024851920483998331, 0.076211098309921857, 0.33350159990662143, 0.05848681875390066, -0.01124625366625983, -0.11949708889554184, -0.093346229924875346, -0.15752601847543721, -0.012986566995229053, -0.20611222419007907, 0.0912503158792304, -0.24388753291692064, 0.021263913622168149, 0.086924557391025059, 0.21504048993667954, -0.053589685456102175, -0.015834236684686274, -0.061710561428211214, -0.12611650272142233, -0.18795071508200051, 0.16374704063757076, 0.15863123179351624, 0.19703699056047452, 0.34045480782446214, 0.084430302926531753, 0.056832587162494286, 0.29560124131307175, 0.13905412929621824, -0.20546320205359042, -0.17996650672943534, -0.33687940511762088 };

	double v2[ 50 ];

  static const double a2[ 50 ][ 45 ] = {
		{ 0.012116665484604066, -0.029478348714572229, 0.035628124562786093, -0.029868291200756774, 4.4809399577067788E-5, -0.014327215834108236, 0.015892027250513044, -0.12492697152817094, -0.031223557883815943, -0.044063035122475537, 0.0040904986950476147, -0.078768320653327076, 0.026207296529852223, 0.00366582999723986, -0.0082058743197017427, 0.016789252799277083, -0.06111911409104865, -0.03407336536399113, -0.047899778024147972, -0.02625048290345891, -0.05214933448668678, 0.024057707280476178, 0.10538096854898142, 0.05972500507208095, 0.021421753488960495, -0.15075231562514496, 0.19874779910802681, 0.0060855374446356419, -0.088548351918847759, -0.029912122562984689, -0.029216221202457742, -0.038429937706843914, 0.04857927188575218, -0.02182539650339535, 0.035986961332470285, -0.034105649576769904, 0.092218598018006359, -0.069765211904929875, -0.035135417500049587, 0.057552754797801947, -0.044084250198787533, 0.023349737170912922, -0.011538900945338226, 0.013072868811447567, 0.01022296601748727 },
		{ 0.019583510997436413, -0.052040845675677738, 0.058212546670183586, -0.03966299961297385, -0.0019068156845082663, -7.8951081023338439E-5, -0.016646548643760195, -0.067812216632219507, -0.047539914117476108, -3.686981999482069E-5, -0.019860240666615989, -0.056868075345691826, 0.047272991836137285, 0.0035076836028534674, 0.0033928899075047554, 0.040523231771691204, -0.048971683386223958, -0.033214335414855654, -0.015720110748009487, -0.029877699136155185, -0.0020473536823993571, -0.010054189702701192, 0.036973146348836645, 0.018158371776993427, 0.0090751402564573438, -0.035244376332934146, 0.15939738793545252, 0.030651182003260315, -0.086398803701622187, -0.025876149058382394, -0.0011398447001491402, -0.028535103871286705, 0.05171280094337842, -0.0016736363624494485, 0.0076263395783786906, -0.0041745837602637607, 0.069909509565552475, -0.032675843828095213, -0.052888330355021, 0.01102399789018565, -0.018378445468885229, 0.034820362950457012, -0.036575164821951556, -0.0353843079604502, 0.01059986843311064 },
		{ -0.016350924839581556, 0.047346056936069046, -0.052694086837772561, 0.038812289500533906, 0.00351979249578264, 0.0092884291588685979, 0.027247382850581914, 0.027264346879162616, 0.044126727120739825, 0.0055252525710258928, 0.02508643670669862, 0.034760885147787517, -0.0319140813608338, -0.010532840619760897, 0.011540732107110636, -0.025783175261279934, 0.020622360900288645, 0.014535696941216389, 0.0088525985395082156, 0.034582887008390364, 0.0016919619079097366, 0.019774235377541766, -0.016869203438250883, -0.011872140137964747, -0.011164594869350794, -0.021000743778555511, -0.075596615946532775, -0.029307852555546691, 0.0560965603829606, 0.011865216597816751, 0.013716847073227253, 0.017681845111678741, -0.030011290668360887, -0.024096670880512223, 0.017361538013868895, -0.015315712831449523, -0.037663115802690023, 0.010828028272477993, 0.055219054210005024, 0.014233975103047043, 0.018958351493891162, -0.035361770172067923, 0.045214502507017848, 0.039276123305636984, -0.019421649888589455 },
		{ 0.0111961541867694, -0.049687112062638115, 0.049111229735122566, -0.033259398376130511, 0.040168985908367032, -0.026150128839174273, -0.049834555092049052, -0.036509773610597547, -0.045784769024935765, -0.061108856577668194, 0.018010895223392887, -0.035782447687591853, 0.023701653299391893, 0.026746376525092277, -0.042779488943260016, 0.042189834431469259, 0.0069239397830904012, -0.00048306111865138848, -0.0621807409499853, -0.044985668186230683, 0.061161527764517332, -0.016918762959256903, 0.026654019580815619, 0.04667190264801039, 0.02812688967460782, 0.024212050553181784, 0.063261490135527884, -0.013938862076136694, -0.056822008332799823, -0.015568779701215132, -0.072771118663517861, -0.056598412220907092, 0.023784152598744095, 0.042040199398781196, -0.045759661284042887, 0.031322707329768, 0.037716917183019769, 0.0015191291590071758, -0.054689683797980407, -0.011340950152842802, -0.043031735141754165, 0.028371419552020438, -0.036376631503376361, -0.067849891449043992, 0.029316164530329022 },
		{ -0.0072194354843015248, 0.0097151649286814148, -0.0035198142958316457, 0.00854265591107658, -0.016904560997260506, 0.03807417440681219, 0.015436635901894848, 0.043619036879172027, 0.030900946693503416, 0.046292757306167907, -0.01120498566257064, 0.0526308150571768, -0.018981614705672063, -0.011193874286038149, 0.050162955251031, -0.0056700713977439823, -0.006224616150083552, -0.004795281290464885, 0.068159281016225218, 0.017888567094635849, -0.016932885478654722, 0.0017929556438430422, -0.065660184616976369, -0.070402811151894035, -0.032651749294774678, 0.026883457959492311, 0.029087900967310343, 0.029503137420629797, 0.020505051100844354, 0.0086874538968249541, 0.062861833010687568, 0.051355547933250978, -0.020280161730417037, -0.026955412863010592, 0.030048085896699748, -0.025049722195289958, -0.027235426916937259, -0.0033123874948346311, 0.017639230723461446, -0.016528782766913473, 0.046307883287993741, -0.0023059934150928951, 0.013555099960338331, 0.0035686794202685417, -0.0108654610397062 },
		{ 0.017592466562078676, 0.0011772341555204614, 0.016652753362745982, -0.01253059896334156, -0.0662880346818602, 0.11042574318283561, 0.014569864894487784, -0.017894131431060816, 0.015276463828121036, 0.24995857194494062, -0.12770511761322476, -0.016122479168898016, 0.08103516478626438, -0.013954239057255183, 0.11047283102587901, -0.0073759983865260923, -0.0608185848035377, 0.019769386730546561, 0.18836687658346379, -0.0031033627203924247, 0.0073380475447327263, -0.0055351054913884944, -0.13213360241421118, -0.16105113673105678, -0.084575219950138808, 0.085022012526318386, -0.43456113382160638, 0.12633493991569825, -0.071495042488102165, -0.009747798359963018, 0.15989226845812773, 0.13709859967663152, 0.0300542538829723, 0.035821608729726478, -0.0091758060545105938, -0.020916401178127896, 0.072697037869541445, -0.02886434735001801, -0.018940439953096775, 0.028613835604936761, 0.10517521763956873, 0.019546755840374595, 0.0066619917513619795, -0.018023359080430776, -0.062260578566931679 },
		{ -0.014096102017626346, 0.018405716231614595, -0.025898050659250594, 0.012114186193982452, 0.017593879934387057, -0.018911751460167634, -0.00083757190268338047, 0.13319853593610509, 0.028634619915030066, -0.012223118717429909, 0.033589594303503847, 0.0829190416301782, -0.039296594078593386, 0.0076175379331610386, -0.021465567117134588, -0.039914669851625423, 0.05185932612142613, 0.035737211571763905, -0.0054649473835426824, 0.023134743661404722, 0.030879802951873432, -0.013084506790422239, -0.035014060545668475, 0.0057908977349299959, 0.014401469027562929, 0.11752624643320138, -0.092631934754737313, -0.021474374808158676, 0.097362716357971266, 0.03321482765138415, -0.030333551593353613, 0.00307266860822294, -0.0409461161066852, 0.054853553147530668, -0.040683407815559812, 0.0453082125542139, -0.070842898811338623, 0.070925632233961422, 0.026388872367130906, -0.070826190028426939, -0.0093901857731057432, -0.024688081712946792, 0.0017508182609894854, 0.00097862795767700078, 0.025752555097096157 },
		{ 0.01611263865411067, -0.054114561548725272, 0.062177217107285028, -0.040450114134850029, 0.029326003239113121, 0.033131283513085148, -0.0050473352606580412, -0.17345793265026763, -0.038341109742476474, 0.0051209443492084335, -0.012446745631160832, -0.0833715822072994, 0.073821859058689418, 0.00975303822766076, 0.021068933489884031, 0.02799657925783167, -0.045348901720352844, -0.021524435317260868, 0.00299000952223329, -0.058950878393944375, 0.024876688487795784, 0.032363397602842091, 0.064224770659278882, 0.0047196997817094549, -0.0045412670322612144, -0.15686785354320931, 0.0615996209460283, 0.018567234110687291, -0.12950005607980261, -0.054600202882748784, 0.020661329234960026, 0.014681458634642949, 0.065165681223915237, -0.027118411937165951, 0.019486907781780326, -0.035841780846055764, 0.10650334406167768, -0.088262943294276128, -0.067355210291333623, 0.067300603589803581, 0.0019015323151379939, 0.028459442959511166, 0.0060708068577174827, -0.05327718665969388, -0.004927333757976389 },
		{ 0.015206929002393087, 0.03994745134670933, -0.03694953996282295, 0.017682811427457444, -0.10396486027491775, 0.053926916815782044, 0.087780632610016812, 0.010620889971941761, 0.036708781824056617, 0.081499399265411235, -0.13408197272656319, -0.028867141161421729, 0.05642267875028583, -0.055307038009150655, 0.086488082785687731, -0.040978826273816653, -0.10795591757951729, -0.032358398471213731, 0.069229932773231848, 0.040399563152524895, -0.12154780709323333, 0.010399770698421219, -0.061172032192716712, -0.06548163396457192, -0.039418541062662439, -0.014066291818366074, -0.091694795471118334, 0.0768427763810422, -0.066889577155094318, 0.016656027542989529, 0.09321904309294593, 0.033174064434536236, 0.0028011558276675595, -0.10254303105798236, 0.07382538144529309, -0.074659656901529745, 0.071577838797554569, -0.042624027056252442, 0.036336145265189905, 0.060104959825509306, 0.053478572235998764, 0.0095208661465751923, 0.020411803690721597, 0.083577168631596824, -0.050925794920036009 },
		{ 0.0015451020842035966, -0.01640158408958255, 0.019753791941615868, -0.011578574827826323, -0.0031173534453323346, -0.041084809771259063, 0.014895709273768671, -0.058681911948362177, -0.031261521415394219, -0.0425717414626797, 0.0098169583941675447, -0.032685875569989239, 0.022069215561183391, -0.0061087224530298485, -0.024317881493230296, 0.007382328795965315, -0.043839694466509579, -0.033877227472002475, -0.012415755985025988, 0.0032785677956114613, -0.043353465671326988, -0.014174988351389148, 0.080314786271230842, 0.053117860349264294, 0.016125801373710443, -0.082156928679830238, 0.25154481325669692, -0.00071323711197408791, -0.062067318260940953, -0.0080041053042556477, -0.04848336550898881, -0.0459078680139511, 0.031598275156900162, -0.027177415091471934, 0.028942675224910935, -0.015656646100579687, 0.0029325991617346276, -0.015984070299161716, -0.011616553524541913, 0.0027898732745131635, -0.017541560861510611, 0.017653530386185851, -0.01396495989675069, 0.023468263268870984, 0.02384914070219956 },
		{ -0.021952294839939303, 0.026133312443096007, -0.044912469367294289, 0.031099845651275698, 0.013472721173390448, -0.039612800792428265, 0.011694472225242447, 0.097428578670261071, 0.039343194789374786, -0.047969398946712115, 0.034285442599816335, 0.070833130651620946, -0.04010467209400824, 0.0026373626645610146, 0.00014531455190979305, -0.03043230094854012, 0.03413630603617291, 0.012152401083277331, -0.026191312404810443, 0.024817718694931653, 0.032657586665165238, 0.017890053062892462, -0.04110239725518567, 0.0090676980922490824, 0.0095915342016312331, 0.055539169927663973, -0.0197310885471282, -0.037301415230747662, 0.094035070766557372, 0.024821455322416437, -0.025037954151702173, -0.0040428026362790059, -0.071610996717894976, -0.0058006946765678013, 0.0061577153215705447, 0.014841074028182522, -0.074767159120411, 0.027101266412364298, 0.034334084111207544, -0.027848705268835369, 0.00030882701246514363, -0.033346854729465143, 0.023017876090550591, 0.018727085521776911, 0.012696813866186145 },
		{ 0.013599247867062294, -0.021422984724060287, 0.033270871559522811, -0.026987471040330425, -0.0030465575517464203, 0.030368864601861976, -0.015665770053539869, -0.050995215635108858, -0.03313770106655467, 0.039112479167075505, -0.046358801287009754, -0.0627690060035871, 0.053593763591312647, 0.0066391846384476309, 0.0079546776066958813, 0.015180397542829838, 0.0024872081449940722, 0.015191739098060583, 0.022096037492937377, -0.0435408814241963, 0.029561263990259507, -0.014039727169709438, 0.011456956050730811, -0.02726537184190414, -0.0043380537511108928, 0.01006376052307666, -0.11525708755350925, 0.038920378231952726, -0.024105607517235252, -0.018527573477142019, 0.023158880083303551, 0.0038197330290645052, 0.036417646918707344, 0.050155183176323392, -0.024891340626615945, 0.01760886488799366, 0.054568382045042552, -0.010976247176479907, -0.035681022749563279, 0.016000357852274397, 0.0074974518871458963, 0.035538443077394871, -0.020441234051642474, -0.026981894988284658, -0.018989945587383785 },
		{ -0.0036581047185973645, -0.033852121844987856, 0.052821648556767582, -0.025238936839484628, 0.068657321155613155, -0.036506769258744125, -0.072808363877508123, 0.0099560761090666562, -0.030506183762027968, -0.046183368747974893, 0.074597470726029014, 0.030767962844913562, -0.031294795920913505, 0.036092793224332564, -0.048396002354231984, 0.033616176754628552, 0.065206342747098772, 0.030303969102331442, -0.037868427332145592, -0.047917114070328629, 0.096920748765536885, -0.015613665932859072, 0.0049807716507618907, 0.029266404406173587, 0.01980276821253955, 0.0656010225224392, 0.020578651621808913, -0.038349540806097142, 0.041096795477180918, -0.0016877842879831507, -0.040703267014543534, -0.02410321346455999, 0.010450118062681764, 0.09427331458662512, -0.0711739803909954, 0.057570722577683267, -0.045692003721637821, 0.043959940092004421, -0.0278562738589239, -0.051920153639438874, -0.035006987863827908, 0.0078893804065162977, -0.0220649102098678, -0.078283530710370919, 0.031347295718867262 },
		{ -0.0022948302964040448, 0.016215020679020479, -0.0054751985135995654, 0.0082354959308635328, -0.030136506288083263, 0.091487467044003526, -0.0029613703048231381, 0.032514042561485872, 0.027402005971713553, 0.14833974929766003, -0.086442827528043428, 0.010934079627280968, 0.058676364616639866, -0.011072764945528593, 0.073626914441638228, 0.00061816172025821688, -0.035151993445250246, 0.023840498612962894, 0.12884757386390103, 0.016909646991861413, 0.0077515247267315388, -0.0015167784002954874, -0.10931533130655718, -0.12997717876605328, -0.064663692453107227, 0.082714582055847558, -0.34444524752380135, 0.076419367018316128, -0.013659858447943551, -0.014488348480746304, 0.10798383664891319, 0.10297040571366181, 0.0010570307837073772, 0.034424166629499695, -0.015185019598267615, -0.01238869329097469, 0.024995222693361647, -0.0035404122064489033, 0.011744295126441782, -0.0091684968234033472, 0.091490636195439762, -0.0012395032520854294, 0.017822765093571797, -0.027105814771307041, -0.042079104358868226 },
		{ -0.0038983757995343002, 0.022543225541287088, -0.023091523657429648, 0.013531185108232081, -0.025977995791705438, -0.007139508698765543, 0.048176009310616594, -0.031757826753435622, 0.018443406833120367, -0.0020907770918910762, -0.020025472914050428, -0.0019942454962546133, -0.0080416359402157821, -0.015170473334654258, 0.016510834500612612, -0.02639192165031716, -0.0467204411846325, -0.01666625494401848, 0.022951254124042968, 0.020002404881574135, -0.077529542058455053, 0.0041133733100241991, 0.022594661271242177, -0.010755841552602754, -0.0090819789681948013, -0.062135944725289126, 0.053027025463473126, 0.0045023632336849417, -0.022666290027372034, -0.00076693101191370027, 0.0086072713036864715, -0.012457804914215663, -0.010639881478574427, -0.055000445305743335, 0.051350958976628933, -0.035829249353516321, 0.0007049142416966531, -0.033240617633112195, 0.016408292031752442, 0.023549782127601216, 0.01513053880597059, 0.0058423574911591978, 0.011198609800258039, 0.040411567710617942, -0.010482273181798306 },
		{ 0.0064574119606904115, -0.01066028670298869, 0.0042848884809306214, -0.0051737654951079393, 0.013005159132439043, -0.018798994786623429, -0.0084564978303475844, 0.001921993260391115, -0.019536326760290962, -0.040074074227444655, 0.0059587354647184518, -0.013346668273460479, 0.01105606606833209, 0.0088038727005862061, -0.048559620358306313, -0.00084401462252321313, 0.00702076705507295, 0.00596348500794673, -0.046728902506310482, -0.016755734487926537, 0.0055977353775407005, -0.0023896972703730422, 0.0345594163354665, 0.06546202246414333, 0.030006920654099108, 0.022362217232422769, 0.0028531941807580044, -0.025538957083683568, -0.010159001945828586, -0.00062768920199134624, -0.058670629396929151, -0.03500087092502803, -0.00013024531332481669, 0.025549944769301235, -0.030940189052351776, 0.016000769999899887, 0.00052373836827146383, 0.026120544735020006, -0.0075394179621649142, -0.016432794603691864, -0.033008719292176046, -7.93780073778767E-5, -0.020769772999647364, 0.004349480163008991, 0.019840304700738626 },
		{ -0.009467121545316504, 0.0009475521429393619, 0.0024597459351871648, 0.0028624338943644763, -0.020059453698502219, 0.018160027181136375, -0.028906114174897024, 0.058203643837818723, 0.00829576001123087, 0.075853022921073959, -0.0094916359670942, 0.049977850381234368, -0.014099560271192984, 0.0033442642452104375, 0.046220952696690673, 0.0038661951256613175, 0.015767618423563619, 0.021059733112524, 0.068967401325729538, 0.0022332448739867256, 0.01801065987603085, -0.018469475243910648, -0.088358519277413519, -0.06707575176334464, -0.032201611250039924, 0.099539532546646547, -0.15336828035827979, 0.026109652499682275, 0.013533299892563917, 0.027704874475964703, 0.0397126772474414, 0.052052449796316888, -0.047511675361325043, 0.050825056505047338, -0.020262630751963655, 0.0088936224648703231, -0.039244928439913035, 0.034683933257649513, 0.0089521921842153646, -0.030565985688466888, 0.034407872999690413, -0.00063827659431864325, 0.0047553261635088348, 0.0010183244572180743, -0.016834003918093129 },
		{ -0.0053113225172162857, 0.016575431308292535, -0.0052837692001091651, 0.0039557658865457188, -0.0047438603296029325, 0.05884486986217502, -0.0058116485239636129, 0.057233600082805182, 0.02377060273761436, 0.096375171140475835, -0.04524135498833174, 0.026781307973916129, -0.02205088931324516, 0.0044819590698692707, 0.031159924502151385, -0.00059515665307148275, 0.025344164177799753, 0.053710610601899751, 0.0760667079511176, -0.010099476820951156, 0.040229870765824738, 0.010675505033598098, -0.081552994114159047, -0.0803102559412016, -0.031319617242046814, 0.094287466094178846, -0.31353097865322932, 0.033896480892220644, 0.043089809107775275, 0.00562387782353672, 0.067816681637169043, 0.064732939858516469, -0.022078202380349576, 0.056559589811322628, -0.035952545833993495, 0.013080086062799649, -0.01388813153038714, 0.013286377640558018, 0.012688684596323468, -0.013466644994216601, 0.040064093018387877, -0.011587385172930873, 0.017928357296526545, -0.041701249773180828, -0.033756153488777245 },
		{ -0.00662908469333348, 0.023614246194462445, -0.032683619407645415, 0.010764585842846576, 0.0076517662512290245, -0.013212912052656057, 0.0082030888185516647, 0.042701270126421206, 0.020661808823957191, -0.022104122791159888, 0.012313498654190385, -0.00017849820523219026, -0.050198110097494239, -0.0026581664410131704, -0.013880781199166848, -0.023337666289309326, 0.015910272212193266, 0.015122093935477815, -0.043028996859516895, -0.0015850564559257772, 0.011200552456688622, 0.0051325732621934628, -0.0060261214918634221, 0.033031721578561261, 0.024291365538298904, 0.020427840245640122, 0.040796632679345565, -0.025487797460428605, 0.06606898366284783, 0.0066362231445479735, -0.023632487533147287, -0.047199096278131264, -0.014856974273196974, -0.03047706527299554, 0.0048452818360349165, 0.007392966378786758, -0.030514349668264249, 0.018130356358278529, 0.019974839640326994, 0.004377730675412945, -0.031594932270495556, -0.015556861795042001, 0.011899944728454093, 0.0036436023041284929, -0.0078187389122542552 },
		{ 0.003371393801270725, -0.026689039957477081, 0.025839748572056852, -0.015596433839626545, 0.02605498639472326, -0.026456099176530551, -0.050707859575951225, 0.02701431199706367, -0.019375768582601028, -0.060448650251327986, 0.025446393085879645, 0.0024671611987849926, -1.197832891253821E-5, 0.020876854315809209, -0.032248111741137953, 0.0232489436227079, 0.030628516408990652, 0.011256607837812956, -0.03923940855531946, -0.020088812183157761, 0.048973850624059671, -0.01892111881048536, -0.0086167696120936169, 0.019530707478437534, 0.011183630047898119, 0.074266412102066942, 0.011601798885858041, -0.023441614517220342, -0.0090671972446991975, 0.0062586675843155854, -0.056804172494110065, -0.028524779516615747, 0.00096635101169014641, 0.064837950150724508, -0.042677493020334113, 0.036160989422994227, -0.0016541849123939147, 0.03687577379841, -0.017345376977714409, -0.030877909395840151, -0.021840205144169067, 0.012225816993171696, -0.028765326144336759, -0.049564135444870393, 0.022594819484729987 },
		{ 0.0079468379837879955, 0.022236779031610265, -0.027549164075492624, 0.017019007086739125, -0.033814375278816408, 0.015602742206285432, 0.071992348416933444, -0.057203405996179207, 0.019561875907100557, 0.033443282098431594, -0.021006589968672596, -0.032454522666993613, 0.040492927343220367, -0.027349754702173545, 0.035685971026731433, -0.0019534293325799757, -0.078993873998844985, -0.038913618721514089, 0.0038398242907516072, 0.034557485684188413, -0.063935101421299184, 0.022275848770108408, 0.03901605967258992, 0.0009644283289335304, -0.0040308538873852582, -0.1138915259198086, 0.092573644662677326, 0.0055590967157521957, -0.0465346462719161, -0.019189151095432382, 0.024866700432070284, -0.0036863850436127249, 0.0070267655304966752, -0.094755007370192879, 0.084929167596119848, -0.0439659628717664, 0.043012490097564457, -0.054270147595109966, 0.013200285248218067, 0.056707579441156307, 0.012723258212046009, -0.000724256960114964, 0.024398059544756094, 0.06865662366753511, -0.02392705691612633 },
		{ -0.013838371318788588, -0.00094849988242938087, -0.024190963553565967, 0.034045607689828325, 0.053053572470833192, -0.10006543588992932, 0.019248776948433771, -0.00089654703199779106, -0.0057179150706795885, -0.22631669196067583, 0.10832609693756216, 0.022638445313254645, -0.013451294229861339, 0.0042914914951710835, -0.10438002786782184, -0.021123583956939303, 0.018578537226808113, -0.045623051538497821, -0.16250862506825786, 0.016625674651626794, -0.026933275515112019, 0.024901331853440897, 0.11860849383566145, 0.1439687006825745, 0.070233078627646869, -0.060657157983564594, 0.32739384297454965, -0.11747556265622443, 0.040618524547071579, -0.0058113962952909739, -0.16299566214302755, -0.12667281855774104, 0.0085981389998899729, -0.056422704341695058, 0.025732764144426087, 0.0024045390517240302, -0.022098950413241791, 0.022630617364396852, 0.013681286397342863, -0.02418368484240738, -0.061348983785604487, -0.019079917911723129, -0.010645113268034337, 0.049130435533180215, 0.071181686659883989 },
		{ 0.0075828993836393032, 0.023925807104284505, -0.026108073014152904, 0.01842918315540408, -0.028754768456046166, 0.00581931390465661, 0.058895169120000021, -0.041464849090560557, 0.0058502405550315576, 0.026658756249544675, -0.018264451694197048, -0.027020092606120712, 0.012237059180152637, -0.025112405494398582, 0.023305536366722109, -0.010251358152168264, -0.037962745800208954, -0.023501913686094156, 4.4602776218901957E-5, 0.026042445374558309, -0.054741127220765622, 0.013197041398966613, 0.025665899800050351, 0.011534976342820448, 0.0011333813976515447, -0.081312961160402569, 0.0720055309700474, -0.0021290186681735962, -0.01785793897466274, -0.0070457580961969871, 0.027374632660443727, -0.0097673072503397067, 0.0075216576887030843, -0.078501137529829082, 0.052812388093155166, -0.027792763491404908, 0.031202223366157498, -0.03744010103909326, 0.016629066735700433, 0.049856896789533457, -0.00029847891339187236, -0.005656385373215736, 0.021137103854406713, 0.06598684803950397, -0.015876876400310335 },
		{ 0.0096226041574014667, 0.0019724365117503354, -0.006484309196308896, -0.0041410814539313773, 0.002661449915332573, 0.024246860436239154, 0.012942646117429417, -0.04391201624943266, -0.00495056990263818, 0.017020763560103808, -0.015581109954664794, -0.033435000925609276, 0.013644040445868089, -0.00020666517418486589, -0.006619946648024861, -0.026374999462821187, 0.014365918909364469, 0.013666319607874981, 0.0089558843206731086, -0.0129785083673866, 0.0020403125677995967, 0.011345549721234275, 0.027078691985563962, 0.0049610668125386363, 0.0030847149327884347, -0.036622791155004865, -0.090244774420384186, 0.022176064749754223, 0.0026174204775466941, -0.017339770466103487, 0.012568209365380879, 0.0082412679564308738, 0.014232969603577275, -0.0022572212865074582, -0.013133339093856819, -0.0094733331300462346, 0.021983264517554574, -0.02543927297915417, -0.0050504971714673014, 0.02571361821663139, 0.0034509434164459044, 0.0082191917736792932, 0.0053390950162210109, 0.0013499976698355942, -0.0052885193110689174 },
		{ -0.0024886878051289096, 0.01983928207010132, -0.035130599420518722, 0.018462907706685086, 0.020779920322104149, 0.00090595053276630751, 0.015585217729714511, 0.026402430098462933, 0.010646880965494581, -0.017441234715889709, 0.035890890914142964, 0.0039107485476792391, -0.039669093064619763, 0.0066804675163823473, -0.019899766140511119, -0.02045944275412306, 0.036568396069688335, 0.018317003427627352, -0.036372308586951792, -0.0013097065567408601, 0.025193032084567957, 0.029654494391220903, 0.014396124835104043, 0.034420962944468454, 0.017357132549966195, 0.021871934608174815, -0.1652206338807175, -0.031969406684176346, 0.0406887884637402, -0.0064947480433200019, -0.022203571887391568, -0.0065300051820109846, -0.022777641823796213, 0.011556832040006133, -0.010734036967095602, 0.015033453461939581, 0.011328498603540902, -7.9048176857032619E-5, 0.012218225325309475, 0.013475726200744794, -0.021466301415691207, -0.02301217368274551, 0.016743436981282205, -0.0082225896683881822, 0.0034713944650305059 },
		{ -0.030904783169958264, 0.051030611231282864, -0.0655282364257609, 0.053149066468538654, 0.035378562847474329, -0.073739079575463912, 0.0074405416705596791, 0.13842559367519283, 0.047785924674076025, -0.14643527884739801, 0.0659903575113572, 0.094937058736581884, -0.076559160375427843, 0.0057843163415531992, -0.073235936549223182, -0.034567906197752622, 0.04135575974714939, 0.010303401730861136, -0.10307292026306192, 0.034553818979717286, 0.026253763540957084, 0.015419821317662055, -0.0144855281727873, 0.060416340602540156, 0.041680960029164715, 0.059896086007545257, 0.13921411004506826, -0.095435093641702343, 0.13458527173539214, 0.019155464321790962, -0.090225760670160679, -0.063867745775313775, -0.074403118053232159, -0.021310886736912373, 0.0016291919073639523, 0.012769395988718377, -0.10969404688847882, 0.059217168524859934, 0.076531403369630518, -0.05236800227262927, -0.022655193941755691, -0.044067166782751743, 0.023356048322244326, 0.016783553087398092, 0.027359084592506921 },
		{ -0.016952343485660947, 0.00090405494987016012, -0.010893783847263727, 0.0045667954603411153, 0.023921374994024597, -0.01293127335352294, -0.035577773952125753, 0.076269767956747891, 0.021719081855706774, -0.023760557177788865, 0.032213298558290658, 0.037606605893948847, -0.018189503308052354, 0.012157028543209155, -0.02107570979133017, -0.0095094209906850437, 0.045578483458965029, 0.026576076611078837, -0.010292329539172479, 0.0012535995415175851, 0.043278873981087304, -0.0054588455899370848, -0.039189681008548506, -0.00564343694794941, -0.002587265771698015, 0.10196970052649997, -0.15018282804760261, -0.01888974296666807, 0.047288952329568694, 0.020102431500686, -0.040374767007798164, 0.0035085671650717343, -0.027012959028789505, 0.059928550226320827, -0.040977871291391091, 0.026797364756300339, -0.032508686354949294, 0.053342990862578543, 0.0018966706591745754, -0.044823874329430372, 0.0028630419869965589, -0.019242156043595152, -0.0027978428482922637, -0.033910522803923794, 0.013224164334673544 },
		{ -0.00076577888518661872, 0.00934294749825185, -0.02001508640747866, 0.0047660817707612735, 0.0089072465885750574, -0.015048616936950152, -0.00011189394946510235, 0.017482044829160853, 0.01130094217012715, -0.031228048170958759, 0.023190109379447349, 0.016138300534364242, -0.037292125904129975, 0.0083907895207860315, -0.027111231627377514, -0.029896815523453944, 0.0042092008333700515, 0.014210302188190463, -0.036541835524251706, -0.00062923748848657589, -0.0024976353952820976, 0.0087007137881773035, 0.032646580047455638, 0.043957405030546425, 0.020400548327015471, 0.00020548655351792334, -0.01629993016738631, -0.026472132120820242, 0.03188121501802077, -0.00048245895532709576, -0.044678659160096328, -0.026306998360833563, -0.027494133921146403, 0.00047315151376179114, -0.0060381635577074939, -0.0038814802000111978, -0.022696154459732245, 0.007179381639565505, 0.014354503130540212, 0.00037370759016663722, -0.030179410391753741, -0.01580425165965876, 0.00359850913418934, 0.012283756308848716, 0.007730919717843298 },
		{ -0.006455085494147088, -0.015318045248716498, 0.011609406357204376, 0.0068135726973256967, 0.042703345332549617, -0.025511262532885357, -0.05167876293657795, 0.09981985653720063, 0.0014010599352842241, -0.029070801889801445, 0.048724092973324314, 0.0652721268834396, -0.049208073255145, 0.033621639016726795, -0.033501247712300682, -0.034158905469394832, 0.090628717638228287, 0.034204381107162486, -0.014056037540615036, -0.02350206203383914, 0.077043749712723653, -0.01078996155532216, -0.046036566596932275, 0.0084728785469151448, 0.004986163673641862, 0.1384195248769963, -0.19610994665033987, -0.027819044600872565, 0.08551694772569679, 0.019475789254809476, -0.029725319270794751, 0.0169152518175215, -0.036783000029266072, 0.092085214919535613, -0.078213182030857684, 0.060606840171304313, -0.06816417738495463, 0.06513924670469, -0.012501207109290258, -0.066681876126716735, -0.0089117828151842691, -0.026331527879635774, -0.012597810904445419, -0.054588058435471987, 0.028996561208054404 },
		{ 0.0146019255227254, 0.0049074823663486627, 0.00071457654015971879, -0.0015572489125734268, -0.0082199340589209739, 0.0009089302829140075, 0.049257781815791525, -0.062035204600482355, -0.011930552673519203, -0.012092772288163431, 0.0023689187777925038, -0.031795069915023039, 0.017202977923364782, -0.011333734039817201, -0.011129056697195263, -0.03714287526820978, -0.025951306852613862, -0.007306889446330401, -0.01100219897163548, 0.013739780074060058, -0.0553613493868831, 0.0091776480184934811, 0.068421665598625864, 0.034331893087923004, 0.014203564147035701, -0.075346262391432536, 0.0698730999728095, 0.023160045892712439, -0.022348578427246833, -0.017318900202601774, -0.0086399871318725927, -0.025889422302278014, 0.023219505982880403, -0.0318447703406178, 0.0051058234047473769, -0.021572191647006314, 0.03423342813639043, -0.032312901054501547, 0.0036545093044862116, 0.020475992564996916, -0.0065412733943150028, 0.00716579417915524, -0.00376676249194666, 0.042329683338109325, -0.00099076513581961142 },
		{ -0.0049350787935228532, 0.007592857485048809, -0.00568340740437318, 0.0013367282690910415, 0.013837438420729466, 0.018214134096592855, -0.046947967147981454, 0.081267482491087409, -2.1919658373973603E-5, 0.021887295895175497, 0.014019806508510219, 0.020712432480311634, -0.0123408586780852, 0.016529310330436319, 0.0044319583828625348, 0.021264802007337526, 0.052255708878532876, 0.03320526937998889, -0.00324123421763435, -0.02187969998640836, 0.063165878781079765, -0.0037103468761717356, -0.062253740895304117, -0.031324790292282925, 0.0039660620677502761, 0.12453294518778157, -0.233289026212659, -0.0035701652614532017, 0.052049739062571029, -0.0022025839654297979, 0.019337039004472787, 0.023763719299378903, -0.039122066974295308, 0.065927814543022323, -0.050982584494589166, 0.042175089205659252, -0.031023255273927096, 0.039278063035308251, 0.0043649791171062576, -0.03349421123777948, 0.0087383882600338969, -0.010104305789132156, 0.0014849038857481216, -0.051906507509859458, -0.013041179558252423 },
		{ 0.014324371888828744, -0.019586834254695661, 0.026975573184754995, -0.026276795293154503, 0.011077125215192251, 0.040544719500118544, -0.00607959850930152, -0.023200203039352729, -0.0077586143598793013, 0.051304201666980541, -0.0011461429966051738, -0.018640530978784955, 0.017459491623801629, 0.013494919328141885, 0.01976006784670448, 0.017475626533282045, 0.00847894182918308, 0.0095499631022127763, 0.014945406091888958, -0.031197951841128564, 0.041291480855340415, 0.016932870187787448, -0.01006874687408716, -0.011384828176369783, -0.0025290337260111505, 0.022206837804150104, -0.056908232797042636, 0.018145186451075095, -0.013353662545218997, -0.026135174580346046, 0.019499367900045694, 0.021756518059643397, 0.0202274147064534, 0.00482389492710543, -0.0062074088756235565, 0.010707388325262978, 0.039703574493399917, -0.020017891943168008, -0.029058339691405709, 0.013107593867916772, 0.0073108407583243771, 0.012626805423551353, -0.005514847524640442, -0.039578091392068311, -0.0029796480676497605 },
		{ 0.012937532741583203, -0.044855978890636021, 0.0487739154957045, -0.027125964746606977, 0.022011235655675334, -0.0035404112251696232, -0.034925896336186445, 0.0048948561662047552, -0.02297066316095463, -0.01846614287418636, 0.01592691133165109, 0.039174564828802216, -0.00019935049761963985, 0.01837591169669215, -0.012957062184053625, 0.0273840032255313, 0.00045507875429515507, -0.00012684501801512164, -0.0071577268314995243, -0.027356210057314204, 0.028032506976708973, -0.00879643451894031, -0.0051870940422062556, 0.021298265973038629, 0.0095310896845904533, 0.063186164870069172, 0.14211136913069675, -0.004739367552292241, -0.008722514178396713, -0.0038630902699356745, -0.01793821295949494, -0.00589030122913879, -0.0090633792458135916, 0.028997988324780635, -0.017475854030503491, 0.015736879830125311, -0.013329995599588031, 0.027222547091968811, -0.013263020768663876, -0.040475668095190084, -0.004373477741282, 0.016783697463214474, -0.029287640924260869, -0.035865564731339609, 0.019507318269737318 },
		{ 0.0025998840454290635, 0.0010192718568242259, -0.0125592026101984, -0.00853457982343789, -0.008561251046449914, -0.021051988621111455, -0.0016673775157599706, -0.019223305510603562, -0.00673897411969733, -0.074697070450421815, 0.015649847203728841, -0.020705272446302072, 0.010394905580393547, -0.0048675096497300171, -0.012297747531930333, 0.035171303837329082, -0.014562835198255011, -0.042477315622599245, -0.055890609244587622, 0.0020407489891356217, -0.031753841956613828, -0.0036186367192755416, 0.037692038295584512, 0.044205920007966866, 0.02161337680853152, -0.0836880768135908, 0.28979168791868937, -0.037796894018840212, -0.02002471146493617, -0.0013727528838005326, -0.036670883234469161, -0.045631770037196741, -0.010924634523409598, -0.087844529635681859, 0.046999764276766554, -0.0099925531335409989, -0.021561435152010987, -0.025238003885497334, 0.00073135081812197508, 0.024175354412982554, -0.039187289063047873, 0.010520902892462013, -0.003836980630082113, 0.023591214323991846, 0.0039236897078553778 },
		{ 0.0063622063720624447, -0.015326402063507522, 0.011355934828435294, -0.021244083468335062, 0.0095179859513562746, 0.034370652451935511, -0.044617165169311329, 0.014903600338412881, -0.021537991967229282, 0.036367359527854123, -0.006024086861376966, -0.026648526662021575, 0.0074966049760157132, 0.011980990174974787, 0.0301034579996756, 0.037527339679437825, 0.019956838305636909, 0.0043549686832638655, 0.0083465723002677118, -0.032239291631435192, 0.047339431279130963, 0.0018210193079202092, -0.041168211448341333, -0.033774065139277838, -0.0024065758788150505, 0.042712981423957359, -0.10596947321274197, 0.016372397099931036, -0.018711092027626178, -0.021059142254417955, 0.036308194555744577, 0.02945402498949037, -0.0062234250985235924, 0.015438929734312067, -0.017151519998470654, 0.016903005704854975, 0.031225248811962366, -0.012202757728691725, -0.012927024555503949, 0.0060755184011485192, 0.0044009750171928727, 0.0025533358487155856, 0.0002804226292502126, -0.056269660350358595, -0.016758842197470088 },
		{ -0.0013975367999518899, -0.019301504016964906, 0.03879298684060755, -0.017864103136083664, -0.001402968507965198, -0.0021756188172317326, -0.018996298290211197, 0.024852475849269792, -0.007603686245512345, 0.0064078787162765863, -0.010428765580213405, 0.017464994053960105, -0.011180098176890871, 0.0098253178059066245, -0.0025524057362164435, 0.027515267673861466, 0.010612031487954221, 0.0062797681571657793, 0.017877853981689489, -0.031131969516881356, 0.0044930186687095535, -0.012143379301065547, -0.020762713703481549, -0.014999380911161465, -0.0053302432220645483, 0.061340541448118707, -0.0056290385567889538, 0.017759169367451155, -0.0074451882963578846, 0.014570012444207393, -0.0027530238692699514, 0.0082361907684476449, -0.0013671088776549471, 0.059648639406605593, -0.026007012414754314, 0.016617085994757086, -0.0099590897854567977, 0.021703159549891431, -0.018197203787453594, -0.034588877206244255, 0.0029022564980717641, 0.0068306708369708195, -0.022665845448112815, -0.012944593924868733, 0.0071523055141300565 },
		{ 0.010594929819698772, -0.0083608867064387249, 0.019831731894469004, -0.013675658259447394, -0.012109411166183389, 0.039404799099462616, 0.022581855875787749, -0.020460401153098561, 0.017590797455989741, 0.0735864892999763, -0.051256797751482584, -0.0076284572046223858, 0.043933588334074354, -0.01175168431173489, 0.024585233843545492, -0.0085589066034059223, -0.020690200805260427, 0.017736101721934648, 0.076478090270487409, -0.0023843487168333324, -0.014172985964341264, -0.012734812479188527, -0.026990113299068041, -0.054313825173471934, -0.034499702452697727, 0.01470528678217016, -0.052332886873976948, 0.059495513481978922, -0.002369375510965813, -0.0020362063283970609, 0.043765684228796113, 0.035277255017874425, 0.016785111862626937, 0.018181058342602884, -0.012985344860497332, -0.0069599667756847523, -0.0015710647623865051, 0.0090532604486087079, -0.0035275537138149612, -0.0076557418182654716, 0.043618841329046955, 0.009810505113350073, -0.0079272578127695652, 0.00325409836196594, -0.014756536971884733 },
		{ -0.018850686296456996, 0.030879262869018563, -0.030111341640732096, 0.029530178451782713, 0.0082467701506722686, -0.05894813592904144, 0.00976217557622252, 0.086587974456127828, 0.029292285156524975, -0.067828514371150339, 0.075441184548983922, 0.089864928887257542, -0.052052915874262522, 0.0028428060659523896, -0.051469101612238805, -0.0491723564905139, 0.059441667879901766, 0.008564607666360833, -0.049039086439253934, 0.044049221223450077, -0.0066090916155993979, -0.020094633606515078, 0.0055148968782840853, 0.044440608898975588, 0.023093509590419736, 0.079696706045275442, 0.03070859409115936, -0.044195502600299114, 0.099917998375871869, 0.052579717618944033, -0.0692375642604211, -0.034620912413394246, -0.047813103662014016, 0.014925953202686065, -0.022731455486207509, 0.032230735409392759, -0.086295076112384766, 0.0801933808915886, 0.035955762227499612, -0.069221974852195334, -0.018461701371033694, -0.032980601554552791, -0.0053969071491592876, 0.048557308247364588, 0.029051622051686887 },
		{ 0.0083799455626497425, -0.0026324042917505422, 0.016900098408791715, -0.017582025643689953, -0.026025839715563444, 0.07149572995757654, 0.0063988217663369373, -0.0054620083132850194, 0.016580477139384313, 0.1205958549877095, -0.07758442994454505, -0.012104810499101934, 0.047315570874653622, -0.016760147334830888, 0.066744478776525326, -0.0035841776186703572, -0.032268808813235329, 0.016455094251352633, 0.11141892814237221, -0.012652940244543057, -0.00826368139475407, -0.013441303581081733, -0.08138254914812118, -0.098843619790259074, -0.0536379152753723, 0.04271614950539198, -0.16987193953577878, 0.086740606595348518, -0.029758612025832736, 0.00064288072373581571, 0.093830120338378689, 0.076325858699346744, 0.017529433890519482, 0.022489041830039261, -0.013939410801686919, -0.0092502767143998049, 0.014481820204064307, 0.0053620410130690282, -0.0025754065204569841, -0.0040856604156263672, 0.055570892271664285, -0.00013703235021860974, 0.0022689048957721383, -0.00045264020912906154, -0.040439213971100223 },
		{ -0.0066241925557364509, -0.013406788159050049, 7.0633777717504667E-5, 0.0030800191132565788, 0.022319263853548786, -0.060166338665135349, -0.0081841171763191584, -0.010338263145888959, -0.016867113974066043, -0.12377606798847293, 0.072887364372813457, 0.020699019531503486, -0.017081729080282174, 0.018693130973416603, -0.058293013307163986, 0.0029138722009035309, 0.016031258375730006, -0.019183043498365686, -0.10084678922724921, 0.0007248543820100598, -0.005350302239638228, 0.0046675862354763021, 0.077286636281412982, 0.096920876664833649, 0.043144251255647276, -0.011988676852215071, 0.14041755406108547, -0.0648057729105915, -0.0009160553221567463, 0.011694473472271956, -0.10061388495562704, -0.065761710060288928, -0.01680382451845985, -0.014588313281630924, -0.00030314218030982963, 0.0054875311698603917, 0.0016748715484476726, 0.008684740729718355, -0.013208974283393895, -0.0052233063528791413, -0.049009885293127144, -0.010694634780721114, -0.014123811895396731, 0.0057942221288337682, 0.0342234864559633 },
		{ 0.0054249168939745612, -0.032405576820450976, 0.041478906708231893, -0.017856815408649973, -0.00868867249709228, -0.030180785022221851, -0.010449676777578952, -0.011199942420801183, -0.030904640207016259, -0.029986393675853953, -0.0092912984424767323, 0.015921072423122042, 0.00649003581367912, -0.00051392332812175492, 0.0022788413450640678, 0.019123947217204004, -0.037611862415477489, -0.033693497630272282, -0.0084778473231368646, -0.012467096168547003, -0.008307467941988507, -0.00071171774270470552, 0.010289872945279636, 0.014853982544882568, 0.0092227951293315773, 0.0025265362180136811, 0.23401153465963476, -0.0013540696775894499, -0.034043358694618137, 0.0030252498449784659, -0.012443938889922016, -0.0098124710206281482, 0.014260800686582446, -0.0089193359508384622, 0.013387314345069548, -0.003076756247560067, 0.0098347292224687625, -0.00055736779408026846, -0.02282456524451278, -0.022277488600336477, -0.0079736200854791765, 0.01386196504324517, -0.022084332283946361, 0.0063830103093049312, 0.015352048553334932 },
		{ -0.0023832586709735008, -0.012226554112534845, 0.004821305851756409, -0.0068050397116606823, 0.027193021471317334, -0.059039196945299986, -0.0010222411416195829, -0.033945310593863694, -0.017455842808451156, -0.11254341972761624, 0.067978927862598071, -0.01329088258079845, 0.01115656695347788, 0.0171809173261427, -0.065629016749313124, 0.014693225251182487, 0.018065898045625452, -0.013935856426088291, -0.10187438006003162, -0.013050667542723278, 0.0010377323176026213, -0.00071139445533106329, 0.094313770480582856, 0.10262427012405875, 0.047090251669945384, -0.032775637162958352, 0.12116671226205046, -0.061292138260921343, -0.012320663134131589, -4.1922794384115193E-5, -0.10211566309750994, -0.079699671829431024, -0.0055052679371290578, -0.011879251146098946, -0.003496937677493784, 0.013544481604177547, -0.00034408776659159129, 0.0067807382943481757, -0.020062878331397919, 0.0080623845657187432, -0.056487499671550194, 0.0027231139501449164, -0.017233574709808019, 0.0050875056989624495, 0.03219005489714484 },
		{ 0.020222450921674302, -0.04681441041381066, 0.063263907062346122, -0.044414791621853363, -0.016904513608353166, 0.013614500110034643, -0.0046623977557640559, -0.16562629945168211, -0.054656256757818121, 0.01884732468668824, -0.048861354892669087, -0.087811262441571947, 0.0726588090041508, -0.0029542241632297484, 0.025989114924835913, 0.073381966538516419, -0.0783568786902674, -0.046864353578996557, 0.0036344892537820864, -0.050442184463842368, -0.036561775350177621, 0.013354098025959904, 0.058541653471435605, 0.022803843222867534, 0.00067172952129989716, -0.1230920617916083, 0.2280455785519824, 0.038707927268375159, -0.15005922929975213, -0.030978082516814498, 0.022878752032168739, -0.016781190555505178, 0.067407050196286261, -0.0440757677615599, 0.04239555526561032, -0.048066747888122631, 0.10859684972082025, -0.078615218693027941, -0.0575246536653738, 0.05862500621829398, -0.019752785010077826, 0.04856869190275042, -0.02071333551370054, 0.012789613012847091, -0.004064354478594814 },
		{ 0.007910106991343686, -0.022320209331144113, 0.020798594926616763, -0.01532484828064278, 0.0173177228551684, -0.0099144626743916282, -0.036614787190385981, -0.0081037020507592865, -0.031908967651920778, -0.028061411583664507, 0.024521847615517938, -0.026998580716073625, 0.020246299210910403, 0.019310659555500535, -0.038828351682203111, 0.03583070177263796, 0.021423576811936476, -0.005076382638127091, -0.036348553679742532, -0.020029665393860505, 0.0525300188818924, -0.012366086600746434, 0.020942568493070163, 0.035488136884209633, 0.025721860226869335, 0.029523548216817803, 0.0412208404343419, -0.022181065411136524, -0.011767135122255924, -0.020312001557455935, -0.042154827632328237, -0.026898388139369487, 0.013063625153110988, 0.027698585558799, -0.021991471720735966, 0.032636536111271354, 0.018847734038958914, 0.0078360882476580917, -0.030791568564993054, -0.0087489447379758851, -0.02597905072785384, 0.019254532464741344, -0.0236530384177425, -0.036951000570156718, 0.021216724207712451 },
		{ 0.0054581520789202679, -0.03284137790796831, 0.041388046417431, -0.018340534464538132, -0.0075268998614110693, 0.019247840323984296, 0.0034812154230638524, 0.007840307895492768, -0.0011006374184323029, -0.011043067587558342, -0.0042683607765405859, 0.054421906485060852, 0.0068147764875412927, -0.0027491960682659833, 0.025487786819587743, 0.0080111211682273691, -0.024402205150529575, -0.018529919556104498, 0.028986873503595634, -0.015973550366128617, -0.025812746988018866, -0.00038927725710760359, -0.028034128535057528, -0.025214449383461058, -0.011948521865254861, 0.02248361807053682, 0.17980334158134526, 0.030729800080040486, -0.0239051223170358, 0.012961859534489296, 0.01280826913244322, 0.026107388988532206, -0.0051879680322508637, -0.013575007328112426, 0.0064929873533031719, -0.019546632136500527, 0.0029940335748287164, 0.011168011653069781, -0.021383799128294977, -0.037106711879300543, 0.013253103630468839, 0.0038839944366597123, -0.017672111206647672, 0.00476379121643974, 0.0167956880233296 },
		{ 0.006397357406211503, -0.016310434863123689, 0.013374818212289342, -0.019488836534602857, 0.029378965449089519, -0.052525147201077357, 0.0064423558261258789, -0.0756301042089379, -0.026185744742124453, -0.10023134631589807, 0.046752501332082863, -0.056077612664331811, -0.022024249812463481, 0.0066851044317652571, -0.044688193619762111, 0.0076770180238799474, -0.011701928141132101, -0.0246196983332236, -0.092966384300587568, -0.01439433064185639, -0.033381945941824809, 0.013572499169898942, 0.13282794666887296, 0.088618288460822842, 0.044575436028416869, -0.13313438011694165, 0.21641712376745376, -0.034445735447428807, -0.035196880482010626, -0.018728693264238296, -0.0700870530462459, -0.078005569118293511, 0.022923999623639009, -0.016443102363824961, 0.024450327412744579, -0.017035080936561572, 0.024604831295032006, -0.047402209814646226, -0.0086979639516643874, 0.031730223027503134, -0.06604562348762906, 0.013311289805781341, -0.00968053371043241, 0.0097690607899530678, 0.019237208274597836 },
		{ 0.015248475226363471, -0.041249870352248329, 0.041471172065756928, -0.036845305458223225, 0.008657032458253805, 0.018271658651038504, -0.036255609024149905, -0.044104541684544286, -0.041110936660172188, 0.022673971023082967, -0.035497260993697662, -0.060871276780403889, 0.045805612156590909, 0.00895979137221722, 0.010020207450981731, 0.047866590391448374, -0.012813046623035214, -0.012804627857568725, -0.0070803009273484093, -0.03306230793006016, 0.035036820693575443, -0.0028427670077540604, 0.011397345682333465, -0.01059789916095707, 0.0076076591127585353, -0.016388515823765853, 0.060572777538944175, 0.01439112640343366, -0.059792344040753613, -0.039295521352038718, 0.013243137782313782, 0.0061900275107398539, 0.04577358675677607, 0.0055714532900707986, 0.0015549758082173224, 0.00035380097842665885, 0.075442105182662436, -0.039251989549129222, -0.032867136646097438, 0.026196912526663889, -0.010758418391781887, 0.032617647725744985, -0.016122657196493573, -0.055984590112400118, -0.002235071964940185 },
		{ -0.0087438844518952352, 0.037640934999272659, -0.042244837024120477, 0.023537801817223702, 0.0036130534800784195, 0.022903603488715184, -0.0022338840077500747, 0.037746344681804142, 0.038031725017466836, 0.059453827513590227, -0.009874427006008454, 0.019779205916014956, -0.01330365480152324, -0.0059632032518472768, 0.019828061109287875, -0.027575714452483274, 0.038031288965642562, 0.033210615382113146, 0.053257240069752319, 0.012788279264560295, 0.0063754746570427318, 0.0021345773712347269, -0.062717672120950974, -0.062902126535971672, -0.033395171036853757, 0.030601980052006796, -0.21800593609109536, 0.006571050335960173, 0.047909563427075613, 0.0023492055842246323, 0.037168150514015806, 0.046993404624111133, -0.03176605829551004, 0.0071852640902117324, -0.016045291523774131, -0.00050741664390369209, -0.039907637346325721, 0.018870614791270543, 0.040142190617078889, -0.0038685541239528768, 0.043261919784866994, -0.021329738064219662, 0.031208645763253384, -0.0015900524321167677, -0.020896442517160007 },
		{ -0.016332848964072676, 0.038762062108793849, -0.0443921520299234, 0.033129581133033506, 0.016339114465390641, -0.013519479289437017, 0.0087574603650272427, 0.103599408143829, 0.0428978562638904, -0.023774565125539413, 0.036384741635808557, 0.084857203250948116, -0.030965931322816716, -0.00035664836982199561, -0.023604341945668381, -0.045327822997445873, 0.0591407639222815, 0.037000340262124114, -0.016451821450837765, 0.053118797286467172, 0.030735600750739837, -0.0087834517269001038, -0.037758601482860039, -0.0044659863056230719, 0.0030944116999756777, 0.0861060757286423, -0.11350205646297995, -0.036310729253533243, 0.10087931899044805, 0.025845629333383809, -0.023472790243179435, 0.0076640289923353014, -0.047970124835328309, 0.031565355735810326, -0.030863776697651103, 0.02813740111463767, -0.063643953593416014, 0.055480336377528895, 0.052328728902742053, -0.040392362676430323, 0.0034888083074103736, -0.035496156732426488, 0.01914396641849726, 0.0039787664871059605, 0.0090368234691439037 },
		{ -0.0066165578004783272, 0.01438512849158698, -0.019757370062275504, 0.026642897926599324, 0.00854952390966707, 0.032497100480128042, 0.0019754619540935706, 0.14191128595608829, 0.04805562016954508, 0.04202079655106021, 0.01378916765765611, 0.094965726513981316, -0.019446304810994918, -0.006899879861034134, 0.00149436173661281, -0.041079383516760534, 0.0726125053824072, 0.044288551448974332, 0.036197949772484618, 0.02902857923973105, 0.050181466008535285, -0.024984616037098853, -0.10155427995526825, -0.055459135716772648, -0.015424327210622724, 0.1992657455553993, -0.19896064257177939, 0.016595343739536254, 0.10677163025737624, 0.040541778961638734, 0.012207224146837429, 0.040812982683876148, -0.047986046862271189, 0.060563710521414259, -0.0571702572390341, 0.046762169480924348, -0.053222887200199895, 0.095017738190339887, 0.021208651819864394, -0.085884047263144914, 0.036375992425913536, -0.034139467845795225, -0.0021551328591638982, -0.016099264296997189, 0.010657609201036363 }
	};

	static const double b2[ 45 ] = { 0.051933756862425566, -0.13352402917228884, 0.14389670707993116, -0.11217264638925904, -0.0024519581384152261, -0.0304506850944222, -0.067806193014844335, -0.17593835115568018, -0.13422584517586841, -0.063065215642878328, -0.044855821396546446, -0.14313182273622133, 0.098924034202180433, 0.031319585012413532, -0.015343331627225648, 0.10011076213382004, -0.080321569609751564, -0.054523443619834196, -0.084516300457012944, -0.10986751840964636, 0.026521518481122667, -0.016188149524004616, 0.11885034137184577, 0.091214836712203234, 0.055635359393274436, -0.0992988550416824, 0.41545934884336011, 0.020727908082042276, -0.18862464329229117, -0.048089615230161968, -0.026933717091237753, -0.089549469732947737, 0.11573387577895169, 0.039298045540403795, -0.0037627461696782796, 0.0092124864322265868, 0.15036677551357491, -0.081042673623696576, -0.12927845515793218, 0.051413357737975243, -0.086105615885071179, 0.10409730074802558, -0.10151236211947827, -0.062102377729233355, 0.027368189285879242 };

	double v3[ 45 ];

	static const double a3[ 45 ][ 15 ] = {
		{ 0.030429911710792273, -0.011471135454710082, -0.0041012922514453085, 0.060713435248879406, 0.0042893956888992, 0.023435233516808464, 0.0067517724677054695, 0.027186495359614003, 0.015083935116398354, -0.024601660844353317, -0.0019376448815856269, -0.050726973396056314, 0.05596515507170903, 0.020614116644298984, -0.061362346707494735 },
		{ -0.051913410370812429, -0.0846454874406107, 0.027548723588435936, -0.076918651105045463, -0.044150782760130645, -0.065605327213256279, 0.048258060752739723, 0.04989908381330229, 0.11548460882106283, 0.091428164041598342, 0.023668589932228059, 0.084056465010753048, -0.0021655777483999888, -0.095142322899718218, 0.12743983237982914 },
		{ 0.062736261769938517, 0.10285806405833192, -0.014436085031533666, 0.073359947949067111, 0.093739879003863907, 0.078100916716378352, -0.042402247179985988, -0.032087187263508107, -0.13446144833266344, -0.13449122661678309, -0.021507810192825216, -0.095620005281284864, -0.012307600952700417, 0.1013316855040016, -0.14501473472793067 },
		{ -0.055933646735485433, -0.069351575380872485, 0.02356199846199674, -0.064312364355896087, -0.043241936796195264, -0.0535435723550283, 0.02345289461798555, 0.0044862073503098262, 0.0896511894770307, 0.08076061383794747, 0.005010654147537188, 0.066066688065052781, -0.00796877960921767, -0.081415202207654913, 0.1178068259596682 },
		{ 0.0421713410552741, 0.022846249045793441, -0.070203810951339, -0.058113116211339605, 0.037577628411446413, -0.000617719493330107, -0.028130632748420821, -0.13649787053773582, -0.0095521101626357616, -0.028000721785167007, -0.0086116508010165779, 0.021929645958017639, -0.020121267493434212, 0.024959721072243739, 0.0337575060528831 },
		{ 0.060548518479893412, -0.047694381576188591, 0.089227254979461226, 0.037369559884887861, 0.021481432682225715, 0.01229610528645317, 0.15392572877044597, 0.21755560865719314, 0.081679045279520371, 0.047045981681828004, -0.00050021176345416032, 0.055484570637561147, 0.01592009665131771, -0.045300985218761254, -0.030801066708636728 },
		{ -0.11961038603564135, -0.094110897255196918, 0.051128383439387755, 0.061452191948732686, -0.084013871462305614, -0.065347048373818384, 0.012018277173287874, 0.0954042077274219, 0.11454243649346799, 0.10680510177526467, 0.054932962387892026, -0.030584461468478864, 0.072787250497511016, -0.083880514004292653, 0.065866051581520535 },
		{ 0.07530827953916408, -0.0036406854599983314, 0.10628548585194755, -0.098532450728155238, 0.047523858970537312, 0.015640911046345848, 0.12774370643464483, 0.056794058471655293, 0.00023143253303290547, 0.025997236132136489, -0.018863686154381682, 0.19084282899988891, 0.0064966719920620886, 0.029819179058784576, 0.44257607241604707 },
		{ -0.05499061310488905, -0.085149591573317529, 0.038104105193558828, -0.079058820522947909, -0.018662407322421165, -0.05948876698050018, 0.039690527262874052, 0.061030562681481555, 0.12486257607498134, 0.083184824122443685, 0.01010874243596689, 0.084657718951017868, -0.0067967422428056711, -0.098947651585489238, 0.15987087461455926 },
		{ 0.065559330783364678, -0.0049025883810794044, 0.22067806439506696, 0.1144917956762372, 0.0047551320265096786, -0.0042533831351278736, 0.14149727226202924, 0.35505898209156955, 0.0061289606468675518, 0.0016740748202837578, -0.015060042670749197, -0.075344200900218025, 0.0020480039127023435, -0.0029382006162338545, -0.1421138095255641 },
		{ -0.048947582773189266, 0.023917483491164473, -0.10827685198602598, -0.10111255316638476, 0.0020282783387239925, 0.0014226148423994564, -0.11241746070560635, -0.29848243671758684, -0.041616278768950618, -0.010159644726691565, -0.023901215172210041, 0.047067958210588907, -0.0094318411871759048, 0.022622744160240949, 0.12137032231742648 },
		{ -0.028486417913353539, -0.035954413812772881, 0.014057321250040259, -0.17164086391860026, 0.03601112332376815, -0.016982922453067689, 0.0331954647847284, -0.00093435460966549027, 0.055907517268346085, 0.025746356764197273, -0.008086612530806005, 0.23060718588575527, -0.05302461552295408, -0.06949512577863419, 0.34775182105097319 },
		{ 0.015258456307422851, 0.014256116138848283, 0.079039030930735055, 0.090153142523193161, -0.035921665443793528, 0.025768214477583332, 0.034606943252055808, 0.13755471034840527, -0.011108914135090185, -0.0320150141904274, 0.0079968233190727579, -0.10631918788074685, -0.035237464013310266, -0.00422525813100657, -0.23538980567464141 },
		{ 0.082500798357196453, 0.03976786406837373, -0.04912037664746155, -0.0086811509241653713, 0.029014481044039614, 0.016520932935122568, 0.00092358143242100951, -0.06239748256012894, -0.05698163826824109, -0.039491975972077813, 0.0013310781859463389, 0.036425369137304271, -0.021698817132354432, 0.0174063683112195, -0.005094268081278326 },
		{ -0.057019759203445548, -0.017912568196362222, 0.14144834313096494, 0.030541017937002507, -0.0241523768541347, -7.3856323349179266E-5, 0.070593872654212048, 0.22169775288592511, 0.0210941643033798, 0.0076297656021086973, 0.00024269414048257832, -0.041202151783949266, 0.00096964239105648152, -0.013666797331092054, -0.03425855965631134 },
		{ 0.034481689622051073, 0.063364052724519823, 0.017185699215834987, 0.0029118305771845413, 0.0288111188177203, 0.046919368167217, -0.013082535935437773, -0.0090351088001599522, -0.075846553431634156, -0.081766122184560822, -0.043865841765178187, -0.034942909502333336, -0.036131830768170718, 0.053740401985593, -0.23338826175231298 },
		{ 0.10420088840260999, 0.025197072500151761, -0.050595499300874577, -0.10685162324834716, 0.027274219616837528, 0.0077981514566190209, 0.044865341787861113, -0.12826238304881921, -0.010586488397607095, -0.030483251356596408, -0.0465271674623787, 0.095965547476508165, -0.081109332620384209, 0.012090686983744397, 0.15598185661443184 },
		{ 0.0854567446764708, -0.040164213514901341, 0.05021667445805901, -0.038740998147728023, 0.021019684595962162, 0.00098740529939323287, 0.080174434167158123, 0.023810990903546395, 0.044926597686074723, 0.010004559095727507, -0.049531790597423013, 0.10252771392663909, -0.015755297321779556, -0.04342246228670938, 0.098019907461799047 },
		{ 0.0064067337286297489, -0.02281917774681335, 0.20562684401349235, 0.076166738544654125, 0.013156651917070767, -0.019642093756164641, 0.10201670627294276, 0.27223340408619834, 0.033169466619679677, 0.036190140650589758, 0.00063030001667352463, -0.038571302779441867, 0.0021567877170354556, -0.021728727292568365, -0.0446787531985041 },
		{ -0.10105821022000235, -0.0744146671193959, 0.046696068909185734, -0.05210440292067197, -0.019444157392881273, -0.0417596180404804, 0.021311213852070684, 0.027391468627373013, 0.092590051287716274, 0.08333844298094753, 0.031400373700296517, 0.079958200317442565, -0.014710973152306229, -0.075619503613849989, 0.14508999989742807 },
		{ 0.1345145624488171, 0.066399225675715393, -0.020873520669454085, -0.092373441375967327, 0.10215833323994743, 0.067048069209119238, -0.00826583749212706, -0.13062285650215127, -0.063222060771570512, -0.06469559680241288, -0.039200999368335931, 0.035767572132862496, -0.098421003177130664, 0.039164170501568854, 0.071509117136192152 },
		{ 0.017644291231186458, -0.037946338305616542, -0.027850462430960114, 0.023800568841611757, -0.024123495732664469, -0.01324841843712183, 0.026859649461505326, -5.5277709885126112E-5, 0.043510939998041349, 0.034705667712323189, 0.027847573481882267, 0.02038441110041065, 0.073148537597028593, -0.026164547256083769, -0.0086630441603520736 },
		{ -0.016018599458302318, 0.032163952731496116, -0.23462415089884034, -0.0080295082175469868, -0.028141486618301553, -0.0022721501220701826, -0.17949886310503477, -0.23837177951832045, -0.017828724154644135, -0.029678053516879088, 0.011860798920263114, -0.013596713993374435, -0.042055962748004674, 0.024951496298478076, -0.11053824991838804 },
		{ -0.021125066960171916, 0.022634692635585669, -0.22238563933047661, -0.040467095352758629, -0.020910075283433187, 0.023301298249947703, -0.18986278627605568, -0.29583582418208554, -0.015784354058010819, -0.018138031303046014, 0.021577558831286404, 0.054910393378086626, -0.011236683021233852, 0.00877845442564174, 0.011293218422952528 },
		{ 0.033318252005182514, 0.020719465834775984, -0.12792906108730964, 0.021057142081778328, -0.017611716499011634, 0.027722293849198255, -0.087127861492544353, -0.16347350406315936, -0.0016538498368028217, -0.04163273298455443, -4.4122236048622172E-5, 0.022542722295022317, 0.029773156950222365, 0.028255409200216557, -0.0064297109959932913 },
		{ 0.13032132776269278, 0.029486997734477036, 0.16332994419296604, -0.067402719802974756, 0.12731345004674788, 0.043564859696755991, 0.19050134675436309, 0.10376226039806809, -0.040174601962661795, -0.037054538682762178, -0.039553639889794968, 0.089007903281781542, -0.0030199586993138717, 0.02835152072725404, 0.37515543273094387 },
		{ -0.094392741678245531, 0.017797341184873312, -0.27857642682726336, -0.53886436862893849, 0.058667067729699096, 0.001662607309318723, -0.35887108206518881, -0.35355420397347687, -0.032647150100124436, -0.076565004134741252, -0.001217780790233033, 0.48654988696783519, -0.15372530842579893, -0.02444735226150993, 0.25016273096570418 },
		{ 0.0026457381701077347, 0.011503924597638342, 0.1175603091616461, 0.034449601606574821, 0.028064359945938141, 0.0029181064860448404, 0.072943735239683546, 0.20770252698394578, -0.0047107207351138655, 0.0010824087352990079, -0.00018978879988645404, -0.078693103140536791, -0.017609345479548164, 0.013428369299658831, -0.076776690858054719 },
		{ 0.072706670894673128, -0.04900765407884642, 0.0024113786661052851, -0.15749961683901406, 0.010935389462336421, -0.011489211811422114, 0.094299245160691283, -0.05887373214146744, 0.074204763983954578, 0.043154904894156139, -0.027889784187642739, 0.19909407864010623, -0.023996367549743363, -0.04964260941219524, 0.39792944573521749 },
		{ -0.095686216938245222, -0.000991452114913601, -0.0034268163483982786, -0.058425655805285082, 0.0076160801517539938, 0.012708897583423711, -0.00914058662922086, -0.0061216291575799178, -0.021971429955153523, 0.022548676750343111, -0.014756013724273043, 0.0047929212862059681, -0.03592983409862624, -0.004683126370967146, 0.12556906623480349 },
		{ 0.03315691059765058, -0.0023519550433419223, 0.19541166949962244, 0.06592382717211312, -0.013395786776405757, -0.0066995916131926585, 0.093901030427433183, 0.31604501979219052, 0.013862810910415304, -0.016190608459731855, 0.0009666629000670984, -0.024806461952027355, -0.018708758063849102, 0.040748601788698477, -0.062586727780247153 },
		{ 0.078588164567668026, -0.018642592509085343, 0.15894262110100582, 0.064318454076294773, 0.017944246208215019, -0.0246121317980917, 0.13801427050162152, 0.17258484955250644, 0.022313814925521976, 0.033838101686673561, 0.011276175817526109, 0.0092439218821910463, -0.0090510622660181419, -0.00058079351841038582, 0.035939242192066123 },
		{ 0.032370438851218319, 0.061822785949998939, 2.2515656240178291E-5, 0.099550022468243349, -0.0030864300219358397, 0.018953146883096055, -0.048377341088583513, 0.015933803919361514, -0.0897687238797186, -0.058522417849391731, 0.025731037885089292, -0.11241165053444906, 0.0059934965747593374, 0.077519430096226208, -0.2125710085180659 },
		{ 0.10061253641216726, 0.044380993192921707, -0.042016458099369734, 0.03488672119631131, 0.057978048567032335, -0.010894289148146817, -0.0052159918954473887, -0.15945661827242988, -0.049022217287297172, -0.010722578352527381, -0.06788693665175656, 0.070318917238598649, -0.183942245083176, 0.0023318797185769638, 0.15653164813420128 },
		{ -0.15720922393602241, -0.037645451743024479, 0.00092029746123738546, 0.067773784369160581, -0.10079357069313977, -0.014132630808582061, -0.052332495987918094, 0.06360946049359549, 0.040215434604489862, 0.030086985906411665, 0.028544095328113462, -0.096901421865764434, 0.066034772969815053, -0.026245836866611687, -0.078371481092602127 },
		{ 0.074062718004924782, 0.049244353051645089, -0.023056656806572436, -0.044252882308371995, 0.062231272277009156, 0.022650263842589755, -0.0013871006359365832, -0.10585632816962336, -0.034387979334577691, -0.048673570479462552, -0.039770196070150719, 0.02821044999189519, -0.044689385888829314, 0.046135317580077134, 0.066628352381758668 },
		{ -0.059659805195959238, 0.0060193436990479852, 0.0060430643440244894, 0.093467970558001129, -0.084121872167893283, 0.021474104640702263, -0.028906670239839186, 0.10986507248713982, -0.016416946140447582, -0.0031706388100968641, -0.024142520265522237, -0.16347010386539823, -0.0068807006172086956, -0.009484951819502159, -0.32097682564372548 },
		{ 0.025027597690085727, 0.023116577698323981, 0.025647750389152277, -0.12432191956313283, 0.089919395416061254, -0.0010032178808474397, 0.034154216316645537, -0.073258150208852735, -0.0082717048276847, -0.0085013713591728639, -0.040036418368419674, 0.10654199196017296, -0.0718813141234634, 0.031808510122176871, 0.23115055590279554 },
		{ -0.075488177284467092, -0.071776660282181742, 0.046468262983681195, -0.092264487663929889, -0.027683218739383321, -0.072597712633641717, 0.067382637821066, 0.02321485385353372, 0.0947743774627414, 0.098261285016872743, 0.029554223554100867, 0.12241571210613868, -0.024534925571359945, -0.0703515457144001, 0.17761444776547267 },
		{ -0.049027426344744357, -0.016233511692686296, 0.0025759638168079985, 0.10159921434214048, -0.11211756309722204, -0.012775672474882057, -0.020566830347744265, 0.077375715953660065, -0.00901379843145126, 0.018255542784233136, 0.018419875764159, -0.098256794011679527, 0.026959425877853813, -0.019700315948286546, -0.21723340664627469 },
		{ -0.020940553543628962, -0.04054794998219742, 0.14727098125947621, -0.012125578793213689, 0.017852107734153692, -0.039878508384314948, 0.096235470430363346, 0.17788554639715984, 0.049756934384053396, 0.057923047790416141, -0.00038416347031019148, 0.0033584995501507457, -0.00959222750780903, -0.036174102197861, 0.030642141424393261 },
		{ 0.034699909577602606, 0.051338155809273957, -0.00072752642213258934, 0.064923450941197211, -0.0016719255983632354, 0.052965507802081313, -0.037177940935703119, 0.013456246605129727, -0.055606306967696015, -0.05425776251520692, -0.017599910825923836, -0.074992072717836242, 0.0093885723375285012, 0.048547335615615493, -0.13482402222190409 },
		{ -0.043079593064996757, -0.052204622617255239, 0.018383175102716375, -0.043010836505462433, -0.026918120343519612, -0.056104371864350709, 0.029348832949610125, 0.043945454130226734, 0.062402159322316457, 0.062207673704178408, 0.029038007062175294, 0.0489008760884029, 0.010826822062293517, -0.048948940132146929, 0.046782516272861584 },
		{-0.15387908842516629, -0.073666239607598, 0.010689026646281461, 0.061503377425313747, -0.11226161852298806, -0.017443019416759083, 0.00891507284904646, 0.055735512430850107, 0.084809779500954877, 0.10033908241902251, 0.05068512638056779, -0.02935751603086947, 0.0774266173283182, -0.064498073470460063, 0.06618956627205716 },
		{ 0.01400914038724874, 0.022935140332806577, -0.066490517571514182, -0.00061334513341812318, 0.023068981418270219, 0.04112798978856741, -0.076056709248761692, -0.12567592114978585, -0.022355279112106059, -0.031655532130436738, -0.00592269608855733, 0.0055763285280232226, 0.023941474706561956, 0.026576132940705939, 0.042327903821042796 }
	};
	static const double b3[ 15 ] = { 0.1509159451715264, 0.17903151894955524, -0.054927917643813065, 0.17645237894737631, 0.085211944671718623, 0.15677750602438928, -0.042998239500351622, -0.097006998042479781, -0.21956481772061825, -0.18646731717119161, -0.068028663170147621, -0.1808114370372878, 0.0043667745382358356, 0.17910630761441323, -0.25518080682851763 };

	double v4[ 15 ];

	static const double a4[ 15 ][ 3 ] = {
		{ 0.34160071894678073, -0.37217567814406566, -0.073438174683426644 },
		{ 0.2360609485928192, -0.081649836378150389, -0.30137781342003234 },
		{ -0.16528594248952178, -0.2014456466455789, -0.614197570055916 },
		{ -0.66044563888392227, -0.2741627898810618, -0.22696915173429449 },
		{ 0.29294219502893032, -0.28563231191573507, 0.03767334742521089 },
		{ 0.201416479572921, -0.06654539462054769, -0.28039792906531968 },
		{ -0.37742707540375364, -0.16743165065926807, -0.36936426969147412 },
		{ 0.12804178546710865, -0.30228226041438511, -0.87011334739184454 },
		{ -0.27609956512411193, 0.0735728127187767, 0.36535986126768921 },
		{ -0.36049372632169013, 0.14942055814470945, 0.26730928760533818 },
		{ -0.23004922243116888, 0.098427096299886385, 0.16286408757819881 },
		{ 0.76831346894351393, 0.15776570668042308, 0.14926347945128754 },
		{ -0.3120979030694791, 0.1159212556012188, -0.15780333650988518 },
		{ 0.16572871133416767, -0.12086663847069315, -0.22639716104184104 },
		{ 1.0773969011791018, 0.074120128836744542, 0.028591035907610123 }
	};
	static const double b4[ 3 ] = { 0.35004631807090653, -0.11700815424551032, -0.54618213782366509 };

	static const double ao[ 3 ] = { 0.236034641103551, 5.09429075779317, 1.80332235534696 };
	static const double bo[ 3 ] = { 3.1380831408776, 0.887875288683602, -0.348013856812933 };

	/* Input */
	for ( int k = 0; k < 4; k++ ) {
		v1[ k ] = ( X[ k ] - bi[ k ] ) * ai[ k ];
	}

	/* Layer 1 */
	/* Sigmoid Symmetric Transfer Function */
	for ( int k = 0; k < 50; k++ ) {
		double d = 0.;
		for ( int l = 0; l < 4; l++ ) {
			d += a1[ l ][ k ] * v1[ l ];
		}

		v2[ k ] = 2. / ( 1. + exp( -2. * ( d + b1[ k ] ) ) ) - 1.;
	}

	/* Layer 2 */
	/* Sigmoid Symmetric Transfer Function */
	for ( int k = 0; k < 45; k++ ) {
		double d = 0.;
		for ( int l = 0; l < 50; l++ ) {
			d += a2[ l ][ k ] * v2[ l ];
		}

		v3[ k ] = 2. / ( 1. + exp( -2. * ( d + b2[ k ] ) ) ) - 1.;
	}

	/* Layer 3 */
	/* Sigmoid Symmetric Transfer Function */
	for ( int k = 0; k < 15; k++ ) {
		double d = 0.;
		for ( int l = 0; l < 45; l++ ) {
			d += a3[ l ][ k ] * v3[ l ];
		}

		v4[ k ] = 2. / ( 1. + exp( -2. * ( d + b3[ k ] ) ) ) - 1.;
	}

	/* Output 1 */
	/* Map Standard Deviation and Mean Output Reverse-Processing Function */
	for ( int k = 0; k < 3; k++ ) {
		double d = 0.;
		for ( int l = 0; l < 15; l++ ) {
			d += a4[ l ][ k ] * v4[ l ];
		}

		Y[ k ] = ( ( d + b4[ k ] ) / ao[ k ] ) + bo[ k ];
	}
}
