# src/django_core_micha/auth/adapters.py
from allauth.socialaccount.adapter import DefaultSocialAccountAdapter
from django.contrib.auth import get_user_model
from allauth.account.adapter import DefaultAccountAdapter
from django_core_micha.auth.security import set_security_level
from allauth.mfa.adapter import DefaultMFAAdapter
from django.conf import settings
from rest_framework.permissions import BasePermission
from django_core_micha.auth.security import get_security_level, is_level_sufficient
import logging
logger = logging.getLogger(__name__)

class InvitationOnlySocialAdapter(DefaultSocialAccountAdapter):
    def is_open_for_signup(self, request, sociallogin):
        User = get_user_model()
        email = sociallogin.user.email
        # Erlaubt Login nur, wenn User schon existiert
        return User.objects.filter(email__iexact=email).exists()
    

class CoreAccountAdapter(DefaultAccountAdapter):
    def login(self, request, user):
        response = super().login(request, user)

        # Sicherheits-Logik basierend auf dem URL-Namen
        if request.resolver_match:
            # Wir holen uns den vollständigen Namen inkl. Namespace
            # Bei allauth headless oft: "headless:mfa:authenticate_webauthn" o.ä.
            view_name = request.resolver_match.view_name 
            
            logger.debug(f"Login via View-Name: {view_name}")

            # PRÜFUNG: Ist es ein WebAuthn/Passkey Login?
            # Hinweis: Der genaue String hängt von deiner Allauth-Version ab.
            # Suche nach "webauthn" und "login" oder "authenticate"
            if "webauthn" in view_name and ("login" in view_name or "authenticate" in view_name):
                set_security_level(request, "strong")
            else:
                set_security_level(request, "basic")
        
        return response
    
class CoreMFAAdapter(DefaultMFAAdapter):
    def on_authentication_success(self, request, user, **kwargs):
        """
        Wird aufgerufen, wenn ein MFA-Schritt (TOTP, Recovery, WebAuthn als 2. Faktor)
        erfolgreich war – egal ob im klassischen oder Headless-Flow.
        """
        # Erfolgreiche MFA => Session ist STRONG
        if getattr(request, "user", None) and request.user.is_authenticated:
            set_security_level(request, "strong")
        return super().on_authentication_success(request, user, **kwargs)

class RequireStrongSecurity(BasePermission):
    required_level = "strong"

    def has_permission(self, request, view):
        current = get_security_level(request)
        if is_level_sufficient(current, self.required_level):
            return True

        # Nicht ausreichend – je nach Setting blockieren oder nur „markieren“
        if getattr(settings, "SECURITY_ENFORCE_STRONG_AUTH", False):
            # Harte Policy: Zugriff verweigern
            return False

        # Weiche Policy: Zugriff erlauben, aber später im UI warnen
        return True