# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['colour']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'colouralpha',
    'version': '0.1.6',
    'description': 'Python color representations manipulation library (RGB, HSL, web, ...)',
    'long_description': '===========\nColourAlpha\n===========\n\nConverts and manipulates common color representation (RGB, HSL, web, ...)\n\nThis project is forked from `Colour <https://github.com/vaab/colour/>`_.\n\nFeatures\n========\n\n- Damn simple and pythonic way to manipulate color representation (see\n  examples below)\n\n- Full conversion between RGB, HSL, 6-digit hex, 3-digit hex, human color\n\n- One object (``Color``) or bunch of single purpose function (``rgb2hex``,\n  ``hsl2rgb`` ...)\n\n- ``web`` format that use the smallest representation between\n  6-digit (e.g. ``#fa3b2c``), 3-digit (e.g. ``#fbb``), fully spelled\n  color (e.g. ``white``), following `W3C color naming`_ for compatible\n  CSS or HTML color specifications.\n\n- smooth intuitive color scale generation choosing N color gradients.\n\n- can pick colors for you to identify objects of your application.\n\n\n.. _W3C color naming: http://www.w3.org/TR/css3-color/#svg-color\n\n\nInstallation\n============\n\nYou don\'t need to download the GIT version of the code as ``colouralpha`` is\navailable on the PyPI. So you should be able to run:\n\n.. code-block::\n\n    pip install colouralpha\n\nIf you have downloaded the GIT sources, then you could add the ``colour.py``\ndirectly to one of your ``site-packages`` (thanks to a symlink). Or install\nthe current version via:\n\n.. code-block::\n\n    poetry install\n\nAnd if you don\'t have the GIT sources but would like to get the latest\nmain or branch from gitlab, you could also:\n\n.. code-block::\n\n    pip install git+https://gitlab.com/marcin-serwin/colouralpha.git\n\nOr even select a specific revision (branch/tag/commit):\n\n.. code-block::\n\n    pip install git+https://gitlab.com/marcin-serwin/colouralpha.git@main\n\n\nUsage\n=====\n\nTo get complete demo of each function, please read the source code which is\nheavily documented and provide a lot of examples in doctest format.\n\nYou may also access generated documentation `here <https://colouralpha.readthedocs.io/en/latest/>`_.\n\nHere is a reduced sample of a common usage scenario:\n\n\nInstantiation\n-------------\n\nLet\'s create blue color:\n\n.. code-block:: python\n\n    >>> from colour import Color\n    >>> c = Color("blue")\n    >>> c\n    <Color blue>\n\nPlease note that all of these are equivalent examples to create the red color:\n\n.. code-block:: python\n\n    Color("red")           ## human, web compatible representation\n    Color(red=1)           ## default amount of blue and green is 0.0\n    Color("blue", hue=0)   ## hue of blue is 0.66, hue of red is 0.0\n    Color("#f00")          ## standard 3 hex digit web compatible representation\n    Color("#ff0000")       ## standard 6 hex digit web compatible representation\n    Color(hue=0, saturation=1, luminance=0.5)\n    Color(hsl=(0, 1, 0.5)) ## full 3-uple HSL specification\n    Color(rgb=(1, 0, 0))   ## full 3-uple RGB specification\n    Color(Color("red"))    ## recursion doesn\'t break object\n\n\nReading values\n--------------\n\nSeveral representations are accessible:\n\n.. code-block:: python\n\n    >>> c.hex\n    \'#00f\'\n    >>> c.hsl  # doctest: +ELLIPSIS\n    (0.66..., 1.0, 0.5)\n    >>> c.rgb\n    (0.0, 0.0, 1.0)\n\nAnd their different parts are also independently accessible, as the different\namount of red, blue, green, in the RGB format:\n\n.. code-block:: python\n\n    >>> c.red\n    0.0\n    >>> c.blue\n    1.0\n    >>> c.green\n    0.0\n\nOr the hue, saturation and luminance of the HSL representation:\n\n.. code-block:: python\n\n    >>> c.hue  # doctest: +ELLIPSIS\n    0.66...\n    >>> c.saturation\n    1.0\n    >>> c.luminance\n    0.5\n\nA note on the ``.hex`` property, it\'ll return the smallest valid value\nwhen possible. If you are only interested by the long value, use\n``.hex_l``:\n\n.. code-block:: python\n\n    >>> c.hex_l\n    \'#0000ff\'\n\n\nModifying color objects\n-----------------------\n\nAll of these properties are read/write, so let\'s add some red to this color:\n\n.. code-block:: python\n\n    >>> c.red = 1\n    >>> c\n    <Color magenta>\n\nWe might want to de-saturate this color:\n\n.. code-block:: python\n\n    >>> c.saturation = 0.5\n    >>> c\n    <Color #bf40bf>\n\nAnd of course, the string conversion will give the web representation which is\nhuman, or 3-digit, or 6-digit hex representation depending which is usable:\n\n.. code-block:: python\n\n    >>> "%s" % c\n    \'#bf40bf\'\n\n    >>> c.luminance = 1\n    >>> "%s" % c\n    \'white\'\n\n\nRanges of colors\n----------------\n\nYou can get some color scale of variation between two ``Color`` objects quite\neasily. Here, is the color scale of the rainbow between red and blue:\n\n.. code-block:: python\n\n    >>> red = Color("red")\n    >>> blue = Color("blue")\n    >>> list(red.range_to(blue, 5))\n    [<Color red>, <Color yellow>, <Color lime>, <Color cyan>, <Color blue>]\n\nOr the different amount of gray between black and white:\n\n.. code-block:: python\n\n    >>> black = Color("black")\n    >>> white = Color("white")\n    >>> list(black.range_to(white, 6))\n    [<Color black>, <Color #333>, <Color #666>, <Color #999>, <Color #ccc>, <Color white>]\n\n\nIf you have to create graphical representation with color scale\nbetween red and green (\'lime\' color is full green):\n\n.. code-block:: python\n\n    >>> lime = Color("lime")\n    >>> list(red.range_to(lime, 5))\n    [<Color red>, <Color #ff7f00>, <Color yellow>, <Color chartreuse>, <Color lime>]\n\nNotice how naturally, the yellow is displayed in human format and in\nthe middle of the scale. And that the quite unusual (but compatible)\n\'chartreuse\' color specification has been used in place of the\nhexadecimal representation.\n\n\nColor comparison\n----------------\n\nSane default\n~~~~~~~~~~~~\n\nColor comparison is a vast subject. However, it might seem quite straightforward for\nyou. ``Colour`` uses a configurable default way of comparing color that might suit\nyour needs:\n\n.. code-block:: python\n\n    >>> Color("red") == Color("#f00") == Color("blue", hue=0)\n    True\n\nThe default comparison algorithm focuses only on the "web" representation which is\nequivalent to comparing the long hex representation (e.g. #FF0000) or to be more\nspecific, it is equivalent to compare the amount of red, green, and blue composition\nof the RGB representation, each of these value being quantized to a 256 value scale.\n\nThis default comparison is a practical and convenient way to measure the actual\ncolor equivalence on your screen, or in your video card memory.\n\nBut this comparison wouldn\'t make the difference between a black red, and a\nblack blue, which both are black:\n\n.. code-block:: python\n\n   >>> black_red = Color("red", luminance=0)\n   >>> black_blue = Color("blue", luminance=0)\n\n   >>> black_red == black_blue\n   True\n\n\nCustomization\n~~~~~~~~~~~~~\n\nBut, this is not the sole way to compare two colors. As I\'m quite lazy, I\'m providing\nyou a way to customize it to your needs. Thus:\n\n.. code-block:: python\n\n   >>> from colour import RGB_equivalence, HSL_equivalence\n   >>> Color.equality = staticmethod(HSL_equivalence)\n   >>> black_red = Color("red", luminance=0)\n   >>> black_blue = Color("blue", luminance=0)\n\n   >>> black_red == black_blue\n   False\n\nYou may also set it per instance:\n\n.. code-block:: python\n\n   >>> black_red = Color("red", luminance=0, equality=HSL_equivalence)\n   >>> black_blue = Color("blue", luminance=0, equality=HSL_equivalence)\n\n   >>> black_red == black_blue\n   False\n\nAs you might have already guessed, the sane default is ``RGB_equivalence``, so:\n\n.. code-block:: python\n\n   >>> Color.equality = staticmethod(RGB_equivalence)\n   >>> black_red = Color("red", luminance=0)\n   >>> black_blue = Color("blue", luminance=0)\n\n   >>> black_red == black_blue\n   True\n\nHere\'s how you could implement your unique comparison function:\n\n.. code-block:: python\n\n   >>> saturation_equivalence = lambda c1, c2: c1.saturation == c2.saturation\n   >>> red = Color("red", equality=saturation_equivalence)\n   >>> blue = Color("blue", equality=saturation_equivalence)\n   >>> white = Color("white", equality=saturation_equivalence)\n\n   >>> red == blue\n   True\n   >>> white == red\n   False\n\nNote: When comparing 2 colors with equality set per instance, *only* the \nequality function *of the first color will be used*. Thus:\n\n.. code-block:: python\n\n   >>> black_red = Color("red", luminance=0, equality=RGB_equivalence)\n   >>> black_blue = Color("blue", luminance=0, equality=HSL_equivalence)\n\n   >>> black_red == black_blue\n   True\n\nBut reverse operation is not equivalent !:\n\n.. code-block:: python\n\n   >>> black_blue == black_red\n   False\n\n\nEquality to non-Colour objects\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\nAs a side note, whatever your custom equality function is, it won\'t be\nused if you compare to anything else than a ``Colour`` instance:\n\n.. code-block:: python\n\n    >>> red = Color("red", equality=lambda c1, c2: True)\n    >>> blue = Color("blue", equality=lambda c1, c2: True)\n\nNote that these instances would compare as equal to any other color:\n\n.. code-block:: python\n\n    >>> red == blue\n    True\n\nBut on another non-Colour object:\n\n.. code-block:: python\n\n    >>> red == None\n    False\n    >>> red != None\n    True\n\nActually, ``Colour`` instances will, politely enough, leave\nthe other side of the equality have a chance to decide of the output,\n(by executing its own ``__eq__``), so:\n\n.. code-block:: python\n\n    >>> class OtherColorImplem(object):\n    ...     def __init__(self, color):\n    ...         self.color = color\n    ...     def __eq__(self, other):\n    ...         return self.color == other.web\n\n    >>> alien_red = OtherColorImplem("red")\n    >>> red == alien_red\n    True\n    >>> blue == alien_red\n    False\n\nAnd inequality (using ``__ne__``) are also polite:\n\n.. code-block:: python\n\n    >>> class AnotherColorImplem(OtherColorImplem):\n    ...     def __ne__(self, other):\n    ...         return self.color != other.web\n\n    >>> new_alien_red = AnotherColorImplem("red")\n    >>> red != new_alien_red\n    False\n    >>> blue != new_alien_red\n    True\n\n\nPicking arbitrary color for a python object\n-------------------------------------------\n\nBasic Usage\n~~~~~~~~~~~\n\nSometimes, you just want to pick a color for an object in your application\noften to visually identify this object. Thus, the picked color should be the\nsame for same objects, and different for different object:\n\n.. code-block:: python\n\n    >>> foo = object()\n    >>> bar = object()\n\n    >>> Color(pick_for=foo)  # doctest: +ELLIPSIS\n    <Color ...>\n    >>> Color(pick_for=foo) == Color(pick_for=foo)\n    True\n    >>> Color(pick_for=foo) == Color(pick_for=bar)\n    False\n\nOf course, although there\'s a tiny probability that different strings yield the\nsame color, most of the time, different inputs will produce different colors.\n\nAdvanced Usage\n~~~~~~~~~~~~~~\n\nYou can customize your color picking algorithm by providing a ``picker``. A\n``picker`` is a callable that takes an object, and returns something that can\nbe instantiated as a color by ``Color``:\n\n.. code-block:: python\n\n    >>> my_picker = lambda obj: "red" if isinstance(obj, int) else "blue"\n    >>> Color(pick_for=3, picker=my_picker, pick_key=None)\n    <Color red>\n    >>> Color(pick_for="foo", picker=my_picker, pick_key=None)\n    <Color blue>\n\nYou might want to use a particular picker, but enforce how the picker will\nidentify two object as the same (or not). So there\'s a ``pick_key`` attribute\nthat is provided and defaults as equivalent of ``hash`` method and if hash is\nnot supported by your object, it\'ll default to the ``str`` of your object salted\nwith the class name.\n\nThus:\n\n.. code-block:: python\n\n    >>> class MyObj(str): pass\n    >>> my_obj_color = Color(pick_for=MyObj("foo"))\n    >>> my_str_color = Color(pick_for="foo")\n    >>> my_obj_color == my_str_color\n    False\n\nPlease make sure your object is hashable or "stringable" before using the\n``RGB_color_picker`` picking mechanism or provide another color picker. Nearly\nall python object are hashable by default so this shouldn\'t be an issue (e.g. \ninstances of ``object`` and subclasses are hashable).\n\nNeither ``hash`` nor ``str`` are perfect solution. So feel free to use\n``pick_key`` at ``Color`` instantiation time to set your way to identify\nobjects, for instance:\n\n.. code-block:: python\n\n    >>> a = object()\n    >>> b = object()\n    >>> Color(pick_for=a, pick_key=id) == Color(pick_for=b, pick_key=id)\n    False\n\nWhen choosing a pick key, you should closely consider if you want your color\nto be consistent between runs (this is NOT the case with the last example),\nor consistent with the content of your object if it is a mutable object.\n\nDefault value of ``pick_key`` and ``picker`` ensures that the same color will\nbe attributed to same object between different run on different computer for\nmost python object.\n\n\nColor factory\n-------------\n\nAs you might have noticed, there are few attributes that you might want to see\nattached to all of your colors as ``equality`` for equality comparison support,\nor ``picker``, ``pick_key`` to configure your object color picker.\n\nYou can create a customized ``Color`` factory thanks to the ``make_color_factory``:\n\n.. code-block:: python\n\n    >>> from colour import make_color_factory, HSL_equivalence, RGB_color_picker\n\n    >>> get_color = make_color_factory(\n    ...    equality=HSL_equivalence,\n    ...    picker=RGB_color_picker,\n    ...    pick_key=str,\n    ... )\n\nAll color created thanks to ``CustomColor`` class instead of the default one\nwould get the specified attributes by default:\n\n.. code-block:: python\n\n    >>> black_red = get_color("red", luminance=0)\n    >>> black_blue = get_color("blue", luminance=0)\n\nOf course, these are always instances of ``Color`` class:\n\n.. code-block:: python\n\n    >>> isinstance(black_red, Color)\n    True\n\nEquality was changed from normal defaults, so:\n\n.. code-block:: python\n\n    >>> black_red == black_blue\n    False\n\nThis because the default equivalence of ``Color`` was set to\n``HSL_equivalence``.\n\n\nContributing\n============\n\nAny suggestion or issue is welcome. Push request are very welcome,\nplease check out the guidelines.\n\n\nPush Request Guidelines\n-----------------------\n\nYou can send any code. I\'ll look at it and will integrate it myself in\nthe code base and leave you as the author. This process can take time and\nit\'ll take less time if you follow the following guidelines:\n\n- check your code with PEP8 or pylint. Try to stick to 80 columns wide.\n- separate your commits per smallest concern.\n- each commit should pass the tests (to allow easy bisect)\n- each functionality/bugfix commit should contain the code, tests,\n  and doc.\n- prior minor commit with typographic or code cosmetic changes are\n  very welcome. These should be tagged in their commit summary with\n  ``!minor``.\n- the commit message should follow gitchangelog rules (check the git\n  log to get examples)\n- if the commit fixes an issue or finished the implementation of a\n  feature, please mention it in the summary.\n\nIf you have some questions about guidelines which is not answered here,\nplease check the current ``git log``, you might find previous commit that\nwould show you how to deal with your issue.\n\n\nLicense\n=======\n\nCurrent license\n---------------\n\nCopyright (c) 2022 Marcin Serwin.\n\nLicensed under `BSD License <https://gitlab.com/marcin-serwin/colouralpha/-/blob/main/LICENSE/>`_.\n\nOriginal license\n----------------\n\nThe project was forked from `Colour <https://github.com/vaab/colour/>`__. \n\nCopyright (c) 2012-2017 Valentin Lab.\n\nLicensed under the `BSD License <https://gitlab.com/marcin-serwin/colouralpha/-/blob/main/LICENSE.old>`__.\n',
    'author': 'Marcin Serwin',
    'author_email': 'marcin.serwin0@protonmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
