# auto generated by update_py.py

from base64 import b64encode
import requests
import json
import time
import traceback
import hashlib
import queue
import threading
import tempfile

import pandas as pd

from tlclient.trader.config import Configurator


class WechatBot:

    _SEND_PARTIALLY_UNIT = 4000

    def __init__(self, config: Configurator, section=None):
        configs = config.get_wechat_settings()
        self.wechat_webhook = configs.webhook if section is None else getattr(configs, section, None)
        assert self.wechat_webhook is not None, 'could not find wechat_webhook config'
        self._wechat_err_msg_webhook = getattr(configs, 'webhook_err_msg', None)
        if self._wechat_err_msg_webhook is None:
            self._wechat_err_msg_webhook = self.wechat_webhook

        self._content_queue = queue.Queue()
        self._content_cv = threading.Condition()
        self._content_t = threading.Thread(target=self._content_worker, name='content_worker', daemon=True)
        self._content_t.start()

    def send_async(self, markdown_content, webhook=None):
        with self._content_cv:
            self._content_queue.put((markdown_content, webhook))
            self._content_cv.notify_all()

    def _content_worker(self):
        while True:
            with self._content_cv:
                self._content_cv.wait_for(lambda: not self._content_queue.empty())
                markdown_content, webhook = self._content_queue.get()
                self._content_queue.task_done()
            self.send(markdown_content, webhook, True)

    def send(self, markdown_content, webhook=None, nowait=False):
        if webhook is None and self.wechat_webhook is None:
            return

        print(markdown_content)
        i = 0
        while i < len(markdown_content):
            md_part = markdown_content[i:i+WechatBot._SEND_PARTIALLY_UNIT]
            if i > 0:
                # 非第一部分 加一行注解
                md_part = '**(接上文)**' + md_part
            if i + len(md_part) < len(markdown_content):
                # 如果没办法一次性过把剩余部分发完 倒着找一下换行符
                pos = md_part.rfind('\n')
                if pos != -1:
                    # 如果找到了 这次只发送到这里
                    md_part = md_part[:pos]
                    i += pos
                else:
                    # 否则就直接全发走
                    i += len(md_part)
                # 加一个还有后文的注解
                md_part += '\n**(接下文)**'
            else:
                # 表示这次可以一次性把剩余部分全发完
                i += len(md_part)
            # Send markdown content
            # print(md_part)
            message = {
                'msgtype': 'markdown',
                'markdown': {
                    'content': md_part
                }
            }
            try:
                res = requests.post(url=self.wechat_webhook if webhook is None else webhook, data=json.dumps(message), timeout=20)
                print(res)
                if not nowait:
                    time.sleep(1)
            except Exception as e:
                print(e)
                traceback.print_exc()

    def send_simple_msg(self, msg: str, title: str):
        markdown_content = f'### <font color=\"warning\">{title}</font>'
        markdown_content += f'\n><font color=\"info\">{msg}</font>'
        self.send_async(markdown_content)

    def send_dict(self, data, title: str):
        markdown_content = f'### <font color=\"warning\">{title}</font>'
        for k, v in data.items():
            markdown_content += f'\n><font color=\"info\">{k}: </font>**{v}**'
        self.send_async(markdown_content)

    def send_image(self, content: str, webhook=None):
        the_base64 = b64encode(content).decode()

        h = hashlib.new('md5', content)
        the_md5 = h.hexdigest()
        message = {
            'msgtype': 'image',
            'image': {
                'base64': the_base64,
                'md5': the_md5
            }
        }
        try:
            res = requests.post(url=self.wechat_webhook if webhook is None else webhook, data=json.dumps(message), timeout=20)
            print(res)
        except Exception as e:
            print(e)
            traceback.print_exc()

    def send_dataframe(self, df: pd.DataFrame, title: str):
        import dataframe_image as dfi

        try:
            with tempfile.TemporaryFile() as fp:
                dfi.export(df.rename_axis(index=title), fp, table_conversion='matplotlib')
                fp.seek(0)
                content = fp.read()
        except Exception as e:
            print(e)
            return
        self.send_image(content)
