# auto generated by update_py.py

import json
import threading
from typing import List, Set

import tlclient.trader.message_common as message_common
import tlclient.trader.message_market as message_market
import tlclient.trader.message_trade as message_trade

from tlclient.linker.constant import FistType
from tlclient.linker.fist import Fist
from tlclient.linker.pb_msg import comm_pb, frame_pb
from tlclient.linker.timer import Timer
from tlclient.linker.utility import bytify
from tlclient.trader.constant import (AssetType, BarType, ExchangeID, MarginMode,
                             MsgType, OffsetFlag, OrderType, TradingStyle,
                             OrderStatus, FundTransType, Side)
from tlclient.trader.data_manager.dm_client import DMClient
from tlclient.trader.pb_msg import message_pb


class Client(Fist):

    def __init__(self, name, env_name, addr, curve_server_key='', *, fist_type=None):
        Fist.__init__(self, name, FistType.CLIENT if fist_type is None else fist_type, env_name, curve_server_key=curve_server_key)
        self.set_master_addr(addr)
        self.create_fist()
        self.trade_router = None
        self.market_router = None
        self.oms_name = None
        self.dm_client = None
        self._msg_callbacks = {}

        self._order_id_lock = threading.Lock()
        self._order_id_inserted: Set[int] = set()

    def init_trade(self, router_name):
        self.reg_sub(router_name)
        self.reg_req(router_name)
        self.trade_router = router_name

    def init_market(self, router_name):
        self.reg_sub(router_name)
        self.reg_req(router_name)
        self.market_router = router_name

    def auto_init_trade(self):
        self.auto_reg_req(FistType.TRADE_ROUTER)
        self.auto_reg_sub(FistType.TRADE_ROUTER)
        self.logger.info('[auto_init] to auto reg trade router...')

    def auto_init_market(self):
        self.auto_reg_req(FistType.MARKET_ROUTER)
        self.auto_reg_sub(FistType.MARKET_ROUTER)
        self.logger.info('[auto_init] to auto reg market router...')

    def init_oms(self, oms_name):
        self.reg_req(oms_name)
        self.oms_name = oms_name

    def init_dm(self, api_key, secret_key):
        self.dm_client = DMClient(api_key, secret_key)

    #####################
    # trading functions #
    #####################

    def get_order_info(self, order_id):
        if self.oms_name is None:
            self.logger.error('[get_order_info] oms has not been inited!')
            return None
        req = message_pb.ReqOrderInfo()
        req.order_id = order_id
        req.sender = self.fist_name
        ret = self.req(self.oms_name, req, message_pb.MsgType.MSG_TYPE_REQ_ORDER_INFO, 0)
        err_id = ret.get_err_id()
        if err_id == 0:
            return ret.get_obj(message_pb.RtnOrder)
        else:
            self.logger.error('[get_order_info] error occurred (eid){}'.format(err_id))
            return None

    def get_position(self, tg_name, sub_account=''):
        if self.oms_name is None:
            self.logger.error('[get_position] oms has not been inited!')
            return None
        req = message_pb.ReqPosition()
        req.account_id = tg_name
        req.sub_account = sub_account
        req.sender = self.fist_name
        ret = self.req(self.oms_name, req, message_pb.MsgType.MSG_TYPE_REQ_POSITION, 0)
        err_id = ret.get_err_id()
        if err_id == 0:
            return ret.get_obj(message_pb.GatewayPosition)
        else:
            self.logger.error('[get_position] error occurred (eid){}'.format(err_id))
            return None

    def get_account(self, tg_name, sub_account=''):
        if self.oms_name is None:
            self.logger.error('[get_account] oms has not been inited!')
            return None
        req = message_pb.ReqAccount()
        req.account_id = tg_name
        req.sub_account = sub_account
        req.sender = self.fist_name
        ret = self.req(self.oms_name, req, message_pb.MsgType.MSG_TYPE_REQ_ACCOUNT, 0)
        err_id = ret.get_err_id()
        if err_id == 0:
            return ret.get_obj(message_pb.RspAccount)
        else:
            self.logger.error('[get_account] error occurred (eid){}'.format(err_id))
            return None

    # return order_id
    def insert_order(self, tg_name, exchange, ticker, price, volume, order_type, direction, sub_account='',
                     offset_flag=OffsetFlag.NOT_AVAILABLE, asset_type=AssetType.NOT_AVAILABLE,
                     margin_mode=MarginMode.NOT_AVAILABLE, lever_rate=0, contract_id: str = '',
                     repay_type: message_pb.CreditRepayType = message_pb.CreditRepayType.CREDIT_REPAY_TYPE_UNSPECIFIED,
                     currency: message_pb.Currency = message_pb.Currency.CURRENCY_UNSPECIFIED):
        req = message_pb.ReqOrderInsert()
        req.account_id = tg_name
        req.sub_account = sub_account
        req.sender = self.fist_name
        req.parent_id = -1

        req.exchange = exchange
        req.offset_flag = offset_flag
        req.order_type = order_type
        req.security_type = asset_type
        req.side = direction
        req.symbol = ticker

        req.entrust_vol = volume
        req.entrust_price = price

        req.margin_mode = margin_mode
        req.lever_rate = lever_rate

        req.contract_id = contract_id
        req.repay_type = repay_type
        req.currency = currency

        ret = self.req(self.trade_router, req, message_pb.MsgType.MSG_TYPE_REQ_ORDER_INSERT, 0)
        err_id = ret.get_err_id()
        if err_id == 0:
            order_id: int = ret.get_req_id()
            with self._order_id_lock:
                # record every order inserted by us
                self._order_id_inserted.add(order_id)
            return order_id
        else:
            self.logger.error(f'[order_insert] error occurred (eid){err_id} (tg_name){tg_name}')
            return -1

    def insert_algo_order(self, tg_name: str, exchange: ExchangeID, ticker: str, volume: float, order_type: OrderType, direction: Side, start_time: str, end_time: str, asset_type=AssetType.NOT_AVAILABLE,
                          currency: message_pb.Currency = message_pb.Currency.CURRENCY_UNSPECIFIED,
                          **algo_args):
        '''
        提交算法单请求

        Parameters
        ----------
        tg_name : str
            tg的名字
        exchange : ExchangeID
            交易所ID
        ticker : str
            标的ID
        volume : float
            数量
            
        order_type : OrderType
            委托类型
        direction : Side
            方向
        currency : Currency
            币种
        start_time : str
            开始时间
        end_time : str
            结束时间
        asset_type : AssetType
            资产类型
        algo_args : Dict[str, Any]
            算法单其他参数

        Returns
        -------
        -1 if failed, or req_id(int) if succeed
        '''

        req = message_pb.ReqOrderInsert()
        req.account_id = tg_name
        req.sender = self.fist_name
        req.parent_id = -1

        req.exchange = exchange
        req.order_type = order_type
        req.security_type = asset_type
        req.side = direction
        req.symbol = ticker
        req.entrust_vol = volume
        req.currency = currency

        order_data = {}
        # add necessary algo args
        order_data['start_time'] = start_time
        order_data['end_time'] = end_time
        # update extra algo args
        order_data.update(algo_args)
        req.algo_param = json.dumps(order_data)
        ret = self.req(self.trade_router, req, message_pb.MsgType.MSG_TYPE_REQ_ORDER_INSERT_J, 0)
        err_id = ret.get_err_id()
        if err_id == 0:
            return ret.get_req_id()
        else:
            self.logger.error('[insert_algo_order] error occurred (eid){}'.format(err_id))
            return -1

    def insert_algo_order_twap(self, tg_name, exchange, ticker, volume, direction, start_time, end_time, sec_interval, trading_style=TradingStyle.NEUTRAL, asset_type=AssetType.NOT_AVAILABLE):
        return self.insert_algo_order(tg_name, exchange, ticker, volume, direction, OrderType.TWAP, start_time, end_time, sec_interval, trading_style, asset_type=asset_type)  # no extra args

    def insert_basket_order(self, basket_name, tg_name, basket_order, child_order_type, **algo_args):
        basket_order.basket_name = basket_name
        basket_order.tg_name = tg_name
        basket_order.order_id = -1
        basket_order.client_name = self.fist_name
        # set child order type
        # if child order type is algo type, then the algo args are necessary
        basket_order.set_child_order_type(child_order_type)
        basket_order.set_algo_info(**algo_args)
        # send out order as json str
        basket_order_str = json.dumps(basket_order.to_dict())
        ret = self.req(self.trade_router, basket_order_str, MsgType.REQ_ORDER_INSERT_BASKET, 0)
        err_id = ret.get_err_id()
        if err_id == 0:
            return ret.get_req_id()
        else:
            self.logger.error('[insert_basket_order] error occurred (eid){}'.format(err_id))
            return -1

    def insert_basket_order_w_twap(self, basket_name, tg_name, basket_order, start_time, end_time, sec_interval, trading_style):
        start_nano = Timer.get_nano(start_time)
        end_nano = Timer.get_nano(end_time)
        return self.insert_basket_order(basket_name, tg_name, basket_order, OrderType.TWAP, trading_style=trading_style, start_nano=start_nano, end_nano=end_nano, sec_interval=sec_interval)

    def cancel_order(self, order_id, tg_name):
        req = message_pb.ReqOrderCancel()
        req.order_id = order_id
        req.account_id = tg_name

        ret = self.req(self.trade_router, req, message_pb.MsgType.MSG_TYPE_REQ_ORDER_CANCEL, 0)
        err_id = ret.get_err_id()
        if err_id == 0:
            return ret.get_req_id()
        else:
            self.logger.error('[cancel_order] error occurred (eid){}'.format(err_id))
            return -1

    def cancel_algo_order(self, strg_id: str, cancel_inserted: bool, tg_name: str):
        '''
        提交撤销算法单请求

        Parameters
        ----------
        strg_id : str
            算法单ID
        cancel_inserted : bool
            是否需要撤销该算法单已下的订单
        tg_name : str
            tg的名字

        Returns
        -------
        -1 if failed, or req_id(int) if succeed
        '''
        req = message_pb.ReqOrderCancel()
        req.order_id = strg_id
        req.account_id = tg_name
        req.cancel_inserted = cancel_inserted
        ret = self.req(self.trade_router, req, message_pb.MsgType.MSG_TYPE_REQ_ORDER_CANCEL_J, 0)
        err_id = ret.get_err_id()
        if err_id == 0:
            return ret.get_req_id()
        else:
            self.logger.error('[cancel_algo_order] error occurred (eid){}'.format(err_id))
            return -1

    def insert_credit_cash_repay(self, tg_name: str, amount: float, contract_id: str = '', repay_type: message_pb.CreditRepayType = message_pb.CreditRepayType.CREDIT_REPAY_TYPE_UNSPECIFIED, sub_account: str = ''):
        '''
        提交现金还款请求

        Parameters
        ----------
        tg_name : str
            tg的名字
        amount : float
            现金还款金额
        contract_id : str
            指定负债合约ID(可选)
        repay_type : CreditRepayType
            偿还类型(可选) 请参考 constant.py
        sub_account : str
            子账户名称

        Returns
        -------
        -1 if failed, or req_id(int) if succeed
        '''
        req = message_pb.ReqCreditCashRepay()
        req.account_id = tg_name
        req.sub_account = sub_account
        req.sender = self.fist_name
        req.amount = amount
        req.contract_id = contract_id
        req.repay_type = repay_type

        ret = self.req(self.trade_router, req, message_pb.MsgType.MSG_TYPE_REQ_CREDIT_CASH_REPAY, 0)
        err_id = ret.get_err_id()
        if err_id == 0:
            return ret.get_req_id()
        else:
            self.logger.error('[credit_cash_repay] error occurred (eid){}'.format(err_id))
            return -1

    def insert_credit_extend_debt_date(self, tg_name: str, contract_id: str, defer_days: int, sub_account: str = ''):
        '''
        提交合约负债展期请求

        Parameters
        ----------
        tg_name : str
            tg的名字
        contract_id : str
            指定负债合约ID
        defer_days : int
            展期天数
        sub_account : str
            子账户名称

        Returns
        -------
        -1 if failed, or req_id(int) if succeed
        '''
        req = message_pb.ReqCreditExtendDebtDate()
        req.account_id = tg_name
        req.sub_account = sub_account
        req.sender = self.fist_name
        req.contract_id = contract_id
        req.defer_days = defer_days

        ret = self.req(self.trade_router, req, message_pb.MsgType.MSG_TYPE_REQ_CREDIT_EXTEND_DEBT_DATE, 0)
        err_id = ret.get_err_id()
        if err_id == 0:
            return ret.get_req_id()
        else:
            self.logger.error('[credit_extend_debt_date] error occurred (eid){}'.format(err_id))
            return -1

    def cancel_all_orders_from_us(self, tg_name):
        with self._order_id_lock:
            for order_id in self._order_id_inserted:
                if self.cancel_order(order_id=order_id, tg_name=tg_name) == -1:
                    self.logger.warning(f'[cancel_all_orders] cancel order {order_id} failed')

    def req_position(self, tg_name, sub_account=''):
        req = message_pb.ReqPosition()
        req.account_id = tg_name
        req.sub_account = sub_account
        req.sender = self.fist_name
        ret = self.req(self.trade_router, req, message_pb.MsgType.MSG_TYPE_REQ_POSITION, 0)
        err_id = ret.get_err_id()
        if err_id == 0:
            return ret.get_req_id()
        else:
            self.logger.error('[req_position] error occurred (eid){} (tg){}'.format(err_id, tg_name))
            return -1

    def req_account(self, tg_name, sub_account=''):
        req = message_pb.ReqAccount()
        req.account_id = tg_name
        req.sub_account = sub_account
        req.sender = self.fist_name
        ret = self.req(self.trade_router, req, message_pb.MsgType.MSG_TYPE_REQ_ACCOUNT, 0)
        err_id = ret.get_err_id()
        if err_id == 0:
            return ret.get_req_id()
        else:
            self.logger.error('[req_account] error occurred (eid){} (tg){}'.format(err_id, tg_name))
            return -1

    def req_active_orders(self, tg_name, sub_account=''):
        req = message_pb.ReqActiveOrders()
        req.account_id = tg_name
        req.sub_account = sub_account
        req.sender = self.fist_name
        ret = self.req(self.trade_router, req, message_pb.MsgType.MSG_TYPE_REQ_ACTIVE_ORDERS, 0)
        err_id = ret.get_err_id()
        if err_id == 0:
            return ret.get_req_id()
        else:
            self.logger.error('[req_active_orders] error occurred (eid){} (tg){}'.format(err_id, tg_name))
            return -1

    def req_cancel_active_orders(self, tg_name, sub_account='', req_active_orders_id=-1):
        req = message_pb.ReqCancelActiveOrders()
        req.active_orders_id = req_active_orders_id
        req.account_id = tg_name
        req.sub_account = sub_account
        req.sender = self.fist_name
        ret = self.req(self.trade_router, req, message_pb.MsgType.MSG_TYPE_REQ_CANCEL_ACTIVE_ORDERS, 0)
        err_id = ret.get_err_id()
        if err_id == 0:
            return ret.get_req_id()
        else:
            self.logger.error('[req_cancel_active_orders] error occurred (eid){} (tg){}'.format(err_id, tg_name))
            return -1

    def req_history_trades(self, tg_name, sub_account=''):
        req = message_pb.ReqHistoryTrades()
        req.account_id = tg_name
        req.sub_account = sub_account
        req.sender = self.fist_name
        ret = self.req(self.trade_router, req, message_pb.MsgType.MSG_TYPE_REQ_HISTORY_TRADES, 0)
        err_id = ret.get_err_id()
        if err_id == 0:
            return ret.get_req_id()
        else:
            self.logger.error('[req_hist_trades] error occurred (eid){} (tg){}'.format(err_id, tg_name))
            return -1

    def req_trades_today(self, tg_name: str, sub_account: str = '') -> int:
        '''
        请求查询当日成交

        Parameters
        ----------
        tg_name : str
            tg的名字
        sub_account : str
            子账户名称

        Returns
        -------
        -1 if failed, or req_id(int) if succeed
        '''
        req = message_pb.ReqTradesToday()
        req.account_id = tg_name
        req.sub_account = sub_account
        req.sender = self.fist_name
        ret = self.req(self.trade_router, req, message_pb.MsgType.MSG_TYPE_REQ_TRADES_TODAY, 0)
        err_id = ret.get_err_id()
        if err_id == 0:
            return ret.get_req_id()
        else:
            self.logger.error(f'[req_trades_today] error occurred (eid){err_id} (tg){tg_name} (sub_account){sub_account}')
            return -1

    def req_fund_trans(self, tg_name: str, fund_account: str, password: str, amount: float, fund_trans_type: FundTransType, sub_account: str = ''):
        '''
        资金划拨请求

        Parameters
        ----------
        tg_name : str
            tg的名字
        fund_account : str
            资金账户代码
        password : str
            资金账户密码
        amount : float
            金额
        fund_trans_type : FundTransType
            资金流转类型，请参考constant.py
        sub_account : str
            子账户名称

        Returns
        -------
        -1 if failed, or req_id(int) if succeed
        '''
        req = message_pb.ReqFundTransfer()
        req.account_id = tg_name
        req.sub_account = sub_account
        req.sender = self.fist_name
        req.fund_account = fund_account
        req.password = password
        req.amount = amount
        req.fund_trans_type = fund_trans_type
        ret = self.req(self.trade_router, req, message_pb.MsgType.MSG_TYPE_REQ_FUND_TRANS, 0)
        err_id = ret.get_err_id()
        if err_id == 0:
            return ret.get_req_id()
        else:
            self.logger.error('[req_fund_trans] error occurred (eid){} (tg){}'.format(err_id, tg_name))
            return -1

    def req_fund_trans_info(self, tg_name: str, serial_id: int = 0, sub_account=''):
        '''
        请求查询资金划拨

        Parameters
        ----------
        tg_name : str
            tg的名字
        serial_id : int
            资金内转编号 可以为0，表示查询所有资金划拨记录；如果不为0，则请求查询特定的资金划拨记录
        sub_account : str
            子账户名称

        Returns
        -------
        -1 if failed, or req_id(int) if succeed
        '''
        req = message_pb.ReqFundTransferInfo()
        req.account_id = tg_name
        req.sub_account = sub_account
        req.sender = self.fist_name
        req.serial_id = serial_id
        ret = self.req(self.trade_router, req, message_pb.MsgType.MSG_TYPE_REQ_FUND_TRANS_INFO, 0)
        err_id = ret.get_err_id()
        if err_id == 0:
            return ret.get_req_id()
        else:
            self.logger.error('[req_fund_trans_info] error occurred (eid){} (tg){}'.format(err_id, tg_name))
            return -1

    def req_fund_trans_from_bank_to_future(self, tg_name: str, bank_id: str, broker_id: str, bank_account: str, account: str, password: str, trade_amount: float, sub_account: str = ''):
        '''
        请求银行资金转期货

        Parameters
        ----------
        tg_name : str
            tg的名字
        bank_id : str
            银行代码
        broker_id : str
            期商代码
        bank_account : str
            银行账号
        account : str
            期货账号
        password : str
            期货密码
        trade_amount : float
            转账金额
        sub_account : str
            子账户名称

        Returns
        -------
        -1 if failed, or req_id(int) if succeed
        '''
        req = message_pb.ReqFundTransFromBankToFuture()
        req.account_id = tg_name
        req.sub_account = sub_account
        req.sender = self.fist_name
        req.bank_id = bank_id
        req.broker_id = broker_id
        req.bank_account = bank_account
        req.account = account
        req.password = password
        req.trade_amount = trade_amount
        ret = self.req(self.trade_router, req, message_pb.MsgType.MSG_TYPE_REQ_FUND_TRANS_FROM_BANK_TO_FUTURE, 0)
        err_id = ret.get_err_id()
        if err_id == 0:
            return ret.get_req_id()
        else:
            self.logger.error('[req_fund_trans_from_bank_to_future] error occurred (eid){} (tg){}'.format(err_id, tg_name))
            return -1

    def req_ipo_info(self, tg_name, sub_account=''):
        '''
        请求查询今日新股申购信息列表

        Parameters
        ----------
        tg_name : str
            tg的名字
        sub_account : str
            子账户名称

        Returns
        -------
        -1 if failed, or req_id(int) if succeed
        '''
        req = message_pb.ReqIPOInfo()
        req.account_id = tg_name
        req.sub_account = sub_account
        req.sender = self.fist_name
        ret = self.req(self.trade_router, req, message_pb.MsgType.MSG_TYPE_REQ_IPO_INFO, 0)
        err_id = ret.get_err_id()
        if err_id == 0:
            return ret.get_req_id()
        else:
            self.logger.error('[req_ipo_info] error occurred (eid){} (tg){}'.format(err_id, tg_name))
            return -1

    def req_ipo_quota_info(self, tg_name, sub_account=''):
        '''
        请求查询用户新股申购额度信息

        Parameters
        ----------
        tg_name : str
            tg的名字
        sub_account : str
            子账户名称

        Returns
        -------
        -1 if failed, or req_id(int) if succeed
        '''
        req = message_pb.ReqIPOQuotaInfo()
        req.account_id = tg_name
        req.sub_account = sub_account
        req.sender = self.fist_name
        ret = self.req(self.trade_router, req, message_pb.MsgType.MSG_TYPE_REQ_IPO_QUOTA_INFO, 0)
        err_id = ret.get_err_id()
        if err_id == 0:
            return ret.get_req_id()
        else:
            self.logger.error('[req_ipo_quota_info] error occurred (eid){} (tg){}'.format(err_id, tg_name))
            return -1

    def req_credit_fund_info(self, tg_name, sub_account=''):
        '''
        请求查询信用账户特有信息，除资金账户以外的信息

        Parameters
        ----------
        tg_name : str
            tg的名字
        sub_account : str
            子账户名称

        Returns
        -------
        -1 if failed, or req_id(int) if succeed
        '''
        req = message_pb.ReqCreditFundInfo()
        req.account_id = tg_name
        req.sub_account = sub_account
        req.sender = self.fist_name
        ret = self.req(self.trade_router, req, message_pb.MsgType.MSG_TYPE_REQ_CREDIT_FUND_INFO, 0)
        err_id = ret.get_err_id()
        if err_id == 0:
            return ret.get_req_id()
        else:
            self.logger.error('[req_credit_fund_info] error occurred (eid){} (tg){}'.format(err_id, tg_name))
            return -1

    def req_credit_active_contracts(self, tg_name, sub_account=''):
        '''
        请求查询信用账户所有未了结负债合约信息

        Parameters
        ----------
        tg_name : str
            tg的名字
        sub_account : str
            子账户名称

        Returns
        -------
        -1 if failed, or req_id(int) if succeed
        '''
        req = message_pb.ReqCreditActiveContracts()
        req.account_id = tg_name
        req.sub_account = sub_account
        req.sender = self.fist_name
        ret = self.req(self.trade_router, req, message_pb.MsgType.MSG_TYPE_REQ_CREDIT_ACTIVE_CONTRACTS, 0)
        err_id = ret.get_err_id()
        if err_id == 0:
            return ret.get_req_id()
        else:
            self.logger.error('[req_credit_active_contracts] error occurred (eid){} (tg){}'.format(err_id, tg_name))
            return -1

    def req_credit_ticker_assign_info(self, tg_name, ticker='', exchange=ExchangeID.NOT_AVAILABLE, sub_account=''):
        '''
        请求查询信用账户可融券头寸信息信息

        Parameters
        ----------
        tg_name : str
            tg的名字
        ticker : str
            标的代码(可选)
        exchange : ExchangeID
            交易所代码(可选) 请参考constant.py
        sub_account : str
            子账户名称

        Returns
        -------
        -1 if failed, or req_id(int) if succeed
        '''
        req = message_pb.ReqCreditTickerAssignInfo()
        req.account_id = tg_name
        req.sub_account = sub_account
        req.sender = self.fist_name
        req.symbol = ticker
        req.exchange = exchange
        ret = self.req(self.trade_router, req, message_pb.MsgType.MSG_TYPE_REQ_CREDIT_TICKER_ASSIGN_INFO, 0)
        err_id = ret.get_err_id()
        if err_id == 0:
            return ret.get_req_id()
        else:
            self.logger.error('[req_credit_ticker_assign_info] error occurred (eid){} (tg){}'.format(err_id, tg_name))
            return -1

    def req_credit_excess_stock(self, tg_name, ticker='', exchange=ExchangeID.NOT_AVAILABLE, sub_account=''):
        '''
        请求查询信用账户余券信息信息

        Parameters
        ----------
        tg_name : str
            tg的名字
        ticker : str
            标的代码(可选)
        exchange : ExchangeID
            交易所代码(可选) 请参考constant.py
        sub_account : str
            子账户名称

        Returns
        -------
        -1 if failed, or req_id(int) if succeed
        '''
        req = message_pb.ReqCreditExcessStock()
        req.account_id = tg_name
        req.sub_account = sub_account
        req.sender = self.fist_name
        req.symbol = ticker
        req.exchange = exchange
        ret = self.req(self.trade_router, req, message_pb.MsgType.MSG_TYPE_REQ_CREDIT_EXCESS_STOCK, 0)
        err_id = ret.get_err_id()
        if err_id == 0:
            return ret.get_req_id()
        else:
            self.logger.error('[req_credit_excess_stock] error occurred (eid){} (tg){}'.format(err_id, tg_name))
            return -1

    # subscription
    def subscribe_snap(self, exchange, ticker):
        return self.subscribe(exchange, ticker, MsgType.MKT_SNAP)

    def subscribe_bar(self, exchange, ticker, bar_type=BarType.MIN_15):
        return self.subscribe(exchange, ticker, MsgType.MKT_BAR, bar_type=bar_type)

    def subscribe_index(self, exchange, ticker):
        return self.subscribe(exchange, ticker, MsgType.MKT_INDEX)

    def subscribe_order(self, exchange, ticker):
        return self.subscribe(exchange, ticker, MsgType.MKT_ORDER)

    def subscribe_trade(self, exchange, ticker):
        return self.subscribe(exchange, ticker, MsgType.MKT_TRADE)

    def subscribe(self, exchange, ticker, msg_type, bar_type=BarType.NOT_AVAILABLE):
        return self.subscribe_batch([exchange], [ticker], msg_type, bar_type)

    def subscribe_batch(self, exchanges, tickers, msg_type=MsgType.MKT_SNAP, bar_type=BarType.NOT_AVAILABLE):
        if not isinstance(tickers, list):
            tickers = [tickers]
        if not isinstance(exchanges, list):
            exchanges = [exchanges for i in range(0, len(tickers))]
        if len(exchanges) != len(tickers):
            self.logger.error('[subscribe] exchanges and tickers, size mismatch')
            return False
        ss = json.dumps([{'exchange': ExchangeID.read(exchanges[i]), 'ticker': tickers[i], 'msg_type': msg_type, 'bar_type': bar_type} for i in range(0, len(tickers))])
        rsp = self.req(self.market_router, ss, MsgType.MKT_SUBSCRIBE, 0)
        if rsp.get_err_id() != 0:
            self.logger.error('[subscribe] failed (err_id){} (msg){}'.format(rsp.get_err_id(), ss))
            return False
        else:
            self.logger.info('[subscribe] success')
            return True

    def _req_param_info(self, server_name: str, file_list: List[str]):
        msg = message_pb.ReqParamInfo()
        msg.file_names.extend(file_list)
        ret = self.req(server_name, msg, message_pb.MsgType.MSG_TYPE_PARAM_REQ_INFO, 0)
        err_id = ret.get_err_id()
        if err_id != 0:
            self.logger.error(f'[req_param_info] error occurred (eid){err_id} (server_name){server_name}')
            return
        return ret.get_obj(message_pb.RspParamInfo)

    # to override
    # trade
    def on_rsp_order_insert(self, obj, frame_nano):
        pass

    def on_rsp_order_cancel(self, obj, frame_nano):
        pass

    def on_rsp_credit_cash_repay(self, obj, frame_nano):
        '''
        处理信用交易现金还款

        obj: RspCreditCashRepay
        '''
        pass

    def on_rsp_credit_extend_debt_date(self, obj, frame_nano):
        '''
        处理信用交易负债合约展期

        obj: RspCreditExtendDebtDate
        '''
        pass

    def on_rtn_order(self, obj, frame_nano):
        pass

    def on_rtn_trade(self, obj, frame_nano):
        pass

    def on_rtn_credit_cash_repay(self, obj, frame_nano):
        '''
        处理信用交易现金还款

        obj: RtnCreditCashRepay
        '''
        pass

    def on_rtn_credit_extend_debt_date(self, obj, frame_nano):
        '''
        处理信用交易负债合约展期

        obj: RtnCreditExtendDebtDate
        '''
        pass

    def on_rsp_position(self, obj, frame_nano):
        pass

    def on_rsp_account(self, obj, frame_nano):
        pass

    def on_rsp_active_orders(self, obj, frame_nano):
        pass

    def on_rsp_cancel_active_orders(self, obj, frame_nano):
        pass

    def on_rsp_history_trades(self, obj, frame_nano):
        pass

    def on_rsp_trades_today(self, obj, frame_nano):
        '''
        处理请求查询当日成交的回报

        obj: GatewayTrades
        '''
        pass

    def on_rsp_fund_trans_info(self, obj, frame_nano):
        '''
        处理请求查询资金划拨的回报

        obj: RspFundTransferInfo
        '''
        pass

    def on_rsp_fund_trans(self, obj, frame_nano):
        '''
        处理资金划拨请求的回报

        obj: RspFundTransferInfo
        '''
        pass

    def on_rsp_fund_trans_from_bank_to_future(self, obj, frame_nano):
        '''
        处理银行资金转期货请求的回报

        obj: RspFundTransFromBankToFuture
        '''
        pass

    def on_rsp_ipo_info(self, obj, frame_nano):
        '''
        处理请求查询今日新股申购信息列表的回报

        obj: RspIPOInfo
        '''
        pass

    def on_rsp_ipo_quota_info(self, obj, frame_nano):
        '''
        处理请求查询用户新股申购额度信息的回报

        obj: RspIPOQuotaInfo
        '''
        pass

    def on_rsp_credit_fund_info(self, obj, frame_nano):
        '''
        处理请求查询信用账户特有信息，除资金账户以外的信息

        obj: RspCreditFundInfo
        '''
        pass

    def on_rsp_credit_active_contracts(self, obj, frame_nano):
        '''
        处理请求查询信用账户负债合约信息

        obj: RspCreditActiveContracts
        '''
        pass

    def on_rsp_credit_ticker_assign_info(self, obj, frame_nano):
        '''
        处理请求查询信用账户可融券头寸信息

        obj: RspCreditTickerAssignInfo
        '''
        pass

    def on_rsp_credit_excess_stock(self, obj, frame_nano):
        '''
        处理请求查询用户新股申购额度信息的回报

        obj: RspCreditExcessStock
        '''
        pass

    # to override
    # market
    def on_mkt_snap(self, obj, msg_type, frame_nano):
        pass

    def on_mkt_bar(self, obj, msg_type, frame_nano):
        pass

    def on_mkt_vol(self, obj, msg_type, frame_nano):
        pass

    def on_mkt_index(self, obj, msg_type, frame_nano):
        pass

    def on_mkt_order(self, obj, msg_type, frame_nano):
        pass

    def on_mkt_trade(self, obj, msg_type, frame_nano):
        pass

    # to override
    # system status
    def on_gateway_connection_change(self, obj, frame_nano):
        pass

    def on_gateway_heart_beat(self, obj, frame_nano):
        pass

    def on_param_info(self, obj, frame_nano):
        pass

    ######################
    # internal functions #
    ######################
    def _on_mkt_snap(self, obj, msg_type, frame_nano):
        self.on_mkt_snap(obj, msg_type, frame_nano)

    def _on_rsp_order_insert(self, obj, frame_nano):
        if obj.err_id != frame_pb.ERR_TYPE_NO_ERR:
            with self._order_id_lock:
                # if the order was inserted failed, we should not record it anymore
                self._order_id_inserted.discard(obj.order_id)
        self.on_rsp_order_insert(obj, frame_nano)

    def _on_rsp_order_cancel(self, obj, frame_nano):
        self.on_rsp_order_cancel(obj, frame_nano)

    def _on_rtn_order(self, obj, frame_nano):
        if obj.order_status >= OrderStatus.REJECTED and obj.order_status <= OrderStatus.ALL_TRADED:
            with self._order_id_lock:
                # if the order was rejected or cancelled or all traded, we should not record it anymore
                self._order_id_inserted.discard(obj.order_id)
        self.on_rtn_order(obj, frame_nano)

    def _on_rtn_trade(self, obj, frame_nano):
        self.on_rtn_trade(obj, frame_nano)

    def register_msg_callback(self, msg_type, callback):
        # callback should have the args list exactly matched (data: dict, msg_type: int, frame_nano: int)
        if msg_type in self._msg_callbacks:
            self.logger.warn('[reg_msg_callback] replacing callback (msg_type){}/{}'.format(msg_type, MsgType.read(msg_type)))
        self._msg_callbacks[msg_type] = callback

    def on_pub_frame(self, f):
        msg_type = f.get_msg_type()
        frame_nano = f.get_nano()

        msg_callback = self._msg_callbacks.get(msg_type)
        if msg_callback is not None:
            msg_callback(f, msg_type, frame_nano)

        elif MsgType.is_market_data_type(msg_type):
            if msg_type == MsgType.MKT_SNAP:
                snap_obj = f.get_obj(message_market.MktSnap)
                self._on_mkt_snap(snap_obj, msg_type, frame_nano)

            elif msg_type == MsgType.MKT_SNAP_PLUS:
                snap_obj = f.get_obj(message_market.MktSnapPlus)
                self._on_mkt_snap(snap_obj, msg_type, frame_nano)

            elif msg_type == MsgType.MKT_SNAP_FUT:
                snap_obj = f.get_obj(message_market.MktSnapFut)
                self._on_mkt_snap(snap_obj, msg_type, frame_nano)

            elif msg_type == MsgType.MKT_SNAP_OPT:
                snap_obj = f.get_obj(message_market.MktSnapOpt)
                self._on_mkt_snap(snap_obj, msg_type, frame_nano)

            elif msg_type == MsgType.MKT_SNAP_AGG:
                agg_d = json.loads(f.get_string())
                snap_obj = message_market.MktSnapAgg(agg_d)
                self._on_mkt_snap(snap_obj, msg_type, frame_nano)

            elif msg_type == MsgType.MKT_INDEX:
                idx_obj = f.get_obj(message_market.MktIndex)
                self.on_mkt_index(idx_obj, msg_type, frame_nano)

            elif msg_type == MsgType.MKT_ORDER:
                order_obj = f.get_obj(message_market.MktOrder)
                self.on_mkt_order(order_obj, msg_type, frame_nano)

            elif msg_type == MsgType.MKT_TRADE:
                trade_obj = f.get_obj(message_market.MktTrade)
                self.on_mkt_trade(trade_obj, msg_type, frame_nano)

            elif msg_type == MsgType.MKT_BAR:
                bar_obj = f.get_obj(message_market.MktBar)
                self.on_mkt_bar(bar_obj, msg_type, frame_nano)

            elif msg_type == MsgType.MKT_BAR_GEN:
                bar_obj = f.get_obj(message_market.MktBarGen)
                self.on_mkt_bar(bar_obj, msg_type, frame_nano)

            elif msg_type == MsgType.MKT_VOL:
                vol_obj = f.get_obj(message_market.MktVol)
                self.on_mkt_vol(vol_obj, msg_type, frame_nano)

        elif MsgType.is_trading_data_type(msg_type):
            if msg_type == message_pb.MsgType.MSG_TYPE_RSP_ORDER_INSERT:
                rsp_ord_obj = f.get_obj(message_pb.RspOrderInsert)
                self._on_rsp_order_insert(rsp_ord_obj, frame_nano)

            elif msg_type == message_pb.MsgType.MSG_TYPE_RSP_ORDER_CANCEL:
                rsp_cancel_obj = f.get_obj(message_pb.RspOrderCancel)
                self._on_rsp_order_cancel(rsp_cancel_obj, frame_nano)

            elif msg_type == message_pb.MsgType.MSG_TYPE_RSP_CREDIT_CASH_REPAY:
                rsp_credit_cash_repay_obj = f.get_obj(message_pb.RspCreditCashRepay)
                self.on_rsp_credit_cash_repay(rsp_credit_cash_repay_obj, frame_nano)

            elif msg_type == message_pb.MsgType.MSG_TYPE_RSP_CREDIT_EXTEND_DEBT_DATE:
                rsp_credit_extend_debt_date_obj = f.get_obj(message_pb.RspCreditExtendDebtDate)
                self.on_rsp_credit_extend_debt_date(rsp_credit_extend_debt_date_obj, frame_nano)

            elif msg_type == message_pb.MsgType.MSG_TYPE_RTN_ORDER:
                ord_object = f.get_obj(message_pb.RtnOrder)
                self._on_rtn_order(ord_object, frame_nano)

            elif msg_type == message_pb.MsgType.MSG_TYPE_RTN_TRADE:
                trd_obj = f.get_obj(message_pb.RtnTrade)
                self._on_rtn_trade(trd_obj, frame_nano)

            elif msg_type == message_pb.MsgType.MSG_TYPE_RTN_CREDIT_CASH_REPAY:
                credit_cash_repay_obj = f.get_obj(message_pb.RtnCreditCashRepay)
                self.on_rtn_credit_cash_repay(credit_cash_repay_obj, frame_nano)

            elif msg_type == message_pb.MsgType.MSG_TYPE_RTN_CREDIT_EXTEND_DEBT_DATE:
                credit_extend_debt_date_obj = f.get_obj(message_pb.RtnCreditExtendDebtDate)
                self.on_rtn_credit_extend_debt_date(credit_extend_debt_date_obj, frame_nano)

            elif msg_type == message_pb.MsgType.MSG_TYPE_RSP_POSITION:
                pos_obj = f.get_obj(message_pb.GatewayPosition)
                self.on_rsp_position(pos_obj, frame_nano)

            elif msg_type == message_pb.MsgType.MSG_TYPE_RSP_ACCOUNT:
                acc_obj = f.get_obj(message_pb.RspAccount)
                self.on_rsp_account(acc_obj, frame_nano)

            elif msg_type == message_pb.MsgType.MSG_TYPE_RSP_ACTIVE_ORDERS:
                ods_obj = f.get_obj(message_pb.GatewayOrders)
                self.on_rsp_active_orders(ods_obj, frame_nano)

            elif msg_type == message_pb.MsgType.MSG_TYPE_RSP_CANCEL_ACTIVE_ORDERS:
                rsp_obj = f.get_obj(message_pb.RspCancelActiveOrders)
                self.on_rsp_cancel_active_orders(rsp_obj, frame_nano)

            elif msg_type == message_pb.MsgType.MSG_TYPE_RSP_HISTORY_TRADES:
                rsp_obj = f.get_obj(message_pb.RspHistoryTrades)
                self.on_rsp_history_trades(rsp_obj, frame_nano)

            elif msg_type == message_pb.MsgType.MSG_TYPE_RSP_TRADES_TODAY:
                rsp_obj = f.get_obj(message_pb.GatewayTrades)
                self.on_rsp_trades_today(rsp_obj, frame_nano)

            elif msg_type == message_pb.MsgType.MSG_TYPE_RSP_FUND_TRANS_INFO:
                rsp_obj = f.get_obj(message_pb.RspFundTransferInfo)
                self.on_rsp_fund_trans_info(rsp_obj, frame_nano)

            elif msg_type == message_pb.MsgType.MSG_TYPE_RSP_FUND_TRANS:
                rsp_obj = f.get_obj(message_pb.RspFundTransferInfo)
                self.on_rsp_fund_trans(rsp_obj, frame_nano)

            elif msg_type == message_pb.MsgType.MSG_TYPE_RSP_FUND_TRANS_FROM_BANK_TO_FUTURE:
                rsp_obj = f.get_obj(message_pb.RspFundTransFromBankToFuture)
                self.on_rsp_fund_trans_from_bank_to_future(rsp_obj, frame_nano)

            elif msg_type == message_pb.MsgType.MSG_TYPE_RSP_IPO_INFO:
                rsp_obj = f.get_obj(message_pb.RspIPOInfo)
                self.on_rsp_ipo_info(rsp_obj, frame_nano)

            elif msg_type == message_pb.MsgType.MSG_TYPE_RSP_IPO_QUOTA_INFO:
                rsp_obj = f.get_obj(message_pb.RspIPOQuotaInfo)
                self.on_rsp_ipo_quota_info(rsp_obj, frame_nano)

            elif msg_type == message_pb.MsgType.MSG_TYPE_RSP_CREDIT_FUND_INFO:
                rsp_credit_fund_info_obj = f.get_obj(message_pb.RspCreditFundInfo)
                self.on_rsp_credit_fund_info(rsp_credit_fund_info_obj, frame_nano)

            elif msg_type == message_pb.MsgType.MSG_TYPE_RSP_CREDIT_ACTIVE_CONTRACTS:
                rsp_credit_active_contracts_obj = f.get_obj(message_pb.RspCreditActiveContracts)
                self.on_rsp_credit_active_contracts(rsp_credit_active_contracts_obj, frame_nano)

            elif msg_type == message_pb.MsgType.MSG_TYPE_RSP_CREDIT_TICKER_ASSIGN_INFO:
                rsp_credit_ticker_assign_info_obj = f.get_obj(message_pb.RspCreditTickerAssignInfo)
                self.on_rsp_credit_ticker_assign_info(rsp_credit_ticker_assign_info_obj, frame_nano)

            elif msg_type == message_pb.MsgType.MSG_TYPE_RSP_CREDIT_EXCESS_STOCK:
                rsp_credit_excess_stock_obj = f.get_obj(message_pb.RspCreditExcessStock)
                self.on_rsp_credit_excess_stock(rsp_credit_excess_stock_obj, frame_nano)

        elif MsgType.is_system_status_data_type(msg_type):
            if msg_type == MsgType.GTW_CONNECTION:
                gtw_con_obj = f.get_obj(message_common.GatewayConnectionStatus)
                self.on_gateway_connection_change(gtw_con_obj, frame_nano)

        elif msg_type == frame_pb.MsgType.MSG_TYPE_FIST_HEART_BEAT:
            gtw_hb = f.get_obj(comm_pb.MsgHeartBeat)
            self.on_gateway_heart_beat(gtw_hb, frame_nano)

        elif msg_type == message_pb.MsgType.MSG_TYPE_PARAM_RSP_INFO:
            rsp = f.get_obj(message_pb.RspParamInfo)
            self.on_param_info(rsp, frame_nano)
