"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RouteSpec = void 0;
const shared_interfaces_1 = require("./shared-interfaces");
/**
 * (experimental) Used to generate specs with different protocols for a RouteSpec.
 *
 * @experimental
 */
class RouteSpec {
    /**
     * (experimental) Creates an HTTP Based RouteSpec.
     *
     * @experimental
     */
    static http(options) {
        return new HttpRouteSpec(options, shared_interfaces_1.Protocol.HTTP);
    }
    /**
     * (experimental) Creates an HTTP2 Based RouteSpec.
     *
     * @experimental
     */
    static http2(options) {
        return new HttpRouteSpec(options, shared_interfaces_1.Protocol.HTTP2);
    }
    /**
     * (experimental) Creates a TCP Based RouteSpec.
     *
     * @experimental
     */
    static tcp(options) {
        return new TcpRouteSpec(options);
    }
    /**
     * (experimental) Creates a GRPC Based RouteSpec.
     *
     * @experimental
     */
    static grpc(options) {
        return new GrpcRouteSpec(options);
    }
}
exports.RouteSpec = RouteSpec;
class HttpRouteSpec extends RouteSpec {
    constructor(props, protocol) {
        super();
        this.protocol = protocol;
        this.match = props.match;
        this.weightedTargets = props.weightedTargets;
        this.timeout = props.timeout;
    }
    bind(_scope) {
        const prefixPath = this.match ? this.match.prefixPath : '/';
        if (prefixPath[0] != '/') {
            throw new Error(`Prefix Path must start with \'/\', got: ${prefixPath}`);
        }
        const httpConfig = {
            action: {
                weightedTargets: renderWeightedTargets(this.weightedTargets),
            },
            match: {
                prefix: prefixPath,
            },
            timeout: renderTimeout(this.timeout),
        };
        return {
            httpRouteSpec: this.protocol === shared_interfaces_1.Protocol.HTTP ? httpConfig : undefined,
            http2RouteSpec: this.protocol === shared_interfaces_1.Protocol.HTTP2 ? httpConfig : undefined,
        };
    }
}
class TcpRouteSpec extends RouteSpec {
    constructor(props) {
        super();
        this.weightedTargets = props.weightedTargets;
        this.timeout = props.timeout;
    }
    bind(_scope) {
        return {
            tcpRouteSpec: {
                action: {
                    weightedTargets: renderWeightedTargets(this.weightedTargets),
                },
                timeout: renderTimeout(this.timeout),
            },
        };
    }
}
class GrpcRouteSpec extends RouteSpec {
    constructor(props) {
        super();
        this.weightedTargets = props.weightedTargets;
        this.match = props.match;
        this.timeout = props.timeout;
    }
    bind(_scope) {
        return {
            grpcRouteSpec: {
                action: {
                    weightedTargets: renderWeightedTargets(this.weightedTargets),
                },
                match: {
                    serviceName: this.match.serviceName,
                },
                timeout: renderTimeout(this.timeout),
            },
        };
    }
}
/**
* Utility method to add weighted route targets to an existing route
*/
function renderWeightedTargets(weightedTargets) {
    const renderedTargets = [];
    for (const t of weightedTargets) {
        renderedTargets.push({
            virtualNode: t.virtualNode.virtualNodeName,
            weight: t.weight || 1,
        });
    }
    return renderedTargets;
}
/**
 * Utility method to construct a route timeout object
 */
function renderTimeout(timeout) {
    return timeout
        ? {
            idle: (timeout === null || timeout === void 0 ? void 0 : timeout.idle) !== undefined
                ? {
                    unit: 'ms',
                    value: timeout === null || timeout === void 0 ? void 0 : timeout.idle.toMilliseconds(),
                }
                : undefined,
            perRequest: (timeout === null || timeout === void 0 ? void 0 : timeout.perRequest) !== undefined
                ? {
                    unit: 'ms',
                    value: timeout === null || timeout === void 0 ? void 0 : timeout.perRequest.toMilliseconds(),
                }
                : undefined,
        }
        : undefined;
}
//# sourceMappingURL=data:application/json;base64,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