"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const cf = require("@aws-cdk/aws-cloudfront");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
test('Cloudfront Distribution Included', () => {
    const stack = new core_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        websiteFolder: 'website',
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            CustomErrorResponses: [
                {
                    ErrorCode: 403,
                    ResponseCode: 200,
                    ResponsePagePath: '/index.html',
                },
                {
                    ErrorCode: 404,
                    ResponseCode: 200,
                    ResponsePagePath: '/index.html',
                },
            ],
            DefaultCacheBehavior: {
                ViewerProtocolPolicy: 'redirect-to-https',
            },
            DefaultRootObject: 'index.html',
            HttpVersion: 'http2',
            IPV6Enabled: true,
            PriceClass: 'PriceClass_100',
            ViewerCertificate: {
                CloudFrontDefaultCertificate: true,
            },
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        's3:GetObject*',
                        's3:GetBucket*',
                        's3:List*',
                    ],
                    Effect: 'Allow',
                    Principal: {
                        CanonicalUser: {
                            'Fn::GetAtt': [
                                'spaDeployOriginAccessIdentityEDA4C19C',
                                'S3CanonicalUserId',
                            ],
                        },
                    },
                    Resource: [
                        {
                            'Fn::GetAtt': [
                                'spaDeployWebsiteBucket1E4C4442',
                                'Arn',
                            ],
                        },
                        {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'spaDeployWebsiteBucket1E4C4442',
                                            'Arn',
                                        ],
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    ],
                }
            ],
        },
    }));
});
test('Cloudfront Distribution Included - with non-default error-doc cfg set', () => {
    const stack = new core_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        errorDoc: 'error.html',
        websiteFolder: 'website',
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
            ErrorDocument: 'error.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            CustomErrorResponses: [
                {
                    ErrorCode: 403,
                    ResponseCode: 200,
                    ResponsePagePath: '/error.html',
                },
                {
                    ErrorCode: 404,
                    ResponseCode: 200,
                    ResponsePagePath: '/error.html',
                },
            ],
            DefaultCacheBehavior: {
                ViewerProtocolPolicy: 'redirect-to-https',
            },
            DefaultRootObject: 'index.html',
            HttpVersion: 'http2',
            IPV6Enabled: true,
            PriceClass: 'PriceClass_100',
            ViewerCertificate: {
                CloudFrontDefaultCertificate: true,
            },
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        's3:GetObject*',
                        's3:GetBucket*',
                        's3:List*',
                    ],
                    Effect: 'Allow',
                    Principal: {
                        CanonicalUser: {
                            'Fn::GetAtt': [
                                'spaDeployOriginAccessIdentityEDA4C19C',
                                'S3CanonicalUserId',
                            ],
                        },
                    },
                    Resource: [
                        {
                            'Fn::GetAtt': [
                                'spaDeployWebsiteBucket1E4C4442',
                                'Arn',
                            ],
                        },
                        {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'spaDeployWebsiteBucket1E4C4442',
                                            'Arn',
                                        ],
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    ],
                }
            ],
        },
    }));
});
test('Cloudfront With Custom Cert and Aliases', () => {
    const stack = new core_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        certificateARN: 'arn:1234',
        cfAliases: ['www.test.com'],
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            Aliases: [
                'www.test.com',
            ],
            ViewerCertificate: {
                AcmCertificateArn: 'arn:1234',
                SslSupportMethod: 'sni-only',
            },
        },
    }));
});
test('Basic Site Setup', () => {
    const stack = new core_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 's3:GetObject',
                    Effect: 'Allow',
                    Principal: '*',
                }
            ],
        },
    }));
});
test('Basic Site Setup with Error Doc set', () => {
    const stack = new core_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createBasicSite({
        indexDoc: 'index.html',
        errorDoc: 'error.html',
        websiteFolder: 'website',
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
            ErrorDocument: 'error.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 's3:GetObject',
                    Effect: 'Allow',
                    Principal: '*',
                }
            ],
        },
    }));
});
test('Basic Site Setup, Encrypted Bucket', () => {
    const stack = new core_1.Stack();
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true })
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: 'AES256',
                    },
                },
            ],
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 's3:GetObject',
                    Effect: 'Allow',
                    Principal: '*',
                }
            ],
        },
    }));
});
test('Cloudfront With Encrypted Bucket', () => {
    const stack = new core_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true });
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        certificateARN: 'arn:1234',
        cfAliases: ['www.test.com'],
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: 'AES256',
                    },
                },
            ],
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            Aliases: [
                'www.test.com',
            ],
            ViewerCertificate: {
                AcmCertificateArn: 'arn:1234',
                SslSupportMethod: 'sni-only',
            },
        },
    }));
});
test('Cloudfront With Custom Defined Behaviors', () => {
    const stack = new core_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        cfBehaviors: [
            {
                isDefaultBehavior: true,
                allowedMethods: cf.CloudFrontAllowedMethods.ALL,
                forwardedValues: {
                    queryString: true,
                    cookies: { forward: 'all' },
                    headers: ['*'],
                },
            },
            {
                pathPattern: '/virtual-path',
                allowedMethods: cf.CloudFrontAllowedMethods.GET_HEAD,
                cachedMethods: cf.CloudFrontAllowedCachedMethods.GET_HEAD,
            },
        ],
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            CacheBehaviors: [
                {
                    AllowedMethods: ['GET', 'HEAD'],
                    CachedMethods: ['GET', 'HEAD'],
                    PathPattern: '/virtual-path',
                },
            ],
            DefaultCacheBehavior: {
                AllowedMethods: ['DELETE', 'GET', 'HEAD', 'OPTIONS', 'PATCH', 'POST', 'PUT'],
                ForwardedValues: {
                    Cookies: { Forward: 'all' },
                    Headers: ['*'],
                    QueryString: true,
                },
                TargetOriginId: 'origin1'
            },
        },
    }));
});
test('Basic Site Setup, IP Filter', () => {
    const stack = new core_1.Stack();
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true, ipFilter: true, ipList: ['1.1.1.1'] })
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: 'AES256',
                    },
                },
            ],
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 's3:GetObject',
                    Condition: {
                        IpAddress: {
                            'aws:SourceIp': [
                                '1.1.1.1',
                            ],
                        },
                    },
                    Effect: 'Allow',
                    Principal: '*',
                }
            ],
        },
    }));
});
test('Create From Hosted Zone', () => {
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'testStack', {
        env: {
            region: 'us-east-1',
            account: '1234',
        },
    });
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true })
        .createSiteFromHostedZone({
        zoneName: 'cdkspadeploy.com',
        indexDoc: 'index.html',
        websiteFolder: 'website',
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: 'AES256',
                    },
                },
            ],
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            Aliases: [
                'www.cdkspadeploy.com',
            ],
            ViewerCertificate: {
                SslSupportMethod: 'sni-only',
            },
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        's3:GetObject*',
                        's3:GetBucket*',
                        's3:List*',
                    ],
                    Effect: 'Allow',
                    Principal: {
                        CanonicalUser: {
                            'Fn::GetAtt': [
                                'spaDeployOriginAccessIdentityEDA4C19C',
                                'S3CanonicalUserId',
                            ],
                        },
                    },
                    Resource: [
                        {
                            'Fn::GetAtt': [
                                'spaDeployWebsiteBucket1E4C4442',
                                'Arn',
                            ],
                        },
                        {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'spaDeployWebsiteBucket1E4C4442',
                                            'Arn',
                                        ],
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    ],
                }
            ],
        },
    }));
});
test('Create From Hosted Zone with Error Bucket', () => {
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'testStack', {
        env: {
            region: 'us-east-1',
            account: '1234',
        },
    });
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true })
        .createSiteFromHostedZone({
        zoneName: 'cdkspadeploy.com',
        indexDoc: 'index.html',
        errorDoc: 'error.html',
        websiteFolder: 'website',
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: 'AES256',
                    },
                },
            ],
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
            ErrorDocument: 'error.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            Aliases: [
                'www.cdkspadeploy.com',
            ],
            ViewerCertificate: {
                SslSupportMethod: 'sni-only',
            },
        },
    }));
});
test('Basic Site Setup, URL Output Enabled With Name', () => {
    const stack = new core_1.Stack();
    const exportName = 'test-export-name';
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', {})
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        exportWebsiteUrlOutput: true,
        exportWebsiteUrlName: exportName,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveOutput({
        exportName,
    }));
});
test('Basic Site Setup, URL Output Enabled With No Name', () => {
    const stack = new core_1.Stack();
    const exportName = 'test-export-name';
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', {})
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        exportWebsiteUrlOutput: true,
    });
    // THEN
    assert_1.expect(stack).notTo(assert_1.haveOutput({
        exportName,
    }));
});
test('Basic Site Setup, URL Output Not Enabled', () => {
    const stack = new core_1.Stack();
    const exportName = 'test-export-name';
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', {})
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        exportWebsiteUrlOutput: false,
    });
    // THEN
    assert_1.expect(stack).notTo(assert_1.haveOutput({
        exportName,
    }));
});
//# sourceMappingURL=data:application/json;base64,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