"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const tasks = require("../../lib");
let stack;
let vpc;
let cluster;
/* eslint-disable quote-props */
beforeEach(() => {
    // GIVEN
    stack = new core_1.Stack();
    vpc = new ec2.Vpc(stack, 'Vpc');
    cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
    cluster.addCapacity('Capacity', {
        instanceType: new ec2.InstanceType('t3.medium'),
    });
});
test('Cannot create a Fargate task with a fargate-incompatible task definition', () => {
    const taskDefinition = new ecs.TaskDefinition(stack, 'TD', {
        memoryMiB: '512',
        cpu: '256',
        compatibility: ecs.Compatibility.EC2,
    });
    taskDefinition.addContainer('TheContainer', {
        image: ecs.ContainerImage.fromRegistry('foo/bar'),
        memoryLimitMiB: 256,
    });
    expect(() => new tasks.EcsRunTask(stack, 'task', { cluster, taskDefinition, launchTarget: new tasks.EcsFargateLaunchTarget() }).toStateJson()).toThrowError(/Supplied TaskDefinition is not compatible with Fargate/);
});
test('Cannot create a Fargate task without a default container', () => {
    const taskDefinition = new ecs.TaskDefinition(stack, 'TD', {
        memoryMiB: '512',
        cpu: '256',
        compatibility: ecs.Compatibility.FARGATE,
    });
    expect(() => new tasks.EcsRunTask(stack, 'task', { cluster, taskDefinition, launchTarget: new tasks.EcsFargateLaunchTarget() }).toStateJson()).toThrowError(/must have at least one essential container/);
});
test('Running a Fargate Task', () => {
    const taskDefinition = new ecs.TaskDefinition(stack, 'TD', {
        memoryMiB: '512',
        cpu: '256',
        compatibility: ecs.Compatibility.FARGATE,
    });
    const containerDefinition = taskDefinition.addContainer('TheContainer', {
        image: ecs.ContainerImage.fromRegistry('foo/bar'),
        memoryLimitMiB: 256,
    });
    // WHEN
    const runTask = new tasks.EcsRunTask(stack, 'RunFargate', {
        integrationPattern: sfn.IntegrationPattern.RUN_JOB,
        cluster,
        taskDefinition,
        containerOverrides: [
            {
                containerDefinition,
                environment: [{ name: 'SOME_KEY', value: sfn.JsonPath.stringAt('$.SomeKey') }],
            },
        ],
        launchTarget: new tasks.EcsFargateLaunchTarget({
            platformVersion: ecs.FargatePlatformVersion.VERSION1_4,
        }),
    });
    new sfn.StateMachine(stack, 'SM', {
        definition: runTask,
    });
    // THEN
    expect(stack.resolve(runTask.toStateJson())).toEqual({
        End: true,
        Parameters: {
            Cluster: { 'Fn::GetAtt': ['ClusterEB0386A7', 'Arn'] },
            LaunchType: 'FARGATE',
            NetworkConfiguration: {
                AwsvpcConfiguration: {
                    SecurityGroups: [{ 'Fn::GetAtt': ['RunFargateSecurityGroup709740F2', 'GroupId'] }],
                    Subnets: [{ Ref: 'VpcPrivateSubnet1Subnet536B997A' }, { Ref: 'VpcPrivateSubnet2Subnet3788AAA1' }],
                },
            },
            PlatformVersion: '1.4.0',
            TaskDefinition: { Ref: 'TD49C78F36' },
            Overrides: {
                ContainerOverrides: [
                    {
                        Environment: [
                            {
                                Name: 'SOME_KEY',
                                'Value.$': '$.SomeKey',
                            },
                        ],
                        Name: 'TheContainer',
                    },
                ],
            },
        },
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::ecs:runTask.sync',
                ],
            ],
        },
        Type: 'Task',
    });
    expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 'ecs:RunTask',
                    Effect: 'Allow',
                    Resource: { Ref: 'TD49C78F36' },
                },
                {
                    Action: ['ecs:StopTask', 'ecs:DescribeTasks'],
                    Effect: 'Allow',
                    Resource: '*',
                },
                {
                    Action: 'iam:PassRole',
                    Effect: 'Allow',
                    Resource: { 'Fn::GetAtt': ['TDTaskRoleC497AFFC', 'Arn'] },
                },
                {
                    Action: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                { Ref: 'AWS::Partition' },
                                ':events:',
                                { Ref: 'AWS::Region' },
                                ':',
                                { Ref: 'AWS::AccountId' },
                                ':rule/StepFunctionsGetEventsForECSTaskRule',
                            ],
                        ],
                    },
                },
            ],
        },
    });
});
test('Running an EC2 Task with bridge network', () => {
    const taskDefinition = new ecs.TaskDefinition(stack, 'TD', {
        compatibility: ecs.Compatibility.EC2,
    });
    const containerDefinition = taskDefinition.addContainer('TheContainer', {
        image: ecs.ContainerImage.fromRegistry('foo/bar'),
        memoryLimitMiB: 256,
    });
    // WHEN
    const runTask = new tasks.EcsRunTask(stack, 'Run', {
        integrationPattern: sfn.IntegrationPattern.RUN_JOB,
        cluster,
        taskDefinition,
        containerOverrides: [
            {
                containerDefinition,
                environment: [{ name: 'SOME_KEY', value: sfn.JsonPath.stringAt('$.SomeKey') }],
            },
        ],
        launchTarget: new tasks.EcsEc2LaunchTarget(),
    });
    new sfn.StateMachine(stack, 'SM', {
        definition: runTask,
    });
    // THEN
    expect(stack.resolve(runTask.toStateJson())).toEqual({
        End: true,
        Parameters: {
            Cluster: { 'Fn::GetAtt': ['ClusterEB0386A7', 'Arn'] },
            LaunchType: 'EC2',
            TaskDefinition: { Ref: 'TD49C78F36' },
            Overrides: {
                ContainerOverrides: [
                    {
                        Environment: [
                            {
                                Name: 'SOME_KEY',
                                'Value.$': '$.SomeKey',
                            },
                        ],
                        Name: 'TheContainer',
                    },
                ],
            },
        },
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::ecs:runTask.sync',
                ],
            ],
        },
        Type: 'Task',
    });
    expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 'ecs:RunTask',
                    Effect: 'Allow',
                    Resource: { Ref: 'TD49C78F36' },
                },
                {
                    Action: ['ecs:StopTask', 'ecs:DescribeTasks'],
                    Effect: 'Allow',
                    Resource: '*',
                },
                {
                    Action: 'iam:PassRole',
                    Effect: 'Allow',
                    Resource: { 'Fn::GetAtt': ['TDTaskRoleC497AFFC', 'Arn'] },
                },
                {
                    Action: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                { Ref: 'AWS::Partition' },
                                ':events:',
                                { Ref: 'AWS::Region' },
                                ':',
                                { Ref: 'AWS::AccountId' },
                                ':rule/StepFunctionsGetEventsForECSTaskRule',
                            ],
                        ],
                    },
                },
            ],
        },
    });
});
test('Running an EC2 Task with placement strategies', () => {
    const taskDefinition = new ecs.TaskDefinition(stack, 'TD', {
        compatibility: ecs.Compatibility.EC2,
    });
    taskDefinition.addContainer('TheContainer', {
        image: ecs.ContainerImage.fromRegistry('foo/bar'),
        memoryLimitMiB: 256,
    });
    // WHEN
    const runTask = new tasks.EcsRunTask(stack, 'Run', {
        integrationPattern: sfn.IntegrationPattern.RUN_JOB,
        cluster,
        taskDefinition,
        launchTarget: new tasks.EcsEc2LaunchTarget({
            placementStrategies: [ecs.PlacementStrategy.spreadAcrossInstances(), ecs.PlacementStrategy.packedByCpu(), ecs.PlacementStrategy.randomly()],
            placementConstraints: [ecs.PlacementConstraint.memberOf('blieptuut')],
        }),
    });
    new sfn.StateMachine(stack, 'SM', {
        definition: runTask,
    });
    // THEN
    expect(stack.resolve(runTask.toStateJson())).toEqual({
        End: true,
        Parameters: {
            Cluster: { 'Fn::GetAtt': ['ClusterEB0386A7', 'Arn'] },
            LaunchType: 'EC2',
            TaskDefinition: { Ref: 'TD49C78F36' },
            PlacementConstraints: [{ Type: 'memberOf', Expression: 'blieptuut' }],
            PlacementStrategy: [{ Field: 'instanceId', Type: 'spread' }, { Field: 'cpu', Type: 'binpack' }, { Type: 'random' }],
        },
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::ecs:runTask.sync',
                ],
            ],
        },
        Type: 'Task',
    });
});
test('Running an EC2 Task with overridden number values', () => {
    const taskDefinition = new ecs.TaskDefinition(stack, 'TD', {
        compatibility: ecs.Compatibility.EC2,
    });
    const containerDefinition = taskDefinition.addContainer('TheContainer', {
        image: ecs.ContainerImage.fromRegistry('foo/bar'),
        memoryLimitMiB: 256,
    });
    // WHEN
    const runTask = new tasks.EcsRunTask(stack, 'Run', {
        integrationPattern: sfn.IntegrationPattern.RUN_JOB,
        cluster,
        taskDefinition,
        containerOverrides: [
            {
                containerDefinition,
                command: sfn.JsonPath.listAt('$.TheCommand'),
                cpu: 5,
                memoryLimit: sfn.JsonPath.numberAt('$.MemoryLimit'),
            },
        ],
        launchTarget: new tasks.EcsEc2LaunchTarget(),
    });
    // THEN
    expect(stack.resolve(runTask.toStateJson())).toEqual({
        End: true,
        Parameters: {
            Cluster: { 'Fn::GetAtt': ['ClusterEB0386A7', 'Arn'] },
            LaunchType: 'EC2',
            TaskDefinition: { Ref: 'TD49C78F36' },
            Overrides: {
                ContainerOverrides: [
                    {
                        'Command.$': '$.TheCommand',
                        Cpu: 5,
                        'Memory.$': '$.MemoryLimit',
                        Name: 'TheContainer',
                    },
                ],
            },
        },
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::ecs:runTask.sync',
                ],
            ],
        },
        Type: 'Task',
    });
});
//# sourceMappingURL=data:application/json;base64,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