"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KongDataplaneTaskdefinition = void 0;
const aws_ecs_1 = require("aws-cdk-lib/aws-ecs");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
// import * as kong_core from '../../../../kong-core/lib';
const kong_core = require("kong-core");
class KongDataplaneTaskdefinition extends aws_ecs_1.FargateTaskDefinition {
    constructor(scope, id, props) {
        super(scope, id, props.kongTaskProps);
        const dataPlanePort = 8000;
        const clusterEndpoint = props.cluster_dns;
        const telemetryEndpoint = props.telemetry_dns;
        let env = {};
        env.KONG_DATABASE = 'off';
        env.KONG_LOG_LEVEL = 'debug';
        env.KONG_PROXY_ACCESS_LOG = '/dev/stdout';
        env.KONG_PROXY_ERROR_LOG = '/dev/stderr';
        env.KONG_ROLE = 'data_plane';
        env.KONG_CLUSTER_SERVER_NAME = 'kong-cp.internal';
        env.KONG_CLUSTER_CA_CERT = '/certs/ca.crt';
        env.KONG_CLUSTER_CERT = '/certs/cluster.crt';
        env.KONG_CLUSTER_CERT_KEY = '/certs/cluster.key';
        env.KONG_CLUSTER_CONTROL_PLANE = clusterEndpoint;
        env.KONG_CLUSTER_TELEMETRY_ENDPOINT = telemetryEndpoint;
        env.KONG_CLUSTER_MTLS = 'pki';
        this.addToExecutionRolePolicy(new aws_iam_1.PolicyStatement({
            resources: ['*'],
            actions: [
                'ecr:GetAuthorizationToken',
                'ecr:BatchCheckLayerAvailability',
                'ecr:GetDownloadUrlForLayer',
                'ecr:BatchGetImage',
            ],
        }));
        this.addToTaskRolePolicy(new aws_iam_1.PolicyStatement({
            resources: ['*'],
            actions: [
                'logs:CreateLogGroup',
                'logs:CreateLogStream',
                'logs:PutLogEvents',
                'acm-pca:Get*',
                'acm:ExportCertificate',
            ],
        }));
        const kongContainer = this.addContainer('KongDataPlaneContainer', {
            image: aws_ecs_1.ContainerImage.fromRegistry(props.image),
            environment: env,
            logging: aws_ecs_1.LogDrivers.firelens({
                options: {
                    'Name': 'cloudwatch',
                    'region': props.cluster.stack.region,
                    'log_group_name': props.cluster.clusterName + '-KongDatalPlaneContainer',
                    'auto_create_group': 'true',
                    'log_stream_prefix': 'from-data-plane-',
                    'log-driver-buffer-limit': '2097152',
                },
            }),
            secrets: props.secrets,
            essential: true,
            portMappings: [{ containerPort: dataPlanePort, hostPort: dataPlanePort, protocol: aws_ecs_1.Protocol.TCP }],
        });
        const kongCertImage = new kong_core.CustomImage(this, 'KongCustomImage', {
            imageName: 'docker.io/kong/kong-gateway:2.8.1.0-alpine',
        });
        const kongContainerCerts = this.addContainer('KongDataPlaneContainerCerts', {
            image: kongCertImage.kongCustomImage,
            essential: false,
            cpu: 256,
            environment: {
                certArn: props.certificates.certificate.ref,
                rootCAArn: props.certificates.certificate.certificateAuthorityArn || '',
            },
            logging: aws_ecs_1.LogDrivers.firelens({
                options: {
                    'Name': 'cloudwatch',
                    'region': props.cluster.stack.region,
                    'log_group_name': props.cluster.clusterName + '-KongDataPlaneContainerCerts',
                    'auto_create_group': 'true',
                    'log_stream_prefix': 'from-data-plane-',
                    'log-driver-buffer-limit': '2097152',
                },
            }),
            command: ['sh', '/scripts/getCertificates.sh'],
        });
        kongContainer.addContainerDependencies({
            container: kongContainerCerts,
            condition: aws_ecs_1.ContainerDependencyCondition.SUCCESS,
        });
        //Add the firelens log router.
        this.addFirelensLogRouter('KongFirelens', {
            image: aws_ecs_1.ContainerImage.fromRegistry('amazon/aws-for-fluent-bit'),
            firelensConfig: {
                type: aws_ecs_1.FirelensLogRouterType.FLUENTBIT,
            },
        });
        const mountpoints = {
            containerPath: '/certs',
            readOnly: false,
            sourceVolume: 'certs',
        };
        kongContainer.addMountPoints(mountpoints);
        kongContainerCerts.addMountPoints(mountpoints);
        this.addVolume({
            name: mountpoints.sourceVolume,
        });
    }
}
exports.KongDataplaneTaskdefinition = KongDataplaneTaskdefinition;
//# sourceMappingURL=data:application/json;base64,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