"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const cdk = require("@aws-cdk/core");
const nodeunit_shim_1 = require("nodeunit-shim");
const lib_1 = require("../lib");
nodeunit_shim_1.nodeunitShim({
    'basic volume'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new lib_1.Volume(stack, 'Volume', {
            availabilityZone: 'us-east-1a',
            size: cdk.Size.gibibytes(8),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Volume', {
            AvailabilityZone: 'us-east-1a',
            MultiAttachEnabled: false,
            Size: 8,
            VolumeType: 'gp2',
        }, assert_1.ResourcePart.Properties));
        test.done();
    },
    'fromVolumeAttributes'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const encryptionKey = new kms.Key(stack, 'Key');
        const volumeId = 'vol-000000';
        const availabilityZone = 'us-east-1a';
        // WHEN
        const volume = lib_1.Volume.fromVolumeAttributes(stack, 'Volume', {
            volumeId,
            availabilityZone,
            encryptionKey,
        });
        // THEN
        test.strictEqual(volume.volumeId, volumeId);
        test.strictEqual(volume.availabilityZone, availabilityZone);
        test.strictEqual(volume.encryptionKey, encryptionKey);
        test.done();
    },
    'tagged volume'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const volume = new lib_1.Volume(stack, 'Volume', {
            availabilityZone: 'us-east-1a',
            size: cdk.Size.gibibytes(8),
        });
        // WHEN
        cdk.Tags.of(volume).add('TagKey', 'TagValue');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Volume', {
            AvailabilityZone: 'us-east-1a',
            MultiAttachEnabled: false,
            Size: 8,
            VolumeType: 'gp2',
            Tags: [{
                    Key: 'TagKey',
                    Value: 'TagValue',
                }],
        }, assert_1.ResourcePart.Properties));
        test.done();
    },
    'autoenableIO'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new lib_1.Volume(stack, 'Volume', {
            availabilityZone: 'us-east-1a',
            size: cdk.Size.gibibytes(500),
            autoEnableIo: true,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Volume', {
            AutoEnableIO: true,
        }, assert_1.ResourcePart.Properties));
        test.done();
    },
    'encryption'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new lib_1.Volume(stack, 'Volume', {
            availabilityZone: 'us-east-1a',
            size: cdk.Size.gibibytes(500),
            encrypted: true,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Volume', {
            Encrypted: true,
        }, assert_1.ResourcePart.Properties));
        test.done();
    },
    'encryption with kms'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const encryptionKey = new kms.Key(stack, 'Key');
        // WHEN
        new lib_1.Volume(stack, 'Volume', {
            availabilityZone: 'us-east-1a',
            size: cdk.Size.gibibytes(500),
            encrypted: true,
            encryptionKey,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Volume', {
            Encrypted: true,
            KmsKeyId: {
                'Fn::GetAtt': [
                    'Key961B73FD',
                    'Arn',
                ],
            },
        }, assert_1.ResourcePart.Properties));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::KMS::Key', {
            KeyPolicy: {
                Statement: [
                    {},
                    {
                        Effect: 'Allow',
                        Principal: {
                            AWS: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':iam::',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':root',
                                    ],
                                ],
                            },
                        },
                        Resource: '*',
                        Action: [
                            'kms:DescribeKey',
                            'kms:GenerateDataKeyWithoutPlainText',
                        ],
                        Condition: {
                            StringEquals: {
                                'kms:ViaService': {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'ec2.',
                                            {
                                                Ref: 'AWS::Region',
                                            },
                                            '.amazonaws.com',
                                        ],
                                    ],
                                },
                                'kms:CallerAccount': {
                                    Ref: 'AWS::AccountId',
                                },
                            },
                        },
                    },
                ],
            },
        }));
        test.done();
    },
    'encryption with kms from snapshot'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const encryptionKey = new kms.Key(stack, 'Key');
        // WHEN
        new lib_1.Volume(stack, 'Volume', {
            availabilityZone: 'us-east-1a',
            size: cdk.Size.gibibytes(500),
            encrypted: true,
            encryptionKey,
            snapshotId: 'snap-1234567890',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::KMS::Key', {
            KeyPolicy: {
                Statement: [
                    {},
                    {
                        Action: [
                            'kms:DescribeKey',
                            'kms:GenerateDataKeyWithoutPlainText',
                            'kms:ReEncrypt*',
                        ],
                    },
                ],
            },
        }));
        test.done();
    },
    'iops'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new lib_1.Volume(stack, 'Volume', {
            availabilityZone: 'us-east-1a',
            size: cdk.Size.gibibytes(500),
            iops: 500,
            volumeType: lib_1.EbsDeviceVolumeType.PROVISIONED_IOPS_SSD,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Volume', {
            Iops: 500,
            VolumeType: 'io1',
        }, assert_1.ResourcePart.Properties));
        test.done();
    },
    'multi-attach'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new lib_1.Volume(stack, 'Volume', {
            availabilityZone: 'us-east-1a',
            size: cdk.Size.gibibytes(500),
            iops: 500,
            volumeType: lib_1.EbsDeviceVolumeType.PROVISIONED_IOPS_SSD,
            enableMultiAttach: true,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Volume', {
            MultiAttachEnabled: true,
        }, assert_1.ResourcePart.Properties));
        test.done();
    },
    'snapshotId'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new lib_1.Volume(stack, 'Volume', {
            availabilityZone: 'us-east-1a',
            snapshotId: 'snap-00000000',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Volume', {
            SnapshotId: 'snap-00000000',
        }, assert_1.ResourcePart.Properties));
        test.done();
    },
    'volume: standard'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new lib_1.Volume(stack, 'Volume', {
            availabilityZone: 'us-east-1a',
            size: cdk.Size.gibibytes(500),
            volumeType: lib_1.EbsDeviceVolumeType.MAGNETIC,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Volume', {
            VolumeType: 'standard',
        }, assert_1.ResourcePart.Properties));
        test.done();
    },
    'volume: io1'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new lib_1.Volume(stack, 'Volume', {
            availabilityZone: 'us-east-1a',
            size: cdk.Size.gibibytes(500),
            volumeType: lib_1.EbsDeviceVolumeType.PROVISIONED_IOPS_SSD,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Volume', {
            VolumeType: 'io1',
        }, assert_1.ResourcePart.Properties));
        test.done();
    },
    'volume: gp2'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new lib_1.Volume(stack, 'Volume', {
            availabilityZone: 'us-east-1a',
            size: cdk.Size.gibibytes(500),
            volumeType: lib_1.EbsDeviceVolumeType.GENERAL_PURPOSE_SSD,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Volume', {
            VolumeType: 'gp2',
        }, assert_1.ResourcePart.Properties));
        test.done();
    },
    'volume: st1'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new lib_1.Volume(stack, 'Volume', {
            availabilityZone: 'us-east-1a',
            size: cdk.Size.gibibytes(500),
            volumeType: lib_1.EbsDeviceVolumeType.THROUGHPUT_OPTIMIZED_HDD,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Volume', {
            VolumeType: 'st1',
        }, assert_1.ResourcePart.Properties));
        test.done();
    },
    'volume: sc1'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new lib_1.Volume(stack, 'Volume', {
            availabilityZone: 'us-east-1a',
            size: cdk.Size.gibibytes(500),
            volumeType: lib_1.EbsDeviceVolumeType.COLD_HDD,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Volume', {
            VolumeType: 'sc1',
        }, assert_1.ResourcePart.Properties));
        test.done();
    },
    'grantAttachVolume to any instance'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const role = new aws_iam_1.Role(stack, 'Role', { assumedBy: new aws_iam_1.AccountRootPrincipal() });
        const volume = new lib_1.Volume(stack, 'Volume', {
            availabilityZone: 'us-east-1a',
            size: cdk.Size.gibibytes(8),
        });
        // WHEN
        volume.grantAttachVolume(role);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [{
                        Action: 'ec2:AttachVolume',
                        Effect: 'Allow',
                        Resource: [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':ec2:',
                                        {
                                            Ref: 'AWS::Region',
                                        },
                                        ':',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':volume/',
                                        {
                                            Ref: 'VolumeA92988D3',
                                        },
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':ec2:',
                                        {
                                            Ref: 'AWS::Region',
                                        },
                                        ':',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':instance/*',
                                    ],
                                ],
                            },
                        ],
                    }],
            },
        }));
        test.done();
    },
    'grantAttachVolume to any instance with encryption'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const role = new aws_iam_1.Role(stack, 'Role', { assumedBy: new aws_iam_1.AccountRootPrincipal() });
        const encryptionKey = new kms.Key(stack, 'Key');
        const volume = new lib_1.Volume(stack, 'Volume', {
            availabilityZone: 'us-east-1a',
            size: cdk.Size.gibibytes(8),
            encrypted: true,
            encryptionKey,
        });
        // WHEN
        volume.grantAttachVolume(role);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::KMS::Key', {
            KeyPolicy: {
                Statement: [
                    {},
                    {},
                    {
                        Effect: 'Allow',
                        Principal: {
                            AWS: {
                                'Fn::GetAtt': [
                                    'Role1ABCC5F0',
                                    'Arn',
                                ],
                            },
                        },
                        Action: 'kms:CreateGrant',
                        Condition: {
                            Bool: {
                                'kms:GrantIsForAWSResource': true,
                            },
                            StringEquals: {
                                'kms:ViaService': {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'ec2.',
                                            {
                                                Ref: 'AWS::Region',
                                            },
                                            '.amazonaws.com',
                                        ],
                                    ],
                                },
                                'kms:GrantConstraintType': 'EncryptionContextSubset',
                            },
                        },
                        Resource: '*',
                    },
                ],
            },
        }));
        test.done();
    },
    'grantAttachVolume to any instance with KMS.fromKeyArn() encryption'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const role = new aws_iam_1.Role(stack, 'Role', { assumedBy: new aws_iam_1.AccountRootPrincipal() });
        const kmsKey = new kms.Key(stack, 'Key');
        // kmsKey policy is not strictly necessary for the test.
        // Demonstrating how to properly construct the Key.
        const principal = new kms.ViaServicePrincipal(`ec2.${stack.region}.amazonaws.com`, new aws_iam_1.AccountRootPrincipal()).withConditions({
            StringEquals: {
                'kms:CallerAccount': stack.account,
            },
        });
        kmsKey.grant(principal, 
        // Describe & Generate are required to be able to create the CMK-encrypted Volume.
        'kms:DescribeKey', 'kms:GenerateDataKeyWithoutPlainText', 
        // ReEncrypt is required for when the CMK is rotated.
        'kms:ReEncrypt*');
        const encryptionKey = kms.Key.fromKeyArn(stack, 'KeyArn', kmsKey.keyArn);
        const volume = new lib_1.Volume(stack, 'Volume', {
            availabilityZone: 'us-east-1a',
            size: cdk.Size.gibibytes(8),
            encrypted: true,
            encryptionKey,
        });
        // WHEN
        volume.grantAttachVolume(role);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [
                    {},
                    {
                        Effect: 'Allow',
                        Action: 'kms:CreateGrant',
                        Resource: {
                            'Fn::GetAtt': [
                                'Key961B73FD',
                                'Arn',
                            ],
                        },
                        Condition: {
                            Bool: {
                                'kms:GrantIsForAWSResource': true,
                            },
                            StringEquals: {
                                'kms:ViaService': {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'ec2.',
                                            {
                                                Ref: 'AWS::Region',
                                            },
                                            '.amazonaws.com',
                                        ],
                                    ],
                                },
                                'kms:GrantConstraintType': 'EncryptionContextSubset',
                            },
                        },
                    },
                ],
            },
        }));
        test.done();
    },
    'grantAttachVolume to specific instances'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const role = new aws_iam_1.Role(stack, 'Role', { assumedBy: new aws_iam_1.AccountRootPrincipal() });
        const vpc = new lib_1.Vpc(stack, 'Vpc');
        const instance1 = new lib_1.Instance(stack, 'Instance1', {
            vpc,
            instanceType: new lib_1.InstanceType('t3.small'),
            machineImage: lib_1.MachineImage.latestAmazonLinux({ generation: lib_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
            availabilityZone: 'us-east-1a',
        });
        const instance2 = new lib_1.Instance(stack, 'Instance2', {
            vpc,
            instanceType: new lib_1.InstanceType('t3.small'),
            machineImage: lib_1.MachineImage.latestAmazonLinux({ generation: lib_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
            availabilityZone: 'us-east-1a',
        });
        const volume = new lib_1.Volume(stack, 'Volume', {
            availabilityZone: 'us-east-1a',
            size: cdk.Size.gibibytes(8),
        });
        // WHEN
        volume.grantAttachVolume(role, [instance1, instance2]);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [{
                        Action: 'ec2:AttachVolume',
                        Effect: 'Allow',
                        Resource: [
                            {},
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':ec2:',
                                        {
                                            Ref: 'AWS::Region',
                                        },
                                        ':',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':instance/',
                                        {
                                            Ref: 'Instance14BC3991D',
                                        },
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':ec2:',
                                        {
                                            Ref: 'AWS::Region',
                                        },
                                        ':',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':instance/',
                                        {
                                            Ref: 'Instance255F35265',
                                        },
                                    ],
                                ],
                            },
                        ],
                    }],
            },
        }));
        test.done();
    },
    'grantAttachVolume to instance self'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new lib_1.Vpc(stack, 'Vpc');
        const instance = new lib_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new lib_1.InstanceType('t3.small'),
            machineImage: lib_1.MachineImage.latestAmazonLinux({ generation: lib_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
            availabilityZone: 'us-east-1a',
        });
        const volume = new lib_1.Volume(stack, 'Volume', {
            availabilityZone: 'us-east-1a',
            size: cdk.Size.gibibytes(8),
        });
        // WHEN
        volume.grantAttachVolumeByResourceTag(instance.grantPrincipal, [instance]);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [{
                        Action: 'ec2:AttachVolume',
                        Effect: 'Allow',
                        Resource: [
                            {},
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':ec2:',
                                        {
                                            Ref: 'AWS::Region',
                                        },
                                        ':',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':instance/*',
                                    ],
                                ],
                            },
                        ],
                        Condition: {
                            'ForAnyValue:StringEquals': {
                                'ec2:ResourceTag/VolumeGrantAttach-B2376B2BDA': 'b2376b2bda65cb40f83c290dd844c4aa',
                            },
                        },
                    }],
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Volume', {
            Tags: [
                {
                    Key: 'VolumeGrantAttach-B2376B2BDA',
                    Value: 'b2376b2bda65cb40f83c290dd844c4aa',
                },
            ],
        }, assert_1.ResourcePart.Properties));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
            Tags: [
                {},
                {
                    Key: 'VolumeGrantAttach-B2376B2BDA',
                    Value: 'b2376b2bda65cb40f83c290dd844c4aa',
                },
            ],
        }, assert_1.ResourcePart.Properties));
        test.done();
    },
    'grantAttachVolume to instance self with suffix'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new lib_1.Vpc(stack, 'Vpc');
        const instance = new lib_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new lib_1.InstanceType('t3.small'),
            machineImage: lib_1.MachineImage.latestAmazonLinux({ generation: lib_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
            availabilityZone: 'us-east-1a',
        });
        const volume = new lib_1.Volume(stack, 'Volume', {
            availabilityZone: 'us-east-1a',
            size: cdk.Size.gibibytes(8),
        });
        // WHEN
        volume.grantAttachVolumeByResourceTag(instance.grantPrincipal, [instance], 'TestSuffix');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [{
                        Action: 'ec2:AttachVolume',
                        Effect: 'Allow',
                        Resource: [
                            {},
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':ec2:',
                                        {
                                            Ref: 'AWS::Region',
                                        },
                                        ':',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':instance/*',
                                    ],
                                ],
                            },
                        ],
                        Condition: {
                            'ForAnyValue:StringEquals': {
                                'ec2:ResourceTag/VolumeGrantAttach-TestSuffix': 'b2376b2bda65cb40f83c290dd844c4aa',
                            },
                        },
                    }],
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Volume', {
            Tags: [
                {
                    Key: 'VolumeGrantAttach-TestSuffix',
                    Value: 'b2376b2bda65cb40f83c290dd844c4aa',
                },
            ],
        }, assert_1.ResourcePart.Properties));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
            Tags: [
                {},
                {
                    Key: 'VolumeGrantAttach-TestSuffix',
                    Value: 'b2376b2bda65cb40f83c290dd844c4aa',
                },
            ],
        }, assert_1.ResourcePart.Properties));
        test.done();
    },
    'grantDetachVolume to any instance'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const role = new aws_iam_1.Role(stack, 'Role', { assumedBy: new aws_iam_1.AccountRootPrincipal() });
        const volume = new lib_1.Volume(stack, 'Volume', {
            availabilityZone: 'us-east-1a',
            size: cdk.Size.gibibytes(8),
        });
        // WHEN
        volume.grantDetachVolume(role);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [{
                        Action: 'ec2:DetachVolume',
                        Effect: 'Allow',
                        Resource: [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':ec2:',
                                        {
                                            Ref: 'AWS::Region',
                                        },
                                        ':',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':volume/',
                                        {
                                            Ref: 'VolumeA92988D3',
                                        },
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':ec2:',
                                        {
                                            Ref: 'AWS::Region',
                                        },
                                        ':',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':instance/*',
                                    ],
                                ],
                            },
                        ],
                    }],
            },
        }));
        test.done();
    },
    'grantDetachVolume from specific instance'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const role = new aws_iam_1.Role(stack, 'Role', { assumedBy: new aws_iam_1.AccountRootPrincipal() });
        const vpc = new lib_1.Vpc(stack, 'Vpc');
        const instance1 = new lib_1.Instance(stack, 'Instance1', {
            vpc,
            instanceType: new lib_1.InstanceType('t3.small'),
            machineImage: lib_1.MachineImage.latestAmazonLinux({ generation: lib_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
            availabilityZone: 'us-east-1a',
        });
        const instance2 = new lib_1.Instance(stack, 'Instance2', {
            vpc,
            instanceType: new lib_1.InstanceType('t3.small'),
            machineImage: lib_1.MachineImage.latestAmazonLinux({ generation: lib_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
            availabilityZone: 'us-east-1a',
        });
        const volume = new lib_1.Volume(stack, 'Volume', {
            availabilityZone: 'us-east-1a',
            size: cdk.Size.gibibytes(8),
        });
        // WHEN
        volume.grantDetachVolume(role, [instance1, instance2]);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [{
                        Action: 'ec2:DetachVolume',
                        Effect: 'Allow',
                        Resource: [
                            {},
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':ec2:',
                                        {
                                            Ref: 'AWS::Region',
                                        },
                                        ':',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':instance/',
                                        {
                                            Ref: 'Instance14BC3991D',
                                        },
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':ec2:',
                                        {
                                            Ref: 'AWS::Region',
                                        },
                                        ':',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':instance/',
                                        {
                                            Ref: 'Instance255F35265',
                                        },
                                    ],
                                ],
                            },
                        ],
                    }],
            },
        }));
        test.done();
    },
    'grantDetachVolume from instance self'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new lib_1.Vpc(stack, 'Vpc');
        const instance = new lib_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new lib_1.InstanceType('t3.small'),
            machineImage: lib_1.MachineImage.latestAmazonLinux({ generation: lib_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
            availabilityZone: 'us-east-1a',
        });
        const volume = new lib_1.Volume(stack, 'Volume', {
            availabilityZone: 'us-east-1a',
            size: cdk.Size.gibibytes(8),
        });
        // WHEN
        volume.grantDetachVolumeByResourceTag(instance.grantPrincipal, [instance]);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [{
                        Action: 'ec2:DetachVolume',
                        Effect: 'Allow',
                        Resource: [
                            {},
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':ec2:',
                                        {
                                            Ref: 'AWS::Region',
                                        },
                                        ':',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':instance/*',
                                    ],
                                ],
                            },
                        ],
                        Condition: {
                            'ForAnyValue:StringEquals': {
                                'ec2:ResourceTag/VolumeGrantDetach-B2376B2BDA': 'b2376b2bda65cb40f83c290dd844c4aa',
                            },
                        },
                    }],
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Volume', {
            Tags: [
                {
                    Key: 'VolumeGrantDetach-B2376B2BDA',
                    Value: 'b2376b2bda65cb40f83c290dd844c4aa',
                },
            ],
        }, assert_1.ResourcePart.Properties));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
            Tags: [
                {},
                {
                    Key: 'VolumeGrantDetach-B2376B2BDA',
                    Value: 'b2376b2bda65cb40f83c290dd844c4aa',
                },
            ],
        }, assert_1.ResourcePart.Properties));
        test.done();
    },
    'grantDetachVolume from instance self with suffix'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new lib_1.Vpc(stack, 'Vpc');
        const instance = new lib_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new lib_1.InstanceType('t3.small'),
            machineImage: lib_1.MachineImage.latestAmazonLinux({ generation: lib_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
            availabilityZone: 'us-east-1a',
        });
        const volume = new lib_1.Volume(stack, 'Volume', {
            availabilityZone: 'us-east-1a',
            size: cdk.Size.gibibytes(8),
        });
        // WHEN
        volume.grantDetachVolumeByResourceTag(instance.grantPrincipal, [instance], 'TestSuffix');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [{
                        Action: 'ec2:DetachVolume',
                        Effect: 'Allow',
                        Resource: [
                            {},
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':ec2:',
                                        {
                                            Ref: 'AWS::Region',
                                        },
                                        ':',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':instance/*',
                                    ],
                                ],
                            },
                        ],
                        Condition: {
                            'ForAnyValue:StringEquals': {
                                'ec2:ResourceTag/VolumeGrantDetach-TestSuffix': 'b2376b2bda65cb40f83c290dd844c4aa',
                            },
                        },
                    }],
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Volume', {
            Tags: [
                {
                    Key: 'VolumeGrantDetach-TestSuffix',
                    Value: 'b2376b2bda65cb40f83c290dd844c4aa',
                },
            ],
        }, assert_1.ResourcePart.Properties));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
            Tags: [
                {},
                {
                    Key: 'VolumeGrantDetach-TestSuffix',
                    Value: 'b2376b2bda65cb40f83c290dd844c4aa',
                },
            ],
        }, assert_1.ResourcePart.Properties));
        test.done();
    },
    'validation fromVolumeAttributes'(test) {
        // GIVEN
        let idx = 0;
        const stack = new cdk.Stack();
        const volume = new lib_1.Volume(stack, 'Volume', {
            availabilityZone: 'us-east-1a',
            size: cdk.Size.gibibytes(8),
        });
        // THEN
        test.doesNotThrow(() => {
            lib_1.Volume.fromVolumeAttributes(stack, `Volume${idx++}`, {
                volumeId: volume.volumeId,
                availabilityZone: volume.availabilityZone,
            });
        });
        test.doesNotThrow(() => {
            lib_1.Volume.fromVolumeAttributes(stack, `Volume${idx++}`, {
                volumeId: 'vol-0123456789abcdefABCDEF',
                availabilityZone: 'us-east-1a',
            });
        });
        test.throws(() => {
            lib_1.Volume.fromVolumeAttributes(stack, `Volume${idx++}`, {
                volumeId: ' vol-0123456789abcdefABCDEF',
                availabilityZone: 'us-east-1a',
            });
        }, '`volumeId` does not match expected pattern. Expected `vol-<hexadecmial value>` (ex: `vol-05abe246af`) or a Token');
        test.throws(() => {
            lib_1.Volume.fromVolumeAttributes(stack, `Volume${idx++}`, {
                volumeId: 'vol-0123456789abcdefABCDEF ',
                availabilityZone: 'us-east-1a',
            });
        }, '`volumeId` does not match expected pattern. Expected `vol-<hexadecmial value>` (ex: `vol-05abe246af`) or a Token');
        test.done();
    },
    'validation required props'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const key = new kms.Key(stack, 'Key');
        let idx = 0;
        // THEN
        test.throws(() => {
            new lib_1.Volume(stack, `Volume${idx++}`, {
                availabilityZone: 'us-east-1a',
            });
        }, 'Must provide at least one of `size` or `snapshotId`');
        test.doesNotThrow(() => {
            new lib_1.Volume(stack, `Volume${idx++}`, {
                availabilityZone: 'us-east-1a',
                size: cdk.Size.gibibytes(8),
            });
        });
        test.doesNotThrow(() => {
            new lib_1.Volume(stack, `Volume${idx++}`, {
                availabilityZone: 'us-east-1a',
                snapshotId: 'snap-000000000',
            });
        });
        test.doesNotThrow(() => {
            new lib_1.Volume(stack, `Volume${idx++}`, {
                availabilityZone: 'us-east-1a',
                size: cdk.Size.gibibytes(8),
                snapshotId: 'snap-000000000',
            });
        });
        test.throws(() => {
            new lib_1.Volume(stack, `Volume${idx++}`, {
                availabilityZone: 'us-east-1a',
                size: cdk.Size.gibibytes(8),
                encryptionKey: key,
            });
        }, '`encrypted` must be true when providing an `encryptionKey`.');
        test.throws(() => {
            new lib_1.Volume(stack, `Volume${idx++}`, {
                availabilityZone: 'us-east-1a',
                size: cdk.Size.gibibytes(8),
                encrypted: false,
                encryptionKey: key,
            });
        }, '`encrypted` must be true when providing an `encryptionKey`.');
        test.doesNotThrow(() => {
            new lib_1.Volume(stack, `Volume${idx++}`, {
                availabilityZone: 'us-east-1a',
                size: cdk.Size.gibibytes(8),
                encrypted: true,
                encryptionKey: key,
            });
        });
        test.done();
    },
    'validation snapshotId'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const volume = new lib_1.Volume(stack, 'ForToken', {
            availabilityZone: 'us-east-1a',
            size: cdk.Size.gibibytes(8),
        });
        let idx = 0;
        // THEN
        test.doesNotThrow(() => {
            // Should not throw if we provide a Token for the snapshotId
            new lib_1.Volume(stack, `Volume${idx++}`, {
                availabilityZone: 'us-east-1a',
                snapshotId: volume.volumeId,
            });
        });
        test.doesNotThrow(() => {
            new lib_1.Volume(stack, `Volume${idx++}`, {
                availabilityZone: 'us-east-1a',
                snapshotId: 'snap-0123456789abcdefABCDEF',
            });
        });
        test.throws(() => {
            new lib_1.Volume(stack, `Volume${idx++}`, {
                availabilityZone: 'us-east-1a',
                snapshotId: ' snap-1234',
            });
        }, '`snapshotId` does match expected pattern. Expected `snap-<hexadecmial value>` (ex: `snap-05abe246af`) or Token');
        test.throws(() => {
            new lib_1.Volume(stack, `Volume${idx++}`, {
                availabilityZone: 'us-east-1a',
                snapshotId: 'snap-1234 ',
            });
        }, '`snapshotId` does match expected pattern. Expected `snap-<hexadecmial value>` (ex: `snap-05abe246af`) or Token');
        test.done();
    },
    'validation iops'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        let idx = 0;
        // THEN
        // Test: Type of volume
        for (const volumeType of [
            lib_1.EbsDeviceVolumeType.GENERAL_PURPOSE_SSD,
            lib_1.EbsDeviceVolumeType.THROUGHPUT_OPTIMIZED_HDD,
            lib_1.EbsDeviceVolumeType.COLD_HDD,
            lib_1.EbsDeviceVolumeType.MAGNETIC,
        ]) {
            test.throws(() => {
                new lib_1.Volume(stack, `Volume${idx++}`, {
                    availabilityZone: 'us-east-1a',
                    size: cdk.Size.gibibytes(500),
                    iops: 100,
                    volumeType,
                });
            }, '`iops` may only be specified if the `volumeType` is `PROVISIONED_IOPS_SSD`/`IO1`');
        }
        // Test: iops in range
        test.throws(() => {
            new lib_1.Volume(stack, `Volume${idx++}`, {
                availabilityZone: 'us-east-1a',
                size: cdk.Size.gibibytes(10),
                iops: 99,
                volumeType: lib_1.EbsDeviceVolumeType.PROVISIONED_IOPS_SSD,
            });
        }, '`iops` must be in the range 100 to 64,000, inclusive.');
        test.doesNotThrow(() => {
            new lib_1.Volume(stack, `Volume${idx++}`, {
                availabilityZone: 'us-east-1a',
                size: cdk.Size.gibibytes(10),
                iops: 100,
                volumeType: lib_1.EbsDeviceVolumeType.PROVISIONED_IOPS_SSD,
            });
        });
        test.doesNotThrow(() => {
            new lib_1.Volume(stack, `Volume${idx++}`, {
                availabilityZone: 'us-east-1a',
                size: cdk.Size.gibibytes(1300),
                iops: 64000,
                volumeType: lib_1.EbsDeviceVolumeType.PROVISIONED_IOPS_SSD,
            });
        });
        test.throws(() => {
            new lib_1.Volume(stack, `Volume${idx++}`, {
                availabilityZone: 'us-east-1a',
                size: cdk.Size.gibibytes(1300),
                iops: 64001,
                volumeType: lib_1.EbsDeviceVolumeType.PROVISIONED_IOPS_SSD,
            });
        }, '`iops` must be in the range 100 to 64,000, inclusive.');
        // Test: iops ratio
        test.doesNotThrow(() => {
            new lib_1.Volume(stack, `Volume${idx++}`, {
                availabilityZone: 'us-east-1a',
                size: cdk.Size.gibibytes(10),
                iops: 500,
                volumeType: lib_1.EbsDeviceVolumeType.PROVISIONED_IOPS_SSD,
            });
        });
        test.throws(() => {
            new lib_1.Volume(stack, `Volume${idx++}`, {
                availabilityZone: 'us-east-1a',
                size: cdk.Size.gibibytes(10),
                iops: 501,
                volumeType: lib_1.EbsDeviceVolumeType.PROVISIONED_IOPS_SSD,
            });
        }, '`iops` has a maximum ratio of 50 IOPS/GiB.');
        test.done();
    },
    'validation multi-attach'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        let idx = 0;
        // THEN
        for (const volumeType of [
            lib_1.EbsDeviceVolumeType.GENERAL_PURPOSE_SSD,
            lib_1.EbsDeviceVolumeType.THROUGHPUT_OPTIMIZED_HDD,
            lib_1.EbsDeviceVolumeType.COLD_HDD,
            lib_1.EbsDeviceVolumeType.MAGNETIC,
        ]) {
            test.throws(() => {
                new lib_1.Volume(stack, `Volume${idx++}`, {
                    availabilityZone: 'us-east-1a',
                    size: cdk.Size.gibibytes(500),
                    enableMultiAttach: true,
                    volumeType,
                });
            }, 'multi-attach is supported exclusively on `PROVISIONED_IOPS_SSD` volumes.');
        }
        test.done();
    },
    'validation size in range'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        let idx = 0;
        // THEN
        for (const testData of [
            [lib_1.EbsDeviceVolumeType.GENERAL_PURPOSE_SSD, 1, 16000],
            [lib_1.EbsDeviceVolumeType.PROVISIONED_IOPS_SSD, 4, 16000],
            [lib_1.EbsDeviceVolumeType.THROUGHPUT_OPTIMIZED_HDD, 500, 16000],
            [lib_1.EbsDeviceVolumeType.COLD_HDD, 500, 16000],
            [lib_1.EbsDeviceVolumeType.MAGNETIC, 1, 1000],
        ]) {
            const volumeType = testData[0];
            const min = testData[1];
            const max = testData[2];
            test.throws(() => {
                new lib_1.Volume(stack, `Volume${idx++}`, {
                    availabilityZone: 'us-east-1a',
                    size: cdk.Size.gibibytes(min - 1),
                    volumeType,
                });
            }, `\`${volumeType}\` volumes must be between ${min} GiB and ${max} GiB in size.`);
            test.doesNotThrow(() => {
                new lib_1.Volume(stack, `Volume${idx++}`, {
                    availabilityZone: 'us-east-1a',
                    size: cdk.Size.gibibytes(min),
                    volumeType,
                });
            });
            test.doesNotThrow(() => {
                new lib_1.Volume(stack, `Volume${idx++}`, {
                    availabilityZone: 'us-east-1a',
                    size: cdk.Size.gibibytes(max),
                    volumeType,
                });
            });
            test.throws(() => {
                new lib_1.Volume(stack, `Volume${idx++}`, {
                    availabilityZone: 'us-east-1a',
                    size: cdk.Size.gibibytes(max + 1),
                    volumeType,
                });
            }, `\`${volumeType}\` volumes must be between ${min} GiB and ${max} GiB in size.`);
        }
        test.done();
    },
});
//# sourceMappingURL=data:application/json;base64,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