"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserData = void 0;
const core_1 = require("@aws-cdk/core");
const machine_image_1 = require("./machine-image");
/**
 * Instance User Data.
 */
class UserData {
    /**
     * Create a userdata object for Linux hosts.
     */
    static forLinux(options = {}) {
        return new LinuxUserData(options);
    }
    /**
     * Create a userdata object for Windows hosts.
     */
    static forWindows() {
        return new WindowsUserData();
    }
    /**
     * Create a userdata object with custom content.
     */
    static custom(content) {
        const userData = new CustomUserData();
        userData.addCommands(content);
        return userData;
    }
    /**
     *
     */
    static forOperatingSystem(os) {
        switch (os) {
            case machine_image_1.OperatingSystemType.LINUX: return UserData.forLinux();
            case machine_image_1.OperatingSystemType.WINDOWS: return UserData.forWindows();
            case machine_image_1.OperatingSystemType.UNKNOWN: throw new Error('Cannot determine UserData for unknown operating system type');
        }
    }
}
exports.UserData = UserData;
/**
 * Linux Instance User Data
 */
class LinuxUserData extends UserData {
    constructor(props = {}) {
        super();
        this.props = props;
        this.lines = [];
        this.onExitLines = [];
    }
    addCommands(...commands) {
        this.lines.push(...commands);
    }
    addOnExitCommands(...commands) {
        this.onExitLines.push(...commands);
    }
    render() {
        const shebang = this.props.shebang !== undefined ? this.props.shebang : '#!/bin/bash';
        return [shebang, ...(this.renderOnExitLines()), ...this.lines].join('\n');
    }
    addS3DownloadCommand(params) {
        const s3Path = `s3://${params.bucket.bucketName}/${params.bucketKey}`;
        const localPath = (params.localFile && params.localFile.length !== 0) ? params.localFile : `/tmp/${params.bucketKey}`;
        this.addCommands(`mkdir -p $(dirname '${localPath}')`, `aws s3 cp '${s3Path}' '${localPath}'`);
        return localPath;
    }
    addExecuteFileCommand(params) {
        this.addCommands('set -e', `chmod +x '${params.filePath}'`, `'${params.filePath}' ${params.arguments}`);
    }
    addSignalOnExitCommand(resource) {
        const stack = core_1.Stack.of(resource);
        const resourceID = stack.getLogicalId(resource.node.defaultChild);
        this.addOnExitCommands(`/opt/aws/bin/cfn-signal --stack ${stack.stackName} --resource ${resourceID} --region ${stack.region} -e $exitCode || echo 'Failed to send Cloudformation Signal'`);
    }
    renderOnExitLines() {
        if (this.onExitLines.length > 0) {
            return ['function exitTrap(){', 'exitCode=$?', ...this.onExitLines, '}', 'trap exitTrap EXIT'];
        }
        return [];
    }
}
/**
 * Windows Instance User Data
 */
class WindowsUserData extends UserData {
    constructor() {
        super();
        this.lines = [];
        this.onExitLines = [];
    }
    addCommands(...commands) {
        this.lines.push(...commands);
    }
    addOnExitCommands(...commands) {
        this.onExitLines.push(...commands);
    }
    render() {
        return `<powershell>${[...(this.renderOnExitLines()),
            ...this.lines,
            ...(this.onExitLines.length > 0 ? ['throw "Success"'] : [])].join('\n')}</powershell>`;
    }
    addS3DownloadCommand(params) {
        const localPath = (params.localFile && params.localFile.length !== 0) ? params.localFile : `C:/temp/${params.bucketKey}`;
        this.addCommands(`mkdir (Split-Path -Path '${localPath}' ) -ea 0`, `Read-S3Object -BucketName '${params.bucket.bucketName}' -key '${params.bucketKey}' -file '${localPath}' -ErrorAction Stop`);
        return localPath;
    }
    addExecuteFileCommand(params) {
        this.addCommands(`&'${params.filePath}' ${params.arguments}`, `if (!$?) { Write-Error 'Failed to execute the file "${params.filePath}"' -ErrorAction Stop }`);
    }
    addSignalOnExitCommand(resource) {
        const stack = core_1.Stack.of(resource);
        const resourceID = stack.getLogicalId(resource.node.defaultChild);
        this.addOnExitCommands(`cfn-signal --stack ${stack.stackName} --resource ${resourceID} --region ${stack.region} --success ($success.ToString().ToLower())`);
    }
    renderOnExitLines() {
        if (this.onExitLines.length > 0) {
            return ['trap {', '$success=($PSItem.Exception.Message -eq "Success")', ...this.onExitLines, 'break', '}'];
        }
        return [];
    }
}
/**
 * Custom Instance User Data
 */
class CustomUserData extends UserData {
    constructor() {
        super();
        this.lines = [];
    }
    addCommands(...commands) {
        this.lines.push(...commands);
    }
    addOnExitCommands() {
        throw new Error('CustomUserData does not support addOnExitCommands, use UserData.forLinux() or UserData.forWindows() instead.');
    }
    render() {
        return this.lines.join('\n');
    }
    addS3DownloadCommand() {
        throw new Error('CustomUserData does not support addS3DownloadCommand, use UserData.forLinux() or UserData.forWindows() instead.');
    }
    addExecuteFileCommand() {
        throw new Error('CustomUserData does not support addExecuteFileCommand, use UserData.forLinux() or UserData.forWindows() instead.');
    }
    addSignalOnExitCommand() {
        throw new Error('CustomUserData does not support addSignalOnExitCommand, use UserData.forLinux() or UserData.forWindows() instead.');
    }
}
//# sourceMappingURL=data:application/json;base64,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