"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const fs = require("fs-extra");
const dev_env_1 = require("../dev-env");
const file_1 = require("../file");
const gitpod_1 = require("../gitpod");
const logging = require("../logging");
const util_1 = require("./util");
// This is duplicated vs exported
const GITPOD_FILE = '.gitpod.yml';
const DEVCONTAINER_FILE = '.devcontainer.json';
logging.disable();
describe('dev environment enable/disable', () => {
    test('given gitpod and devContainer are false', () => {
        // GIVEN
        const project = new util_1.TestProject({
            gitpod: false,
            devContainer: false,
        });
        // WHEN
        project.synth();
        // THEN
        const gitpodFilePath = path.join(project.outdir, GITPOD_FILE);
        const devContainerFilePath = path.join(project.outdir, DEVCONTAINER_FILE);
        expect(fs.existsSync(gitpodFilePath)).toBeFalsy();
        expect(fs.existsSync(devContainerFilePath)).toBeFalsy();
    });
    test('no gitpod/devcontainer files if they are empty', () => {
        // WHEN
        const project = new util_1.TestProject({
            gitpod: true,
            devContainer: true,
        });
        // THEN
        project.synth();
        const gitpodFilePath = path.join(project.outdir, GITPOD_FILE);
        const devContainerFilePath = path.join(project.outdir, DEVCONTAINER_FILE);
        expect(fs.existsSync(gitpodFilePath)).toBeFalsy();
        expect(fs.existsSync(devContainerFilePath)).toBeFalsy();
    });
    test('given gitpod and devContainer are true', () => {
        var _a, _b;
        // GIVEN
        const project = new util_1.TestProject({
            gitpod: true,
            devContainer: true,
        });
        // WHEN
        (_a = project.gitpod) === null || _a === void 0 ? void 0 : _a.addDockerImage({ image: 'foo' });
        (_b = project.devContainer) === null || _b === void 0 ? void 0 : _b.addPorts('1234');
        // THEN
        project.synth();
        const gitpodFilePath = path.join(project.outdir, GITPOD_FILE);
        const devContainerFilePath = path.join(project.outdir, DEVCONTAINER_FILE);
        expect(fs.existsSync(gitpodFilePath)).toBeTruthy();
        expect(fs.existsSync(devContainerFilePath)).toBeTruthy();
    });
});
describe('dev environment docker options', () => {
    test('given an image', () => {
        var _a, _b;
        // GIVEN
        const project = new util_1.TestProject({
            gitpod: true,
            devContainer: true,
        });
        // WHEN
        (_a = project.gitpod) === null || _a === void 0 ? void 0 : _a.addDockerImage(dev_env_1.DevEnvironmentDockerImage.fromImage('jsii/superchain'));
        (_b = project.devContainer) === null || _b === void 0 ? void 0 : _b.addDockerImage(dev_env_1.DevEnvironmentDockerImage.fromImage('jsii/uberchain'));
        // THEN
        const gitpodSnapshot = util_1.synthSnapshot(project)[GITPOD_FILE];
        expect(gitpodSnapshot).toContain('image: jsii/superchain');
        const devContainerSnapshot = util_1.synthSnapshot(project)[DEVCONTAINER_FILE];
        expect(devContainerSnapshot).toStrictEqual({
            '//': file_1.FileBase.PROJEN_MARKER,
            'image': 'jsii/uberchain',
        });
    });
    test('given a docker file dep', () => {
        var _a, _b;
        // GIVEN
        const project = new util_1.TestProject({
            gitpod: true,
            devContainer: true,
        });
        // WHEN
        (_a = project.gitpod) === null || _a === void 0 ? void 0 : _a.addDockerImage(dev_env_1.DevEnvironmentDockerImage.fromFile('.gitpod.Dockerfile'));
        (_b = project.devContainer) === null || _b === void 0 ? void 0 : _b.addDockerImage(dev_env_1.DevEnvironmentDockerImage.fromFile('Dockerfile'));
        // THEN
        const gitpodSnapshot = util_1.synthSnapshot(project)[GITPOD_FILE];
        expect(gitpodSnapshot).toContain('image:');
        expect(gitpodSnapshot).toContain('file: .gitpod.Dockerfile');
        const devContainerSnapshot = util_1.synthSnapshot(project)[DEVCONTAINER_FILE];
        expect(devContainerSnapshot).toStrictEqual({
            '//': file_1.FileBase.PROJEN_MARKER,
            'build': { dockerfile: 'Dockerfile' },
        });
    });
});
describe('dev environment tasks', () => {
    test('given custom task', () => {
        var _a, _b;
        // GIVEN
        const project = new util_1.TestProject({
            gitpod: true,
            devContainer: true,
        });
        // WHEN
        const task = project.addTask('gitpod-test', { exec: 'text' });
        (_a = project.gitpod) === null || _a === void 0 ? void 0 : _a.addTasks(task);
        (_b = project.devContainer) === null || _b === void 0 ? void 0 : _b.addTasks(task);
        // THEN
        const gitpodSnapshot = util_1.synthSnapshot(project)[GITPOD_FILE];
        expect(gitpodSnapshot).toContain('command');
        expect(gitpodSnapshot).toContain('gitpod-test');
        const devContainerSnapshot = util_1.synthSnapshot(project)[DEVCONTAINER_FILE];
        expect(devContainerSnapshot.postCreateCommand).toContain('gitpod-test');
    });
    test('given gitpod task options', () => {
        var _a;
        // GIVEN
        const project = new util_1.TestProject({
            gitpod: true,
            devContainer: true,
        });
        // WHEN
        const task = project.addTask('gitpod-test', { exec: 'text' });
        (_a = project.gitpod) === null || _a === void 0 ? void 0 : _a.addCustomTask({
            init: 'echo Initializing',
            openIn: gitpod_1.GitpodOpenIn.LEFT,
            openMode: gitpod_1.GitpodOpenMode.SPLIT_BOTTOM,
            command: `npx projen ${task.name}`,
        });
        // THEN
        const snapshot = util_1.synthSnapshot(project)[GITPOD_FILE];
        expect(snapshot).toContain('init: echo Initializing');
        expect(snapshot).toContain('openIn: left');
        expect(snapshot).toContain('openMode: split-bottom');
        expect(snapshot).toContain('command: npx projen gitpod-test');
    });
});
test('dev environment ports', () => {
    var _a, _b;
    // GIVEN
    const project = new util_1.TestProject({
        gitpod: true,
        devContainer: true,
    });
    // WHEN
    (_a = project.gitpod) === null || _a === void 0 ? void 0 : _a.addPorts('8080', '3000-3999');
    (_b = project.devContainer) === null || _b === void 0 ? void 0 : _b.addPorts('8080', '3000');
    // THEN
    const gitpodSnapshot = util_1.synthSnapshot(project)[GITPOD_FILE];
    expect(gitpodSnapshot).toContain('port: "8080"');
    expect(gitpodSnapshot).toContain('port: 3000-3999');
    const devContainerSnapshot = util_1.synthSnapshot(project)[DEVCONTAINER_FILE];
    expect(devContainerSnapshot).toStrictEqual({
        '//': file_1.FileBase.PROJEN_MARKER,
        'forwardPorts': ['8080', '3000'],
    });
});
test('gitpod prebuilds config', () => {
    var _a;
    // GIVEN
    const project = new util_1.TestProject({
        gitpod: true,
        devContainer: false,
    });
    // WHEN
    (_a = project.gitpod) === null || _a === void 0 ? void 0 : _a.addPrebuilds({
        master: true,
        branches: true,
        pullRequestsFromForks: true,
        addBadge: false,
    });
    // THEN
    const gitpodSnapshot = util_1.synthSnapshot(project)[GITPOD_FILE];
    expect(gitpodSnapshot).toContain('github');
    expect(gitpodSnapshot).toContain('prebuilds');
    expect(gitpodSnapshot).toContain('master');
    expect(gitpodSnapshot).toContain('branches');
    expect(gitpodSnapshot).toContain('pullRequestsFromForks');
    expect(gitpodSnapshot).toContain('addBadge');
});
test('dev environment vscode extensions', () => {
    var _a, _b;
    // GIVEN
    const project = new util_1.TestProject({
        gitpod: true,
        devContainer: true,
    });
    // WHEN
    (_a = project.gitpod) === null || _a === void 0 ? void 0 : _a.addVscodeExtensions('dbaeumer.vscode-eslint@2.1.13:5sYlSD6wJi5s3xqD8hupUw==');
    (_b = project.devContainer) === null || _b === void 0 ? void 0 : _b.addVscodeExtensions('dbaeumer.vscode-eslint');
    // THEN
    const gitpodSnapshot = util_1.synthSnapshot(project)[GITPOD_FILE];
    expect(gitpodSnapshot).toContain('extensions:');
    expect(gitpodSnapshot).toContain('dbaeumer.vscode-eslint@2.1.13:5sYlSD6wJi5s3xqD8hupUw==');
    const devContainerSnapshot = util_1.synthSnapshot(project)[DEVCONTAINER_FILE];
    expect(devContainerSnapshot).toStrictEqual({
        '//': file_1.FileBase.PROJEN_MARKER,
        'extensions': ['dbaeumer.vscode-eslint'],
    });
});
//# sourceMappingURL=data:application/json;base64,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