"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SlackApprovalAction = void 0;
const path = require("path");
const aws_apigateway_1 = require("@aws-cdk/aws-apigateway");
const aws_codepipeline_1 = require("@aws-cdk/aws-codepipeline");
const aws_codepipeline_actions_1 = require("@aws-cdk/aws-codepipeline-actions");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_sns_1 = require("@aws-cdk/aws-sns");
const aws_sns_subscriptions_1 = require("@aws-cdk/aws-sns-subscriptions");
const channel_types_1 = require("./channel-types");
class SlackApprovalAction extends aws_codepipeline_actions_1.Action {
    constructor(props) {
        super({
            ...props,
            category: aws_codepipeline_1.ActionCategory.APPROVAL,
            provider: 'Manual',
            artifactBounds: {
                minInputs: 0,
                maxInputs: 0,
                minOutputs: 0,
                maxOutputs: 0,
            },
        });
        this.props = props;
        this.props = props;
    }
    /**
     * The method called when an Action is attached to a Pipeline.
     *
     * This method is guaranteed to be called only once for each Action instance.
     */
    bound(scope, stage, options) {
        const environment = {
            SLACK_BOT_TOKEN: this.props.slackBotToken,
            SLACK_SIGNING_SECRET: this.props.slackSigningSecret,
            SLACK_CHANNEL: this.props.slackChannel,
            SLACK_CHANNEL_ID: this.props.slackChannelId,
            SLACK_CHANNEL_TYPES: (this.props.slackChannelTypes || [channel_types_1.ChannelTypes.PUBLIC]).join(','),
            SLACK_BOT_NAME: this.props.slackBotName || 'buildbot',
            SLACK_BOT_ICON: this.props.slackBotIcon || ':robot_face:',
        };
        const approvalRequester = new aws_lambda_1.Function(scope, 'SlackApprovalRequesterFunction', {
            runtime: aws_lambda_1.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: aws_lambda_1.Code.asset(path.join(__dirname, 'lambdas', 'approval-requester')),
            environment,
        });
        const topic = new aws_sns_1.Topic(scope, 'SlackApprovalTopic');
        topic.grantPublish(options.role);
        topic.addSubscription(new aws_sns_subscriptions_1.LambdaSubscription(approvalRequester));
        const approvalHandler = new aws_lambda_1.Function(scope, 'SlackApprovalHandlerFunction', {
            runtime: aws_lambda_1.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, 'lambdas', 'approval-handler')),
            environment,
        });
        const api = new aws_apigateway_1.RestApi(scope, 'SlackApprovalApi');
        api.root.addProxy({
            defaultIntegration: new aws_apigateway_1.LambdaIntegration(approvalHandler),
        });
        approvalHandler.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ['codepipeline:PutApprovalResult'],
            resources: [`${stage.pipeline.pipelineArn}/${stage.stageName}/${this.props.actionName}`],
        }));
        return {
            configuration: {
                NotificationArn: topic.topicArn,
                CustomData: this.props.additionalInformation,
                ExternalEntityLink: this.props.externalEntityLink,
            },
        };
    }
}
exports.SlackApprovalAction = SlackApprovalAction;
//# sourceMappingURL=data:application/json;base64,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