# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['dunamai']

package_data = \
{'': ['*']}

install_requires = \
['packaging>=20.9']

extras_require = \
{':python_version < "3.8"': ['importlib-metadata>=1.6.0']}

entry_points = \
{'console_scripts': ['dunamai = dunamai.__main__:main']}

setup_kwargs = {
    'name': 'dunamai',
    'version': '1.15.0',
    'description': 'Dynamic version generation',
    'long_description': '\n# Dunamai\nDunamai is a Python 3.5+ library and command line tool for producing dynamic,\nstandards-compliant version strings, derived from tags in your version\ncontrol system. This facilitates uniquely identifying nightly or per-commit\nbuilds in continuous integration and releasing new versions of your software\nsimply by creating a tag.\n\nDunamai is also available as a [GitHub Action](https://github.com/marketplace/actions/run-dunamai).\n\n## Features\n* Version control system support:\n  * [Git](https://git-scm.com) (2.7.0+ is recommended, but versions as old as 1.8.2.3 will work with some reduced functionality)\n  * [Mercurial](https://www.mercurial-scm.org)\n  * [Darcs](http://darcs.net)\n  * [Subversion](https://subversion.apache.org)\n  * [Bazaar](https://bazaar.canonical.com/en)\n  * [Fossil](https://www.fossil-scm.org/home/doc/trunk/www/index.wiki)\n  * [Pijul](https://pijul.org)\n* Version styles:\n  * [PEP 440](https://www.python.org/dev/peps/pep-0440)\n  * [Semantic Versioning](https://semver.org)\n  * [Haskell Package Versioning Policy](https://pvp.haskell.org)\n  * Custom output formats\n* Can be used for projects written in any programming language.\n  For Python, this means you do not need a setup.py.\n\n## Usage\n### Installation\n```\npip install dunamai\n```\n\n### CLI\n```console\n# Suppose you are on commit g29045e8, 7 commits after the v0.2.0 tag.\n\n# Auto-detect the version control system and generate a version:\n$ dunamai from any\n0.2.0.post7.dev0+g29045e8\n\n# Or use an explicit VCS and style:\n$ dunamai from git --no-metadata --style semver\n0.2.0-post.7\n\n# Custom formats:\n$ dunamai from any --format "v{base}+{distance}.{commit}"\nv0.2.0+7.g29045e8\n\n# If you\'d prefer to frame the version in terms of progress toward the next\n# release rather than distance from the latest one, you can bump it:\n$ dunamai from any --bump\n0.2.1.dev7+g29045e8\n\n# Validation of custom formats:\n$ dunamai from any --format "v{base}" --style pep440\nVersion \'v0.2.0\' does not conform to the PEP 440 style\n\n# Validate your own freeform versions:\n$ dunamai check 0.01.0 --style semver\nVersion \'0.01.0\' does not conform to the Semantic Versioning style\n\n# More info:\n$ dunamai --help\n$ dunamai from --help\n$ dunamai from git --help\n```\n\n### Library\n\n```python\nfrom dunamai import Version, Style\n\n# Let\'s say you\'re on commit g644252b, which is tagged as v0.1.0.\nversion = Version.from_git()\nassert version.serialize() == "0.1.0"\n\n# Let\'s say there was a v0.1.0rc5 tag 44 commits ago\n# and you have some uncommitted changes.\nversion = Version.from_any_vcs()\nassert version.serialize() == "0.1.0rc5.post44.dev0+g644252b"\nassert version.serialize(metadata=False) == "0.1.0rc5.post44.dev0"\nassert version.serialize(dirty=True) == "0.1.0rc5.post44.dev0+g644252b.dirty"\nassert version.serialize(style=Style.SemVer) == "0.1.0-rc.5.post.44+g644252b"\n```\n\nThe `serialize()` method gives you an opinionated, PEP 440-compliant default\nthat ensures that versions for untagged commits are compatible with Pip\'s\n`--pre` flag. The individual parts of the version are also available for you\nto use and inspect as you please:\n\n```python\nassert version.base == "0.1.0"\nassert version.stage == "rc"\nassert version.revision == 5\nassert version.distance == 44\nassert version.commit == "g644252b"\nassert version.dirty is True\n\n# Available if the latest tag includes metadata, like v0.1.0+linux:\nassert version.tagged_metadata == "linux"\n```\n\n### Tips\nBy default, the "v" prefix on the tag is required, unless you specify\na custom tag pattern. You can either write a regular expression:\n\n```console\n$ dunamai from any --pattern "(?P<base>\\d+\\.\\d+\\.\\d+)"\n```\n\n```python\nfrom dunamai import Version\n\nversion = Version.from_any_vcs(pattern=r"(?P<base>\\d+\\.\\d+\\.\\d+)")\n```\n\n...or use a named preset:\n\n```console\n$ dunamai from any --pattern default-unprefixed\n```\n\n```python\nfrom dunamai import Version, Pattern\n\nversion = Version.from_any_vcs(pattern=Pattern.DefaultUnprefixed)\n```\n\n### VCS archives\nSometimes, you may only have access to an archive of a repository (e.g., a zip file) without the full history.\nDunamai can still detect a version in some of these cases:\n\n* For Git, you can configure `git archive` to produce a file with some metadata for Dunamai.\n\n  Add a `.git_archival.json` file to the root of your repository with this content:\n  ```\n  {\n    "hash-full": "$Format:%H$",\n    "hash-short": "$Format:%h$",\n    "timestamp": "$Format:%cI$",\n    "refs": "$Format:%D$",\n    "describe": "$Format:%(describe:tags=true,match=v[0-9]*)$"\n  }\n  ```\n\n  Add this line to your `.gitattributes` file.\n  If you don\'t already have this file, add it to the root of your repository:\n  ```\n  .git_archival.json  export-subst\n  ```\n\n* For Mercurial, Dunamai will detect and use an `.hg_archival.txt` file created by `hg archive`.\n  It will also recognize `.hgtags` if present.\n\n### Custom formats\nHere are the available substitutions for custom formats. If you have a tag like\n`v9!0.1.2-beta.3+other`, then:\n\n* `{base}` = `0.1.2`\n* `{stage}` = `beta`\n* `{revision}` = `3`\n* `{distance}` is the number of commits since the last\n* `{commit}` is the commit hash (defaults to short form, unless you use `--full-commit`)\n* `{dirty}` expands to either "dirty" or "clean" if you have uncommitted modified files\n* `{tagged_metadata}` = `other`\n* `{epoch}` = `9`\n* `{branch}` = `feature/foo`\n* `{branch_escaped}` = `featurefoo`\n* `{timestamp}` is in the format `YYYYmmddHHMMSS` as UTC\n\nIf you specify a substitution, its value will always be included in the output.\nFor conditional formatting, you can do something like this (Bash):\n\n```bash\ndistance=$(dunamai from any --format "{distance}")\nif [ "$distance" = "0" ]; then\n    dunamai from any --format "v{base}"\nelse\n    dunamai from any --format "v{base}+{distance}.{dirty}"\nfi\n```\n\n## Comparison to Versioneer\n[Versioneer](https://github.com/warner/python-versioneer) is another great\nlibrary for dynamic versions, but there are some design decisions that\nprompted the creation of Dunamai as an alternative:\n\n* Versioneer requires a setup.py file to exist, or else `versioneer install`\n  will fail, rendering it incompatible with non-setuptools-based projects\n  such as those using Poetry or Flit. Dunamai can be used regardless of the\n  project\'s build system.\n* Versioneer has a CLI that generates Python code which needs to be committed\n  into your repository, whereas Dunamai is just a normal importable library\n  with an optional CLI to help statically include your version string.\n* Versioneer produces the version as an opaque string, whereas Dunamai provides\n  a Version class with discrete parts that can then be inspected and serialized\n  separately.\n* Versioneer provides customizability through a config file, whereas Dunamai\n  aims to offer customizability through its library API and CLI for both\n  scripting support and use in other libraries.\n\n## Integration\n* Setting a `__version__` statically:\n\n  ```console\n  $ echo "__version__ = \'$(dunamai from any)\'" > your_library/_version.py\n  ```\n  ```python\n  # your_library/__init__.py\n  from your_library._version import __version__\n  ```\n\n  Or dynamically (but Dunamai becomes a runtime dependency):\n\n  ```python\n  # your_library/__init__.py\n  import dunamai as _dunamai\n  __version__ = _dunamai.get_version("your-library", third_choice=_dunamai.Version.from_any_vcs).serialize()\n  ```\n\n* setup.py (no install-time dependency on Dunamai as long as you use wheels):\n\n  ```python\n  from setuptools import setup\n  from dunamai import Version\n\n  setup(\n      name="your-library",\n      version=Version.from_any_vcs().serialize(),\n  )\n  ```\n\n  Or you could use a static inclusion approach as in the prior example.\n\n* [Poetry](https://poetry.eustace.io):\n\n  ```console\n  $ poetry version $(dunamai from any)\n  ```\n\n  Or you can use the [poetry-dynamic-versioning](https://github.com/mtkennerly/poetry-dynamic-versioning) plugin.\n\n## Other notes\n* When using Git, remember that lightweight tags do not store their creation time.\n  Therefore, if a commit has multiple lightweight tags,\n  we cannot reliably determine which one should be considered the newest.\n  The solution is to use annotated tags instead.\n* When using Git, the initial commit must **not** be both tagged and empty\n  (i.e., created with `--allow-empty`). This is related to a reporting issue\n  in Git. For more info, [click here](https://github.com/mtkennerly/dunamai/issues/14).\n',
    'author': 'Matthew T. Kennerly',
    'author_email': 'mtkennerly@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/mtkennerly/dunamai',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.5,<4.0',
}


setup(**setup_kwargs)
