# AUTOGENERATED! DO NOT EDIT! File to edit: 02_alamode.ipynb (unless otherwise specified).

__all__ = ['run_alamode', 'gen', 'make_disp_force']

# Cell
import sys
import ase.io
from ase.units import Bohr
import spglib
import click
from collections import OrderedDict

import numpy as np
from numpy import array, zeros, arange, savetxt
from numpy import dot, mod
from numpy.linalg import inv
from numpy.random import choice
import ase
import ase.io
import ase.units as units
from ase.atoms import Atoms
import glob
import click
import sys
from .core import read_potim, normalize_traj, read_traj
import os.path
from contextlib import ExitStack

# Cell
def run_alamode(d='phon', prefix='cryst', kpath='cryst', dfset='DFSET', sc='../sc/CONTCAR',
                o=1, n=0, c2=10, born=None, charge=None):
    fit_cmd = f'/home/jochym/Projects/alamode-tools/devel/make-gen.py opt -p {prefix} -n {sc} -f {dfset} -o {o} --c2 {c2} -d {n}'.split()
    b = ''
    if charge is None:
        charge = prefix
    if born is not None:
        b = f'-b {born} -c {charge}'
    phon_cmd = f'/home/jochym/Projects/alamode-tools/devel/make-gen.py phon -p {prefix} -n {sc} {b} -k {kpath}.path'.split()
    alm_cmd = f'/home/jochym/public/bin/alm {prefix}_fit.in'.split()
    anph_cmd = f'/home/jochym/public/bin/anphon {prefix}_phon.in'.split()

    with open(f'{d}/{prefix}_fit.in', 'w') as ff:
        fit = subprocess.run(fit_cmd, cwd=d, stdout=ff, stderr=subprocess.PIPE)

    with open(f'{d}/{prefix}_phon.in', 'w') as ff:
        phon = subprocess.run(phon_cmd, cwd=d, stdout=ff, stderr=subprocess.PIPE)

    alm = subprocess.run(alm_cmd, cwd=d, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    anph = subprocess.run(anph_cmd, cwd=d, stdout=subprocess.PIPE, stderr=subprocess.PIPE)

    for p, l in zip((fit, phon, alm, anph), ('fit', 'phon', 'alm', 'anphon')):
        if p.stdout is not None:
            with open(f'{d}/{prefix}_{l}.log', 'wt') as lf:
                lf.write(p.stdout.decode())
        if p.stderr is not None and len(p.stderr) > 0:
            with open(f'{d}/{prefix}_{l}.err', 'wt') as lf:
                lf.write(p.stderr.decode())

    return all([r.returncode==0 for r in  (fit, phon, alm, anph)]), fit, phon, alm, anph

# Cell
@click.command()
@click.argument('action', default='opt')
@click.option('-o', '--order', default=1, help='Approximation order (1)')
@click.option('-p', '--prefix', default='CRYST', help='Prefix used in calculations (CRYST)')
@click.option('-n', '--name', default='SPOSCAR', type=click.Path(exists=True), help='Supercell POSCAR file (SPOSCAR)')
@click.option('-s', '--scale', default=1.0, help='Scale of the unit cell (1.0)')
@click.option('-e', '--evec', default=1, help='Print eigenvectors (1)')
@click.option('-m', '--msd', default=1, help='Print mean squere displacement (1)')
@click.option('--c1', default='None', type=str, help='First order interaction cutoff (None)')
@click.option('--c2', default='10', type=str, help='Second order interaction cutoff (10)')
@click.option('--c3', default='10', type=str, help='Third order interaction cutoff (10)')
@click.option('-k', '--kpath', default=None, type=click.Path(exists=True), help='File with reciprocal space path')
@click.option('-g', '--grid', default='10x10x10', help='k-grid for dos calculation (10x10x10)')
@click.option('-d', '--ndat', default=0, help='Number of data points used in fitting (All)', type=int)
@click.option('-f', '--dfset', default='DFSET', help='Name of the DFSET file (DFSET)')
@click.option('-t', '--tmax', default=1000, help='Max temperature (1000)')
@click.option('-c', '--charge', default=None, help='Name of the Born effective charges file (<prefix>.born)')
@click.option('-b', '--born', default=0, help='If non-zero use info from <prefix>.born as Born effective charges.' +
                                              ' Use <born> = [1,2,3] value to select method of non-analytic correction.')
def gen(name, order, prefix, scale, action, evec, msd, tmax, charge, born, ndat, kpath, grid, c1, c2, c3, dfset):
    """Generates gen/opt/phon/dos file depending on the ACTION (default: opt).
       The default values of parameters are enclosed in parethesis.
    """

    tmpl={'gen':
'''
&general
  PREFIX = {prefix}
  MODE = {mode}
  NAT = {nat}
  NKD = {nkd}
  KD = {kd}
/

{dfset}

&interaction
  NORDER = {order}  # 1: harmonic, 2: cubic, ..
/

&cell
  {scale:14.10f} # factor
  {cell} # cell matrix
/

&cutoff
  {cutoff}
/

&position
  {positions}
/
''',
          'phon':
'''
&general
  PREFIX = {prefix}
  MODE = phonons ;
  FCSXML = {prefix}.xml
  NKD = {nkd}
  KD = {kd}
  MASS = {mass}
  TMAX = {tmax}
  EMIN = 0; EMAX = 1000; DELTA_E = 2
  {born}
/


&cell
  {scale:14.10f} # factor
  {cell} # cell matrix
/


&analysis
  PRINTMSD = {msd}
  PRINTEVEC = {evec}
  PDOS = 1
/

&kpoint
  {KMODE}
{KPATH}
/

'''}


    cr = ase.io.read(name)
    mode = ''
    NDATA = ''
    KMODE = 1
    KPATH = '10 10 10'

    cell = cr.get_cell()
    if action == 'dos':
        KMODE = 2
        KPATH = ' '.join(grid.split('x'))
    if action == 'phon':
        KMODE = 1
        if kpath is not None:
            with open(kpath) as kpf:
                KPATH = ''.join(kpf.readlines())
        else :
            print('No k-path given. Using Gamma-X.', file=sys.stderr)
            KPATH = 'G 0.0 0.0 0.0   X 0.0 0.5 0.0   51'
    if action in ['phon','dos']:
        cell=spglib.find_primitive(cr)[0]
        action='phon'
    if action == 'opt':
        if ndat > 0:
            NDATA = f'NDATA = {ndat}'
        dfset = f'&optimize\n  DFSET = {dfset}\n  {NDATA}\n/\n'
    else :
        dfset = ''
    if action in ['gen','opt']:
        mode = {'gen':'suggest', 'opt':'optimize'}[action]
        action = 'gen'


    cell = '\n  '.join([' '.join(['%14.10f' % c for c in v]) for v in cell])

    nat = len(cr.get_atomic_numbers())
    nkd = len(set(cr.get_atomic_numbers()))
    kd = ' '.join(list(OrderedDict.fromkeys(cr.get_chemical_symbols())))
    scl = 1.889725989*scale # in A -> bohr
    elems = {e:n+1 for n, e in enumerate(kd.split())}
    masses = {e:m for e,m in zip(cr.get_chemical_symbols(), cr.get_masses())}

    positions='\n  '.join(['{:d} {:14.10f} {:14.10f} {:14.10f}'.format(elems[e], p[0], p[1], p[2])
        for e, p in zip(cr.get_chemical_symbols(),cr.get_scaled_positions())])

    elm=kd.split()
    cutoff='\n  '.join(['{}-{} {}'.format(a,b, ' '.join([f'{c1}', f'{c2}', f'{c3}'][:order]))
                        for k,a in enumerate(elm) for l,b in enumerate(elm) if k<=l])

    mass = ' '.join(['{:14.10f}'.format(masses[e]) for e in kd.split()])

    if born :
        if charge is None:
            charge=prefix
        born = 'BORNINFO = {charge}.born \n  NONANALYTIC = {born} \n'.format(born=born, charge=charge)
    else :
        born = ''

    print(tmpl[action].format(prefix=prefix, nat=nat, nkd=nkd, kd=kd, order=order, evec=evec, msd=msd,
                              scale=scl, cell=cell, cutoff=cutoff, positions=positions, mass=mass, tmax=tmax,
                              born=born, mode=mode, dfset=dfset, KMODE=KMODE, KPATH=KPATH))

# Cell
@click.command()
@click.option('-p', '--poscar', default='SPOSCAR', type=click.Path(exists=True),
              help='Supercell POSCAR file (SPOSCAR)')
@click.option('-t', '--traj', default='./vasprun.xml', type=click.Path(exists=True),
              help='Trajectory file OUTCAR or vasprun.xml (vasprun.xml)')
@click.option('-s', '--skip', default=0, help='Skip initial time steps (0)')
@click.option('-n', '--number', default=50, help='Number of configurations (50)')
@click.option('-d', '--disp', default='disp.dat', help='Displacement file (disp.dat)')
@click.option('-f', '--force', default='force.dat', help='Forces file (force.dat)')
@click.option('-c', '--configs', is_flag=True, default=False, help='Write disp_xxx.POSCAR files')
@click.option('-r', '--report', default=None, type=click.Path(exists=False),
              help='Report selected timestaps to file (no report by default)')
@click.option('--hff', is_flag=True, default=False, help='Write PHONON HFfile')
@click.option('--dff', is_flag=True, default=True, help='Write disp/force files for alamode')
def make_disp_force(poscar, traj, skip, number, disp, force, configs, report, hff, dff):
    """Generates displacement and forces files from the MD trajectory"""
    rpt=None
    base=ase.io.read(poscar)
    dn, fn = os.path.split(traj)
    if fn :
        md=read_traj(dn, skip, fn)
    else :
        md=read_traj(dn, skip)
    pos=md['pos']
    spos=md['spos']
    tr=md['trj']
    p0=pos.mean(axis=0)
    sp0=spos.mean(axis=0)
    n = 0

    print(f'#Writing output files from {number} random steps out of {traj} file.')
    print('#Time steps:')

    with ExitStack() as estack:
        if hff :
            hffile = estack.enter_context(open('HFfile','tw'))

        if dff :
            dsp = estack.enter_context(open(disp, 'bw'))
            frc = estack.enter_context(open(force, 'bw'))

        if configs and report is not None:
            rpt = estack.enter_context(open(report, 'tw'))

        for k in sorted(choice(arange(pos.shape[0]), number, replace=False)):
            if dff :
                savetxt(dsp, (pos[k]-p0)/units.Bohr, fmt='%24.18f')
                savetxt(frc, tr[k].get_forces()/(units.Ry/units.Bohr), fmt='%24.18f')
            if hff :
                for p,d in zip(base.get_scaled_positions(),spos[k]-sp0):
                    if not np.allclose(d,0.0,atol=1e-05):
                        hffile.write((3*'%9.5f '+3*'%10.6f'+'\n')%(tuple(p)+tuple(d)))
                for i,(s,p,f) in enumerate(zip(base.get_chemical_symbols(),base.get_scaled_positions(),tr[k].get_forces())):
                        hffile.write((f'{i+1:4} {s[0]}'+3*'%8.5f'+3*'%10.6f'+'\n')%(tuple(p)+tuple(f)))
            print(f'{k+skip}', end=' ')
            sys.stdout.flush()
            if configs :
                a=Atoms(tr[k])
                a.set_pbc(False)
                a.set_positions(pos[k])
                ase.io.write(f'disp_{n:04d}_{k:04d}.POSCAR', a, direct=True, vasp5=True)
                if rpt is not None :
                    rpt.write(f'{k+skip}\n')
            n+=1
    print()
    print(f'#Finished: {number} configs extracted')