# -*- coding: utf-8 -*-
# Copyright 2022 Google LLC
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
import proto  # type: ignore

from google.protobuf import timestamp_pb2  # type: ignore


__protobuf__ = proto.module(
    package="google.cloud.osconfig.v1alpha",
    manifest={
        "VulnerabilityReport",
        "GetVulnerabilityReportRequest",
        "ListVulnerabilityReportsRequest",
        "ListVulnerabilityReportsResponse",
        "CVSSv3",
    },
)


class VulnerabilityReport(proto.Message):
    r"""This API resource represents the vulnerability report for a
    specified Compute Engine virtual machine (VM) instance at a given
    point in time.

    For more information, see `Vulnerability
    reports <https://cloud.google.com/compute/docs/instances/os-inventory-management#vulnerability-reports>`__.

    Attributes:
        name (str):
            Output only. The ``vulnerabilityReport`` API resource name.

            Format:
            ``projects/{project_number}/locations/{location}/instances/{instance_id}/vulnerabilityReport``
        vulnerabilities (Sequence[google.cloud.osconfig_v1alpha.types.VulnerabilityReport.Vulnerability]):
            Output only. List of vulnerabilities
            affecting the VM.
        update_time (google.protobuf.timestamp_pb2.Timestamp):
            Output only. The timestamp for when the last
            vulnerability report was generated for the VM.
    """

    class Vulnerability(proto.Message):
        r"""A vulnerability affecting the VM instance.

        Attributes:
            details (google.cloud.osconfig_v1alpha.types.VulnerabilityReport.Vulnerability.Details):
                Contains metadata as per the upstream feed of
                the operating system and NVD.
            installed_inventory_item_ids (Sequence[str]):
                Corresponds to the ``INSTALLED_PACKAGE`` inventory item on
                the VM. This field displays the inventory items affected by
                this vulnerability. If the vulnerability report was not
                updated after the VM inventory update, these values might
                not display in VM inventory. For some distros, this field
                may be empty.
            available_inventory_item_ids (Sequence[str]):
                Corresponds to the ``AVAILABLE_PACKAGE`` inventory item on
                the VM. If the vulnerability report was not updated after
                the VM inventory update, these values might not display in
                VM inventory. If there is no available fix, the field is
                empty. The ``inventory_item`` value specifies the latest
                ``SoftwarePackage`` available to the VM that fixes the
                vulnerability.
            create_time (google.protobuf.timestamp_pb2.Timestamp):
                The timestamp for when the vulnerability was
                first detected.
            update_time (google.protobuf.timestamp_pb2.Timestamp):
                The timestamp for when the vulnerability was
                last modified.
            items (Sequence[google.cloud.osconfig_v1alpha.types.VulnerabilityReport.Vulnerability.Item]):
                List of items affected by the vulnerability.
        """

        class Details(proto.Message):
            r"""Contains metadata information for the vulnerability. This
            information is collected from the upstream feed of the operating
            system.

            Attributes:
                cve (str):
                    The CVE of the vulnerability. CVE cannot be
                    empty and the combination of <cve,
                    classification> should be unique across
                    vulnerabilities for a VM.
                cvss_v2_score (float):
                    The CVSS V2 score of this vulnerability. CVSS
                    V2 score is on a scale of 0 - 10 where 0
                    indicates low severity and 10 indicates high
                    severity.
                cvss_v3 (google.cloud.osconfig_v1alpha.types.CVSSv3):
                    The full description of the CVSSv3 for this
                    vulnerability from NVD.
                severity (str):
                    Assigned severity/impact ranking from the
                    distro.
                description (str):
                    The note or description describing the
                    vulnerability from the distro.
                references (Sequence[google.cloud.osconfig_v1alpha.types.VulnerabilityReport.Vulnerability.Details.Reference]):
                    Corresponds to the references attached to the
                    ``VulnerabilityDetails``.
            """

            class Reference(proto.Message):
                r"""A reference for this vulnerability.

                Attributes:
                    url (str):
                        The url of the reference.
                    source (str):
                        The source of the reference e.g. NVD.
                """

                url = proto.Field(proto.STRING, number=1,)
                source = proto.Field(proto.STRING, number=2,)

            cve = proto.Field(proto.STRING, number=1,)
            cvss_v2_score = proto.Field(proto.FLOAT, number=2,)
            cvss_v3 = proto.Field(proto.MESSAGE, number=3, message="CVSSv3",)
            severity = proto.Field(proto.STRING, number=4,)
            description = proto.Field(proto.STRING, number=5,)
            references = proto.RepeatedField(
                proto.MESSAGE,
                number=6,
                message="VulnerabilityReport.Vulnerability.Details.Reference",
            )

        class Item(proto.Message):
            r"""OS inventory item that is affected by a vulnerability or
            fixed as a result of a vulnerability.

            Attributes:
                installed_inventory_item_id (str):
                    Corresponds to the ``INSTALLED_PACKAGE`` inventory item on
                    the VM. This field displays the inventory items affected by
                    this vulnerability. If the vulnerability report was not
                    updated after the VM inventory update, these values might
                    not display in VM inventory. For some operating systems,
                    this field might be empty.
                available_inventory_item_id (str):
                    Corresponds to the ``AVAILABLE_PACKAGE`` inventory item on
                    the VM. If the vulnerability report was not updated after
                    the VM inventory update, these values might not display in
                    VM inventory. If there is no available fix, the field is
                    empty. The ``inventory_item`` value specifies the latest
                    ``SoftwarePackage`` available to the VM that fixes the
                    vulnerability.
                fixed_cpe_uri (str):
                    The recommended `CPE
                    URI <https://cpe.mitre.org/specification/>`__ update that
                    contains a fix for this vulnerability.
                upstream_fix (str):
                    The upstream OS patch, packages or KB that
                    fixes the vulnerability.
            """

            installed_inventory_item_id = proto.Field(proto.STRING, number=1,)
            available_inventory_item_id = proto.Field(proto.STRING, number=2,)
            fixed_cpe_uri = proto.Field(proto.STRING, number=3,)
            upstream_fix = proto.Field(proto.STRING, number=4,)

        details = proto.Field(
            proto.MESSAGE,
            number=1,
            message="VulnerabilityReport.Vulnerability.Details",
        )
        installed_inventory_item_ids = proto.RepeatedField(proto.STRING, number=2,)
        available_inventory_item_ids = proto.RepeatedField(proto.STRING, number=3,)
        create_time = proto.Field(
            proto.MESSAGE, number=4, message=timestamp_pb2.Timestamp,
        )
        update_time = proto.Field(
            proto.MESSAGE, number=5, message=timestamp_pb2.Timestamp,
        )
        items = proto.RepeatedField(
            proto.MESSAGE, number=6, message="VulnerabilityReport.Vulnerability.Item",
        )

    name = proto.Field(proto.STRING, number=1,)
    vulnerabilities = proto.RepeatedField(
        proto.MESSAGE, number=2, message=Vulnerability,
    )
    update_time = proto.Field(proto.MESSAGE, number=3, message=timestamp_pb2.Timestamp,)


class GetVulnerabilityReportRequest(proto.Message):
    r"""A request message for getting the vulnerability report for
    the specified VM.

    Attributes:
        name (str):
            Required. API resource name for vulnerability resource.

            Format:
            ``projects/{project}/locations/{location}/instances/{instance}/vulnerabilityReport``

            For ``{project}``, either ``project-number`` or
            ``project-id`` can be provided. For ``{instance}``, either
            Compute Engine ``instance-id`` or ``instance-name`` can be
            provided.
    """

    name = proto.Field(proto.STRING, number=1,)


class ListVulnerabilityReportsRequest(proto.Message):
    r"""A request message for listing vulnerability reports for all
    VM instances in the specified location.

    Attributes:
        parent (str):
            Required. The parent resource name.

            Format:
            ``projects/{project}/locations/{location}/instances/-``

            For ``{project}``, either ``project-number`` or
            ``project-id`` can be provided.
        page_size (int):
            The maximum number of results to return.
        page_token (str):
            A pagination token returned from a previous call to
            ``ListVulnerabilityReports`` that indicates where this
            listing should continue from.
        filter (str):
            If provided, this field specifies the criteria that must be
            met by a ``vulnerabilityReport`` API resource to be included
            in the response.
    """

    parent = proto.Field(proto.STRING, number=1,)
    page_size = proto.Field(proto.INT32, number=2,)
    page_token = proto.Field(proto.STRING, number=3,)
    filter = proto.Field(proto.STRING, number=4,)


class ListVulnerabilityReportsResponse(proto.Message):
    r"""A response message for listing vulnerability reports for all
    VM instances in the specified location.

    Attributes:
        vulnerability_reports (Sequence[google.cloud.osconfig_v1alpha.types.VulnerabilityReport]):
            List of vulnerabilityReport objects.
        next_page_token (str):
            The pagination token to retrieve the next
            page of vulnerabilityReports object.
    """

    @property
    def raw_page(self):
        return self

    vulnerability_reports = proto.RepeatedField(
        proto.MESSAGE, number=1, message="VulnerabilityReport",
    )
    next_page_token = proto.Field(proto.STRING, number=2,)


class CVSSv3(proto.Message):
    r"""Common Vulnerability Scoring System version 3.
    For details, see
    https://www.first.org/cvss/specification-document

    Attributes:
        base_score (float):
            The base score is a function of the base
            metric scores.
            https://www.first.org/cvss/specification-document#Base-Metrics
        exploitability_score (float):
            The Exploitability sub-score equation is
            derived from the Base Exploitability metrics.
            https://www.first.org/cvss/specification-document#2-1-Exploitability-Metrics
        impact_score (float):
            The Impact sub-score equation is derived from
            the Base Impact metrics.
        attack_vector (google.cloud.osconfig_v1alpha.types.CVSSv3.AttackVector):
            This metric reflects the context by which
            vulnerability exploitation is possible.
        attack_complexity (google.cloud.osconfig_v1alpha.types.CVSSv3.AttackComplexity):
            This metric describes the conditions beyond
            the attacker's control that must exist in order
            to exploit the vulnerability.
        privileges_required (google.cloud.osconfig_v1alpha.types.CVSSv3.PrivilegesRequired):
            This metric describes the level of privileges
            an attacker must possess before successfully
            exploiting the vulnerability.
        user_interaction (google.cloud.osconfig_v1alpha.types.CVSSv3.UserInteraction):
            This metric captures the requirement for a
            human user, other than the attacker, to
            participate in the successful compromise of the
            vulnerable component.
        scope (google.cloud.osconfig_v1alpha.types.CVSSv3.Scope):
            The Scope metric captures whether a
            vulnerability in one vulnerable component
            impacts resources in components beyond its
            security scope.
        confidentiality_impact (google.cloud.osconfig_v1alpha.types.CVSSv3.Impact):
            This metric measures the impact to the
            confidentiality of the information resources
            managed by a software component due to a
            successfully exploited vulnerability.
        integrity_impact (google.cloud.osconfig_v1alpha.types.CVSSv3.Impact):
            This metric measures the impact to integrity
            of a successfully exploited vulnerability.
        availability_impact (google.cloud.osconfig_v1alpha.types.CVSSv3.Impact):
            This metric measures the impact to the
            availability of the impacted component resulting
            from a successfully exploited vulnerability.
    """

    class AttackVector(proto.Enum):
        r"""This metric reflects the context by which vulnerability
        exploitation is possible.
        """
        ATTACK_VECTOR_UNSPECIFIED = 0
        ATTACK_VECTOR_NETWORK = 1
        ATTACK_VECTOR_ADJACENT = 2
        ATTACK_VECTOR_LOCAL = 3
        ATTACK_VECTOR_PHYSICAL = 4

    class AttackComplexity(proto.Enum):
        r"""This metric describes the conditions beyond the attacker's
        control that must exist in order to exploit the vulnerability.
        """
        ATTACK_COMPLEXITY_UNSPECIFIED = 0
        ATTACK_COMPLEXITY_LOW = 1
        ATTACK_COMPLEXITY_HIGH = 2

    class PrivilegesRequired(proto.Enum):
        r"""This metric describes the level of privileges an attacker
        must possess before successfully exploiting the vulnerability.
        """
        PRIVILEGES_REQUIRED_UNSPECIFIED = 0
        PRIVILEGES_REQUIRED_NONE = 1
        PRIVILEGES_REQUIRED_LOW = 2
        PRIVILEGES_REQUIRED_HIGH = 3

    class UserInteraction(proto.Enum):
        r"""This metric captures the requirement for a human user, other
        than the attacker, to participate in the successful compromise
        of the vulnerable component.
        """
        USER_INTERACTION_UNSPECIFIED = 0
        USER_INTERACTION_NONE = 1
        USER_INTERACTION_REQUIRED = 2

    class Scope(proto.Enum):
        r"""The Scope metric captures whether a vulnerability in one
        vulnerable component impacts resources in components beyond its
        security scope.
        """
        SCOPE_UNSPECIFIED = 0
        SCOPE_UNCHANGED = 1
        SCOPE_CHANGED = 2

    class Impact(proto.Enum):
        r"""The Impact metrics capture the effects of a successfully
        exploited vulnerability on the component that suffers the worst
        outcome that is most directly and predictably associated with
        the attack.
        """
        IMPACT_UNSPECIFIED = 0
        IMPACT_HIGH = 1
        IMPACT_LOW = 2
        IMPACT_NONE = 3

    base_score = proto.Field(proto.FLOAT, number=1,)
    exploitability_score = proto.Field(proto.FLOAT, number=2,)
    impact_score = proto.Field(proto.FLOAT, number=3,)
    attack_vector = proto.Field(proto.ENUM, number=5, enum=AttackVector,)
    attack_complexity = proto.Field(proto.ENUM, number=6, enum=AttackComplexity,)
    privileges_required = proto.Field(proto.ENUM, number=7, enum=PrivilegesRequired,)
    user_interaction = proto.Field(proto.ENUM, number=8, enum=UserInteraction,)
    scope = proto.Field(proto.ENUM, number=9, enum=Scope,)
    confidentiality_impact = proto.Field(proto.ENUM, number=10, enum=Impact,)
    integrity_impact = proto.Field(proto.ENUM, number=11, enum=Impact,)
    availability_impact = proto.Field(proto.ENUM, number=12, enum=Impact,)


__all__ = tuple(sorted(__protobuf__.manifest))
