import shutil
import gzip
import os
import xml.etree.ElementTree as ET
import pandas

__author__ = "Francesco Del Carratore"
__version__ = "0.4.16"
__maintainer__ = "Francesco Del Carratore"
__email__ = "francesco.delcarratore@gmail.com"

def ReadPeakML(filename):
    """
    Loading data from a PeakML file
    
    Parameters
    ----------
    filename : string with the name (including path) of the .peakml file.
                It is important to use the allpeaks file generated by the
                mzmatch anlysis, for the proper use of the IPA method.
    
    Returns
    -------
    df : pandas dataframe (necessary)
         A dataframe containing the MS1 data including the following columns:
            -ids: an unique id for each feature
            -rel.ids:   relation ids. In a previous step of the data processing
                        pipeline, features are clustered based on peak shape
                        similarity/retention time. Features in the same
                        cluster are likely to come from the same metabolite.
                        All isotope patterns must be in the same rel.id
                        cluster.
            -mzs: mass-to-charge ratios, usually the average across
                  different samples.
            -RTs: retention times in seconds, usually the average across
                  different samples.
            -Ints: representative (e.g., maximum or average) intensity detected
                   for each feature across samples (either peak area or peak
                   intensity)
    """
    print('loading '+filename+'...')
    name = filename.split('.peakml')[0]
    shutil.copy(filename,name+'.gz')
    with gzip.open(name+'.gz', 'rb') as f_in:
        with open(name, 'wb') as f_out:
            shutil.copyfileobj(f_in, f_out)
    tree = ET.parse(name)
    root = tree.getroot()
    os.remove(name)
    os.remove(name+'.gz')
    peaks = root[1]
    N = len(peaks)
    ids=[None]*N
    rel_ids=[None]*N
    mzs=[None]*N
    RTs=[None]*N
    Int=[None]*N
    print('parsing '+str(N)+' peaks...')
    for k in range(0,N):
        peak = peaks[k]
        annotations=peak.find('annotations')
        for ann in annotations:
            lab=ann[0].text
            if lab=='id':
                ids[k]=int(ann[1].text)
            if lab=='relation.id':
                rel_ids[k]=int(ann[1].text)
        mzs[k]=float(peak.find('mass').text)
        RTs[k]=float(peak.find('retentiontime').text)
        Int[k]=float(peak.find('intensity').text)
        df=pandas.DataFrame(list(zip(ids,rel_ids,mzs,RTs,Int)),
                            columns=['ids','rel.ids','mzs','RTs','Int'])
            
            
        
        
    return(df)