"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const core_1 = require("@aws-cdk/core");
const process_1 = require("process");
class AlexaSkillStack extends core_1.Stack {
    constructor(parent, config) {
        var _a;
        super(parent, config.skillName, { env: { account: process_1.env.CDK_DEFAULT_ACCOUNT, region: process_1.env.CDK_DEFAULT_REGION } });
        this.templateOptions.description = `The Alexa Skill ${config.skillName}`;
        const assetBucket = new aws_s3_1.Bucket(this, 'AssetBucket', {
            bucketName: `${this.account}-${config.skillName}-${this.region}-assets`,
        });
        assetBucket.grantPublicAccess();
        const userTable = new aws_dynamodb_1.Table(this, 'AttributesTable', {
            billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
            partitionKey: {
                type: aws_dynamodb_1.AttributeType.STRING,
                name: config.userAttribute || 'id',
            },
        });
        const versionName = `${new Date().getTime()}`;
        const skillFunction = new aws_lambda_1.Function(this, 'SkillFunction', {
            code: config.codeAsset,
            description: `Skill Function for ${config.skillName} - updated ${versionName}`,
            timeout: core_1.Duration.seconds(10),
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: (_a = config.codeHandler) !== null && _a !== void 0 ? _a : 'dist/index.handler',
            memorySize: 128,
            environment: {
                ...config.environment,
                TABLE_NAME: userTable.tableName,
                ASSET_BUCKET: assetBucket.bucketName,
                ASSET_BUCKET_URL: assetBucket.bucketRegionalDomainName,
                SKILL_ID: config.skillId,
            }
        });
        userTable.grantReadWriteData(skillFunction);
        const version = skillFunction.addVersion(versionName);
        version.node.tryFindChild('Resource').cfnOptions.deletionPolicy = core_1.CfnDeletionPolicy.RETAIN;
        version.node.tryFindChild('Resource').cfnOptions.updateReplacePolicy = core_1.CfnDeletionPolicy.RETAIN;
        const skillFunctionPermission = new aws_lambda_1.CfnPermission(this, 'SkillFunctionPermission', {
            action: 'lambda:invokeFunction',
            functionName: version.functionName,
            principal: 'alexa-appkit.amazon.com',
        });
        skillFunctionPermission.cfnOptions.deletionPolicy = core_1.CfnDeletionPolicy.RETAIN;
        skillFunctionPermission.cfnOptions.updateReplacePolicy = core_1.CfnDeletionPolicy.RETAIN;
        new core_1.CfnOutput(this, 'SkillId', { value: config.skillId });
        new core_1.CfnOutput(this, 'EndpointArn', { value: version.functionArn });
        new core_1.CfnOutput(this, 'TableName', { value: userTable.tableName });
        new core_1.CfnOutput(this, 'AssetBucketName', { value: assetBucket.bucketName });
    }
}
exports.AlexaSkillStack = AlexaSkillStack;
//# sourceMappingURL=data:application/json;base64,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