// Helper function for creating default graph params. Defaults to a Bar chart, 
import React from "react";
import { intersection } from "../../../utils/arrays";
import { getDisplayColumnHeader } from "../../../utils/columnHeaders";
import { isDatetimeDtype } from "../../../utils/dtypes";
import { convertStringToFloatOrUndefined } from "../../../utils/numbers";
import { convertToStringOrUndefined } from "../../../utils/strings";
import DropdownItem from "../../elements/DropdownItem";
import { GRAPHS_THAT_HAVE_BARMODE, GRAPHS_THAT_HAVE_HISTFUNC, GRAPHS_THAT_HAVE_LINE_SHAPE, GRAPHS_THAT_HAVE_POINTS, GraphType, GRAPH_SAFETY_FILTER_CUTOFF } from "./GraphSetupTab";
// Note: these should match the constants in Python as well
const DO_NOT_CHANGE_PAPER_BGCOLOR_DEFAULT = '#FFFFFF';
const DO_NOT_CHANGE_PLOT_BGCOLOR_DEFAULT = '#E6EBF5';
const DO_NOT_CHANGE_TITLE_FONT_COLOR_DEFAULT = '#2F3E5D';
// unless a graph type is provided
export const getDefaultGraphParams = (sheetDataArray, sheetIndex, graphType) => {
    const safetyFilter = getDefaultSafetyFilter(sheetDataArray, sheetIndex);
    graphType = graphType || GraphType.BAR;
    return {
        graphPreprocessing: {
            safety_filter_turned_on_by_user: safetyFilter
        },
        graphCreation: {
            graph_type: graphType,
            sheet_index: sheetIndex,
            x_axis_column_ids: [],
            y_axis_column_ids: [],
            color: undefined,
            facet_col_column_id: undefined,
            facet_row_column_id: undefined,
            facet_col_wrap: undefined,
            facet_col_spacing: undefined,
            facet_row_spacing: undefined,
            // Params that are only available to some graph types
            points: GRAPHS_THAT_HAVE_POINTS.includes(graphType) ? 'outliers' : undefined,
            line_shape: GRAPHS_THAT_HAVE_LINE_SHAPE.includes(graphType) ? 'linear' : undefined,
            nbins: undefined,
            histnorm: undefined,
            histfunc: GRAPHS_THAT_HAVE_HISTFUNC.includes(graphType) ? 'count' : undefined
        },
        graphStyling: {
            title: {
                title: undefined,
                visible: true,
                title_font_color: DO_NOT_CHANGE_TITLE_FONT_COLOR_DEFAULT
            },
            xaxis: {
                title: undefined,
                visible: true,
                title_font_color: DO_NOT_CHANGE_TITLE_FONT_COLOR_DEFAULT,
                type: undefined,
                showgrid: true,
                gridwidth: undefined,
                rangeslider: {
                    visible: true,
                },
            },
            yaxis: {
                title: undefined,
                visible: true,
                title_font_color: DO_NOT_CHANGE_TITLE_FONT_COLOR_DEFAULT,
                type: undefined,
                showgrid: true,
                gridwidth: undefined,
            },
            showlegend: true,
            legend: {
                title: {
                    text: undefined
                },
                orientation: 'v',
                x: undefined,
                y: undefined,
            },
            paper_bgcolor: DO_NOT_CHANGE_PAPER_BGCOLOR_DEFAULT,
            plot_bgcolor: DO_NOT_CHANGE_PLOT_BGCOLOR_DEFAULT,
            // Params that are only available to some graph types
            barmode: GRAPHS_THAT_HAVE_BARMODE.includes(graphType) ? 'group' : undefined,
            barnorm: undefined,
        }
    };
};
// Helper function for getting the default safety filter status
export const getDefaultSafetyFilter = (sheetDataArray, sheetIndex) => {
    return sheetDataArray[sheetIndex] === undefined || sheetDataArray[sheetIndex].numRows > GRAPH_SAFETY_FILTER_CUTOFF;
};
/*
    A helper function for getting the params for the graph for this sheet when
    opening the graphing taskpane, or when switching to a sheet.

    Notably, will filter oout any columns that are no longer in the dataset,
    which stops the user from having invalid columns selected in their graph
    params.
*/
export const getGraphParams = (graphDataDict, graphID, selectedSheetIndex, sheetDataArray) => {
    var _a;
    const graphParamsCopy = JSON.parse(JSON.stringify((_a = graphDataDict[graphID]) === null || _a === void 0 ? void 0 : _a.graphParams));
    // If the graph already exists, get the data source sheet index from the graph params.
    // Otherwise create a new graph of the selectedSheetIndex
    const graphDataSourceSheetIndex = graphParamsCopy !== undefined ? graphParamsCopy.graphCreation.sheet_index : selectedSheetIndex;
    // If the graph already exists, retrieve the graph params that still make sense. In other words, 
    // if a column was previously included in the graph and it no longer exists, remove it from the graph. 
    if (graphParamsCopy !== undefined) {
        // Filter out column headers that no longer exist
        const validColumnIDs = sheetDataArray[graphDataSourceSheetIndex] !== undefined ? sheetDataArray[graphDataSourceSheetIndex].data.map(c => c.columnID) : [];
        const xAxisColumnIDs = intersection(validColumnIDs, graphParamsCopy.graphCreation.x_axis_column_ids);
        const yAxisColumnIDs = intersection(validColumnIDs, graphParamsCopy.graphCreation.y_axis_column_ids);
        const color = graphParamsCopy.graphCreation.color !== undefined && validColumnIDs.includes(graphParamsCopy.graphCreation.color) ? graphParamsCopy.graphCreation.color : undefined;
        return Object.assign(Object.assign({}, graphParamsCopy), { graphCreation: Object.assign(Object.assign({}, graphParamsCopy.graphCreation), { x_axis_column_ids: xAxisColumnIDs, y_axis_column_ids: yAxisColumnIDs, color: color }) });
    }
    // If the graph does not already exist, create a default graph.
    return getDefaultGraphParams(sheetDataArray, graphDataSourceSheetIndex);
};
// Returns a list of dropdown items. Selecting them sets the color attribute of the graph.
// Option 'None' always comes first.
export const getColorDropdownItems = (graphSheetIndex, columnIDsMapArray, columnDtypesMap, setColor) => {
    const NoneOption = [(React.createElement(DropdownItem, { key: 'None', title: 'None', onClick: () => setColor(undefined) }))];
    const columnDropdownItems = Object.keys(columnIDsMapArray[graphSheetIndex] || {}).map(columnID => {
        const columnHeader = columnIDsMapArray[graphSheetIndex][columnID];
        // Plotly doesn't support setting the color as a date series, so we disable date series dropdown items
        const disabled = isDatetimeDtype(columnDtypesMap[columnID]);
        return (React.createElement(DropdownItem, { key: columnID, title: getDisplayColumnHeader(columnHeader), onClick: () => setColor(columnID), disabled: disabled, subtext: disabled ? 'Dates cannot be used as the color breakdown property' : '', hideSubtext: true, displaySubtextOnHover: true }));
    });
    return NoneOption.concat(columnDropdownItems);
};
export const getGraphTypeFullName = (graphType) => {
    switch (graphType) {
        case GraphType.BAR: return 'Bar chart';
        case GraphType.BOX: return 'Box plot';
        case GraphType.DENSITY_CONTOUR: return 'Density contour';
        case GraphType.DENSITY_HEATMAP: return 'Density heatmap';
        case GraphType.ECDF: return 'ECDF';
        case GraphType.HISTOGRAM: return 'Histogram';
        case GraphType.LINE: return 'Line chart';
        case GraphType.SCATTER: return 'Scatter plot';
        case GraphType.STRIP: return 'Strip plot';
        case GraphType.VIOLIN: return 'Violin plot';
    }
};
export const convertFrontendtoBackendGraphParams = (graphParamsFrontend) => {
    const graphCreationParams = graphParamsFrontend.graphCreation;
    const graphStylingParams = graphParamsFrontend.graphStyling;
    const x = Object.assign(Object.assign({}, graphParamsFrontend), { graphCreation: Object.assign(Object.assign({}, graphParamsFrontend.graphCreation), { facet_col_wrap: convertStringToFloatOrUndefined(graphCreationParams.facet_col_wrap), facet_col_spacing: convertStringToFloatOrUndefined(graphCreationParams.facet_col_spacing), facet_row_spacing: convertStringToFloatOrUndefined(graphCreationParams.facet_row_spacing), nbins: convertStringToFloatOrUndefined(graphCreationParams.nbins) }), graphStyling: Object.assign(Object.assign({}, graphParamsFrontend.graphStyling), { xaxis: Object.assign(Object.assign({}, graphParamsFrontend.graphStyling.xaxis), { gridwidth: convertStringToFloatOrUndefined(graphStylingParams.xaxis.gridwidth) }), yaxis: Object.assign(Object.assign({}, graphParamsFrontend.graphStyling.yaxis), { gridwidth: convertStringToFloatOrUndefined(graphStylingParams.yaxis.gridwidth) }), legend: Object.assign(Object.assign({}, graphParamsFrontend.graphStyling.legend), { x: convertStringToFloatOrUndefined(graphStylingParams.legend.x), y: convertStringToFloatOrUndefined(graphStylingParams.legend.y) }) }) });
    return x;
};
export const convertBackendtoFrontendGraphParams = (graphParamsFrontend) => {
    const graphCreationParams = graphParamsFrontend.graphCreation;
    const graphStylingParams = graphParamsFrontend.graphStyling;
    return Object.assign(Object.assign({}, graphParamsFrontend), { graphCreation: Object.assign(Object.assign({}, graphParamsFrontend.graphCreation), { facet_col_wrap: convertToStringOrUndefined(graphCreationParams.facet_col_wrap), facet_col_spacing: convertToStringOrUndefined(graphCreationParams.facet_col_spacing), facet_row_spacing: convertToStringOrUndefined(graphCreationParams.facet_row_spacing), nbins: convertToStringOrUndefined(graphCreationParams.nbins) }), graphStyling: Object.assign(Object.assign({}, graphParamsFrontend.graphStyling), { xaxis: Object.assign(Object.assign({}, graphParamsFrontend.graphStyling.xaxis), { gridwidth: convertToStringOrUndefined(graphStylingParams.xaxis.gridwidth) }), yaxis: Object.assign(Object.assign({}, graphParamsFrontend.graphStyling.yaxis), { gridwidth: convertToStringOrUndefined(graphStylingParams.yaxis.gridwidth) }), legend: Object.assign(Object.assign({}, graphParamsFrontend.graphStyling.legend), { x: convertToStringOrUndefined(graphStylingParams.legend.x), y: convertToStringOrUndefined(graphStylingParams.legend.y) }) }) });
};
//# sourceMappingURL=graphUtils.js.map