// Copyright (c) Mito
import React, { Fragment } from 'react';
import Row from '../../layout/Row';
import Col from '../../layout/Col';
import Select from '../../elements/Select';
import DropdownItem from '../../elements/DropdownItem';
import AxisSection, { GraphAxisType } from './AxisSection';
import Toggle from '../../elements/Toggle';
import { getColorDropdownItems, getDefaultGraphParams, getDefaultSafetyFilter, getGraphTypeFullName } from './graphUtils';
import { getDisplayColumnHeader } from '../../../utils/columnHeaders';
import Tooltip from '../../elements/Tooltip';
import DataframeSelect from '../../elements/DataframeSelect';
import CollapsibleSection from '../../layout/CollapsibleSection';
import Input from '../../elements/Input';
import { updateObjectWithPartialObject } from '../../../utils/objects';
export var GraphType;
(function (GraphType) {
    GraphType["BAR"] = "bar";
    GraphType["LINE"] = "line";
    GraphType["SCATTER"] = "scatter";
    GraphType["HISTOGRAM"] = "histogram";
    GraphType["DENSITY_HEATMAP"] = "density heatmap";
    GraphType["DENSITY_CONTOUR"] = "density contour";
    GraphType["BOX"] = "box";
    GraphType["VIOLIN"] = "violin";
    GraphType["STRIP"] = "strip";
    GraphType["ECDF"] = "ecdf";
})(GraphType || (GraphType = {}));
// Graphing a dataframe with more than this number of rows will
// give the user the option to apply the safety filter
// Note: This must be kept in sync with the graphing heuristic in the mitosheet/graph folder
export const GRAPH_SAFETY_FILTER_CUTOFF = 1000;
// Tooltips used to explain the Safety filter toggle
const SAFETY_FILTER_DISABLED_MESSAGE = `Because you’re graphing less than ${GRAPH_SAFETY_FILTER_CUTOFF} rows of data, you can safely graph your data without applying a filter first.`;
const SAFETY_FILTER_ENABLED_MESSAGE = `Turning on Filter to Safe Size only graphs the first ${GRAPH_SAFETY_FILTER_CUTOFF} rows of your dataframe, ensuring that your browser tab won’t crash. Turning off Filter to Safe Size graphs the entire dataframe and may slow or crash your browser tab.`;
const GRAPHS_THAT_DONT_SUPPORT_COLOR = [GraphType.DENSITY_HEATMAP];
// These variables are used to populate the collapsible style section that is 
// specific to each graph type.
export const GRAPHS_THAT_HAVE_NBINS = [GraphType.HISTOGRAM];
export const GRAPHS_THAT_HAVE_BARMODE = [GraphType.BAR, GraphType.HISTOGRAM];
export const GRAPHS_THAT_HAVE_BARNORM = [GraphType.BAR, GraphType.HISTOGRAM];
export const GRAPHS_THAT_HAVE_HISTNORM = [GraphType.HISTOGRAM];
export const GRAPHS_THAT_HAVE_HISTFUNC = [GraphType.HISTOGRAM];
export const GRAPHS_THAT_HAVE_POINTS = [GraphType.BOX, GraphType.VIOLIN];
export const GRAPHS_THAT_HAVE_LINE_SHAPE = [GraphType.LINE];
// This variable is used to figure out which graph types should 
// havve teh specific graph type configuration section
export const GRAPHS_WITH_UNIQUE_CONFIG_OPTIONS = [...new Set([
        ...GRAPHS_THAT_HAVE_NBINS,
        ...GRAPHS_THAT_HAVE_BARMODE,
        ...GRAPHS_THAT_HAVE_BARNORM,
        ...GRAPHS_THAT_HAVE_HISTNORM,
        ...GRAPHS_THAT_HAVE_HISTFUNC,
        ...GRAPHS_THAT_HAVE_POINTS,
        ...GRAPHS_THAT_HAVE_LINE_SHAPE
    ])];
/*
    The graph setup tab where the user creates the structure of the graph by
    selecting data
*/
function GraphSetupTab(props) {
    var _a;
    const graphSheetIndex = props.graphParams.graphCreation.sheet_index;
    const graphPreprocessingParams = props.graphParams.graphPreprocessing;
    /*
        Function responsible for updating the selected column headers for each axis.
        Set the columnHeader at the index of the graphAxis selected columns array.
    
        To remove a column, leave the columnHeader empty.
    */
    const updateAxisData = (graphAxis, index, columnID) => {
        // Get the current axis data
        let axisColumnIDs = [];
        if (graphAxis === GraphAxisType.X_AXIS) {
            axisColumnIDs = props.graphParams.graphCreation.x_axis_column_ids;
        }
        else {
            axisColumnIDs = props.graphParams.graphCreation.y_axis_column_ids;
        }
        // Make a copy of the column headers before editing them
        const axisColumnIDsCopy = [...axisColumnIDs];
        if (columnID === undefined) {
            axisColumnIDsCopy.splice(index, 1);
        }
        else {
            axisColumnIDsCopy[index] = columnID;
        }
        // Update the axis data
        if (graphAxis === GraphAxisType.X_AXIS) {
            props.setGraphParams(prevGraphParams => {
                const graphParamsCopy = JSON.parse(JSON.stringify(prevGraphParams));
                return Object.assign(Object.assign({}, graphParamsCopy), { graphCreation: Object.assign(Object.assign({}, graphParamsCopy.graphCreation), { x_axis_column_ids: axisColumnIDsCopy }) });
            });
        }
        else {
            props.setGraphParams(prevGraphParams => {
                const graphParamsCopy = JSON.parse(JSON.stringify(prevGraphParams));
                return Object.assign(Object.assign({}, graphParamsCopy), { graphCreation: Object.assign(Object.assign({}, graphParamsCopy.graphCreation), { y_axis_column_ids: axisColumnIDsCopy }) });
            });
        }
        // Then set increment graphUpdateNumber so we send the graph message
        props.setGraphUpdatedNumber((old) => old + 1);
    };
    const setGraphType = (graphType) => {
        const xAxisColumnIDsCopy = [...props.graphParams.graphCreation.x_axis_column_ids];
        const yAxisColumnIDsCopy = [...props.graphParams.graphCreation.y_axis_column_ids];
        // Update the graph type and reset params that are only available for some graph types
        props.setGraphParams(prevGraphParams => {
            const graphParamsCopy = JSON.parse(JSON.stringify(prevGraphParams));
            return Object.assign(Object.assign({}, graphParamsCopy), { graphCreation: Object.assign(Object.assign({}, graphParamsCopy.graphCreation), { graph_type: graphType, x_axis_column_ids: xAxisColumnIDsCopy, y_axis_column_ids: yAxisColumnIDsCopy, color: GRAPHS_THAT_DONT_SUPPORT_COLOR.includes(graphType) ? undefined : graphParamsCopy.graphCreation.color, points: GRAPHS_THAT_HAVE_POINTS.includes(graphType) ? 'outliers' : undefined, line_shape: GRAPHS_THAT_HAVE_LINE_SHAPE.includes(graphType) ? 'linear' : undefined, nbins: undefined, histnorm: undefined, histfunc: GRAPHS_THAT_HAVE_HISTFUNC.includes(graphType) ? 'count' : undefined }), graphStyling: Object.assign(Object.assign({}, graphParamsCopy.graphStyling), { barmode: GRAPHS_THAT_HAVE_BARMODE.includes(graphType) ? 'group' : undefined, barnorm: undefined }) });
        });
        props.setGraphUpdatedNumber((old) => old + 1);
    };
    const setColor = (newColorColumnID) => {
        props.setGraphParams(prevGraphParams => {
            const graphParamsCopy = JSON.parse(JSON.stringify(prevGraphParams));
            return Object.assign(Object.assign({}, graphParamsCopy), { graphCreation: Object.assign(Object.assign({}, graphParamsCopy.graphCreation), { color: newColorColumnID }) });
        });
        props.setGraphUpdatedNumber((old) => old + 1);
    };
    function updateGraphParam(update) {
        props.setGraphParams(prevGraphParams => {
            return updateObjectWithPartialObject(prevGraphParams, update);
        });
        props.setGraphUpdatedNumber(old => old + 1);
    }
    const colorByColumnTitle = GRAPHS_THAT_DONT_SUPPORT_COLOR.includes(props.graphParams.graphCreation.graph_type)
        ? `${props.graphParams.graphCreation.graph_type} does not support further breaking down data using color.`
        : 'Use an additional column to further breakdown the data by color.';
    const columnIDsMap = props.columnIDsMapArray[graphSheetIndex] || {};
    return (React.createElement(Fragment, null,
        React.createElement("div", { className: 'graph-sidebar-toolbar-content' },
            React.createElement(DataframeSelect, { title: 'Select the data sheet to graph.', sheetDataArray: props.sheetDataArray, sheetIndex: graphSheetIndex, onChange: (newSheetIndex) => {
                    // Reset the graph params for the new sheet, but keep the graph type!
                    const newSheetGraphParams = getDefaultGraphParams(props.sheetDataArray, newSheetIndex, props.graphParams.graphCreation.graph_type);
                    props.setGraphParams(newSheetGraphParams);
                    props.setGraphUpdatedNumber((old) => old + 1);
                } }),
            React.createElement(Row, { justify: 'space-between', align: 'center', title: 'Select one of many Plotly graphs to create.' },
                React.createElement(Col, null,
                    React.createElement("p", { className: 'text-header-3' }, "Chart Type")),
                React.createElement(Col, null,
                    React.createElement(Select, { value: props.graphParams.graphCreation.graph_type, onChange: (graphType) => {
                            setGraphType(graphType);
                        }, width: 'small', dropdownWidth: 'medium' },
                        React.createElement(DropdownItem, { title: GraphType.BAR }),
                        React.createElement(DropdownItem, { title: GraphType.LINE }),
                        React.createElement(DropdownItem, { title: GraphType.SCATTER }),
                        React.createElement(DropdownItem, { title: GraphType.HISTOGRAM }),
                        React.createElement(DropdownItem, { title: GraphType.DENSITY_HEATMAP }),
                        React.createElement(DropdownItem, { title: GraphType.DENSITY_CONTOUR }),
                        React.createElement(DropdownItem, { title: GraphType.BOX }),
                        React.createElement(DropdownItem, { title: GraphType.VIOLIN }),
                        React.createElement(DropdownItem, { title: GraphType.STRIP }),
                        React.createElement(DropdownItem, { title: GraphType.ECDF })))),
            React.createElement(AxisSection, { columnIDsMap: columnIDsMap, graphType: props.graphParams.graphCreation.graph_type, graphAxis: GraphAxisType.X_AXIS, selectedColumnIDs: props.graphParams.graphCreation.x_axis_column_ids, otherAxisSelectedColumnIDs: props.graphParams.graphCreation.y_axis_column_ids, updateAxisData: updateAxisData, mitoAPI: props.mitoAPI }),
            React.createElement(AxisSection, { columnIDsMap: columnIDsMap, graphType: props.graphParams.graphCreation.graph_type, graphAxis: GraphAxisType.Y_AXIS, selectedColumnIDs: props.graphParams.graphCreation.y_axis_column_ids, otherAxisSelectedColumnIDs: props.graphParams.graphCreation.x_axis_column_ids, updateAxisData: updateAxisData, mitoAPI: props.mitoAPI }),
            React.createElement("div", null,
                React.createElement(Row, { justify: 'space-between', align: 'center', title: colorByColumnTitle, suppressTopBottomMargin: true },
                    React.createElement(Col, null,
                        React.createElement(Row, { justify: 'space-between', align: 'center', suppressTopBottomMargin: true },
                            React.createElement("div", { className: 'text-header-3' }, "Color By Column \u00A0"),
                            React.createElement(Tooltip, { title: colorByColumnTitle }))),
                    React.createElement(Col, null,
                        React.createElement(Select, { value: props.graphParams.graphCreation.color ? getDisplayColumnHeader(columnIDsMap[props.graphParams.graphCreation.color]) : 'None', disabled: GRAPHS_THAT_DONT_SUPPORT_COLOR.includes(props.graphParams.graphCreation.graph_type), width: 'small', searchable: true }, getColorDropdownItems(graphSheetIndex, props.columnIDsMapArray, props.columnDtypesMap, setColor))))),
            React.createElement(Row, { justify: 'space-between', align: 'center', title: getDefaultSafetyFilter(props.sheetDataArray, graphSheetIndex) ? SAFETY_FILTER_ENABLED_MESSAGE : SAFETY_FILTER_DISABLED_MESSAGE },
                React.createElement(Col, null,
                    React.createElement(Row, { justify: 'space-between', align: 'center', suppressTopBottomMargin: true },
                        React.createElement("p", { className: 'text-header-3' }, "Filter to safe size \u00A0"),
                        React.createElement(Tooltip, { title: getDefaultSafetyFilter(props.sheetDataArray, graphSheetIndex) ? SAFETY_FILTER_ENABLED_MESSAGE : SAFETY_FILTER_DISABLED_MESSAGE }))),
                React.createElement(Col, null,
                    React.createElement(Toggle, { value: props.graphParams.graphPreprocessing.safety_filter_turned_on_by_user, onChange: () => {
                            updateGraphParam({ graphPreprocessing: { safety_filter_turned_on_by_user: !graphPreprocessingParams.safety_filter_turned_on_by_user } });
                        }, disabled: !getDefaultSafetyFilter(props.sheetDataArray, graphSheetIndex) }))),
            GRAPHS_WITH_UNIQUE_CONFIG_OPTIONS.includes(props.graphParams.graphCreation.graph_type) &&
                React.createElement(CollapsibleSection, { title: getGraphTypeFullName(props.graphParams.graphCreation.graph_type) + ' configuration' },
                    GRAPHS_THAT_HAVE_NBINS.includes(props.graphParams.graphCreation.graph_type) &&
                        React.createElement(Row, { justify: 'space-between', align: 'center', title: 'Number of bins in histogram' },
                            React.createElement(Col, null,
                                React.createElement("p", null, "Number of bins (int)")),
                            React.createElement(Input, { value: ((_a = props.graphParams.graphCreation.nbins) === null || _a === void 0 ? void 0 : _a.toString()) || '', type: 'number', placeholder: '5', onChange: (e) => {
                                    const newNumberBins = e.target.value === '' ? undefined : e.target.value;
                                    updateGraphParam({ graphCreation: { nbins: newNumberBins } });
                                }, width: 'small' })),
                    GRAPHS_THAT_HAVE_BARMODE.includes(props.graphParams.graphCreation.graph_type) &&
                        React.createElement(Row, { justify: 'space-between', align: 'center', title: 'How bars are grouped together when there are multiple' },
                            React.createElement(Col, null,
                                React.createElement(Row, { justify: 'space-between', align: 'center', suppressTopBottomMargin: true },
                                    React.createElement("p", null, "Bar mode"),
                                    React.createElement(Tooltip, { title: 'How bars are grouped together when there are multiple' }))),
                            React.createElement(Select, { value: props.graphParams.graphStyling.barmode || 'group', onChange: (newBarMode) => {
                                    updateGraphParam({ graphStyling: { barmode: newBarMode } });
                                }, width: 'small', dropdownWidth: 'medium' },
                                React.createElement(DropdownItem, { title: 'stack' }),
                                React.createElement(DropdownItem, { title: 'group' }),
                                React.createElement(DropdownItem, { title: 'overlay' }),
                                React.createElement(DropdownItem, { title: 'relative' }))),
                    GRAPHS_THAT_HAVE_BARNORM.includes(props.graphParams.graphCreation.graph_type) &&
                        React.createElement(Row, { justify: 'space-between', align: 'center', title: "Normalize strategy used for each group of bars at a specific location on the graph's domain" },
                            React.createElement(Col, null,
                                React.createElement(Row, { justify: 'space-between', align: 'center', suppressTopBottomMargin: true },
                                    React.createElement("p", null, "Bar normalization"),
                                    React.createElement(Tooltip, { title: "Normalize strategy used for each group of bars at a specific location on the graph's domain" }))),
                            React.createElement(Select, { value: props.graphParams.graphStyling.barnorm || 'none', onChange: (newBarNorm) => {
                                    updateGraphParam({ graphStyling: { barnorm: newBarNorm } });
                                }, width: 'small', dropdownWidth: 'medium' },
                                React.createElement(DropdownItem, { title: 'none' }),
                                React.createElement(DropdownItem, { title: 'fraction', subtext: 'value of each bar divided by the sum of all values at that location' }),
                                React.createElement(DropdownItem, { title: 'percent', subtext: 'fraction multiplied by 100' }))),
                    GRAPHS_THAT_HAVE_HISTNORM.includes(props.graphParams.graphCreation.graph_type) &&
                        React.createElement(Row, { justify: 'space-between', align: 'center', title: 'Normalization strategy used for each graphed series in the histogram' },
                            React.createElement(Col, null,
                                React.createElement(Row, { justify: 'space-between', align: 'center', suppressTopBottomMargin: true },
                                    React.createElement("p", null, "Hist normalization"),
                                    React.createElement(Tooltip, { title: 'Normalization strategy used for each graphed series in the histogram' }))),
                            React.createElement(Select, { value: props.graphParams.graphCreation.histnorm || 'none', onChange: (newHistnorm) => {
                                    updateGraphParam({ graphCreation: { histnorm: newHistnorm } });
                                }, width: 'small', dropdownWidth: 'medium' },
                                React.createElement(DropdownItem, { title: 'none' }),
                                React.createElement(DropdownItem, { title: 'probability', subtext: 'occurrences in bin divided by total number of sample points' }),
                                React.createElement(DropdownItem, { title: 'percent', subtext: 'probabilty multiplied by 100' }),
                                React.createElement(DropdownItem, { title: 'density', subtext: 'occurences in bin divided by bin interval' }),
                                React.createElement(DropdownItem, { title: 'probability density', subtext: 'probability that a point falls into bin' }))),
                    GRAPHS_THAT_HAVE_HISTFUNC.includes(props.graphParams.graphCreation.graph_type) &&
                        React.createElement(Row, { justify: 'space-between', align: 'center', title: 'The metric displayed for each bin of data' },
                            React.createElement(Col, null,
                                React.createElement(Row, { justify: 'space-between', align: 'center', suppressTopBottomMargin: true },
                                    React.createElement("p", null, "Hist function"),
                                    React.createElement(Tooltip, { title: 'The metric displayed for each bin of data' }))),
                            React.createElement(Select, { value: props.graphParams.graphCreation.histfunc || 'count', onChange: (newHistfunc) => {
                                    updateGraphParam({ graphCreation: { histfunc: newHistfunc } });
                                }, width: 'small', dropdownWidth: 'medium' },
                                React.createElement(DropdownItem, { title: 'count', subtext: 'number of values in each bin' }),
                                React.createElement(DropdownItem, { title: 'sum', subtext: 'sum of values in each bin' }),
                                React.createElement(DropdownItem, { title: 'avg', subtext: 'average value in each bin' }),
                                React.createElement(DropdownItem, { title: 'min', subtext: 'min value in each bin' }),
                                React.createElement(DropdownItem, { title: 'max', subtext: 'max value in each bin' }))),
                    GRAPHS_THAT_HAVE_POINTS.includes(props.graphParams.graphCreation.graph_type) &&
                        React.createElement(Row, { justify: 'space-between', align: 'center', title: 'Display outlier points' },
                            React.createElement(Col, null,
                                React.createElement("p", null, "Points")),
                            React.createElement(Select, { value: props.graphParams.graphCreation.points === false ? 'none' : props.graphParams.graphCreation.points !== undefined ? props.graphParams.graphCreation.points : '', onChange: (newPointsString) => {
                                    const newPointsParams = newPointsString === 'false' ? false : newPointsString;
                                    updateGraphParam({ graphCreation: { points: newPointsParams } });
                                }, width: 'small', dropdownWidth: 'medium' },
                                React.createElement(DropdownItem, { title: 'outliers', subtext: 'only display sample points outside the whiskers' }),
                                React.createElement(DropdownItem, { title: 'supsected outliers', id: 'suspectedoutliers', subtext: 'display outlier and suspected outlier points' }),
                                React.createElement(DropdownItem, { title: 'all', subtext: 'display all sample points' }),
                                React.createElement(DropdownItem, { title: 'none', id: 'false', subtext: 'display no individual sample points' }))),
                    GRAPHS_THAT_HAVE_LINE_SHAPE.includes(props.graphParams.graphCreation.graph_type) &&
                        React.createElement(Row, { justify: 'space-between', align: 'center', title: 'The shape of the line' },
                            React.createElement(Col, null,
                                React.createElement("p", null, "Line shape")),
                            React.createElement(Select, { value: props.graphParams.graphCreation.line_shape || 'linear', onChange: (newLineShape) => {
                                    updateGraphParam({ graphCreation: { line_shape: newLineShape } });
                                }, width: 'small', dropdownWidth: 'medium' },
                                React.createElement(DropdownItem, { title: 'linear', subtext: 'straight line between points' }),
                                React.createElement(DropdownItem, { title: 'spline', subtext: 'spline interpolation between points' }),
                                React.createElement(DropdownItem, { title: 'hv', subtext: 'horizontal vertical' }),
                                React.createElement(DropdownItem, { title: 'vh', subtext: 'veritical horizontal' }),
                                React.createElement(DropdownItem, { title: 'hvh', subtext: 'horizontal vertical horizontal' }),
                                React.createElement(DropdownItem, { title: 'vhv', subtext: 'vertical horizontal vertical' })))),
            React.createElement(CollapsibleSection, { title: 'Facet plots' },
                React.createElement("div", null,
                    React.createElement(Row, { justify: 'space-between', align: 'center', title: "Create subplots based on this attribute" },
                        React.createElement(Col, null,
                            React.createElement(Row, { justify: 'space-between', align: 'center', suppressTopBottomMargin: true },
                                React.createElement("p", null, "Facet Column \u00A0"),
                                React.createElement(Tooltip, { title: "Create subplots based on this attribute" }))),
                        React.createElement(Col, null,
                            React.createElement(Select, { value: props.graphParams.graphCreation.facet_col_column_id ? getDisplayColumnHeader(columnIDsMap[props.graphParams.graphCreation.facet_col_column_id]) : 'None', width: 'small', searchable: true }, [React.createElement(DropdownItem, { key: 'None', title: 'None', onClick: () => {
                                        updateGraphParam({ graphCreation: { facet_col_column_id: undefined } });
                                    } })].concat((Object.keys(columnIDsMap) || []).map(columnID => {
                                const columnHeader = columnIDsMap[columnID];
                                return (React.createElement(DropdownItem, { key: columnID, title: getDisplayColumnHeader(columnHeader), onClick: () => {
                                        updateGraphParam({ graphCreation: { facet_col_column_id: columnID } });
                                    } }));
                            })))))),
                React.createElement("div", null,
                    React.createElement(Row, { justify: 'space-between', align: 'center', title: "Create subplots based on this attribute" },
                        React.createElement(Col, null,
                            React.createElement(Row, { justify: 'space-between', align: 'center', suppressTopBottomMargin: true },
                                React.createElement("p", null, "Facet Row \u00A0"),
                                React.createElement(Tooltip, { title: "Create subplots based on this attribute" }))),
                        React.createElement(Col, null,
                            React.createElement(Select, { value: props.graphParams.graphCreation.facet_row_column_id ? getDisplayColumnHeader(columnIDsMap[props.graphParams.graphCreation.facet_row_column_id]) : 'None', width: 'small', searchable: true }, [React.createElement(DropdownItem, { key: 'None', title: 'None', onClick: () => {
                                        updateGraphParam({ graphCreation: { facet_row_column_id: undefined } });
                                    } })].concat((Object.keys(columnIDsMap) || []).map(columnID => {
                                const columnHeader = columnIDsMap[columnID];
                                return (React.createElement(DropdownItem, { key: columnID, title: getDisplayColumnHeader(columnHeader), onClick: () => {
                                        updateGraphParam({ graphCreation: { facet_row_column_id: columnID } });
                                    } }));
                            }))))))))));
}
export default GraphSetupTab;
//# sourceMappingURL=GraphSetupTab.js.map