#!/bin/bash
# Helper script created by Protontricks to run Wine binaries using Steam Runtime
set -o errexit

PROTONTRICKS_PROXY_SCRIPT_PATH="@@script_path@@"

BLACKLISTED_ROOT_DIRS=(
    /bin /dev /lib /lib64 /proc /run /sys /var /usr
)

ADDITIONAL_MOUNT_DIRS=(
    /run/media "$PROTON_PATH" "$WINEPREFIX"
)

WINESERVER_ENV_VARS_TO_COPY=(
    WINEESYNC WINEFSYNC
)

if [[ -n "$PROTONTRICKS_BACKGROUND_WINESERVER"
       && "$PROTONTRICKS_STEAM_RUNTIME" = "bwrap"
       && "$0" = "@@script_path@@"
    ]]; then
    # Check if we're calling 'wineserver -w' inside a bwrap sandbox.
    # If so, prompt our keepalive wineserver to restart itself by creating
    # a 'restart' file inside the temporary directory
    if [[ "$(basename "$0")" = "wineserver"
        && "$1" = "-w"
        ]]; then
        touch "$PROTONTRICKS_TEMP_PATH/restart"
    fi
fi

if [[ -z "$PROTONTRICKS_FIRST_START" ]]; then
    # Try to detect if wineserver is already running, and if so, copy a few
    # environment variables from it to ensure our own Wine processes
    # are able to run at the same time without any issues.
    # This usually happens when the user is running the Steam app and
    # Protontricks at the same time.
    wineserver_found=false

    # Find the correct Wineserver that's using the same prefix
    while read -r pid; do
        if [[ $(xargs -0 -L1 -a "/proc/${pid}/environ" | grep "^WINEPREFIX=${WINEPREFIX}") ]] &> /dev/null; then
            if [[ "$pid" = "$$" ]]; then
                # Don't mistake this very script for a wineserver instance
                continue
            fi
            wineserver_found=true
            wineserver_pid="$pid"
        fi
    done < <(pgrep "wineserver$")

    if [[ "$wineserver_found" = true ]]; then
        # wineserver found, retrieve its environment variables
        wineserver_env_vars=$(xargs -0 -L1 -a "/proc/${wineserver_pid}/environ")

        # Copy the required environment variables found in the
        # existing wineserver process
        for env_name in "${WINESERVER_ENV_VARS_TO_COPY[@]}"; do
            env_declr=$(echo "$wineserver_env_vars" | grep "^${env_name}=" || :)
            if [[ -n "$env_declr" ]]; then
                export "${env_declr?}"
            fi
        done
    fi

    # Enable fsync & esync by default
    if [[ "$wineserver_found" = false ]]; then
        if [[ -z "$WINEFSYNC" ]]; then
            if [[ -z "$PROTON_NO_FSYNC" || "$PROTON_NO_FSYNC" = "0" ]]; then
                export WINEFSYNC=1
            fi
        fi

        if [[ -z "$WINEESYNC" ]]; then
            if [[ -z "$PROTON_NO_ESYNC" || "$PROTON_NO_ESYNC" = "0" ]]; then
                export WINEESYNC=1
            fi
        fi
    fi

    export PROTONTRICKS_FIRST_START=1
fi

# PROTONTRICKS_STEAM_RUNTIME values:
#   bwrap:  Run Wine binaries inside Steam Runtime's bwrap sandbox,
#           modify LD_LIBRARY_PATH to include Proton libraries
#
#   legacy: Modify LD_LIBRARY_PATH to include Steam Runtime *and* Proton
#           libraries. Host library order is adjusted as well.
#
#   off:    Just run the binaries as-is.
if [[ -n "$PROTONTRICKS_INSIDE_STEAM_RUNTIME"
       || "$PROTONTRICKS_STEAM_RUNTIME" = "legacy"
       || "$PROTONTRICKS_STEAM_RUNTIME" = "off"
    ]]; then

    # If either Steam Runtime is enabled, change LD_LIBRARY_PATH
    if [[ "$PROTONTRICKS_STEAM_RUNTIME" = "bwrap" ]]; then
        export LD_LIBRARY_PATH="$LD_LIBRARY_PATH":"$PROTON_LD_LIBRARY_PATH"
    elif [[ "$PROTONTRICKS_STEAM_RUNTIME" = "legacy" ]]; then
        export LD_LIBRARY_PATH="$PROTON_LD_LIBRARY_PATH"
    fi
    exec "$PROTON_DIST_PATH"/bin/@@name@@ "$@" || :
elif [[ "$PROTONTRICKS_STEAM_RUNTIME" = "bwrap" ]]; then
    # Command is being executed outside Steam Runtime and bwrap is enabled.
    # Use "pressure-vessel-launch" to launch it in the existing container.

    # Wait until socket is created
    while [[ ! -S "$PROTONTRICKS_TEMP_PATH/launcher.sock" ]]; do
        sleep 0.25
    done

    exec "$STEAM_RUNTIME_PATH"/pressure-vessel/bin/pressure-vessel-launch \
    --share-pids --socket="$PROTONTRICKS_TEMP_PATH/launcher.sock" \
    --env=PROTONTRICKS_INSIDE_STEAM_RUNTIME=1 \
    --pass-env-matching="*" -- "$PROTONTRICKS_PROXY_SCRIPT_PATH" "$@"
else
    echo "Unknown PROTONTRICKS_STEAM_RUNTIME value $PROTONTRICKS_STEAM_RUNTIME"
    exit 1
fi

