"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Canary = exports.Test = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const s3 = require("aws-cdk-lib/aws-s3");
const cdk = require("aws-cdk-lib");
const synthetics_canned_metrics_generated_1 = require("./synthetics-canned-metrics.generated");
const aws_synthetics_1 = require("aws-cdk-lib/aws-synthetics");
/**
 * Specify a test that the canary should run
 */
class Test {
    /**
     * Construct a Test property
     *
     * @param code The code that the canary should run
     * @param handler The handler of the canary
     */
    constructor(code, handler) {
        this.code = code;
        this.handler = handler;
    }
    /**
     * Specify a custom test with your own code
     *
     * @returns `Test` associated with the specified Code object
     * @param options The configuration options
     */
    static custom(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_synthetics_alpha_CustomTestOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.custom);
            }
            throw error;
        }
        Test.validateHandler(options.handler);
        return new Test(options.code, options.handler);
    }
    /**
     * Verifies that the given handler ends in '.handler'. Returns the handler if successful and
     * throws an error if not.
     *
     * @param handler - the handler given by the user
     */
    static validateHandler(handler) {
        if (!handler.endsWith('.handler')) {
            throw new Error(`Canary Handler must end in '.handler' (${handler})`);
        }
        if (handler.length > 21) {
            throw new Error(`Canary Handler must be less than 21 characters (${handler})`);
        }
    }
}
exports.Test = Test;
_a = JSII_RTTI_SYMBOL_1;
Test[_a] = { fqn: "@aws-cdk/aws-synthetics-alpha.Test", version: "2.45.0-alpha.0" };
/**
 * Define a new Canary
 */
class Canary extends cdk.Resource {
    constructor(scope, id, props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_synthetics_alpha_CanaryProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Canary);
            }
            throw error;
        }
        if (props.canaryName && !cdk.Token.isUnresolved(props.canaryName)) {
            validateName(props.canaryName);
        }
        super(scope, id, {
            physicalName: props.canaryName || cdk.Lazy.string({
                produce: () => this.generateUniqueName(),
            }),
        });
        this.artifactsBucket = props.artifactsBucketLocation?.bucket ?? new s3.Bucket(this, 'ArtifactsBucket', {
            encryption: s3.BucketEncryption.KMS_MANAGED,
            enforceSSL: true,
        });
        this.role = props.role ?? this.createDefaultRole(props);
        if (props.vpc) {
            // Security Groups are created and/or appended in `createVpcConfig`.
            this._connections = new ec2.Connections({});
        }
        const resource = new aws_synthetics_1.CfnCanary(this, 'Resource', {
            artifactS3Location: this.artifactsBucket.s3UrlForObject(props.artifactsBucketLocation?.prefix),
            executionRoleArn: this.role.roleArn,
            startCanaryAfterCreation: props.startAfterCreation ?? true,
            runtimeVersion: props.runtime.name,
            name: this.physicalName,
            schedule: this.createSchedule(props),
            failureRetentionPeriod: props.failureRetentionPeriod?.toDays(),
            successRetentionPeriod: props.successRetentionPeriod?.toDays(),
            code: this.createCode(props),
            runConfig: this.createRunConfig(props),
            vpcConfig: this.createVpcConfig(props),
        });
        this.canaryId = resource.attrId;
        this.canaryState = resource.attrState;
        this.canaryName = this.getResourceNameAttribute(resource.ref);
    }
    /**
     * Access the Connections object
     *
     * Will fail if not a VPC-enabled Canary
     */
    get connections() {
        if (!this._connections) {
            // eslint-disable-next-line max-len
            throw new Error('Only VPC-associated Canaries have security groups to manage. Supply the "vpc" parameter when creating the Canary.');
        }
        return this._connections;
    }
    /**
     * Measure the Duration of a single canary run, in seconds.
     *
     * @param options - configuration options for the metric
     *
     * @default avg over 5 minutes
     */
    metricDuration(options) {
        return new aws_cloudwatch_1.Metric({
            ...synthetics_canned_metrics_generated_1.CloudWatchSyntheticsMetrics.durationMaximum({ CanaryName: this.canaryName }),
            ...{ statistic: 'Average' },
            ...options,
        }).attachTo(this);
    }
    /**
     * Measure the percentage of successful canary runs.
     *
     * @param options - configuration options for the metric
     *
     * @default avg over 5 minutes
     */
    metricSuccessPercent(options) {
        return this.cannedMetric(synthetics_canned_metrics_generated_1.CloudWatchSyntheticsMetrics.successPercentAverage, options);
    }
    /**
     * Measure the number of failed canary runs over a given time period.
     *
     * Default: sum over 5 minutes
     *
     * @param options - configuration options for the metric
     */
    metricFailed(options) {
        return this.cannedMetric(synthetics_canned_metrics_generated_1.CloudWatchSyntheticsMetrics.failedSum, options);
    }
    /**
     * Returns a default role for the canary
     */
    createDefaultRole(props) {
        const prefix = props.artifactsBucketLocation?.prefix;
        // Created role will need these policies to run the Canary.
        // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-synthetics-canary.html#cfn-synthetics-canary-executionrolearn
        const policy = new iam.PolicyDocument({
            statements: [
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: ['s3:ListAllMyBuckets'],
                }),
                new iam.PolicyStatement({
                    resources: [this.artifactsBucket.bucketArn],
                    actions: ['s3:GetBucketLocation'],
                }),
                new iam.PolicyStatement({
                    resources: [this.artifactsBucket.arnForObjects(`${prefix ? prefix + '/*' : '*'}`)],
                    actions: ['s3:PutObject'],
                }),
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: ['cloudwatch:PutMetricData'],
                    conditions: { StringEquals: { 'cloudwatch:namespace': 'CloudWatchSynthetics' } },
                }),
                new iam.PolicyStatement({
                    resources: [this.logGroupArn()],
                    actions: ['logs:CreateLogStream', 'logs:CreateLogGroup', 'logs:PutLogEvents'],
                }),
            ],
        });
        const managedPolicies = [];
        if (props.vpc) {
            // Policy that will have ENI creation permissions
            managedPolicies.push(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaVPCAccessExecutionRole'));
        }
        return new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                canaryPolicy: policy,
            },
            managedPolicies,
        });
    }
    logGroupArn() {
        return cdk.Stack.of(this).formatArn({
            service: 'logs',
            resource: 'log-group',
            arnFormat: cdk.ArnFormat.COLON_RESOURCE_NAME,
            resourceName: '/aws/lambda/cwsyn-*',
        });
    }
    /**
     * Returns the code object taken in by the canary resource.
     */
    createCode(props) {
        const codeConfig = {
            handler: props.test.handler,
            ...props.test.code.bind(this, props.test.handler, props.runtime.family),
        };
        return {
            handler: codeConfig.handler,
            script: codeConfig.inlineCode,
            s3Bucket: codeConfig.s3Location?.bucketName,
            s3Key: codeConfig.s3Location?.objectKey,
            s3ObjectVersion: codeConfig.s3Location?.objectVersion,
        };
    }
    createRunConfig(props) {
        if (!props.environmentVariables) {
            return undefined;
        }
        return {
            environmentVariables: props.environmentVariables,
        };
    }
    /**
     * Returns a canary schedule object
     */
    createSchedule(props) {
        return {
            durationInSeconds: String(`${props.timeToLive?.toSeconds() ?? 0}`),
            expression: props.schedule?.expressionString ?? 'rate(5 minutes)',
        };
    }
    createVpcConfig(props) {
        if (!props.vpc) {
            if (props.vpcSubnets != null || props.securityGroups != null) {
                throw new Error("You must provide the 'vpc' prop when using VPC-related properties.");
            }
            return undefined;
        }
        const { subnetIds } = props.vpc.selectSubnets(props.vpcSubnets);
        if (subnetIds.length < 1) {
            throw new Error('No matching subnets found in the VPC.');
        }
        let securityGroups;
        if (props.securityGroups && props.securityGroups.length > 0) {
            securityGroups = props.securityGroups;
        }
        else {
            const securityGroup = new ec2.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
                description: 'Automatic security group for Canary ' + cdk.Names.uniqueId(this),
            });
            securityGroups = [securityGroup];
        }
        this._connections.addSecurityGroup(...securityGroups);
        return {
            vpcId: props.vpc.vpcId,
            subnetIds,
            securityGroupIds: cdk.Lazy.list({ produce: () => this.connections.securityGroups.map(sg => sg.securityGroupId) }),
        };
    }
    /**
     * Creates a unique name for the canary. The generated name is the physical ID of the canary.
     */
    generateUniqueName() {
        const name = cdk.Names.uniqueId(this).toLowerCase().replace(' ', '-');
        if (name.length <= 21) {
            return name;
        }
        else {
            return name.substring(0, 15) + nameHash(name);
        }
    }
    cannedMetric(fn, props) {
        return new aws_cloudwatch_1.Metric({
            ...fn({ CanaryName: this.canaryName }),
            ...props,
        }).attachTo(this);
    }
}
exports.Canary = Canary;
_b = JSII_RTTI_SYMBOL_1;
Canary[_b] = { fqn: "@aws-cdk/aws-synthetics-alpha.Canary", version: "2.45.0-alpha.0" };
/**
 * Take a hash of the given name.
 *
 * @param name the name to be hashed
 */
function nameHash(name) {
    const md5 = crypto.createHash('sha256').update(name).digest('hex');
    return md5.slice(0, 6);
}
const nameRegex = /^[0-9a-z_\-]+$/;
/**
 * Verifies that the name fits the regex expression: ^[0-9a-z_\-]+$.
 *
 * @param name - the given name of the canary
 */
function validateName(name) {
    if (name.length > 21) {
        throw new Error(`Canary name is too large, must be between 1 and 21 characters, but is ${name.length} (got "${name}")`);
    }
    if (!nameRegex.test(name)) {
        throw new Error(`Canary name must be lowercase, numbers, hyphens, or underscores (got "${name}")`);
    }
}
//# sourceMappingURL=data:application/json;base64,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