import psycopg

from protgraph.export.peptides.pep_postgres import PepPostgres


class PepCitus(PepPostgres):
    """
    A PostGreSQL-Citus - Exporter to export PEPTIDES
    into distributed peptides table

    Those tables will contain all output generated by
    each of the processes. Keep in mind that this table can
    be extremely large, depending on the parmeters set in this tool.

    NOTE: Maybe even exceeding trillion of results for one protein!
    """

    def start_up(self, **kwargs):
        super(PepCitus, self).start_up(**kwargs)
        # Here we generate a connection to postgres
        # and generate the corresponding tables

        # TODO psycopg3 changed the bulk inserts in such a way
        # that no nore then 65535 parameters can be sent at once.
        # We therefore need to limit the number of params here!
        self.pg_max_allowed_entries_to_send = int(65535/29)  # --> deduced by the length of l_peptide_tuple
        self.pg_max_allowed_entries_to_send_compressed = int(65535/30)  # --> deduced by the length of l_peptide_tuple
        self.pg_max_allowed_entries_to_send_meta = int(65535/4)  # --> deduced by the length of l_peptide_tuple

        # Connection and other parameters
        self.host = kwargs["pep_citus_host"]  # Host
        self.port = kwargs["pep_citus_port"]  # Port
        self.user = kwargs["pep_citus_user"]  # User
        self.password = kwargs["pep_citus_password"]  # Password
        self.database = kwargs["pep_citus_database"]  # Database
        self.postgres_no_duplicates = kwargs["pep_citus_no_duplicates"]

        # Initialize connection
        try:
            self.conn = psycopg.connect(
                host=self.host,
                port=self.port,
                user=self.user,
                password=self.password,
                dbname=self.database
            )
            # Set a cursor
            self.cursor = self.conn.cursor(binary=True)
        except Exception as e:
            raise Exception("Could not establish a connection to Postgres Citus (Peptides).", e)

        # Create tables if they not exist
        try:
            self._create_tables(**kwargs)
        except Exception as e:
            raise Exception("Could not create tables in Postgres Citus (Peptides).", e)

    def _create_tables(self, **kwargs):
        """ Create the accessions and peptides and peptides_meta tables """
        try:
            # create accessions, so that we only save numbers in the large table!
            cur = self.conn.cursor()
            cur.execute("""
                create table if not exists accessions (
                    id SERIAl PRIMARY KEY,
                    accession VARCHAR(15) NOT NULL
                );""")
        except Exception as e:
            print("Warning: Failed creating table 'accessions' (Reason: {})".format(str(e)))
        finally:
            self.conn.commit()
            cur.close()

        try:
            # Create the large peptides table containing most information
            cur = self.conn.cursor()
            cur.execute("""
            CREATE TABLE  if not exists peptides (
                id {0},
                weight {1} NOT NULL,
                a_count SMALLINT NOT NULL,
                b_count SMALLINT NOT NULL,
                c_count SMALLINT NOT NULL,
                d_count SMALLINT NOT NULL,
                e_count SMALLINT NOT NULL,
                f_count SMALLINT NOT NULL,
                g_count SMALLINT NOT NULL,
                h_count SMALLINT NOT NULL,
                i_count SMALLINT NOT NULL,
                j_count SMALLINT NOT NULL,
                k_count SMALLINT NOT NULL,
                l_count SMALLINT NOT NULL,
                m_count SMALLINT NOT NULL,
                n_count SMALLINT NOT NULL,
                o_count SMALLINT NOT NULL,
                p_count SMALLINT NOT NULL,
                q_count SMALLINT NOT NULL,
                r_count SMALLINT NOT NULL,
                s_count SMALLINT NOT NULL,
                t_count SMALLINT NOT NULL,
                u_count SMALLINT NOT NULL,
                v_count SMALLINT NOT NULL,
                w_count SMALLINT NOT NULL,
                x_count SMALLINT NOT NULL,
                y_count SMALLINT NOT NULL,
                z_count SMALLINT NOT NULL,
                n_terminus character(1) NOT NULL,
                c_terminus character(1) NOT NULL,
                PRIMARY KEY (id));""".format(
                "BIT(452)" if self.postgres_no_duplicates else "BIGSERIAL",
                "BIGINT" if kwargs["mass_dict_type"] is int else "DOUBLE PRECISION"
                ))
            cur.execute("SELECT create_distributed_table(%s, %s);", ("peptides", "id", ))
        except Exception as e:
            print("Warning: Failed creating table 'peptides' (Reason: {})".format(str(e)))
        finally:
            self.conn.commit()
            cur.close()
            self.peptides_keys = [
                "weight",
                "a_count", "b_count", "c_count", "d_count", "e_count", "f_count", "g_count", "h_count",
                "i_count", "j_count", "k_count", "l_count", "m_count", "n_count", "o_count", "p_count",
                "q_count", "r_count", "s_count", "t_count", "u_count", "v_count", "w_count", "x_count",
                "y_count", "z_count", "n_terminus", "c_terminus"
            ]
            if self.postgres_no_duplicates:
                self.peptides_keys = ["id"] + self.peptides_keys

        try:
            # Create the peptides meta information (can also be extremely large), larger than the peptides tables
            cur = self.conn.cursor()
            cur.execute("""
            CREATE TABLE  if not exists peptides_meta (
                id BIGSERIAL,
                peptides_id {0},
                accession_id INT,
                path INT[] NOT NULL,
                miscleavages INT NOT NULL,
                PRIMARY KEY (id, peptides_id)
            );""".format("BIT(452)" if self.postgres_no_duplicates else "BIGINT"))
            # References to peptide and accession removed for performance reasons
            cur.execute("SELECT create_distributed_table(%s, %s);", ("peptides_meta", "peptides_id", ))
        except Exception as e:
            print("Warning: Failed creating table 'peptides_meta' (Reason: {})".format(str(e)))
        finally:
            self.conn.commit()
            cur.close()
            self.peptides_meta_keys = [
                "peptides_id",
                "accession_id",
                "path",
                "miscleavages"
            ]

        # Set statements for inserting or selecting
        self.statement_accession = "INSERT INTO accessions(accession) VALUES (%s) RETURNING id;"
        self.statement_peptides_inner_values = "(" + ",".join(["%s"]*len(self.peptides_keys)) + ")"
        self.statement_peptides_meta_inner_values = "(" + ",".join(["%s"]*len(self.peptides_meta_keys)) + ")"

    def tear_down(self):
        # Close the connection to postgres citus
        try:
            self.cursor.close()  # Close cursor
            self.conn.close()  # Close connection
        except Exception as e:
            print("Connection to PostgreSQL Citus could not be closed. (Reason: {})".format(str(e)))
