---
jupyter:
  jupytext:
    formats: ipynb,md
    text_representation:
      extension: .md
      format_name: markdown
      format_version: '1.3'
      jupytext_version: 1.13.8
  kernelspec:
    display_name: nanomesh
    language: python
    name: nanomesh
---

```python
%config InlineBackend.rc = {'figure.figsize': (10,6)}
%matplotlib inline
```

## Hello world!

<!-- #raw raw_mimetype="text/restructuredtext" -->
This example shows the workflow for generating a mesh from segmented data, and demonstrates a few of the features of Nanomesh. It uses a synthetic binary image with several rounded blob-like objects generated by :mod:`skimage`.

Image data in Nanomesh is stored as an :class:`nanomesh.Image` type. Calling :class:`nanomesh.Image` will create the appropriate subtype, :class:`nanomesh.Plane` for 2D or :class:`nanomesh.Volume` for 3D data.
<!-- #endraw -->

```python
from skimage.data import binary_blobs
from nanomesh import Image

blobs = binary_blobs(length=100, volume_fraction=0.25, seed=2102)
plane = Image(blobs)

print(plane)
```

<!-- #raw raw_mimetype="text/restructuredtext" -->
:class:`nanomesh.Image` is essentially a container for a :mod:`numpy` array with some methods for image segmentation and visualization.
<!-- #endraw -->

```python
plane.show()
```

<!-- #raw raw_mimetype="text/restructuredtext" -->
Generating a mesh from image data is simple in Nanomesh using :meth:`nanomesh.Plane.generate_mesh`. The options `opts` are passed to the triangulation function (:func:`nanomesh.triangulate`). In this example, we use `q30` to generate a quality mesh with minimum angles of 30°, and `a50` to limit the triangle size to 50 pixels.

The returned `mesh` is a :class:`nanomesh.MeshContainer` with the generated triangles and line segments.
<!-- #endraw -->

```python
mesh = plane.generate_mesh(opts='q30a10')
mesh
```

In the next cell, we plot the triangles.

```python
mesh.plot('triangle')
```

<!-- #raw raw_mimetype="text/restructuredtext" -->
Nanomesh can also calculate cell quality metrics and show them as a colored triangle or histogram plot.

Note that the area shaded in green below highlights the *optimal* range.

Have a look at the :mod:`nanomesh.metrics` submodule or in the example: :doc:`nanopores_calculate_mesh_quality_metrics`.
<!-- #endraw -->

```python
from nanomesh import metrics

triangle_mesh = mesh.get('triangle')

metrics.histogram(triangle_mesh, metric='radius_ratio')
```

<!-- #raw raw_mimetype="text/restructuredtext" -->
Nanomesh uses [meshio](https://github.com/nschloe/meshio) to write data to most meshing formats.
<!-- #endraw -->

```python
mesh.write('mesh.vtk')
```

<!-- #raw raw_mimetype="text/restructuredtext" -->
For more practical examples of how to use Nanomesh, check out the how-to guides and notebooks: :ref:`examples`.
<!-- #endraw -->
