---
jupyter:
  jupytext:
    formats: ipynb,md
    text_representation:
      extension: .md
      format_name: markdown
      format_version: '1.3'
      jupytext_version: 1.13.8
  kernelspec:
    display_name: nanomesh
    language: python
    name: nanomesh
---

```python
%config InlineBackend.rc = {'figure.figsize': (10,6)}
%matplotlib inline
```

## Poisson equation with getfem

This example solves the Poisson problem using [getfem](https://getfem.org/) using data generated by Nanomesh. We solve the Poisson problem $-\Delta u = 1$ with the boundary condition $u=0$.

This an adaptation of [Python getfem demo](https://getfem-examples.readthedocs.io/en/latest/demo_unit_disk.html).

### Setup getfem

First, we must setup the path to the python module ([link](https://getfem.org/python/pygf.html#introduction)), so that getfem can be used in our Nanomesh environment.

We import getfem and generate a mesh to test if it works.

```python
import sys
sys.path.append('../../../getfem/interface/src/python/')

import getfem
m = getfem.Mesh('cartesian', range(0, 3), range(0,3))
print(m)
```

### Generate some data

We use the 2D binary blobs data to generate a triangle mesh. The triangles that belong to the blobs are removed.

```python
from skimage.morphology import disk
from nanomesh.data import binary_blobs2d

data = binary_blobs2d(length=100, seed=96)

from nanomesh import Mesher

mesher = Mesher(data)
mesher.generate_contour(max_edge_dist=3, precision=1)
mesh = mesher.triangulate(opts='q30a25')

triangles = mesh.get('triangle')
triangles.remove_cells(label=2, key='physical')

triangles.plot()
```

### Convert to getfem mesh type

We use the *2D triangulation* [mesh type](https://getfem-examples.readthedocs.io/en/latest/ball_eigen.html?highlight=mesh#Mesh-generation) described by passing the argument `pt2D`. Note that the points and cells arrays must be transposed.


```python
import getfem as gf

p = triangles.points.T
t = triangles.cells.T

mesh = gf.Mesh('pt2D', p, t)
mesh
```

### Poisson's equation

The next cell shows how to solve the Poisson equation. This code was re-used from [here](https://getfem-examples.readthedocs.io/en/latest/demo_unit_disk.html).

```python
import getfem as gf
import numpy as np

OUTER_BOUND = 1

outer_faces = mesh.outer_faces()
mesh.set_region(OUTER_BOUND, outer_faces)

sl = gf.Slice(("none",), mesh, 1)

elements_degree = 2

mfu = gf.MeshFem(mesh, 1)
mfu.set_classical_fem(elements_degree)

mim = gf.MeshIm(mesh, pow(elements_degree, 2))

F = 1.0

md = gf.Model("real")

md.add_fem_variable("u", mfu)
md.add_Laplacian_brick(mim, "u")
md.add_fem_data("F", mfu)
md.set_variable("F", np.repeat(F, mfu.nbdof()))
md.add_source_term_brick(mim, "u", "F")
md.add_Dirichlet_condition_with_multipliers(mim, "u", elements_degree - 1, OUTER_BOUND)

md.solve()
```

### Display result using PyVista

The data can be visualized by saving to a vtk file, and loading that with [PyVista](https://docs.pyvista.org/).

```python
import pyvista as pv

U = md.variable("u")
sl.export_to_vtk("u.vtk", "ascii", mfu, U, "U")

m = pv.read("u.vtk")
contours = m.contour()

p = pv.Plotter()
p.add_mesh(m, show_edges=False)
p.add_mesh(contours, color="black", line_width=1)
p.add_mesh(m.contour(8).extract_largest(), opacity=0.1)
p.show(cpos="xy", jupyter_backend='static')
```
