import * as s3 from 'aws-cdk-lib/aws-s3';
import * as s3_assets from 'aws-cdk-lib/aws-s3-assets';
import { Construct } from 'constructs';
import { RuntimeFamily } from './runtime';
/**
 * (experimental) The code the canary should execute.
 *
 * @experimental
 */
export declare abstract class Code {
    /**
     * (experimental) Specify code inline.
     *
     * @param code The actual handler code (limited to 4KiB).
     * @returns `InlineCode` with inline code.
     * @experimental
     */
    static fromInline(code: string): InlineCode;
    /**
     * (experimental) Specify code from a local path.
     *
     * Path must include the folder structure `nodejs/node_modules/myCanaryFilename.js`.
     *
     * @param assetPath Either a directory or a .zip file.
     * @returns `AssetCode` associated with the specified path.
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html#CloudWatch_Synthetics_Canaries_write_from_scratch
     * @experimental
     */
    static fromAsset(assetPath: string, options?: s3_assets.AssetOptions): AssetCode;
    /**
     * (experimental) Specify code from an s3 bucket.
     *
     * The object in the s3 bucket must be a .zip file that contains
     * the structure `nodejs/node_modules/myCanaryFilename.js`.
     *
     * @param bucket The S3 bucket.
     * @param key The object key.
     * @param objectVersion Optional S3 object version.
     * @returns `S3Code` associated with the specified S3 object.
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html#CloudWatch_Synthetics_Canaries_write_from_scratch
     * @experimental
     */
    static fromBucket(bucket: s3.IBucket, key: string, objectVersion?: string): S3Code;
    /**
     * (experimental) Called when the canary is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @param scope The binding scope.
     * @returns a bound `CodeConfig`.
     * @experimental
     */
    abstract bind(scope: Construct, handler: string, family: RuntimeFamily): CodeConfig;
}
/**
 * (experimental) Configuration of the code class.
 *
 * @experimental
 */
export interface CodeConfig {
    /**
     * (experimental) The location of the code in S3 (mutually exclusive with `inlineCode`).
     *
     * @default - none
     * @experimental
     */
    readonly s3Location?: s3.Location;
    /**
     * (experimental) Inline code (mutually exclusive with `s3Location`).
     *
     * @default - none
     * @experimental
     */
    readonly inlineCode?: string;
}
/**
 * (experimental) Canary code from an Asset.
 *
 * @experimental
 */
export declare class AssetCode extends Code {
    private assetPath;
    private options?;
    private asset?;
    /**
     * @param assetPath The path to the asset file or directory.
     * @experimental
     */
    constructor(assetPath: string, options?: s3_assets.AssetOptions | undefined);
    /**
     * (experimental) Called when the canary is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @experimental
     */
    bind(scope: Construct, handler: string, family: RuntimeFamily): CodeConfig;
    /**
     * Validates requirements specified by the canary resource. For example, the canary code with handler `index.handler`
     * must be found in the file structure `nodejs/node_modules/index.js`.
     *
     * Requires path to be either zip file or directory.
     * Requires asset directory to have the structure 'nodejs/node_modules'.
     * Requires canary file to be directly inside node_modules folder.
     * Requires canary file name matches the handler name.
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html
     *
     * @param handler the canary handler
     */
    private validateCanaryAsset;
}
/**
 * (experimental) Canary code from an inline string.
 *
 * @experimental
 */
export declare class InlineCode extends Code {
    private code;
    /**
     * @experimental
     */
    constructor(code: string);
    /**
     * (experimental) Called when the canary is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @experimental
     */
    bind(_scope: Construct, handler: string, _family: RuntimeFamily): CodeConfig;
}
/**
 * (experimental) S3 bucket path to the code zip file.
 *
 * @experimental
 */
export declare class S3Code extends Code {
    private bucket;
    private key;
    private objectVersion?;
    /**
     * @experimental
     */
    constructor(bucket: s3.IBucket, key: string, objectVersion?: string | undefined);
    /**
     * (experimental) Called when the canary is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @experimental
     */
    bind(_scope: Construct, _handler: string, _family: RuntimeFamily): CodeConfig;
}
