"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const eks = require("../lib");
const kubectl_layer_1 = require("../lib/kubectl-layer");
const util_1 = require("./util");
module.exports = {
    'a default cluster spans all subnets'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        // WHEN
        new eks.Cluster(stack, 'Cluster', { vpc, kubectlEnabled: false, defaultCapacity: 0 });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EKS::Cluster', {
            ResourcesVpcConfig: {
                SubnetIds: [
                    { Ref: 'VPCPublicSubnet1SubnetB4246D30' },
                    { Ref: 'VPCPublicSubnet2Subnet74179F39' },
                    { Ref: 'VPCPrivateSubnet1Subnet8BCA10E0' },
                    { Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A' },
                ],
            },
        }));
        test.done();
    },
    'create custom cluster correctly in any aws region'(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'stack', { env: { region: 'us-east-1' } });
        // WHEN
        const vpc = new ec2.Vpc(stack, 'VPC');
        new eks.Cluster(stack, 'Cluster', { vpc, kubectlEnabled: true, defaultCapacity: 0 });
        const layer = kubectl_layer_1.KubectlLayer.getOrCreate(stack, {});
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-Cluster'));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Serverless::Application', {
            Location: {
                ApplicationId: 'arn:aws:serverlessrepo:us-east-1:903779448426:applications/lambda-layer-kubectl',
            },
        }));
        test.equal(layer.isChina(), false);
        test.done();
    },
    'create custom cluster correctly in any aws region in china'(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'stack', { env: { region: 'cn-north-1' } });
        // WHEN
        const vpc = new ec2.Vpc(stack, 'VPC');
        new eks.Cluster(stack, 'Cluster', { vpc, kubectlEnabled: true, defaultCapacity: 0 });
        new kubectl_layer_1.KubectlLayer(stack, 'NewLayer');
        const layer = kubectl_layer_1.KubectlLayer.getOrCreate(stack);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-Cluster'));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Serverless::Application', {
            Location: {
                ApplicationId: 'arn:aws-cn:serverlessrepo:cn-north-1:487369736442:applications/lambda-layer-kubectl',
            },
        }));
        test.equal(layer.isChina(), true);
        test.done();
    },
    'if "vpc" is not specified, vpc with default configuration will be created'(test) {
        // GIVEN
        const { stack } = util_1.testFixtureNoVpc();
        // WHEN
        new eks.Cluster(stack, 'cluster');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC'));
        test.done();
    },
    'default capacity': {
        'x2 m5.large by default'(test) {
            // GIVEN
            const { stack } = util_1.testFixtureNoVpc();
            // WHEN
            const cluster = new eks.Cluster(stack, 'cluster');
            // THEN
            test.ok(cluster.defaultNodegroup);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EKS::Nodegroup', {
                InstanceTypes: [
                    'm5.large',
                ],
                ScalingConfig: {
                    DesiredSize: 2,
                    MaxSize: 2,
                    MinSize: 2,
                },
            }));
            test.done();
        },
        'quantity and type can be customized'(test) {
            // GIVEN
            const { stack } = util_1.testFixtureNoVpc();
            // WHEN
            const cluster = new eks.Cluster(stack, 'cluster', {
                defaultCapacity: 10,
                defaultCapacityInstance: new ec2.InstanceType('m2.xlarge'),
            });
            // THEN
            test.ok(cluster.defaultNodegroup);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EKS::Nodegroup', {
                ScalingConfig: {
                    DesiredSize: 10,
                    MaxSize: 10,
                    MinSize: 10,
                },
            }));
            // expect(stack).to(haveResource('AWS::AutoScaling::LaunchConfiguration', { InstanceType: 'm2.xlarge' }));
            test.done();
        },
        'defaultCapacity=0 will not allocate at all'(test) {
            // GIVEN
            const { stack } = util_1.testFixtureNoVpc();
            // WHEN
            const cluster = new eks.Cluster(stack, 'cluster', { defaultCapacity: 0 });
            // THEN
            test.ok(!cluster.defaultCapacity);
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup'));
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::AutoScaling::LaunchConfiguration'));
            test.done();
        },
    },
    'creating a cluster tags the private VPC subnets'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        // WHEN
        new eks.Cluster(stack, 'Cluster', { vpc, kubectlEnabled: false, defaultCapacity: 0 });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
            Tags: [
                { Key: 'aws-cdk:subnet-name', Value: 'Private' },
                { Key: 'aws-cdk:subnet-type', Value: 'Private' },
                { Key: 'kubernetes.io/role/internal-elb', Value: '1' },
                { Key: 'Name', Value: 'Stack/VPC/PrivateSubnet1' },
            ],
        }));
        test.done();
    },
    'creating a cluster tags the public VPC subnets'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        // WHEN
        new eks.Cluster(stack, 'Cluster', { vpc, kubectlEnabled: false, defaultCapacity: 0 });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
            MapPublicIpOnLaunch: true,
            Tags: [
                { Key: 'aws-cdk:subnet-name', Value: 'Public' },
                { Key: 'aws-cdk:subnet-type', Value: 'Public' },
                { Key: 'kubernetes.io/role/elb', Value: '1' },
                { Key: 'Name', Value: 'Stack/VPC/PublicSubnet1' },
            ],
        }));
        test.done();
    },
    'adding capacity creates an ASG with tags'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', { vpc, kubectlEnabled: false, defaultCapacity: 0 });
        // WHEN
        cluster.addCapacity('Default', {
            instanceType: new ec2.InstanceType('t2.medium'),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup', {
            Tags: [
                {
                    Key: { 'Fn::Join': ['', ['kubernetes.io/cluster/', { Ref: 'ClusterEB0386A7' }]] },
                    PropagateAtLaunch: true,
                    Value: 'owned',
                },
                {
                    Key: 'Name',
                    PropagateAtLaunch: true,
                    Value: 'Stack/Cluster/Default',
                },
            ],
        }));
        test.done();
    },
    'create nodegroup with existing role'(test) {
        // GIVEN
        const { stack } = util_1.testFixtureNoVpc();
        // WHEN
        const cluster = new eks.Cluster(stack, 'cluster', {
            defaultCapacity: 10,
            defaultCapacityInstance: new ec2.InstanceType('m2.xlarge'),
        });
        const existingRole = new iam.Role(stack, 'ExistingRole', {
            assumedBy: new iam.AccountRootPrincipal(),
        });
        new eks.Nodegroup(stack, 'Nodegroup', {
            cluster,
            nodeRole: existingRole,
        });
        // THEN
        test.ok(cluster.defaultNodegroup);
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EKS::Nodegroup', {
            ScalingConfig: {
                DesiredSize: 10,
                MaxSize: 10,
                MinSize: 10,
            },
        }));
        test.done();
    },
    'adding bottlerocket capacity creates an ASG with tags'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', { vpc, kubectlEnabled: false, defaultCapacity: 0 });
        // WHEN
        cluster.addCapacity('Bottlerocket', {
            instanceType: new ec2.InstanceType('t2.medium'),
            machineImageType: eks.MachineImageType.BOTTLEROCKET,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup', {
            Tags: [
                {
                    Key: { 'Fn::Join': ['', ['kubernetes.io/cluster/', { Ref: 'ClusterEB0386A7' }]] },
                    PropagateAtLaunch: true,
                    Value: 'owned',
                },
                {
                    Key: 'Name',
                    PropagateAtLaunch: true,
                    Value: 'Stack/Cluster/Bottlerocket',
                },
            ],
        }));
        test.done();
    },
    'adding bottlerocket capacity with bootstrapOptions throws error'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', { vpc, kubectlEnabled: false, defaultCapacity: 0 });
        test.throws(() => cluster.addCapacity('Bottlerocket', {
            instanceType: new ec2.InstanceType('t2.medium'),
            machineImageType: eks.MachineImageType.BOTTLEROCKET,
            bootstrapOptions: {},
        }), /bootstrapOptions is not supported for Bottlerocket/);
        test.done();
    },
    'exercise export/import'(test) {
        // GIVEN
        const { stack: stack1, vpc, app } = util_1.testFixture();
        const stack2 = new cdk.Stack(app, 'stack2', { env: { region: 'us-east-1' } });
        const cluster = new eks.Cluster(stack1, 'Cluster', { vpc, kubectlEnabled: false, defaultCapacity: 0 });
        // WHEN
        const imported = eks.Cluster.fromClusterAttributes(stack2, 'Imported', {
            clusterArn: cluster.clusterArn,
            vpc: cluster.vpc,
            clusterEndpoint: cluster.clusterEndpoint,
            clusterName: cluster.clusterName,
            securityGroups: cluster.connections.securityGroups,
            clusterCertificateAuthorityData: cluster.clusterCertificateAuthorityData,
        });
        // this should cause an export/import
        new cdk.CfnOutput(stack2, 'ClusterARN', { value: imported.clusterArn });
        // THEN
        assert_1.expect(stack2).toMatch({
            Outputs: {
                ClusterARN: {
                    Value: {
                        'Fn::ImportValue': 'Stack:ExportsOutputFnGetAttClusterEB0386A7Arn2F2E3C3F',
                    },
                },
            },
        });
        test.done();
    },
    'disabled features when kubectl is disabled'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', { vpc, kubectlEnabled: false, defaultCapacity: 0 });
        test.throws(() => cluster.awsAuth, /Cannot define aws-auth mappings if kubectl is disabled/);
        test.throws(() => cluster.addResource('foo', {}), /Unable to perform this operation since kubectl is not enabled for this cluster/);
        test.throws(() => cluster.addCapacity('boo', { instanceType: new ec2.InstanceType('r5d.24xlarge'), mapRole: true }), /Cannot map instance IAM role to RBAC if kubectl is disabled for the cluster/);
        test.throws(() => new eks.HelmChart(stack, 'MyChart', { cluster, chart: 'chart' }), /Unable to perform this operation since kubectl is not enabled for this cluster/);
        test.throws(() => cluster.openIdConnectProvider, /Cannot specify a OpenID Connect Provider if kubectl is disabled/);
        test.done();
    },
    'mastersRole can be used to map an IAM role to "system:masters" (required kubectl)'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const role = new iam.Role(stack, 'role', { assumedBy: new iam.AnyPrincipal() });
        // WHEN
        new eks.Cluster(stack, 'Cluster', { vpc, mastersRole: role, defaultCapacity: 0 });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource(eks.KubernetesResource.RESOURCE_TYPE, {
            Manifest: {
                'Fn::Join': [
                    '',
                    [
                        '[{"apiVersion":"v1","kind":"ConfigMap","metadata":{"name":"aws-auth","namespace":"kube-system"},"data":{"mapRoles":"[{\\"rolearn\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'roleC7B7E775',
                                'Arn',
                            ],
                        },
                        '\\",\\"username\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'roleC7B7E775',
                                'Arn',
                            ],
                        },
                        '\\",\\"groups\\":[\\"system:masters\\"]}]","mapUsers":"[]","mapAccounts":"[]"}}]',
                    ],
                ],
            },
        }));
        test.done();
    },
    'addResource can be used to apply k8s manifests on this cluster'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', { vpc, defaultCapacity: 0 });
        // WHEN
        cluster.addResource('manifest1', { foo: 123 });
        cluster.addResource('manifest2', { bar: 123 }, { boor: [1, 2, 3] });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource(eks.KubernetesResource.RESOURCE_TYPE, {
            Manifest: '[{"foo":123}]',
        }));
        assert_1.expect(stack).to(assert_1.haveResource(eks.KubernetesResource.RESOURCE_TYPE, {
            Manifest: '[{"bar":123},{"boor":[1,2,3]}]',
        }));
        test.done();
    },
    'kubectl resources can be created in a separate stack'(test) {
        // GIVEN
        const { stack, app } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'cluster'); // cluster is under stack2
        // WHEN resource is under stack2
        const stack2 = new cdk.Stack(app, 'stack2', { env: { account: stack.account, region: stack.region } });
        new eks.KubernetesResource(stack2, 'myresource', {
            cluster,
            manifest: [{ foo: 'bar' }],
        });
        // THEN
        app.synth(); // no cyclic dependency (see https://github.com/aws/aws-cdk/issues/7231)
        // expect a single resource in the 2nd stack
        assert_1.expect(stack2).toMatch({
            Resources: {
                myresource49C6D325: {
                    Type: 'Custom::AWSCDK-EKS-KubernetesResource',
                    Properties: {
                        ServiceToken: {
                            'Fn::ImportValue': 'Stack:ExportsOutputFnGetAttawscdkawseksKubectlProviderNestedStackawscdkawseksKubectlProviderNestedStackResourceA7AEBA6BOutputsStackawscdkawseksKubectlProviderframeworkonEvent8897FD9BArn49BEF20C',
                        },
                        Manifest: '[{\"foo\":\"bar\"}]',
                        ClusterName: { 'Fn::ImportValue': 'Stack:ExportsOutputRefclusterC5B25D0D98D553F5' },
                        RoleArn: { 'Fn::ImportValue': 'Stack:ExportsOutputFnGetAttclusterCreationRole2B3B5002ArnF05122FC' },
                    },
                    UpdateReplacePolicy: 'Delete',
                    DeletionPolicy: 'Delete',
                },
            },
        });
        test.done();
    },
    'when kubectl is enabled (default) adding capacity will automatically map its IAM role'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', { vpc, defaultCapacity: 0 });
        // WHEN
        cluster.addCapacity('default', {
            instanceType: new ec2.InstanceType('t2.nano'),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource(eks.KubernetesResource.RESOURCE_TYPE, {
            Manifest: {
                'Fn::Join': [
                    '',
                    [
                        '[{"apiVersion":"v1","kind":"ConfigMap","metadata":{"name":"aws-auth","namespace":"kube-system"},"data":{"mapRoles":"[{\\"rolearn\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'ClusterdefaultInstanceRoleF20A29CD',
                                'Arn',
                            ],
                        },
                        '\\",\\"username\\":\\"system:node:{{EC2PrivateDNSName}}\\",\\"groups\\":[\\"system:bootstrappers\\",\\"system:nodes\\"]}]","mapUsers":"[]","mapAccounts":"[]"}}]',
                    ],
                ],
            },
        }));
        test.done();
    },
    'addCapacity will *not* map the IAM role if mapRole is false'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', { vpc, defaultCapacity: 0 });
        // WHEN
        cluster.addCapacity('default', {
            instanceType: new ec2.InstanceType('t2.nano'),
            mapRole: false,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.not(assert_1.haveResource(eks.KubernetesResource.RESOURCE_TYPE)));
        test.done();
    },
    'addCapacity will *not* map the IAM role if kubectl is disabled'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', { vpc, kubectlEnabled: false, defaultCapacity: 0 });
        // WHEN
        cluster.addCapacity('default', {
            instanceType: new ec2.InstanceType('t2.nano'),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.not(assert_1.haveResource(eks.KubernetesResource.RESOURCE_TYPE)));
        test.done();
    },
    'outputs': {
        'aws eks update-kubeconfig is the only output synthesized by default'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            new eks.Cluster(stack, 'Cluster');
            // THEN
            const assembly = app.synth();
            const template = assembly.getStackByName(stack.stackName).template;
            test.deepEqual(template.Outputs, {
                ClusterConfigCommand43AAE40F: { Value: { 'Fn::Join': ['', ['aws eks update-kubeconfig --name ', { Ref: 'Cluster9EE0221C' }, ' --region us-east-1']] } },
                ClusterGetTokenCommand06AE992E: { Value: { 'Fn::Join': ['', ['aws eks get-token --cluster-name ', { Ref: 'Cluster9EE0221C' }, ' --region us-east-1']] } },
            });
            test.done();
        },
        'if masters role is defined, it should be included in the config command'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            const mastersRole = new iam.Role(stack, 'masters', { assumedBy: new iam.AccountRootPrincipal() });
            new eks.Cluster(stack, 'Cluster', { mastersRole });
            // THEN
            const assembly = app.synth();
            const template = assembly.getStackByName(stack.stackName).template;
            test.deepEqual(template.Outputs, {
                ClusterConfigCommand43AAE40F: { Value: { 'Fn::Join': ['', ['aws eks update-kubeconfig --name ', { Ref: 'Cluster9EE0221C' }, ' --region us-east-1 --role-arn ', { 'Fn::GetAtt': ['masters0D04F23D', 'Arn'] }]] } },
                ClusterGetTokenCommand06AE992E: { Value: { 'Fn::Join': ['', ['aws eks get-token --cluster-name ', { Ref: 'Cluster9EE0221C' }, ' --region us-east-1 --role-arn ', { 'Fn::GetAtt': ['masters0D04F23D', 'Arn'] }]] } },
            });
            test.done();
        },
        'if `outputConfigCommand=false` will disabled the output'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            const mastersRole = new iam.Role(stack, 'masters', { assumedBy: new iam.AccountRootPrincipal() });
            new eks.Cluster(stack, 'Cluster', {
                mastersRole,
                outputConfigCommand: false,
            });
            // THEN
            const assembly = app.synth();
            const template = assembly.getStackByName(stack.stackName).template;
            test.ok(!template.Outputs); // no outputs
            test.done();
        },
        '`outputClusterName` can be used to synthesize an output with the cluster name'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            new eks.Cluster(stack, 'Cluster', {
                outputConfigCommand: false,
                outputClusterName: true,
            });
            // THEN
            const assembly = app.synth();
            const template = assembly.getStackByName(stack.stackName).template;
            test.deepEqual(template.Outputs, {
                ClusterClusterNameEB26049E: { Value: { Ref: 'Cluster9EE0221C' } },
            });
            test.done();
        },
        '`outputMastersRoleArn` can be used to synthesize an output with the arn of the masters role if defined'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            new eks.Cluster(stack, 'Cluster', {
                outputConfigCommand: false,
                outputMastersRoleArn: true,
                mastersRole: new iam.Role(stack, 'masters', { assumedBy: new iam.AccountRootPrincipal() }),
            });
            // THEN
            const assembly = app.synth();
            const template = assembly.getStackByName(stack.stackName).template;
            test.deepEqual(template.Outputs, {
                ClusterMastersRoleArnB15964B1: { Value: { 'Fn::GetAtt': ['masters0D04F23D', 'Arn'] } },
            });
            test.done();
        },
        'boostrap user-data': {
            'rendered by default for ASGs'(test) {
                // GIVEN
                const { app, stack } = util_1.testFixtureNoVpc();
                const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0 });
                // WHEN
                cluster.addCapacity('MyCapcity', { instanceType: new ec2.InstanceType('m3.xlargs') });
                // THEN
                const template = app.synth().getStackByName(stack.stackName).template;
                const userData = template.Resources.ClusterMyCapcityLaunchConfig58583345.Properties.UserData;
                test.deepEqual(userData, { 'Fn::Base64': { 'Fn::Join': ['', ['#!/bin/bash\nset -o xtrace\n/etc/eks/bootstrap.sh ', { Ref: 'Cluster9EE0221C' }, ' --kubelet-extra-args "--node-labels lifecycle=OnDemand" --use-max-pods true\n/opt/aws/bin/cfn-signal --exit-code $? --stack Stack --resource ClusterMyCapcityASGD4CD8B97 --region us-east-1']] } });
                test.done();
            },
            'not rendered if bootstrap is disabled'(test) {
                // GIVEN
                const { app, stack } = util_1.testFixtureNoVpc();
                const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0 });
                // WHEN
                cluster.addCapacity('MyCapcity', {
                    instanceType: new ec2.InstanceType('m3.xlargs'),
                    bootstrapEnabled: false,
                });
                // THEN
                const template = app.synth().getStackByName(stack.stackName).template;
                const userData = template.Resources.ClusterMyCapcityLaunchConfig58583345.Properties.UserData;
                test.deepEqual(userData, { 'Fn::Base64': '#!/bin/bash' });
                test.done();
            },
            // cursory test for options: see test.user-data.ts for full suite
            'bootstrap options'(test) {
                // GIVEN
                const { app, stack } = util_1.testFixtureNoVpc();
                const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0 });
                // WHEN
                cluster.addCapacity('MyCapcity', {
                    instanceType: new ec2.InstanceType('m3.xlargs'),
                    bootstrapOptions: {
                        kubeletExtraArgs: '--node-labels FOO=42',
                    },
                });
                // THEN
                const template = app.synth().getStackByName(stack.stackName).template;
                const userData = template.Resources.ClusterMyCapcityLaunchConfig58583345.Properties.UserData;
                test.deepEqual(userData, { 'Fn::Base64': { 'Fn::Join': ['', ['#!/bin/bash\nset -o xtrace\n/etc/eks/bootstrap.sh ', { Ref: 'Cluster9EE0221C' }, ' --kubelet-extra-args "--node-labels lifecycle=OnDemand  --node-labels FOO=42" --use-max-pods true\n/opt/aws/bin/cfn-signal --exit-code $? --stack Stack --resource ClusterMyCapcityASGD4CD8B97 --region us-east-1']] } });
                test.done();
            },
            'spot instances': {
                'nodes labeled an tainted accordingly'(test) {
                    // GIVEN
                    const { app, stack } = util_1.testFixtureNoVpc();
                    const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0 });
                    // WHEN
                    cluster.addCapacity('MyCapcity', {
                        instanceType: new ec2.InstanceType('m3.xlargs'),
                        spotPrice: '0.01',
                    });
                    // THEN
                    const template = app.synth().getStackByName(stack.stackName).template;
                    const userData = template.Resources.ClusterMyCapcityLaunchConfig58583345.Properties.UserData;
                    test.deepEqual(userData, { 'Fn::Base64': { 'Fn::Join': ['', ['#!/bin/bash\nset -o xtrace\n/etc/eks/bootstrap.sh ', { Ref: 'Cluster9EE0221C' }, ' --kubelet-extra-args "--node-labels lifecycle=Ec2Spot --register-with-taints=spotInstance=true:PreferNoSchedule" --use-max-pods true\n/opt/aws/bin/cfn-signal --exit-code $? --stack Stack --resource ClusterMyCapcityASGD4CD8B97 --region us-east-1']] } });
                    test.done();
                },
                'if kubectl is enabled, the interrupt handler is added'(test) {
                    // GIVEN
                    const { stack } = util_1.testFixtureNoVpc();
                    const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0 });
                    // WHEN
                    cluster.addCapacity('MyCapcity', {
                        instanceType: new ec2.InstanceType('m3.xlarge'),
                        spotPrice: '0.01',
                    });
                    // THEN
                    assert_1.expect(stack).to(assert_1.haveResource(eks.HelmChart.RESOURCE_TYPE, {
                        Release: 'stackclusterchartspotinterrupthandlerdec62e07',
                        Chart: 'aws-node-termination-handler',
                        Wait: false,
                        Values: '{\"nodeSelector.lifecycle\":\"Ec2Spot\"}',
                        Namespace: 'kube-system',
                        Repository: 'https://aws.github.io/eks-charts',
                    }));
                    test.done();
                },
                'its possible to add two capacities with spot instances and only one stop handler will be installed'(test) {
                    // GIVEN
                    const { stack } = util_1.testFixtureNoVpc();
                    const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0 });
                    // WHEN
                    cluster.addCapacity('Spot1', {
                        instanceType: new ec2.InstanceType('m3.xlarge'),
                        spotPrice: '0.01',
                    });
                    cluster.addCapacity('Spot2', {
                        instanceType: new ec2.InstanceType('m4.xlarge'),
                        spotPrice: '0.01',
                    });
                    // THEN
                    assert_1.expect(stack).to(assert_1.countResources(eks.HelmChart.RESOURCE_TYPE, 1));
                    test.done();
                },
                'if kubectl is disabled, interrupt handler is not added'(test) {
                    // GIVEN
                    const { stack } = util_1.testFixtureNoVpc();
                    const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0, kubectlEnabled: false });
                    // WHEN
                    cluster.addCapacity('MyCapcity', {
                        instanceType: new ec2.InstanceType('m3.xlargs'),
                        spotPrice: '0.01',
                    });
                    // THEN
                    assert_1.expect(stack).notTo(assert_1.haveResource(eks.KubernetesResource.RESOURCE_TYPE));
                    test.done();
                },
            },
        },
        'if bootstrap is disabled cannot specify options'(test) {
            // GIVEN
            const { stack } = util_1.testFixtureNoVpc();
            const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0 });
            // THEN
            test.throws(() => cluster.addCapacity('MyCapcity', {
                instanceType: new ec2.InstanceType('m3.xlargs'),
                bootstrapEnabled: false,
                bootstrapOptions: { awsApiRetryAttempts: 10 },
            }), /Cannot specify "bootstrapOptions" if "bootstrapEnabled" is false/);
            test.done();
        },
        'EksOptimizedImage() with no nodeType always uses STANDARD with LATEST_KUBERNETES_VERSION'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            const LATEST_KUBERNETES_VERSION = '1.14';
            // WHEN
            new eks.EksOptimizedImage().getImage(stack);
            // THEN
            const assembly = app.synth();
            const parameters = assembly.getStackByName(stack.stackName).template.Parameters;
            test.ok(Object.entries(parameters).some(([k, v]) => k.startsWith('SsmParameterValueawsserviceeksoptimizedami') &&
                v.Default.includes('/amazon-linux-2/')), 'EKS STANDARD AMI should be in ssm parameters');
            test.ok(Object.entries(parameters).some(([k, v]) => k.startsWith('SsmParameterValueawsserviceeksoptimizedami') &&
                v.Default.includes(LATEST_KUBERNETES_VERSION)), 'LATEST_KUBERNETES_VERSION should be in ssm parameters');
            test.done();
        },
        'EksOptimizedImage() with specific kubernetesVersion return correct AMI'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            new eks.EksOptimizedImage({ kubernetesVersion: '1.15' }).getImage(stack);
            // THEN
            const assembly = app.synth();
            const parameters = assembly.getStackByName(stack.stackName).template.Parameters;
            test.ok(Object.entries(parameters).some(([k, v]) => k.startsWith('SsmParameterValueawsserviceeksoptimizedami') &&
                v.Default.includes('/amazon-linux-2/')), 'EKS STANDARD AMI should be in ssm parameters');
            test.ok(Object.entries(parameters).some(([k, v]) => k.startsWith('SsmParameterValueawsserviceeksoptimizedami') &&
                v.Default.includes('/1.15/')), 'kubernetesVersion should be in ssm parameters');
            test.done();
        },
        'EKS-Optimized AMI with GPU support when addCapacity'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            new eks.Cluster(stack, 'cluster', {
                defaultCapacity: 0,
            }).addCapacity('GPUCapacity', {
                instanceType: new ec2.InstanceType('g4dn.xlarge'),
            });
            // THEN
            const assembly = app.synth();
            const parameters = assembly.getStackByName(stack.stackName).template.Parameters;
            test.ok(Object.entries(parameters).some(([k, v]) => k.startsWith('SsmParameterValueawsserviceeksoptimizedami') && v.Default.includes('amazon-linux-2-gpu')), 'EKS AMI with GPU should be in ssm parameters');
            test.done();
        },
        'when using custom resource a creation role & policy is defined'(test) {
            // GIVEN
            const { stack } = util_1.testFixture();
            // WHEN
            new eks.Cluster(stack, 'MyCluster', {
                clusterName: 'my-cluster-name',
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-Cluster', {
                Config: {
                    name: 'my-cluster-name',
                    roleArn: { 'Fn::GetAtt': ['MyClusterRoleBA20FE72', 'Arn'] },
                    resourcesVpcConfig: {
                        securityGroupIds: [{ 'Fn::GetAtt': ['MyClusterControlPlaneSecurityGroup6B658F79', 'GroupId'] }],
                        subnetIds: [
                            { Ref: 'MyClusterDefaultVpcPublicSubnet1SubnetFAE5A9B6' },
                            { Ref: 'MyClusterDefaultVpcPublicSubnet2SubnetF6D028A0' },
                            { Ref: 'MyClusterDefaultVpcPrivateSubnet1SubnetE1D0DCDB' },
                            { Ref: 'MyClusterDefaultVpcPrivateSubnet2Subnet11FEA8D0' },
                        ],
                    },
                },
            }));
            // role can be assumed by 3 lambda handlers (2 for the cluster resource and 1 for the kubernetes resource)
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
                AssumeRolePolicyDocument: {
                    Statement: [
                        {
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                AWS: [
                                    {
                                        'Fn::GetAtt': [
                                            'awscdkawseksClusterResourceProviderNestedStackawscdkawseksClusterResourceProviderNestedStackResource9827C454',
                                            'Outputs.StackawscdkawseksClusterResourceProviderOnEventHandlerServiceRole3AEE0A43Arn',
                                        ],
                                    },
                                    {
                                        'Fn::GetAtt': [
                                            'awscdkawseksClusterResourceProviderNestedStackawscdkawseksClusterResourceProviderNestedStackResource9827C454',
                                            'Outputs.StackawscdkawseksClusterResourceProviderIsCompleteHandlerServiceRole8E7F1C11Arn',
                                        ],
                                    },
                                ],
                            },
                        },
                    ],
                    Version: '2012-10-17',
                },
            }));
            // policy allows creation role to pass the cluster role and to interact with the cluster (given we know the explicit cluster name)
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'iam:PassRole',
                            Effect: 'Allow',
                            Resource: {
                                'Fn::GetAtt': [
                                    'MyClusterRoleBA20FE72',
                                    'Arn',
                                ],
                            },
                        },
                        {
                            Action: [
                                'ec2:DescribeSubnets',
                                'ec2:DescribeRouteTables',
                            ],
                            Effect: 'Allow',
                            Resource: '*',
                        },
                        {
                            Action: [
                                'eks:CreateCluster',
                                'eks:DescribeCluster',
                                'eks:DescribeUpdate',
                                'eks:DeleteCluster',
                                'eks:UpdateClusterVersion',
                                'eks:UpdateClusterConfig',
                                'eks:CreateFargateProfile',
                                'eks:TagResource',
                                'eks:UntagResource',
                            ],
                            Effect: 'Allow',
                            Resource: [{
                                    'Fn::Join': [
                                        '',
                                        [
                                            'arn:',
                                            {
                                                Ref: 'AWS::Partition',
                                            },
                                            ':eks:us-east-1:',
                                            {
                                                Ref: 'AWS::AccountId',
                                            },
                                            ':cluster/my-cluster-name',
                                        ],
                                    ],
                                }, {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'arn:',
                                            {
                                                Ref: 'AWS::Partition',
                                            },
                                            ':eks:us-east-1:',
                                            {
                                                Ref: 'AWS::AccountId',
                                            },
                                            ':cluster/my-cluster-name/*',
                                        ],
                                    ],
                                }],
                        },
                        {
                            Action: [
                                'eks:DescribeFargateProfile',
                                'eks:DeleteFargateProfile',
                            ],
                            Effect: 'Allow',
                            Resource: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':eks:us-east-1:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':fargateprofile/my-cluster-name/*',
                                    ],
                                ],
                            },
                        },
                        {
                            Action: 'iam:GetRole',
                            Effect: 'Allow',
                            Resource: '*',
                        },
                        {
                            Action: 'iam:CreateServiceLinkedRole',
                            Effect: 'Allow',
                            Resource: '*',
                        },
                    ],
                    Version: '2012-10-17',
                },
            }));
            test.done();
        },
        'if an explicit cluster name is not provided, the creation role policy is wider (allows interacting with all clusters)'(test) {
            // GIVEN
            const { stack } = util_1.testFixture();
            // WHEN
            new eks.Cluster(stack, 'MyCluster');
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'iam:PassRole',
                            Effect: 'Allow',
                            Resource: {
                                'Fn::GetAtt': [
                                    'MyClusterRoleBA20FE72',
                                    'Arn',
                                ],
                            },
                        },
                        {
                            Action: [
                                'ec2:DescribeSubnets',
                                'ec2:DescribeRouteTables',
                            ],
                            Effect: 'Allow',
                            Resource: '*',
                        },
                        {
                            Action: [
                                'eks:CreateCluster',
                                'eks:DescribeCluster',
                                'eks:DescribeUpdate',
                                'eks:DeleteCluster',
                                'eks:UpdateClusterVersion',
                                'eks:UpdateClusterConfig',
                                'eks:CreateFargateProfile',
                                'eks:TagResource',
                                'eks:UntagResource',
                            ],
                            Effect: 'Allow',
                            Resource: ['*'],
                        },
                        {
                            Action: [
                                'eks:DescribeFargateProfile',
                                'eks:DeleteFargateProfile',
                            ],
                            Effect: 'Allow',
                            Resource: '*',
                        },
                        {
                            Action: 'iam:GetRole',
                            Effect: 'Allow',
                            Resource: '*',
                        },
                        {
                            Action: 'iam:CreateServiceLinkedRole',
                            Effect: 'Allow',
                            Resource: '*',
                        },
                    ],
                    Version: '2012-10-17',
                },
            }));
            test.done();
        },
        'if helm charts are used, its resource provider is allowed to assume the creation role'(test) {
            // GIVEN
            const { stack } = util_1.testFixture();
            const cluster = new eks.Cluster(stack, 'MyCluster', {
                clusterName: 'my-cluster-name',
            });
            // WHEN
            cluster.addChart('MyChart', {
                chart: 'foo',
            });
            // THEN
            // role can be assumed by 4 principals: two for the cluster resource, one
            // for kubernetes resource and one for the helm resource.
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
                AssumeRolePolicyDocument: {
                    Statement: [
                        {
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                AWS: [
                                    {
                                        'Fn::GetAtt': [
                                            'awscdkawseksClusterResourceProviderNestedStackawscdkawseksClusterResourceProviderNestedStackResource9827C454',
                                            'Outputs.StackawscdkawseksClusterResourceProviderOnEventHandlerServiceRole3AEE0A43Arn',
                                        ],
                                    },
                                    {
                                        'Fn::GetAtt': [
                                            'awscdkawseksClusterResourceProviderNestedStackawscdkawseksClusterResourceProviderNestedStackResource9827C454',
                                            'Outputs.StackawscdkawseksClusterResourceProviderIsCompleteHandlerServiceRole8E7F1C11Arn',
                                        ],
                                    },
                                ],
                            },
                        },
                        {
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                AWS: {
                                    'Fn::GetAtt': [
                                        'awscdkawseksKubectlProviderNestedStackawscdkawseksKubectlProviderNestedStackResourceA7AEBA6B',
                                        'Outputs.StackawscdkawseksKubectlProviderHandlerServiceRole2C52B3ECArn',
                                    ],
                                },
                            },
                        },
                    ],
                    Version: '2012-10-17',
                },
            }));
            test.done();
        },
        'coreDnsComputeType will patch the coreDNS configuration to use a "fargate" compute type and restore to "ec2" upon removal'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            new eks.Cluster(stack, 'MyCluster', {
                coreDnsComputeType: eks.CoreDnsComputeType.FARGATE,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-KubernetesPatch', {
                ResourceName: 'deployment/coredns',
                ResourceNamespace: 'kube-system',
                ApplyPatchJson: '{"spec":{"template":{"metadata":{"annotations":{"eks.amazonaws.com/compute-type":"fargate"}}}}}',
                RestorePatchJson: '{"spec":{"template":{"metadata":{"annotations":{"eks.amazonaws.com/compute-type":"ec2"}}}}}',
                ClusterName: {
                    Ref: 'MyCluster8AD82BF8',
                },
                RoleArn: {
                    'Fn::GetAtt': [
                        'MyClusterCreationRoleB5FA4FF3',
                        'Arn',
                    ],
                },
            }));
            test.done();
        },
        'if openIDConnectProvider a new OpenIDConnectProvider resource is created and exposed'(test) {
            // GIVEN
            const { stack } = util_1.testFixtureNoVpc();
            const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0 });
            // WHEN
            const provider = cluster.openIdConnectProvider;
            // THEN
            test.equal(provider, cluster.openIdConnectProvider, 'openIdConnect provider is different and created more than once.');
            assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDKOpenIdConnectProvider', {
                ServiceToken: {
                    'Fn::GetAtt': [
                        'CustomAWSCDKOpenIdConnectProviderCustomResourceProviderHandlerF2C543E0',
                        'Arn',
                    ],
                },
                ClientIDList: [
                    'sts.amazonaws.com',
                ],
                ThumbprintList: [
                    '9e99a48a9960b14926bb7f3b02e22da2b0ab7280',
                ],
                Url: {
                    'Fn::GetAtt': [
                        'Cluster9EE0221C',
                        'OpenIdConnectIssuerUrl',
                    ],
                },
            }));
            test.done();
        },
    }
};
//# sourceMappingURL=data:application/json;base64,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