"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
/**
 * Service Account
 */
class ServiceAccount extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        const { cluster } = props;
        this.serviceAccountName = (_a = props.name) !== null && _a !== void 0 ? _a : this.node.uniqueId.toLowerCase();
        this.serviceAccountNamespace = (_b = props.namespace) !== null && _b !== void 0 ? _b : 'default';
        /* Add conditions to the role to improve security. This prevents other pods in the same namespace to assume the role.
        * See documentation: https://docs.aws.amazon.com/eks/latest/userguide/create-service-account-iam-policy-and-role.html
        */
        const conditions = new core_1.CfnJson(this, 'ConditionJson', {
            value: {
                [`${cluster.clusterOpenIdConnectIssuer}:aud`]: 'sts.amazonaws.com',
                [`${cluster.clusterOpenIdConnectIssuer}:sub`]: `system:serviceaccount:${this.serviceAccountNamespace}:${this.serviceAccountName}`,
            },
        });
        const principal = new aws_iam_1.OpenIdConnectPrincipal(cluster.openIdConnectProvider).withConditions({
            StringEquals: conditions,
        });
        this.role = new aws_iam_1.Role(this, 'Role', { assumedBy: principal });
        this.assumeRoleAction = this.role.assumeRoleAction;
        this.grantPrincipal = this.role.grantPrincipal;
        this.policyFragment = this.role.policyFragment;
        cluster.addResource(`${id}ServiceAccountResource`, {
            apiVersion: 'v1',
            kind: 'ServiceAccount',
            metadata: {
                name: this.serviceAccountName,
                namespace: this.serviceAccountNamespace,
                labels: {
                    'app.kubernetes.io/name': this.serviceAccountName,
                },
                annotations: {
                    'eks.amazonaws.com/role-arn': this.role.roleArn,
                },
            },
        });
    }
    addToPolicy(statement) {
        return this.role.addToPolicy(statement);
    }
    addToPrincipalPolicy(statement) {
        return this.role.addToPrincipalPolicy(statement);
    }
}
exports.ServiceAccount = ServiceAccount;
//# sourceMappingURL=data:application/json;base64,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