"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SimpleSynthAction = void 0;
const path = require("path");
const codebuild = require("../../../aws-codebuild"); // Automatically re-written from '@aws-cdk/aws-codebuild'
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const codepipeline_actions = require("../../../aws-codepipeline-actions"); // Automatically re-written from '@aws-cdk/aws-codepipeline-actions'
const construct_internals_1 = require("../private/construct-internals");
const _util_1 = require("./_util");
/**
 * A standard synth with a generated buildspec
 */
class SimpleSynthAction {
    constructor(props) {
        var _a, _b, _c;
        this.props = props;
        // A number of actionProperties get read before bind() is even called (so before we
        // have made the Project and can construct the actual CodeBuildAction)
        //
        // - actionName
        // - resource
        // - region
        // - category
        // - role
        // - owner
        this._actionProperties = {
            actionName: (_a = props.actionName) !== null && _a !== void 0 ? _a : 'Synth',
            category: codepipeline.ActionCategory.BUILD,
            provider: 'CodeBuild',
            artifactBounds: { minInputs: 0, maxInputs: 5, minOutputs: 0, maxOutputs: 5 },
            inputs: [props.sourceArtifact],
            outputs: [props.cloudAssemblyArtifact, ...((_b = props.additionalArtifacts) !== null && _b !== void 0 ? _b : []).map(a => a.artifact)],
        };
        if (this.props.installCommand && this.props.installCommands) {
            throw new Error('Pass either \'installCommand\' or \'installCommands\', but not both');
        }
        if (this.props.buildCommand && this.props.buildCommands) {
            throw new Error('Pass either \'buildCommand\' or \'buildCommands\', but not both');
        }
        const addls = (_c = props.additionalArtifacts) !== null && _c !== void 0 ? _c : [];
        if (Object.keys(addls).length > 0) {
            if (!props.cloudAssemblyArtifact.artifactName) {
                throw new Error('You must give all output artifacts, including the \'cloudAssemblyArtifact\', names when using \'additionalArtifacts\'');
            }
            for (const addl of addls) {
                if (!addl.artifact.artifactName) {
                    throw new Error('You must give all output artifacts passed to SimpleSynthAction names when using \'additionalArtifacts\'');
                }
            }
        }
    }
    /**
     * Create a standard NPM synth action
     *
     * Uses `npm ci` to install dependencies and `npx cdk synth` to synthesize.
     *
     * If you need a build step, add `buildCommand: 'npm run build'`.
     */
    static standardNpmSynth(options) {
        var _a, _b;
        return new SimpleSynthAction({
            ...options,
            installCommand: (_a = options.installCommand) !== null && _a !== void 0 ? _a : 'npm ci',
            synthCommand: (_b = options.synthCommand) !== null && _b !== void 0 ? _b : 'npx cdk synth',
        });
    }
    /**
     * Create a standard Yarn synth action
     *
     * Uses `yarn install --frozen-lockfile` to install dependencies and `npx cdk synth` to synthesize.
     *
     * If you need a build step, add `buildCommand: 'yarn build'`.
     */
    static standardYarnSynth(options) {
        var _a, _b;
        return new SimpleSynthAction({
            ...options,
            installCommand: (_a = options.installCommand) !== null && _a !== void 0 ? _a : 'yarn install --frozen-lockfile',
            synthCommand: (_b = options.synthCommand) !== null && _b !== void 0 ? _b : 'npx cdk synth',
        });
    }
    /**
     * Exists to implement IAction
     */
    get actionProperties() {
        return this._actionProperties;
    }
    /**
     * Exists to implement IAction
     */
    bind(scope, stage, options) {
        var _a, _b, _c, _d, _e;
        const buildCommands = (_a = this.props.buildCommands) !== null && _a !== void 0 ? _a : [this.props.buildCommand];
        const installCommands = (_b = this.props.installCommands) !== null && _b !== void 0 ? _b : [this.props.installCommand];
        const testCommands = (_c = this.props.testCommands) !== null && _c !== void 0 ? _c : [];
        const synthCommand = this.props.synthCommand;
        const project = new codebuild.PipelineProject(scope, 'CdkBuildProject', {
            projectName: (_d = this.props.projectName) !== null && _d !== void 0 ? _d : this.props.projectName,
            environment: { buildImage: codebuild.LinuxBuildImage.STANDARD_4_0, ...this.props.environment },
            buildSpec: codebuild.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    pre_build: {
                        commands: _util_1.filterEmpty([
                            this.props.subdirectory ? `cd ${this.props.subdirectory}` : '',
                            ...installCommands,
                        ]),
                    },
                    build: {
                        commands: _util_1.filterEmpty([
                            ...buildCommands,
                            ...testCommands,
                            synthCommand,
                        ]),
                    },
                },
                artifacts: renderArtifacts(this),
            }),
            environmentVariables: {
                ..._util_1.copyEnvironmentVariables(...this.props.copyEnvironmentVariables || []),
                ...this.props.environmentVariables,
            },
        });
        if (this.props.rolePolicyStatements !== undefined) {
            this.props.rolePolicyStatements.forEach(policyStatement => {
                project.addToRolePolicy(policyStatement);
            });
        }
        this._action = new codepipeline_actions.CodeBuildAction({
            actionName: this.actionProperties.actionName,
            input: this.props.sourceArtifact,
            outputs: [this.props.cloudAssemblyArtifact, ...((_e = this.props.additionalArtifacts) !== null && _e !== void 0 ? _e : []).map(a => a.artifact)],
            project,
        });
        this._actionProperties = this._action.actionProperties;
        return this._action.bind(scope, stage, options);
        function renderArtifacts(self) {
            var _a;
            // save the generated files in the output artifact
            // This part of the buildspec has to look completely different depending on whether we're
            // using secondary artifacts or not.
            const cloudAsmArtifactSpec = {
                'base-directory': path.join((_a = self.props.subdirectory) !== null && _a !== void 0 ? _a : '.', construct_internals_1.cloudAssemblyBuildSpecDir(scope)),
                'files': '**/*',
            };
            if (self.props.additionalArtifacts) {
                const secondary = {};
                if (!self.props.cloudAssemblyArtifact.artifactName) {
                    throw new Error('When using additional output artifacts, you must also name the CloudAssembly artifact');
                }
                secondary[self.props.cloudAssemblyArtifact.artifactName] = cloudAsmArtifactSpec;
                self.props.additionalArtifacts.forEach((art) => {
                    var _a;
                    if (!art.artifact.artifactName) {
                        throw new Error('You must give the output artifact a name');
                    }
                    secondary[art.artifact.artifactName] = {
                        'base-directory': path.join((_a = self.props.subdirectory) !== null && _a !== void 0 ? _a : '.', art.directory),
                        'files': '**/*',
                    };
                });
                return { 'secondary-artifacts': secondary };
            }
            return cloudAsmArtifactSpec;
        }
    }
    /**
     * Exists to implement IAction
     */
    onStateChange(name, target, options) {
        if (!this._action) {
            throw new Error('Need bind() first');
        }
        return this._action.onStateChange(name, target, options);
    }
}
exports.SimpleSynthAction = SimpleSynthAction;
//# sourceMappingURL=data:application/json;base64,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