"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CdkPipeline = void 0;
const path = require("path");
const codepipeline = require("../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const actions_1 = require("./actions");
const construct_internals_1 = require("./private/construct-internals");
const stage_1 = require("./stage");
/**
 * A Pipeline to deploy CDK apps
 *
 * Defines an AWS CodePipeline-based Pipeline to deploy CDK applications.
 *
 * Automatically manages the following:
 *
 * - Stack dependency order.
 * - Asset publishing.
 * - Keeping the pipeline up-to-date as the CDK apps change.
 * - Using stack outputs later on in the pipeline.
 */
class CdkPipeline extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this._stages = [];
        this._outputArtifacts = {};
        if (!core_1.App.isApp(this.node.root)) {
            throw new Error('CdkPipeline must be created under an App');
        }
        this._cloudAssemblyArtifact = props.cloudAssemblyArtifact;
        const pipelineStack = core_1.Stack.of(this);
        if (props.codePipeline) {
            if (props.pipelineName) {
                throw new Error('Cannot set \'pipelineName\' if an existing CodePipeline is given using \'codePipeline\'');
            }
            this._pipeline = props.codePipeline;
        }
        else {
            this._pipeline = new codepipeline.Pipeline(this, 'Pipeline', {
                pipelineName: props.pipelineName,
                restartExecutionOnUpdate: true,
            });
        }
        if (props.sourceAction && !props.synthAction) {
            // Because of ordering limitations, you can: bring your own Source, bring your own
            // Both, or bring your own Nothing. You cannot bring your own Build (which because of the
            // current CodePipeline API must go BEFORE what we're adding) and then having us add a
            // Source after it. That doesn't make any sense.
            throw new Error('When passing a \'sourceAction\' you must also pass a \'synthAction\' (or a \'codePipeline\' that already has both)');
        }
        if (!props.sourceAction && (!props.codePipeline || props.codePipeline.stages.length < 1)) {
            throw new Error('You must pass a \'sourceAction\' (or a \'codePipeline\' that already has a Source stage)');
        }
        if (!props.synthAction && (!props.codePipeline || props.codePipeline.stages.length < 2)) {
            // This looks like a weirdly specific requirement, but actually the underlying CodePipeline
            // requires that a Pipeline has at least 2 stages. We're just hitching onto upstream
            // requirements to do this check.
            throw new Error('You must pass a \'synthAction\' (or a \'codePipeline\' that already has a Build stage)');
        }
        if (props.sourceAction) {
            this._pipeline.addStage({
                stageName: 'Source',
                actions: [props.sourceAction],
            });
        }
        if (props.synthAction) {
            this._pipeline.addStage({
                stageName: 'Build',
                actions: [props.synthAction],
            });
        }
        this._pipeline.addStage({
            stageName: 'UpdatePipeline',
            actions: [new actions_1.UpdatePipelineAction(this, 'UpdatePipeline', {
                    cloudAssemblyInput: this._cloudAssemblyArtifact,
                    pipelineStackName: pipelineStack.stackName,
                    cdkCliVersion: props.cdkCliVersion,
                    projectName: maybeSuffix(props.pipelineName, '-selfupdate'),
                })],
        });
        this._assets = new AssetPublishing(this, 'Assets', {
            cloudAssemblyInput: this._cloudAssemblyArtifact,
            cdkCliVersion: props.cdkCliVersion,
            pipeline: this._pipeline,
            projectName: maybeSuffix(props.pipelineName, '-publish'),
        });
        core_1.Aspects.of(this).add({ visit: () => this._assets.removeAssetsStageIfEmpty() });
    }
    /**
     * The underlying CodePipeline object
     *
     * You can use this to add more Stages to the pipeline, or Actions
     * to Stages.
     */
    get codePipeline() {
        return this._pipeline;
    }
    /**
     * Access one of the pipeline's stages by stage name
     *
     * You can use this to add more Actions to a stage.
     */
    stage(stageName) {
        return this._pipeline.stage(stageName);
    }
    /**
     * Add pipeline stage that will deploy the given application stage
     *
     * The application construct should subclass `Stage` and can contain any
     * number of `Stacks` inside it that may have dependency relationships
     * on one another.
     *
     * All stacks in the application will be deployed in the appropriate order,
     * and all assets found in the application will be added to the asset
     * publishing stage.
     */
    addApplicationStage(appStage, options = {}) {
        const stage = this.addStage(appStage.stageName);
        stage.addApplication(appStage, options);
        return stage;
    }
    /**
     * Add a new, empty stage to the pipeline
     *
     * Prefer to use `addApplicationStage` if you are intended to deploy a CDK
     * application, but you can use this method if you want to add other kinds of
     * Actions to a pipeline.
     */
    addStage(stageName) {
        const pipelineStage = this._pipeline.addStage({
            stageName,
        });
        const stage = new stage_1.CdkStage(this, stageName, {
            cloudAssemblyArtifact: this._cloudAssemblyArtifact,
            pipelineStage,
            stageName,
            host: {
                publishAsset: this._assets.addPublishAssetAction.bind(this._assets),
                stackOutputArtifact: (artifactId) => this._outputArtifacts[artifactId],
            },
        });
        this._stages.push(stage);
        return stage;
    }
    /**
     * Get the StackOutput object that holds this CfnOutput's value in this pipeline
     *
     * `StackOutput` can be used in validation actions later in the pipeline.
     */
    stackOutput(cfnOutput) {
        const stack = core_1.Stack.of(cfnOutput);
        if (!this._outputArtifacts[stack.artifactId]) {
            // We should have stored the ArtifactPath in the map, but its Artifact
            // property isn't publicly readable...
            this._outputArtifacts[stack.artifactId] = new codepipeline.Artifact(`Artifact_${stack.artifactId}_Outputs`);
        }
        return new stage_1.StackOutput(this._outputArtifacts[stack.artifactId].atPath('outputs.json'), cfnOutput.logicalId);
    }
    /**
     * Validate that we don't have any stacks violating dependency order in the pipeline
     *
     * Our own convenience methods will never generate a pipeline that does that (although
     * this is a nice verification), but a user can also add the stacks by hand.
     */
    validate() {
        const ret = new Array();
        ret.push(...this.validateDeployOrder());
        ret.push(...this.validateRequestedOutputs());
        return ret;
    }
    /**
     * Return all StackDeployActions in an ordered list
     */
    get stackActions() {
        return flatMap(this._pipeline.stages, s => s.actions.filter(isDeployAction));
    }
    *validateDeployOrder() {
        const stackActions = this.stackActions;
        for (const stackAction of stackActions) {
            // For every dependency, it must be executed in an action before this one is prepared.
            for (const depId of stackAction.dependencyStackArtifactIds) {
                const depAction = stackActions.find(s => s.stackArtifactId === depId);
                if (depAction === undefined) {
                    this.node.addWarning(`Stack '${stackAction.stackName}' depends on stack ` +
                        `'${depId}', but that dependency is not deployed through the pipeline!`);
                }
                else if (!(depAction.executeRunOrder < stackAction.prepareRunOrder)) {
                    yield `Stack '${stackAction.stackName}' depends on stack ` +
                        `'${depAction.stackName}', but is deployed before it in the pipeline!`;
                }
            }
        }
    }
    *validateRequestedOutputs() {
        const artifactIds = this.stackActions.map(s => s.stackArtifactId);
        for (const artifactId of Object.keys(this._outputArtifacts)) {
            if (!artifactIds.includes(artifactId)) {
                yield `Trying to use outputs for Stack '${artifactId}', but Stack is not deployed in this pipeline. Add it to the pipeline.`;
            }
        }
    }
}
exports.CdkPipeline = CdkPipeline;
function isDeployAction(a) {
    return a instanceof actions_1.DeployCdkStackAction;
}
function flatMap(xs, f) {
    return Array.prototype.concat([], ...xs.map(f));
}
/**
 * Add appropriate publishing actions to the asset publishing stage
 */
class AssetPublishing extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.publishers = {};
        this.assetRoles = {};
        this._fileAssetCtr = 1;
        this._dockerAssetCtr = 1;
        this.myCxAsmRoot = path.resolve(construct_internals_1.assemblyBuilderOf(construct_internals_1.appOf(this)).outdir);
        // We MUST add the Stage immediately here, otherwise it will be in the wrong place
        // in the pipeline!
        this.stage = this.props.pipeline.addStage({ stageName: 'Assets' });
        this.pipeline = this.props.pipeline;
    }
    /**
     * Make sure there is an action in the stage to publish the given asset
     *
     * Assets are grouped by asset ID (which represent individual assets) so all assets
     * are published in parallel. For each assets, all destinations are published sequentially
     * so that we can reuse expensive operations between them (mostly: building a Docker image).
     */
    addPublishAssetAction(command) {
        // FIXME: this is silly, we need the relative path here but no easy way to get it
        const relativePath = path.relative(this.myCxAsmRoot, command.assetManifestPath);
        // Late-binding here (rather than in the constructor) to prevent creating the role in cases where no asset actions are created.
        if (!this.assetRoles[command.assetType]) {
            this.generateAssetRole(command.assetType);
        }
        let action = this.publishers[command.assetId];
        if (!action) {
            // The asset ID would be a logical candidate for the construct path and project names, but if the asset
            // changes it leads to recreation of a number of Role/Policy/Project resources which is slower than
            // necessary. Number sequentially instead.
            //
            // FIXME: The ultimate best solution is probably to generate a single Project per asset type
            // and reuse that for all assets.
            const id = command.assetType === actions_1.AssetType.FILE ? `FileAsset${this._fileAssetCtr++}` : `DockerAsset${this._dockerAssetCtr++}`;
            // NOTE: It's important that asset changes don't force a pipeline self-mutation.
            // This can cause an infinite loop of updates (see https://github.com/aws/aws-cdk/issues/9080).
            // For that reason, we use the id as the actionName below, rather than the asset hash.
            action = this.publishers[command.assetId] = new actions_1.PublishAssetsAction(this, id, {
                actionName: id,
                cloudAssemblyInput: this.props.cloudAssemblyInput,
                cdkCliVersion: this.props.cdkCliVersion,
                assetType: command.assetType,
                role: this.assetRoles[command.assetType],
            });
            this.stage.addAction(action);
        }
        action.addPublishCommand(relativePath, command.assetSelector);
    }
    /**
     * Remove the Assets stage if it turns out we didn't add any Assets to publish
     */
    removeAssetsStageIfEmpty() {
        if (Object.keys(this.publishers).length === 0) {
            // Hacks to get access to innards of Pipeline
            // Modify 'stages' array in-place to remove Assets stage if empty
            const stages = this.props.pipeline._stages;
            const ix = stages.indexOf(this.stage);
            if (ix > -1) {
                stages.splice(ix, 1);
            }
        }
    }
    /**
     * This role is used by both the CodePipeline build action and related CodeBuild project. Consolidating these two
     * roles into one, and re-using across all assets, saves significant size of the final synthesized output.
     * Modeled after the CodePipeline role and 'CodePipelineActionRole' roles.
     * Generates one role per asset type to separate file and Docker/image-based permissions.
     */
    generateAssetRole(assetType) {
        if (this.assetRoles[assetType]) {
            return this.assetRoles[assetType];
        }
        const rolePrefix = assetType === actions_1.AssetType.DOCKER_IMAGE ? 'Docker' : 'File';
        const assetRole = new iam.Role(this, `${rolePrefix}Role`, {
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal('codebuild.amazonaws.com'), new iam.AccountPrincipal(core_1.Stack.of(this).account)),
        });
        // Logging permissions
        const logGroupArn = core_1.Stack.of(this).formatArn({
            service: 'logs',
            resource: 'log-group',
            sep: ':',
            resourceName: '/aws/codebuild/*',
        });
        assetRole.addToPolicy(new iam.PolicyStatement({
            resources: [logGroupArn],
            actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
        }));
        // CodeBuild report groups
        const codeBuildArn = core_1.Stack.of(this).formatArn({
            service: 'codebuild',
            resource: 'report-group',
            resourceName: '*',
        });
        assetRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'codebuild:CreateReportGroup',
                'codebuild:CreateReport',
                'codebuild:UpdateReport',
                'codebuild:BatchPutTestCases',
            ],
            resources: [codeBuildArn],
        }));
        // CodeBuild start/stop
        assetRole.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: [
                'codebuild:BatchGetBuilds',
                'codebuild:StartBuild',
                'codebuild:StopBuild',
            ],
        }));
        // Publishing role access
        const rolePattern = assetType === actions_1.AssetType.DOCKER_IMAGE
            ? 'arn:*:iam::*:role/*-image-publishing-role-*'
            : 'arn:*:iam::*:role/*-file-publishing-role-*';
        assetRole.addToPolicy(new iam.PolicyStatement({
            actions: ['sts:AssumeRole'],
            resources: [rolePattern],
        }));
        // Artifact access
        this.pipeline.artifactBucket.grantRead(assetRole);
        this.assetRoles[assetType] = assetRole.withoutPolicyUpdates();
        return this.assetRoles[assetType];
    }
}
function maybeSuffix(x, suffix) {
    if (x === undefined) {
        return undefined;
    }
    return `${x}${suffix}`;
}
//# sourceMappingURL=data:application/json;base64,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