"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.makeUniqueId = void 0;
const crypto = require("crypto");
const encoding_1 = require("./encoding");
/**
 * Resources with this ID are hidden from humans
 *
 * They do not appear in the human-readable part of the logical ID,
 * but they are included in the hash calculation.
 */
const HIDDEN_FROM_HUMAN_ID = 'Resource';
/**
 * Resources with this ID are complete hidden from the logical ID calculation.
 */
const HIDDEN_ID = 'Default';
const PATH_SEP = '/';
const HASH_LEN = 8;
const MAX_HUMAN_LEN = 240; // max ID len is 255
const MAX_ID_LEN = 255;
/**
 * Calculates a unique ID for a set of textual components.
 *
 * This is done by calculating a hash on the full path and using it as a suffix
 * of a length-limited "human" rendition of the path components.
 *
 * @param components The path components
 * @returns a unique alpha-numeric identifier with a maximum length of 255
 */
function makeUniqueId(components) {
    components = components.filter(x => x !== HIDDEN_ID);
    if (components.length === 0) {
        throw new Error('Unable to calculate a unique id for an empty set of components');
    }
    // Lazy require in order to break a module dependency cycle
    const unresolvedTokens = components.filter(c => encoding_1.unresolved(c));
    if (unresolvedTokens.length > 0) {
        throw new Error(`ID components may not include unresolved tokens: ${unresolvedTokens.join(',')}`);
    }
    // top-level resources will simply use the `name` as-is in order to support
    // transparent migration of cloudformation templates to the CDK without the
    // need to rename all resources.
    if (components.length === 1) {
        // we filter out non-alpha characters but that is actually a bad idea
        // because it could create conflicts ("A-B" and "AB" will render the same
        // logical ID). sadly, changing it in the 1.x version line is impossible
        // because it will be a breaking change. we should consider for v2.0.
        // https://github.com/aws/aws-cdk/issues/6421
        const candidate = removeNonAlphanumeric(components[0]);
        // if our candidate is short enough, use it as is. otherwise, fall back to
        // the normal mode.
        if (candidate.length <= MAX_ID_LEN) {
            return candidate;
        }
    }
    const hash = pathHash(components);
    const human = removeDupes(components)
        .filter(x => x !== HIDDEN_FROM_HUMAN_ID)
        .map(removeNonAlphanumeric)
        .join('')
        .slice(0, MAX_HUMAN_LEN);
    return human + hash;
}
exports.makeUniqueId = makeUniqueId;
/**
 * Take a hash of the given path.
 *
 * The hash is limited in size.
 */
function pathHash(path) {
    const md5 = crypto.createHash('md5').update(path.join(PATH_SEP)).digest('hex');
    return md5.slice(0, HASH_LEN).toUpperCase();
}
/**
 * Removes all non-alphanumeric characters in a string.
 */
function removeNonAlphanumeric(s) {
    return s.replace(/[^A-Za-z0-9]/g, '');
}
/**
 * Remove duplicate "terms" from the path list
 *
 * If the previous path component name ends with this component name, skip the
 * current component.
 */
function removeDupes(path) {
    const ret = new Array();
    for (const component of path) {
        if (ret.length === 0 || !ret[ret.length - 1].endsWith(component)) {
            ret.push(component);
        }
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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