"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Fn = void 0;
const cloudformation_lang_1 = require("./private/cloudformation-lang");
const intrinsic_1 = require("./private/intrinsic");
const reference_1 = require("./reference");
const stack_trace_1 = require("./stack-trace");
const token_1 = require("./token");
/* eslint-disable max-len */
/**
 * CloudFormation intrinsic functions.
 * http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference.html
 */
class Fn {
    /**
     * The ``Ref`` intrinsic function returns the value of the specified parameter or resource.
     * Note that it doesn't validate the logicalName, it mainly serves paremeter/resource reference defined in a ``CfnInclude`` template.
     * @param logicalName The logical name of a parameter/resource for which you want to retrieve its value.
     */
    static ref(logicalName) {
        return new FnRef(logicalName).toString();
    }
    /** @internal */
    static _ref(logicalId) {
        return new FnRef(logicalId);
    }
    /**
     * The ``Fn::GetAtt`` intrinsic function returns the value of an attribute
     * from a resource in the template.
     * @param logicalNameOfResource The logical name (also called logical ID) of
     * the resource that contains the attribute that you want.
     * @param attributeName The name of the resource-specific attribute whose
     * value you want. See the resource's reference page for details about the
     * attributes available for that resource type.
     * @returns an IResolvable object
     */
    static getAtt(logicalNameOfResource, attributeName) {
        return new FnGetAtt(logicalNameOfResource, attributeName);
    }
    /**
     * The intrinsic function ``Fn::Join`` appends a set of values into a single
     * value, separated by the specified delimiter. If a delimiter is the empty
     * string, the set of values are concatenated with no delimiter.
     * @param delimiter The value you want to occur between fragments. The
     * delimiter will occur between fragments only. It will not terminate the
     * final value.
     * @param listOfValues The list of values you want combined.
     * @returns a token represented as a string
     */
    static join(delimiter, listOfValues) {
        if (listOfValues.length === 0) {
            throw new Error('FnJoin requires at least one value to be provided');
        }
        return new FnJoin(delimiter, listOfValues).toString();
    }
    /**
     * To split a string into a list of string values so that you can select an element from the
     * resulting string list, use the ``Fn::Split`` intrinsic function. Specify the location of splits
     * with a delimiter, such as , (a comma). After you split a string, use the ``Fn::Select`` function
     * to pick a specific element.
     * @param delimiter A string value that determines where the source string is divided.
     * @param source The string value that you want to split.
     * @returns a token represented as a string array
     */
    static split(delimiter, source) {
        // short-circut if source is not a token
        if (!token_1.Token.isUnresolved(source)) {
            return source.split(delimiter);
        }
        return token_1.Token.asList(new FnSplit(delimiter, source));
    }
    /**
     * The intrinsic function ``Fn::Select`` returns a single object from a list of objects by index.
     * @param index The index of the object to retrieve. This must be a value from zero to N-1, where N represents the number of elements in the array.
     * @param array The list of objects to select from. This list must not be null, nor can it have null entries.
     * @returns a token represented as a string
     */
    static select(index, array) {
        if (!token_1.Token.isUnresolved(array)) {
            return array[index];
        }
        return new FnSelect(index, array).toString();
    }
    /**
     * The intrinsic function ``Fn::Sub`` substitutes variables in an input string
     * with values that you specify. In your templates, you can use this function
     * to construct commands or outputs that include values that aren't available
     * until you create or update a stack.
     * @param body A string with variables that AWS CloudFormation substitutes
     * with their associated values at runtime. Write variables as ${MyVarName}.
     * Variables can be template parameter names, resource logical IDs, resource
     * attributes, or a variable in a key-value map. If you specify only template
     * parameter names, resource logical IDs, and resource attributes, don't
     * specify a key-value map.
     * @param variables The name of a variable that you included in the String
     * parameter. The value that AWS CloudFormation substitutes for the associated
     * variable name at runtime.
     * @returns a token represented as a string
     */
    static sub(body, variables) {
        return new FnSub(body, variables).toString();
    }
    /**
     * The intrinsic function ``Fn::Base64`` returns the Base64 representation of
     * the input string. This function is typically used to pass encoded data to
     * Amazon EC2 instances by way of the UserData property.
     * @param data The string value you want to convert to Base64.
     * @returns a token represented as a string
     */
    static base64(data) {
        return new FnBase64(data).toString();
    }
    /**
     * The intrinsic function ``Fn::Cidr`` returns the specified Cidr address block.
     * @param ipBlock  The user-specified default Cidr address block.
     * @param count  The number of subnets' Cidr block wanted. Count can be 1 to 256.
     * @param sizeMask The digit covered in the subnet.
     * @returns a token represented as a string
     */
    static cidr(ipBlock, count, sizeMask) {
        return token_1.Token.asList(new FnCidr(ipBlock, count, sizeMask));
    }
    /**
     * The intrinsic function ``Fn::GetAZs`` returns an array that lists
     * Availability Zones for a specified region. Because customers have access to
     * different Availability Zones, the intrinsic function ``Fn::GetAZs`` enables
     * template authors to write templates that adapt to the calling user's
     * access. That way you don't have to hard-code a full list of Availability
     * Zones for a specified region.
     * @param region The name of the region for which you want to get the
     * Availability Zones. You can use the AWS::Region pseudo parameter to specify
     * the region in which the stack is created. Specifying an empty string is
     * equivalent to specifying AWS::Region.
     * @returns a token represented as a string array
     */
    static getAzs(region) {
        return token_1.Token.asList(new FnGetAZs(region));
    }
    /**
     * The intrinsic function ``Fn::ImportValue`` returns the value of an output
     * exported by another stack. You typically use this function to create
     * cross-stack references. In the following example template snippets, Stack A
     * exports VPC security group values and Stack B imports them.
     * @param sharedValueToImport The stack output value that you want to import.
     * @returns a token represented as a string
     */
    static importValue(sharedValueToImport) {
        return new FnImportValue(sharedValueToImport).toString();
    }
    /**
     * The intrinsic function ``Fn::FindInMap`` returns the value corresponding to
     * keys in a two-level map that is declared in the Mappings section.
     * @returns a token represented as a string
     */
    static findInMap(mapName, topLevelKey, secondLevelKey) {
        return new FnFindInMap(mapName, topLevelKey, secondLevelKey).toString();
    }
    /**
     * Creates a token representing the ``Fn::Transform`` expression
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-transform.html
     * @param macroName The name of the macro to perform the processing
     * @param parameters The parameters to be passed to the macro
     * @returns a token representing the transform expression
     */
    static transform(macroName, parameters) {
        return new FnTransform(macroName, parameters);
    }
    /**
     * Returns true if all the specified conditions evaluate to true, or returns
     * false if any one of the conditions evaluates to false. ``Fn::And`` acts as
     * an AND operator. The minimum number of conditions that you can include is
     * 2, and the maximum is 10.
     * @param conditions conditions to AND
     * @returns an FnCondition token
     */
    static conditionAnd(...conditions) {
        return new FnAnd(...conditions);
    }
    /**
     * Compares if two values are equal. Returns true if the two values are equal
     * or false if they aren't.
     * @param lhs A value of any type that you want to compare.
     * @param rhs A value of any type that you want to compare.
     * @returns an FnCondition token
     */
    static conditionEquals(lhs, rhs) {
        return new FnEquals(lhs, rhs);
    }
    /**
     * Returns one value if the specified condition evaluates to true and another
     * value if the specified condition evaluates to false. Currently, AWS
     * CloudFormation supports the ``Fn::If`` intrinsic function in the metadata
     * attribute, update policy attribute, and property values in the Resources
     * section and Outputs sections of a template. You can use the AWS::NoValue
     * pseudo parameter as a return value to remove the corresponding property.
     * @param conditionId A reference to a condition in the Conditions section. Use
     * the condition's name to reference it.
     * @param valueIfTrue A value to be returned if the specified condition
     * evaluates to true.
     * @param valueIfFalse A value to be returned if the specified condition
     * evaluates to false.
     * @returns an FnCondition token
     */
    static conditionIf(conditionId, valueIfTrue, valueIfFalse) {
        return new FnIf(conditionId, valueIfTrue, valueIfFalse);
    }
    /**
     * Returns true for a condition that evaluates to false or returns false for a
     * condition that evaluates to true. ``Fn::Not`` acts as a NOT operator.
     * @param condition A condition such as ``Fn::Equals`` that evaluates to true
     * or false.
     * @returns an FnCondition token
     */
    static conditionNot(condition) {
        return new FnNot(condition);
    }
    /**
     * Returns true if any one of the specified conditions evaluate to true, or
     * returns false if all of the conditions evaluates to false. ``Fn::Or`` acts
     * as an OR operator. The minimum number of conditions that you can include is
     * 2, and the maximum is 10.
     * @param conditions conditions that evaluates to true or false.
     * @returns an FnCondition token
     */
    static conditionOr(...conditions) {
        return new FnOr(...conditions);
    }
    /**
     * Returns true if a specified string matches at least one value in a list of
     * strings.
     * @param listOfStrings A list of strings, such as "A", "B", "C".
     * @param value A string, such as "A", that you want to compare against a list of strings.
     * @returns an FnCondition token
     */
    static conditionContains(listOfStrings, value) {
        return new FnContains(listOfStrings, value);
    }
    /**
     * Returns true if a specified string matches all values in a list.
     * @param listOfStrings A list of strings, such as "A", "B", "C".
     * @param value A string, such as "A", that you want to compare against a list
     * of strings.
     * @returns an FnCondition token
     */
    static conditionEachMemberEquals(listOfStrings, value) {
        return new FnEachMemberEquals(listOfStrings, value);
    }
    /**
     * Returns true if each member in a list of strings matches at least one value
     * in a second list of strings.
     * @param stringsToCheck A list of strings, such as "A", "B", "C". AWS
     * CloudFormation checks whether each member in the strings_to_check parameter
     * is in the strings_to_match parameter.
     * @param stringsToMatch A list of strings, such as "A", "B", "C". Each member
     * in the strings_to_match parameter is compared against the members of the
     * strings_to_check parameter.
     * @returns an FnCondition token
     */
    static conditionEachMemberIn(stringsToCheck, stringsToMatch) {
        return new FnEachMemberIn(stringsToCheck, stringsToMatch);
    }
    /**
     * Returns all values for a specified parameter type.
     * @param parameterType An AWS-specific parameter type, such as
     * AWS::EC2::SecurityGroup::Id or AWS::EC2::VPC::Id. For more information, see
     * Parameters in the AWS CloudFormation User Guide.
     * @returns a token represented as a string array
     */
    static refAll(parameterType) {
        return token_1.Token.asList(new FnRefAll(parameterType));
    }
    /**
     * Returns an attribute value or list of values for a specific parameter and
     * attribute.
     * @param parameterOrLogicalId The name of a parameter for which you want to
     * retrieve attribute values. The parameter must be declared in the Parameters
     * section of the template.
     * @param attribute The name of an attribute from which you want to retrieve a
     * value.
     * @returns a token represented as a string
     */
    static valueOf(parameterOrLogicalId, attribute) {
        return new FnValueOf(parameterOrLogicalId, attribute).toString();
    }
    /**
     * Returns a list of all attribute values for a given parameter type and
     * attribute.
     * @param parameterType An AWS-specific parameter type, such as
     * AWS::EC2::SecurityGroup::Id or AWS::EC2::VPC::Id. For more information, see
     * Parameters in the AWS CloudFormation User Guide.
     * @param attribute The name of an attribute from which you want to retrieve a
     * value. For more information about attributes, see Supported Attributes.
     * @returns a token represented as a string array
     */
    static valueOfAll(parameterType, attribute) {
        return token_1.Token.asList(new FnValueOfAll(parameterType, attribute));
    }
    constructor() { }
}
exports.Fn = Fn;
/**
 * Base class for tokens that represent CloudFormation intrinsic functions.
 */
class FnBase extends intrinsic_1.Intrinsic {
    constructor(name, value) {
        super({ [name]: value });
    }
}
/**
 * The intrinsic function ``Ref`` returns the value of the specified parameter or resource.
 * When you specify a parameter's logical name, it returns the value of the parameter.
 * When you specify a resource's logical name, it returns a value that you can typically use to refer to that resource, such as a physical ID.
 */
class FnRef extends FnBase {
    /**
     * Creates an ``Ref`` function.
     * @param logicalName The logical name of a parameter/resource for which you want to retrieve its value.
     */
    constructor(logicalName) {
        super('Ref', logicalName);
    }
}
/**
 * The intrinsic function ``Fn::FindInMap`` returns the value corresponding to keys in a two-level
 * map that is declared in the Mappings section.
 */
class FnFindInMap extends FnBase {
    /**
     * Creates an ``Fn::FindInMap`` function.
     * @param mapName The logical name of a mapping declared in the Mappings section that contains the keys and values.
     * @param topLevelKey The top-level key name. Its value is a list of key-value pairs.
     * @param secondLevelKey The second-level key name, which is set to one of the keys from the list assigned to TopLevelKey.
     */
    constructor(mapName, topLevelKey, secondLevelKey) {
        super('Fn::FindInMap', [mapName, topLevelKey, secondLevelKey]);
    }
}
/**
 * The intrinsic function ``Fn::Transform`` specifies a macro to perform custom processing on part of a stack template.
 */
class FnTransform extends FnBase {
    /**
     * creates an ``Fn::Transform`` function.
     * @param macroName The name of the macro to be invoked
     * @param parameters the parameters to pass to it
     */
    constructor(macroName, parameters) {
        super('Fn::Transform', { Name: macroName, Parameters: parameters });
    }
}
/**
 * The ``Fn::GetAtt`` intrinsic function returns the value of an attribute from a resource in the template.
 */
class FnGetAtt extends FnBase {
    /**
     * Creates a ``Fn::GetAtt`` function.
     * @param logicalNameOfResource The logical name (also called logical ID) of the resource that contains the attribute that you want.
     * @param attributeName The name of the resource-specific attribute whose value you want. See the resource's reference page for details about the attributes available for that resource type.
     */
    constructor(logicalNameOfResource, attributeName) {
        super('Fn::GetAtt', [logicalNameOfResource, attributeName]);
    }
}
/**
 * The intrinsic function ``Fn::GetAZs`` returns an array that lists Availability Zones for a
 * specified region. Because customers have access to different Availability Zones, the intrinsic
 * function ``Fn::GetAZs`` enables template authors to write templates that adapt to the calling
 * user's access. That way you don't have to hard-code a full list of Availability Zones for a
 * specified region.
 */
class FnGetAZs extends FnBase {
    /**
     * Creates an ``Fn::GetAZs`` function.
     * @param region The name of the region for which you want to get the Availability Zones.
     *         You can use the AWS::Region pseudo parameter to specify the region in
     *         which the stack is created. Specifying an empty string is equivalent to
     *         specifying AWS::Region.
     */
    constructor(region) {
        super('Fn::GetAZs', region || '');
    }
}
/**
 * The intrinsic function ``Fn::ImportValue`` returns the value of an output exported by another stack.
 * You typically use this function to create cross-stack references. In the following example
 * template snippets, Stack A exports VPC security group values and Stack B imports them.
 */
class FnImportValue extends FnBase {
    /**
     * Creates an ``Fn::ImportValue`` function.
     * @param sharedValueToImport The stack output value that you want to import.
     */
    constructor(sharedValueToImport) {
        super('Fn::ImportValue', sharedValueToImport);
    }
}
/**
 * The intrinsic function ``Fn::Select`` returns a single object from a list of objects by index.
 */
class FnSelect extends FnBase {
    /**
     * Creates an ``Fn::Select`` function.
     * @param index The index of the object to retrieve. This must be a value from zero to N-1, where N represents the number of elements in the array.
     * @param array The list of objects to select from. This list must not be null, nor can it have null entries.
     */
    constructor(index, array) {
        super('Fn::Select', [index, array]);
    }
}
/**
 * To split a string into a list of string values so that you can select an element from the
 * resulting string list, use the ``Fn::Split`` intrinsic function. Specify the location of splits
 * with a delimiter, such as , (a comma). After you split a string, use the ``Fn::Select`` function
 * to pick a specific element.
 */
class FnSplit extends FnBase {
    /**
     * Create an ``Fn::Split`` function.
     * @param delimiter A string value that determines where the source string is divided.
     * @param source The string value that you want to split.
     */
    constructor(delimiter, source) {
        super('Fn::Split', [delimiter, source]);
    }
}
/**
 * The intrinsic function ``Fn::Sub`` substitutes variables in an input string with values that
 * you specify. In your templates, you can use this function to construct commands or outputs
 * that include values that aren't available until you create or update a stack.
 */
class FnSub extends FnBase {
    /**
     * Creates an ``Fn::Sub`` function.
     * @param body A string with variables that AWS CloudFormation substitutes with their
     *       associated values at runtime. Write variables as ${MyVarName}. Variables
     *       can be template parameter names, resource logical IDs, resource attributes,
     *       or a variable in a key-value map. If you specify only template parameter names,
     *       resource logical IDs, and resource attributes, don't specify a key-value map.
     * @param variables The name of a variable that you included in the String parameter.
     *          The value that AWS CloudFormation substitutes for the associated variable name at runtime.
     */
    constructor(body, variables) {
        super('Fn::Sub', variables ? [body, variables] : body);
    }
}
/**
 * The intrinsic function ``Fn::Base64`` returns the Base64 representation of the input string.
 * This function is typically used to pass encoded data to Amazon EC2 instances by way of
 * the UserData property.
 */
class FnBase64 extends FnBase {
    /**
     * Creates an ``Fn::Base64`` function.
     * @param data The string value you want to convert to Base64.
     */
    constructor(data) {
        super('Fn::Base64', data);
    }
}
/**
 * The intrinsic function ``Fn::Cidr`` returns the specified Cidr address block.
 */
class FnCidr extends FnBase {
    /**
     * Creates an ``Fn::Cidr`` function.
     * @param ipBlock  The user-specified default Cidr address block.
     * @param count  The number of subnets' Cidr block wanted. Count can be 1 to 256.
     * @param sizeMask The digit covered in the subnet.
     */
    constructor(ipBlock, count, sizeMask) {
        if (count < 1 || count > 256) {
            throw new Error(`Fn::Cidr's count attribute must be betwen 1 and 256, ${count} was provided.`);
        }
        super('Fn::Cidr', [ipBlock, count, sizeMask]);
    }
}
class FnConditionBase extends intrinsic_1.Intrinsic {
    constructor(type, value) {
        super({ [type]: value });
    }
}
/**
 * Returns true if all the specified conditions evaluate to true, or returns false if any one
 *  of the conditions evaluates to false. ``Fn::And`` acts as an AND operator. The minimum number of
 * conditions that you can include is 2, and the maximum is 10.
 */
class FnAnd extends FnConditionBase {
    constructor(...condition) {
        super('Fn::And', condition);
    }
}
/**
 * Compares if two values are equal. Returns true if the two values are equal or false
 * if they aren't.
 */
class FnEquals extends FnConditionBase {
    /**
     * Creates an ``Fn::Equals`` condition function.
     * @param lhs A value of any type that you want to compare.
     * @param rhs A value of any type that you want to compare.
     */
    constructor(lhs, rhs) {
        super('Fn::Equals', [lhs, rhs]);
    }
}
/**
 * Returns one value if the specified condition evaluates to true and another value if the
 * specified condition evaluates to false. Currently, AWS CloudFormation supports the ``Fn::If``
 * intrinsic function in the metadata attribute, update policy attribute, and property values
 * in the Resources section and Outputs sections of a template. You can use the AWS::NoValue
 * pseudo parameter as a return value to remove the corresponding property.
 */
class FnIf extends FnConditionBase {
    /**
     * Creates an ``Fn::If`` condition function.
     * @param condition A reference to a condition in the Conditions section. Use the condition's name to reference it.
     * @param valueIfTrue A value to be returned if the specified condition evaluates to true.
     * @param valueIfFalse A value to be returned if the specified condition evaluates to false.
     */
    constructor(condition, valueIfTrue, valueIfFalse) {
        super('Fn::If', [condition, valueIfTrue, valueIfFalse]);
    }
}
/**
 * Returns true for a condition that evaluates to false or returns false for a condition that evaluates to true.
 * ``Fn::Not`` acts as a NOT operator.
 */
class FnNot extends FnConditionBase {
    /**
     * Creates an ``Fn::Not`` condition function.
     * @param condition A condition such as ``Fn::Equals`` that evaluates to true or false.
     */
    constructor(condition) {
        super('Fn::Not', [condition]);
    }
}
/**
 * Returns true if any one of the specified conditions evaluate to true, or returns false if
 * all of the conditions evaluates to false. ``Fn::Or`` acts as an OR operator. The minimum number
 * of conditions that you can include is 2, and the maximum is 10.
 */
class FnOr extends FnConditionBase {
    /**
     * Creates an ``Fn::Or`` condition function.
     * @param condition A condition that evaluates to true or false.
     */
    constructor(...condition) {
        super('Fn::Or', condition);
    }
}
/**
 * Returns true if a specified string matches at least one value in a list of strings.
 */
class FnContains extends FnConditionBase {
    /**
     * Creates an ``Fn::Contains`` function.
     * @param listOfStrings A list of strings, such as "A", "B", "C".
     * @param value A string, such as "A", that you want to compare against a list of strings.
     */
    constructor(listOfStrings, value) {
        super('Fn::Contains', [listOfStrings, value]);
    }
}
/**
 * Returns true if a specified string matches all values in a list.
 */
class FnEachMemberEquals extends FnConditionBase {
    /**
     * Creates an ``Fn::EachMemberEquals`` function.
     * @param listOfStrings A list of strings, such as "A", "B", "C".
     * @param value A string, such as "A", that you want to compare against a list of strings.
     */
    constructor(listOfStrings, value) {
        super('Fn::EachMemberEquals', [listOfStrings, value]);
    }
}
/**
 * Returns true if each member in a list of strings matches at least one value in a second
 * list of strings.
 */
class FnEachMemberIn extends FnConditionBase {
    /**
     * Creates an ``Fn::EachMemberIn`` function.
     * @param stringsToCheck A list of strings, such as "A", "B", "C". AWS CloudFormation checks whether each member in the strings_to_check parameter is in the strings_to_match parameter.
     * @param stringsToMatch A list of strings, such as "A", "B", "C". Each member in the strings_to_match parameter is compared against the members of the strings_to_check parameter.
     */
    constructor(stringsToCheck, stringsToMatch) {
        super('Fn::EachMemberIn', [stringsToCheck, stringsToMatch]);
    }
}
/**
 * Returns all values for a specified parameter type.
 */
class FnRefAll extends FnBase {
    /**
     * Creates an ``Fn::RefAll`` function.
     * @param parameterType An AWS-specific parameter type, such as AWS::EC2::SecurityGroup::Id or
     *            AWS::EC2::VPC::Id. For more information, see Parameters in the AWS
     *            CloudFormation User Guide.
     */
    constructor(parameterType) {
        super('Fn::RefAll', parameterType);
    }
}
/**
 * Returns an attribute value or list of values for a specific parameter and attribute.
 */
class FnValueOf extends FnBase {
    /**
     * Creates an ``Fn::ValueOf`` function.
     * @param parameterOrLogicalId The name of a parameter for which you want to retrieve attribute values. The parameter must be declared in the Parameters section of the template.
     * @param attribute The name of an attribute from which you want to retrieve a value.
     */
    constructor(parameterOrLogicalId, attribute) {
        super('Fn::ValueOf', [parameterOrLogicalId, attribute]);
    }
}
/**
 * Returns a list of all attribute values for a given parameter type and attribute.
 */
class FnValueOfAll extends FnBase {
    /**
     * Creates an ``Fn::ValueOfAll`` function.
     * @param parameterType An AWS-specific parameter type, such as AWS::EC2::SecurityGroup::Id or AWS::EC2::VPC::Id. For more information, see Parameters in the AWS CloudFormation User Guide.
     * @param attribute The name of an attribute from which you want to retrieve a value. For more information about attributes, see Supported Attributes.
     */
    constructor(parameterType, attribute) {
        super('Fn::ValueOfAll', [parameterType, attribute]);
    }
}
/**
 * The intrinsic function ``Fn::Join`` appends a set of values into a single value, separated by
 * the specified delimiter. If a delimiter is the empty string, the set of values are concatenated
 * with no delimiter.
 */
class FnJoin {
    /**
     * Creates an ``Fn::Join`` function.
     * @param delimiter The value you want to occur between fragments. The delimiter will occur between fragments only.
     *          It will not terminate the final value.
     * @param listOfValues The list of values you want combined.
     */
    constructor(delimiter, listOfValues) {
        if (listOfValues.length === 0) {
            throw new Error('FnJoin requires at least one value to be provided');
        }
        this.delimiter = delimiter;
        this.listOfValues = listOfValues;
        this.creationStack = stack_trace_1.captureStackTrace();
    }
    resolve(context) {
        if (token_1.Token.isUnresolved(this.listOfValues)) {
            // This is a list token, don't try to do smart things with it.
            return { 'Fn::Join': [this.delimiter, this.listOfValues] };
        }
        const resolved = this.resolveValues(context);
        if (resolved.length === 1) {
            return resolved[0];
        }
        return { 'Fn::Join': [this.delimiter, resolved] };
    }
    toString() {
        return token_1.Token.asString(this, { displayHint: 'Fn::Join' });
    }
    toJSON() {
        return '<Fn::Join>';
    }
    /**
     * Optimization: if an Fn::Join is nested in another one and they share the same delimiter, then flatten it up. Also,
     * if two concatenated elements are literal strings (not tokens), then pre-concatenate them with the delimiter, to
     * generate shorter output.
     */
    resolveValues(context) {
        const resolvedValues = this.listOfValues.map(x => reference_1.Reference.isReference(x) ? x : context.resolve(x));
        return cloudformation_lang_1.minimalCloudFormationJoin(this.delimiter, resolvedValues);
    }
}
//# sourceMappingURL=data:application/json;base64,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