"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EvaluateExpression = void 0;
const path = require("path");
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const sfn = require("../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
/**
 * A Step Functions Task to evaluate an expression
 *
 * OUTPUT: the output of this task is the evaluated expression.
 *
 * @experimental
 */
class EvaluateExpression extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        this.evalFn = createEvalFn(this.props.runtime || lambda.Runtime.NODEJS_10_X, this);
        this.taskPolicies = [
            new iam.PolicyStatement({
                resources: [this.evalFn.functionArn],
                actions: ['lambda:InvokeFunction'],
            }),
        ];
    }
    /**
     * @internal
     */
    _renderTask() {
        const matches = this.props.expression.match(/\$[.\[][.a-zA-Z[\]0-9]+/g);
        let expressionAttributeValues = {};
        if (matches) {
            expressionAttributeValues = matches.reduce((acc, m) => ({
                ...acc,
                [m]: sfn.JsonPath.stringAt(m),
            }), {});
        }
        const parameters = {
            expression: this.props.expression,
            expressionAttributeValues,
        };
        return {
            Resource: this.evalFn.functionArn,
            Parameters: sfn.FieldUtils.renderObject(parameters),
        };
    }
}
exports.EvaluateExpression = EvaluateExpression;
function createEvalFn(runtime, scope) {
    const code = lambda.Code.asset(path.join(__dirname, `eval-${runtime.name}-handler`));
    const lambdaPurpose = 'Eval';
    switch (runtime) {
        case lambda.Runtime.NODEJS_10_X:
            return new lambda.SingletonFunction(scope, 'EvalFunction', {
                runtime,
                handler: 'index.handler',
                uuid: 'a0d2ce44-871b-4e74-87a1-f5e63d7c3bdc',
                lambdaPurpose,
                code,
            });
        // TODO: implement other runtimes
        default:
            throw new Error(`The runtime ${runtime.name} is currently not supported.`);
    }
}
//# sourceMappingURL=data:application/json;base64,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