"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cluster = exports.ClusterType = exports.NodeType = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const secretsmanager = require("../../aws-secretsmanager"); // Automatically re-written from '@aws-cdk/aws-secretsmanager'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const database_secret_1 = require("./database-secret");
const endpoint_1 = require("./endpoint");
const redshift_generated_1 = require("./redshift.generated");
/**
 * Possible Node Types to use in the cluster
 * used for defining {@link ClusterProps.nodeType}.
 */
var NodeType;
(function (NodeType) {
    /**
     * ds2.xlarge
     */
    NodeType["DS2_XLARGE"] = "ds2.xlarge";
    /**
     * ds2.8xlarge
     */
    NodeType["DS2_8XLARGE"] = "ds2.8xlarge";
    /**
     * dc1.large
     */
    NodeType["DC1_LARGE"] = "dc1.large";
    /**
     * dc1.8xlarge
     */
    NodeType["DC1_8XLARGE"] = "dc1.8xlarge";
    /**
     * dc2.large
     */
    NodeType["DC2_LARGE"] = "dc2.large";
    /**
     * dc2.8xlarge
     */
    NodeType["DC2_8XLARGE"] = "dc2.8xlarge";
    /**
     * ra3.16xlarge
     */
    NodeType["RA3_16XLARGE"] = "ra3.16xlarge";
})(NodeType = exports.NodeType || (exports.NodeType = {}));
/**
 * What cluster type to use.
 * Used by {@link ClusterProps.clusterType}
 */
var ClusterType;
(function (ClusterType) {
    /**
     * single-node cluster, the {@link ClusterProps.numberOfNodes} parameter is not required
     */
    ClusterType["SINGLE_NODE"] = "single-node";
    /**
     * multi-node cluster, set the amount of nodes using {@link ClusterProps.numberOfNodes} parameter
     */
    ClusterType["MULTI_NODE"] = "multi-node";
})(ClusterType = exports.ClusterType || (exports.ClusterType = {}));
/**
 * A new or imported clustered database.
 */
class ClusterBase extends core_1.Resource {
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.clusterName,
            targetType: secretsmanager.AttachmentTargetType.REDSHIFT_CLUSTER,
        };
    }
}
/**
 * Create a Redshift cluster a given number of nodes.
 *
 * @resource AWS::Redshift::Cluster
 */
class Cluster extends ClusterBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.vpc = props.vpc;
        this.vpcSubnets = props.vpcSubnets ? props.vpcSubnets : {
            subnetType: ec2.SubnetType.PRIVATE,
        };
        const removalPolicy = props.removalPolicy ? props.removalPolicy : core_1.RemovalPolicy.RETAIN;
        const { subnetIds } = this.vpc.selectSubnets(this.vpcSubnets);
        const subnetGroup = new redshift_generated_1.CfnClusterSubnetGroup(this, 'Subnets', {
            description: `Subnets for ${id} Redshift cluster`,
            subnetIds,
        });
        subnetGroup.applyRemovalPolicy(removalPolicy, {
            applyToUpdateReplacePolicy: true,
        });
        const securityGroups = props.securityGroups !== undefined ?
            props.securityGroups : [new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: 'Redshift security group',
                vpc: this.vpc,
                securityGroupName: 'redshift SG',
            })];
        const securityGroupIds = securityGroups.map(sg => sg.securityGroupId);
        let secret;
        if (!props.masterUser.masterPassword) {
            secret = new database_secret_1.DatabaseSecret(this, 'Secret', {
                username: props.masterUser.masterUsername,
                encryptionKey: props.masterUser.encryptionKey,
            });
        }
        const clusterType = props.clusterType || ClusterType.MULTI_NODE;
        const nodeCount = this.validateNodeCount(clusterType, props.numberOfNodes);
        if (props.encrypted === false && props.encryptionKey !== undefined) {
            throw new Error('Cannot set property encryptionKey without enabling encryption!');
        }
        this.singleUserRotationApplication = secretsmanager.SecretRotationApplication.REDSHIFT_ROTATION_SINGLE_USER;
        this.multiUserRotationApplication = secretsmanager.SecretRotationApplication.REDSHIFT_ROTATION_MULTI_USER;
        let loggingProperties;
        if (props.loggingBucket) {
            loggingProperties = {
                bucketName: props.loggingBucket.bucketName,
                s3KeyPrefix: props.loggingKeyPrefix,
            };
        }
        const cluster = new redshift_generated_1.CfnCluster(this, 'Resource', {
            // Basic
            allowVersionUpgrade: true,
            automatedSnapshotRetentionPeriod: 1,
            clusterType,
            clusterIdentifier: props.clusterName,
            clusterSubnetGroupName: subnetGroup.ref,
            vpcSecurityGroupIds: securityGroupIds,
            port: props.port,
            clusterParameterGroupName: props.parameterGroup && props.parameterGroup.clusterParameterGroupName,
            // Admin
            masterUsername: secret ? secret.secretValueFromJson('username').toString() : props.masterUser.masterUsername,
            masterUserPassword: secret
                ? secret.secretValueFromJson('password').toString()
                : (props.masterUser.masterPassword
                    ? props.masterUser.masterPassword.toString()
                    : 'default'),
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            nodeType: props.nodeType || NodeType.DC2_LARGE,
            numberOfNodes: nodeCount,
            loggingProperties,
            iamRoles: props.roles ? props.roles.map(role => role.roleArn) : undefined,
            dbName: props.defaultDatabaseName || 'default_db',
            publiclyAccessible: false,
            // Encryption
            kmsKeyId: props.encryptionKey && props.encryptionKey.keyArn,
            encrypted: props.encrypted !== undefined ? props.encrypted : true,
        });
        cluster.applyRemovalPolicy(removalPolicy, {
            applyToUpdateReplacePolicy: true,
        });
        this.clusterName = cluster.ref;
        // create a number token that represents the port of the cluster
        const portAttribute = core_1.Token.asNumber(cluster.attrEndpointPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(cluster.attrEndpointAddress, portAttribute);
        if (secret) {
            this.secret = secret.attach(this);
        }
        const defaultPort = ec2.Port.tcp(this.clusterEndpoint.port);
        this.connections = new ec2.Connections({ securityGroups, defaultPort });
    }
    /**
     * Import an existing DatabaseCluster from properties
     */
    static fromClusterAttributes(scope, id, attrs) {
        class Import extends ClusterBase {
            constructor() {
                super(...arguments);
                this.connections = new ec2.Connections({
                    securityGroups: attrs.securityGroups,
                    defaultPort: ec2.Port.tcp(attrs.clusterEndpointPort),
                });
                this.clusterName = attrs.clusterName;
                this.instanceIdentifiers = [];
                this.clusterEndpoint = new endpoint_1.Endpoint(attrs.clusterEndpointAddress, attrs.clusterEndpointPort);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds the single user rotation of the master password to this cluster.
     *
     * @param [automaticallyAfter=Duration.days(30)] Specifies the number of days after the previous rotation
     * before Secrets Manager triggers the next automatic rotation.
     */
    addRotationSingleUser(automaticallyAfter) {
        if (!this.secret) {
            throw new Error('Cannot add single user rotation for a cluster without secret.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this cluster.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: this.secret,
            automaticallyAfter,
            application: this.singleUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
    /**
     * Adds the multi user rotation to this cluster.
     */
    addRotationMultiUser(id, options) {
        if (!this.secret) {
            throw new Error('Cannot add multi user rotation for a cluster without secret.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: options.secret,
            masterSecret: this.secret,
            automaticallyAfter: options.automaticallyAfter,
            application: this.multiUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
    validateNodeCount(clusterType, numberOfNodes) {
        if (clusterType === ClusterType.SINGLE_NODE) {
            // This property must not be set for single-node clusters; be generous and treat a value of 1 node as undefined.
            if (numberOfNodes !== undefined && numberOfNodes !== 1) {
                throw new Error('Number of nodes must be not be supplied or be 1 for cluster type single-node');
            }
            return undefined;
        }
        else {
            const nodeCount = numberOfNodes !== null && numberOfNodes !== void 0 ? numberOfNodes : 2;
            if (nodeCount < 2 || nodeCount > 100) {
                throw new Error('Number of nodes for cluster type multi-node must be at least 2 and no more than 100');
            }
            return nodeCount;
        }
    }
}
exports.Cluster = Cluster;
//# sourceMappingURL=data:application/json;base64,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