"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ParameterGroup = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const rds_generated_1 = require("./rds.generated");
/**
 * A parameter group.
 * Represents both a cluster parameter group,
 * and an instance parameter group.
 *
 * @resource AWS::RDS::DBParameterGroup
 */
class ParameterGroup extends core_1.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const family = props.engine.parameterGroupFamily;
        if (!family) {
            throw new Error("ParameterGroup cannot be used with an engine that doesn't specify a version");
        }
        this.family = family;
        this.description = props.description;
        this.parameters = (_a = props.parameters) !== null && _a !== void 0 ? _a : {};
    }
    /**
     * Imports a parameter group
     */
    static fromParameterGroupName(scope, id, parameterGroupName) {
        class Import extends core_1.Resource {
            bindToCluster(_options) {
                return { parameterGroupName };
            }
            bindToInstance(_options) {
                return { parameterGroupName };
            }
            addParameter(_key, _value) {
                return false;
            }
        }
        return new Import(scope, id);
    }
    bindToCluster(_options) {
        if (!this.clusterCfnGroup) {
            const id = this.instanceCfnGroup ? 'ClusterParameterGroup' : 'Resource';
            this.clusterCfnGroup = new rds_generated_1.CfnDBClusterParameterGroup(this, id, {
                description: this.description || `Cluster parameter group for ${this.family}`,
                family: this.family,
                parameters: core_1.Lazy.anyValue({ produce: () => this.parameters }),
            });
        }
        return {
            parameterGroupName: this.clusterCfnGroup.ref,
        };
    }
    bindToInstance(_options) {
        if (!this.instanceCfnGroup) {
            const id = this.clusterCfnGroup ? 'InstanceParameterGroup' : 'Resource';
            this.instanceCfnGroup = new rds_generated_1.CfnDBParameterGroup(this, id, {
                description: this.description || `Parameter group for ${this.family}`,
                family: this.family,
                parameters: core_1.Lazy.anyValue({ produce: () => this.parameters }),
            });
        }
        return {
            parameterGroupName: this.instanceCfnGroup.ref,
        };
    }
    /**
     * Add a parameter to this parameter group
     *
     * @param key The key of the parameter to be added
     * @param value The value of the parameter to be added
     */
    addParameter(key, value) {
        this.parameters[key] = value;
        return true;
    }
}
exports.ParameterGroup = ParameterGroup;
//# sourceMappingURL=data:application/json;base64,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