"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OptionGroup = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const rds_generated_1 = require("./rds.generated");
/**
 * An option group
 */
class OptionGroup extends core_1.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        /**
         * The connections object for the options.
         */
        this.optionConnections = {};
        const majorEngineVersion = (_a = props.engine.engineVersion) === null || _a === void 0 ? void 0 : _a.majorVersion;
        if (!majorEngineVersion) {
            throw new Error("OptionGroup cannot be used with an engine that doesn't specify a version");
        }
        const optionGroup = new rds_generated_1.CfnOptionGroup(this, 'Resource', {
            engineName: props.engine.engineType,
            majorEngineVersion,
            optionGroupDescription: props.description || `Option group for ${props.engine.engineType} ${majorEngineVersion}`,
            optionConfigurations: this.renderConfigurations(props.configurations),
        });
        this.optionGroupName = optionGroup.ref;
    }
    /**
     * Import an existing option group.
     */
    static fromOptionGroupName(scope, id, optionGroupName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.optionGroupName = optionGroupName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Renders the option configurations specifications.
     */
    renderConfigurations(configurations) {
        const configs = [];
        for (const config of configurations) {
            let configuration = {
                optionName: config.name,
                optionSettings: config.settings && Object.entries(config.settings).map(([name, value]) => ({ name, value })),
                optionVersion: config.version,
            };
            if (config.port) {
                if (!config.vpc) {
                    throw new Error('`port` and `vpc` must be specified together.');
                }
                const securityGroups = config.securityGroups && config.securityGroups.length > 0
                    ? config.securityGroups
                    : [new ec2.SecurityGroup(this, `SecurityGroup${config.name}`, {
                            description: `Security group for ${config.name} option`,
                            vpc: config.vpc,
                        })];
                this.optionConnections[config.name] = new ec2.Connections({
                    securityGroups: securityGroups,
                    defaultPort: ec2.Port.tcp(config.port),
                });
                configuration = {
                    ...configuration,
                    port: config.port,
                    vpcSecurityGroupMemberships: securityGroups.map(sg => sg.securityGroupId),
                };
            }
            configs.push(configuration);
        }
        return configs;
    }
}
exports.OptionGroup = OptionGroup;
//# sourceMappingURL=data:application/json;base64,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