"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseInstanceEngine = exports.SqlServerEngineVersion = exports.OracleEngineVersion = exports.OracleLegacyEngineVersion = exports.PostgresEngineVersion = exports.MysqlEngineVersion = exports.MariaDbEngineVersion = void 0;
const secretsmanager = require("../../aws-secretsmanager"); // Automatically re-written from '@aws-cdk/aws-secretsmanager'
class InstanceEngineBase {
    constructor(props) {
        var _a;
        this.engineType = props.engineType;
        this.singleUserRotationApplication = props.singleUserRotationApplication;
        this.multiUserRotationApplication = props.multiUserRotationApplication;
        this.engineVersion = props.version;
        this.parameterGroupFamily = (_a = props.parameterGroupFamily) !== null && _a !== void 0 ? _a : (this.engineVersion ? `${this.engineType}${this.engineVersion.majorVersion}` : undefined);
    }
    bindToInstance(_scope, options) {
        if (options.timezone) {
            throw new Error(`timezone property can be configured only for Microsoft SQL Server, not ${this.engineType}`);
        }
        return {};
    }
}
/**
 * The versions for the MariaDB instance engines
 * (those returned by {@link DatabaseInstanceEngine.mariaDb}).
 */
class MariaDbEngineVersion {
    constructor(mariaDbFullVersion, mariaDbMajorVersion) {
        this.mariaDbFullVersion = mariaDbFullVersion;
        this.mariaDbMajorVersion = mariaDbMajorVersion;
    }
    /**
     * Create a new MariaDbEngineVersion with an arbitrary version.
     *
     * @param mariaDbFullVersion the full version string,
     *   for example "10.5.28"
     * @param mariaDbMajorVersion the major version of the engine,
     *   for example "10.5"
     */
    static of(mariaDbFullVersion, mariaDbMajorVersion) {
        return new MariaDbEngineVersion(mariaDbFullVersion, mariaDbMajorVersion);
    }
}
exports.MariaDbEngineVersion = MariaDbEngineVersion;
/** Version "10.0" (only a major version, without a specific minor version). */
MariaDbEngineVersion.VER_10_0 = MariaDbEngineVersion.of('10.0', '10.0');
/** Version "10.0.17". */
MariaDbEngineVersion.VER_10_0_17 = MariaDbEngineVersion.of('10.0.17', '10.0');
/** Version "10.0.24". */
MariaDbEngineVersion.VER_10_0_24 = MariaDbEngineVersion.of('10.0.24', '10.0');
/** Version "10.0.28". */
MariaDbEngineVersion.VER_10_0_28 = MariaDbEngineVersion.of('10.0.28', '10.0');
/** Version "10.0.31". */
MariaDbEngineVersion.VER_10_0_31 = MariaDbEngineVersion.of('10.0.31', '10.0');
/** Version "10.0.32". */
MariaDbEngineVersion.VER_10_0_32 = MariaDbEngineVersion.of('10.0.32', '10.0');
/** Version "10.0.34". */
MariaDbEngineVersion.VER_10_0_34 = MariaDbEngineVersion.of('10.0.34', '10.0');
/** Version "10.0.35". */
MariaDbEngineVersion.VER_10_0_35 = MariaDbEngineVersion.of('10.0.35', '10.0');
/** Version "10.1" (only a major version, without a specific minor version). */
MariaDbEngineVersion.VER_10_1 = MariaDbEngineVersion.of('10.1', '10.1');
/** Version "10.1.14". */
MariaDbEngineVersion.VER_10_1_14 = MariaDbEngineVersion.of('10.1.14', '10.1');
/** Version "10.1.19". */
MariaDbEngineVersion.VER_10_1_19 = MariaDbEngineVersion.of('10.1.19', '10.1');
/** Version "10.1.23". */
MariaDbEngineVersion.VER_10_1_23 = MariaDbEngineVersion.of('10.1.23', '10.1');
/** Version "10.1.26". */
MariaDbEngineVersion.VER_10_1_26 = MariaDbEngineVersion.of('10.1.26', '10.1');
/** Version "10.1.31". */
MariaDbEngineVersion.VER_10_1_31 = MariaDbEngineVersion.of('10.1.31', '10.1');
/** Version "10.1.34". */
MariaDbEngineVersion.VER_10_1_34 = MariaDbEngineVersion.of('10.1.34', '10.1');
/** Version "10.2" (only a major version, without a specific minor version). */
MariaDbEngineVersion.VER_10_2 = MariaDbEngineVersion.of('10.2', '10.2');
/** Version "10.2.11". */
MariaDbEngineVersion.VER_10_2_11 = MariaDbEngineVersion.of('10.2.11', '10.2');
/** Version "10.2.12". */
MariaDbEngineVersion.VER_10_2_12 = MariaDbEngineVersion.of('10.2.12', '10.2');
/** Version "10.2.15". */
MariaDbEngineVersion.VER_10_2_15 = MariaDbEngineVersion.of('10.2.15', '10.2');
/** Version "10.2.21". */
MariaDbEngineVersion.VER_10_2_21 = MariaDbEngineVersion.of('10.2.21', '10.2');
/** Version "10.3" (only a major version, without a specific minor version). */
MariaDbEngineVersion.VER_10_3 = MariaDbEngineVersion.of('10.3', '10.3');
/** Version "10.3.8". */
MariaDbEngineVersion.VER_10_3_8 = MariaDbEngineVersion.of('10.3.8', '10.3');
/** Version "10.3.13". */
MariaDbEngineVersion.VER_10_3_13 = MariaDbEngineVersion.of('10.3.13', '10.3');
/** Version "10.3.20". */
MariaDbEngineVersion.VER_10_3_20 = MariaDbEngineVersion.of('10.3.20', '10.3');
/** Version "10.3.23". */
MariaDbEngineVersion.VER_10_3_23 = MariaDbEngineVersion.of('10.3.23', '10.3');
/** Version "10.4" (only a major version, without a specific minor version). */
MariaDbEngineVersion.VER_10_4 = MariaDbEngineVersion.of('10.4', '10.4');
/** Version "10.4.8". */
MariaDbEngineVersion.VER_10_4_8 = MariaDbEngineVersion.of('10.4.8', '10.4');
/** Version "10.4.13". */
MariaDbEngineVersion.VER_10_4_13 = MariaDbEngineVersion.of('10.4.13', '10.4');
class MariaDbInstanceEngine extends InstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'mariadb',
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.MARIADB_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.MARIADB_ROTATION_MULTI_USER,
            version: version
                ? {
                    fullVersion: version.mariaDbFullVersion,
                    majorVersion: version.mariaDbMajorVersion,
                }
                : undefined,
        });
    }
    bindToInstance(scope, options) {
        if (options.domain) {
            throw new Error(`domain property cannot be configured for ${this.engineType}`);
        }
        return super.bindToInstance(scope, options);
    }
}
/**
 * The versions for the MySQL instance engines
 * (those returned by {@link DatabaseInstanceEngine.mysql}).
 */
class MysqlEngineVersion {
    constructor(mysqlFullVersion, mysqlMajorVersion) {
        this.mysqlFullVersion = mysqlFullVersion;
        this.mysqlMajorVersion = mysqlMajorVersion;
    }
    /**
     * Create a new MysqlEngineVersion with an arbitrary version.
     *
     * @param mysqlFullVersion the full version string,
     *   for example "8.1.43"
     * @param mysqlMajorVersion the major version of the engine,
     *   for example "8.1"
     */
    static of(mysqlFullVersion, mysqlMajorVersion) {
        return new MysqlEngineVersion(mysqlFullVersion, mysqlMajorVersion);
    }
}
exports.MysqlEngineVersion = MysqlEngineVersion;
/** Version "5.5" (only a major version, without a specific minor version). */
MysqlEngineVersion.VER_5_5 = MysqlEngineVersion.of('5.5', '5.5');
/** Version "5.5.46". */
MysqlEngineVersion.VER_5_5_46 = MysqlEngineVersion.of('5.5.46', '5.5');
/** Version "5.5.53". */
MysqlEngineVersion.VER_5_5_53 = MysqlEngineVersion.of('5.5.53', '5.5');
/** Version "5.5.57". */
MysqlEngineVersion.VER_5_5_57 = MysqlEngineVersion.of('5.5.57', '5.5');
/** Version "5.5.59". */
MysqlEngineVersion.VER_5_5_59 = MysqlEngineVersion.of('5.5.59', '5.5');
/** Version "5.5.61". */
MysqlEngineVersion.VER_5_5_61 = MysqlEngineVersion.of('5.5.61', '5.5');
/** Version "5.6" (only a major version, without a specific minor version). */
MysqlEngineVersion.VER_5_6 = MysqlEngineVersion.of('5.6', '5.6');
/** Version "5.6.34". */
MysqlEngineVersion.VER_5_6_34 = MysqlEngineVersion.of('5.6.34', '5.6');
/** Version "5.6.35". */
MysqlEngineVersion.VER_5_6_35 = MysqlEngineVersion.of('5.6.35', '5.6');
/** Version "5.6.37". */
MysqlEngineVersion.VER_5_6_37 = MysqlEngineVersion.of('5.6.37', '5.6');
/** Version "5.6.39". */
MysqlEngineVersion.VER_5_6_39 = MysqlEngineVersion.of('5.6.39', '5.6');
/** Version "5.6.40". */
MysqlEngineVersion.VER_5_6_40 = MysqlEngineVersion.of('5.6.40', '5.6');
/** Version "5.6.41". */
MysqlEngineVersion.VER_5_6_41 = MysqlEngineVersion.of('5.6.41', '5.6');
/** Version "5.6.43". */
MysqlEngineVersion.VER_5_6_43 = MysqlEngineVersion.of('5.6.43', '5.6');
/** Version "5.6.44". */
MysqlEngineVersion.VER_5_6_44 = MysqlEngineVersion.of('5.6.44', '5.6');
/** Version "5.6.46". */
MysqlEngineVersion.VER_5_6_46 = MysqlEngineVersion.of('5.6.46', '5.6');
/** Version "5.6.48". */
MysqlEngineVersion.VER_5_6_48 = MysqlEngineVersion.of('5.6.48', '5.6');
/** Version "5.7" (only a major version, without a specific minor version). */
MysqlEngineVersion.VER_5_7 = MysqlEngineVersion.of('5.7', '5.7');
/** Version "5.7.16". */
MysqlEngineVersion.VER_5_7_16 = MysqlEngineVersion.of('5.7.16', '5.7');
/** Version "5.7.17". */
MysqlEngineVersion.VER_5_7_17 = MysqlEngineVersion.of('5.7.17', '5.7');
/** Version "5.7.19". */
MysqlEngineVersion.VER_5_7_19 = MysqlEngineVersion.of('5.7.19', '5.7');
/** Version "5.7.21". */
MysqlEngineVersion.VER_5_7_21 = MysqlEngineVersion.of('5.7.21', '5.7');
/** Version "5.7.22". */
MysqlEngineVersion.VER_5_7_22 = MysqlEngineVersion.of('5.7.22', '5.7');
/** Version "5.7.23". */
MysqlEngineVersion.VER_5_7_23 = MysqlEngineVersion.of('5.7.23', '5.7');
/** Version "5.7.24". */
MysqlEngineVersion.VER_5_7_24 = MysqlEngineVersion.of('5.7.24', '5.7');
/** Version "5.7.25". */
MysqlEngineVersion.VER_5_7_25 = MysqlEngineVersion.of('5.7.25', '5.7');
/** Version "5.7.26". */
MysqlEngineVersion.VER_5_7_26 = MysqlEngineVersion.of('5.7.26', '5.7');
/** Version "5.7.28". */
MysqlEngineVersion.VER_5_7_28 = MysqlEngineVersion.of('5.7.28', '5.7');
/** Version "5.7.30". */
MysqlEngineVersion.VER_5_7_30 = MysqlEngineVersion.of('5.7.30', '5.7');
/** Version "8.0" (only a major version, without a specific minor version). */
MysqlEngineVersion.VER_8_0 = MysqlEngineVersion.of('8.0', '8.0');
/** Version "8.0.11". */
MysqlEngineVersion.VER_8_0_11 = MysqlEngineVersion.of('8.0.11', '8.0');
/** Version "8.0.13". */
MysqlEngineVersion.VER_8_0_13 = MysqlEngineVersion.of('8.0.13', '8.0');
/** Version "8.0.15". */
MysqlEngineVersion.VER_8_0_15 = MysqlEngineVersion.of('8.0.15', '8.0');
/** Version "8.0.16". */
MysqlEngineVersion.VER_8_0_16 = MysqlEngineVersion.of('8.0.16', '8.0');
/** Version "8.0.17". */
MysqlEngineVersion.VER_8_0_17 = MysqlEngineVersion.of('8.0.17', '8.0');
/** Version "8.0.19". */
MysqlEngineVersion.VER_8_0_19 = MysqlEngineVersion.of('8.0.19', '8.0');
class MySqlInstanceEngine extends InstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'mysql',
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER,
            version: version
                ? {
                    fullVersion: version.mysqlFullVersion,
                    majorVersion: version.mysqlMajorVersion,
                }
                : undefined,
        });
    }
}
/**
 * The versions for the PostgreSQL instance engines
 * (those returned by {@link DatabaseInstanceEngine.postgres}).
 */
class PostgresEngineVersion {
    constructor(postgresFullVersion, postgresMajorVersion) {
        this.postgresFullVersion = postgresFullVersion;
        this.postgresMajorVersion = postgresMajorVersion;
    }
    /**
     * Create a new PostgresEngineVersion with an arbitrary version.
     *
     * @param postgresFullVersion the full version string,
     *   for example "13.11"
     * @param postgresMajorVersion the major version of the engine,
     *   for example "13"
     */
    static of(postgresFullVersion, postgresMajorVersion) {
        return new PostgresEngineVersion(postgresFullVersion, postgresMajorVersion);
    }
}
exports.PostgresEngineVersion = PostgresEngineVersion;
/** Version "9.5" (only a major version, without a specific minor version). */
PostgresEngineVersion.VER_9_5 = PostgresEngineVersion.of('9.5', '9.5');
/** Version "9.5.2". */
PostgresEngineVersion.VER_9_5_2 = PostgresEngineVersion.of('9.5.2', '9.5');
/** Version "9.5.4". */
PostgresEngineVersion.VER_9_5_4 = PostgresEngineVersion.of('9.5.4', '9.5');
/** Version "9.5.6". */
PostgresEngineVersion.VER_9_5_6 = PostgresEngineVersion.of('9.5.6', '9.5');
/** Version "9.5.7". */
PostgresEngineVersion.VER_9_5_7 = PostgresEngineVersion.of('9.5.7', '9.5');
/** Version "9.5.9". */
PostgresEngineVersion.VER_9_5_9 = PostgresEngineVersion.of('9.5.9', '9.5');
/** Version "9.5.10". */
PostgresEngineVersion.VER_9_5_10 = PostgresEngineVersion.of('9.5.10', '9.5');
/** Version "9.5.12". */
PostgresEngineVersion.VER_9_5_12 = PostgresEngineVersion.of('9.5.12', '9.5');
/** Version "9.5.13". */
PostgresEngineVersion.VER_9_5_13 = PostgresEngineVersion.of('9.5.13', '9.5');
/** Version "9.5.14". */
PostgresEngineVersion.VER_9_5_14 = PostgresEngineVersion.of('9.5.14', '9.5');
/** Version "9.5.15". */
PostgresEngineVersion.VER_9_5_15 = PostgresEngineVersion.of('9.5.15', '9.5');
/** Version "9.5.16". */
PostgresEngineVersion.VER_9_5_16 = PostgresEngineVersion.of('9.5.16', '9.5');
/** Version "9.5.18". */
PostgresEngineVersion.VER_9_5_18 = PostgresEngineVersion.of('9.5.18', '9.5');
/** Version "9.5.19". */
PostgresEngineVersion.VER_9_5_19 = PostgresEngineVersion.of('9.5.19', '9.5');
/** Version "9.5.20". */
PostgresEngineVersion.VER_9_5_20 = PostgresEngineVersion.of('9.5.20', '9.5');
/** Version "9.5.21". */
PostgresEngineVersion.VER_9_5_21 = PostgresEngineVersion.of('9.5.21', '9.5');
/** Version "9.5.22". */
PostgresEngineVersion.VER_9_5_22 = PostgresEngineVersion.of('9.5.22', '9.5');
/** Version "9.6" (only a major version, without a specific minor version). */
PostgresEngineVersion.VER_9_6 = PostgresEngineVersion.of('9.6', '9.6');
/** Version "9.6.1". */
PostgresEngineVersion.VER_9_6_1 = PostgresEngineVersion.of('9.6.1', '9.6');
/** Version "9.6.2". */
PostgresEngineVersion.VER_9_6_2 = PostgresEngineVersion.of('9.6.2', '9.6');
/** Version "9.6.3". */
PostgresEngineVersion.VER_9_6_3 = PostgresEngineVersion.of('9.6.3', '9.6');
/** Version "9.6.5". */
PostgresEngineVersion.VER_9_6_5 = PostgresEngineVersion.of('9.6.5', '9.6');
/** Version "9.6.6". */
PostgresEngineVersion.VER_9_6_6 = PostgresEngineVersion.of('9.6.6', '9.6');
/** Version "9.6.8". */
PostgresEngineVersion.VER_9_6_8 = PostgresEngineVersion.of('9.6.8', '9.6');
/** Version "9.6.9". */
PostgresEngineVersion.VER_9_6_9 = PostgresEngineVersion.of('9.6.9', '9.6');
/** Version "9.6.10". */
PostgresEngineVersion.VER_9_6_10 = PostgresEngineVersion.of('9.6.10', '9.6');
/** Version "9.6.11". */
PostgresEngineVersion.VER_9_6_11 = PostgresEngineVersion.of('9.6.11', '9.6');
/** Version "9.6.12". */
PostgresEngineVersion.VER_9_6_12 = PostgresEngineVersion.of('9.6.12', '9.6');
/** Version "9.6.14". */
PostgresEngineVersion.VER_9_6_14 = PostgresEngineVersion.of('9.6.14', '9.6');
/** Version "9.6.15". */
PostgresEngineVersion.VER_9_6_15 = PostgresEngineVersion.of('9.6.15', '9.6');
/** Version "9.6.16". */
PostgresEngineVersion.VER_9_6_16 = PostgresEngineVersion.of('9.6.16', '9.6');
/** Version "9.6.17". */
PostgresEngineVersion.VER_9_6_17 = PostgresEngineVersion.of('9.6.17', '9.6');
/** Version "9.6.18". */
PostgresEngineVersion.VER_9_6_18 = PostgresEngineVersion.of('9.6.18', '9.6');
/** Version "10" (only a major version, without a specific minor version). */
PostgresEngineVersion.VER_10 = PostgresEngineVersion.of('10', '10');
/** Version "10.1". */
PostgresEngineVersion.VER_10_1 = PostgresEngineVersion.of('10.1', '10');
/** Version "10.3". */
PostgresEngineVersion.VER_10_3 = PostgresEngineVersion.of('10.3', '10');
/** Version "10.4". */
PostgresEngineVersion.VER_10_4 = PostgresEngineVersion.of('10.4', '10');
/** Version "10.5". */
PostgresEngineVersion.VER_10_5 = PostgresEngineVersion.of('10.5', '10');
/** Version "10.6". */
PostgresEngineVersion.VER_10_6 = PostgresEngineVersion.of('10.6', '10');
/** Version "10.7". */
PostgresEngineVersion.VER_10_7 = PostgresEngineVersion.of('10.7', '10');
/** Version "10.9". */
PostgresEngineVersion.VER_10_9 = PostgresEngineVersion.of('10.9', '10');
/** Version "10.10". */
PostgresEngineVersion.VER_10_10 = PostgresEngineVersion.of('10.10', '10');
/** Version "10.11". */
PostgresEngineVersion.VER_10_11 = PostgresEngineVersion.of('10.11', '10');
/** Version "10.12". */
PostgresEngineVersion.VER_10_12 = PostgresEngineVersion.of('10.12', '10');
/** Version "10.13". */
PostgresEngineVersion.VER_10_13 = PostgresEngineVersion.of('10.13', '10');
/** Version "11" (only a major version, without a specific minor version). */
PostgresEngineVersion.VER_11 = PostgresEngineVersion.of('11', '11');
/** Version "11.1". */
PostgresEngineVersion.VER_11_1 = PostgresEngineVersion.of('11.1', '11');
/** Version "11.2". */
PostgresEngineVersion.VER_11_2 = PostgresEngineVersion.of('11.2', '11');
/** Version "11.4". */
PostgresEngineVersion.VER_11_4 = PostgresEngineVersion.of('11.4', '11');
/** Version "11.5". */
PostgresEngineVersion.VER_11_5 = PostgresEngineVersion.of('11.5', '11');
/** Version "11.6". */
PostgresEngineVersion.VER_11_6 = PostgresEngineVersion.of('11.6', '11');
/** Version "11.7". */
PostgresEngineVersion.VER_11_7 = PostgresEngineVersion.of('11.7', '11');
/** Version "11.8". */
PostgresEngineVersion.VER_11_8 = PostgresEngineVersion.of('11.8', '11');
/** Version "12" (only a major version, without a specific minor version). */
PostgresEngineVersion.VER_12 = PostgresEngineVersion.of('12', '12');
/** Version "12.2". */
PostgresEngineVersion.VER_12_2 = PostgresEngineVersion.of('12.2', '12');
/** Version "12.3". */
PostgresEngineVersion.VER_12_3 = PostgresEngineVersion.of('12.3', '12');
/**
 * The instance engine for PostgreSQL.
 */
class PostgresInstanceEngine extends InstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'postgres',
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER,
            version: version
                ? {
                    fullVersion: version.postgresFullVersion,
                    majorVersion: version.postgresMajorVersion,
                }
                : undefined,
        });
    }
}
/**
 * The versions for the legacy Oracle instance engines
 * (those returned by {@link DatabaseInstanceEngine.oracleSe}
 * and {@link DatabaseInstanceEngine.oracleSe1}).
 * Note: RDS will stop allowing creating new databases with this version in August 2020.
 */
class OracleLegacyEngineVersion {
    constructor(oracleLegacyFullVersion, oracleLegacyMajorVersion) {
        this.oracleLegacyFullVersion = oracleLegacyFullVersion;
        this.oracleLegacyMajorVersion = oracleLegacyMajorVersion;
    }
    static of(oracleLegacyFullVersion, oracleLegacyMajorVersion) {
        return new OracleLegacyEngineVersion(oracleLegacyFullVersion, oracleLegacyMajorVersion);
    }
}
exports.OracleLegacyEngineVersion = OracleLegacyEngineVersion;
/** Version "11.2" (only a major version, without a specific minor version). */
OracleLegacyEngineVersion.VER_11_2 = OracleLegacyEngineVersion.of('11.2', '11.2');
/** Version "11.2.0.2.v2". */
OracleLegacyEngineVersion.VER_11_2_0_2_V2 = OracleLegacyEngineVersion.of('11.2.0.2.v2', '11.2');
/** Version "11.2.0.4.v1". */
OracleLegacyEngineVersion.VER_11_2_0_4_V1 = OracleLegacyEngineVersion.of('11.2.0.4.v1', '11.2');
/** Version "11.2.0.4.v3". */
OracleLegacyEngineVersion.VER_11_2_0_4_V3 = OracleLegacyEngineVersion.of('11.2.0.4.v3', '11.2');
/** Version "11.2.0.4.v4". */
OracleLegacyEngineVersion.VER_11_2_0_4_V4 = OracleLegacyEngineVersion.of('11.2.0.4.v4', '11.2');
/** Version "11.2.0.4.v5". */
OracleLegacyEngineVersion.VER_11_2_0_4_V5 = OracleLegacyEngineVersion.of('11.2.0.4.v5', '11.2');
/** Version "11.2.0.4.v6". */
OracleLegacyEngineVersion.VER_11_2_0_4_V6 = OracleLegacyEngineVersion.of('11.2.0.4.v6', '11.2');
/** Version "11.2.0.4.v7". */
OracleLegacyEngineVersion.VER_11_2_0_4_V7 = OracleLegacyEngineVersion.of('11.2.0.4.v7', '11.2');
/** Version "11.2.0.4.v8". */
OracleLegacyEngineVersion.VER_11_2_0_4_V8 = OracleLegacyEngineVersion.of('11.2.0.4.v8', '11.2');
/** Version "11.2.0.4.v9". */
OracleLegacyEngineVersion.VER_11_2_0_4_V9 = OracleLegacyEngineVersion.of('11.2.0.4.v9', '11.2');
/** Version "11.2.0.4.v10". */
OracleLegacyEngineVersion.VER_11_2_0_4_V10 = OracleLegacyEngineVersion.of('11.2.0.4.v10', '11.2');
/** Version "11.2.0.4.v11". */
OracleLegacyEngineVersion.VER_11_2_0_4_V11 = OracleLegacyEngineVersion.of('11.2.0.4.v11', '11.2');
/** Version "11.2.0.4.v12". */
OracleLegacyEngineVersion.VER_11_2_0_4_V12 = OracleLegacyEngineVersion.of('11.2.0.4.v12', '11.2');
/** Version "11.2.0.4.v13". */
OracleLegacyEngineVersion.VER_11_2_0_4_V13 = OracleLegacyEngineVersion.of('11.2.0.4.v13', '11.2');
/** Version "11.2.0.4.v14". */
OracleLegacyEngineVersion.VER_11_2_0_4_V14 = OracleLegacyEngineVersion.of('11.2.0.4.v14', '11.2');
/** Version "11.2.0.4.v15". */
OracleLegacyEngineVersion.VER_11_2_0_4_V15 = OracleLegacyEngineVersion.of('11.2.0.4.v15', '11.2');
/** Version "11.2.0.4.v16". */
OracleLegacyEngineVersion.VER_11_2_0_4_V16 = OracleLegacyEngineVersion.of('11.2.0.4.v16', '11.2');
/** Version "11.2.0.4.v17". */
OracleLegacyEngineVersion.VER_11_2_0_4_V17 = OracleLegacyEngineVersion.of('11.2.0.4.v17', '11.2');
/** Version "11.2.0.4.v18". */
OracleLegacyEngineVersion.VER_11_2_0_4_V18 = OracleLegacyEngineVersion.of('11.2.0.4.v18', '11.2');
/** Version "11.2.0.4.v19". */
OracleLegacyEngineVersion.VER_11_2_0_4_V19 = OracleLegacyEngineVersion.of('11.2.0.4.v19', '11.2');
/** Version "11.2.0.4.v20". */
OracleLegacyEngineVersion.VER_11_2_0_4_V20 = OracleLegacyEngineVersion.of('11.2.0.4.v20', '11.2');
/** Version "11.2.0.4.v21". */
OracleLegacyEngineVersion.VER_11_2_0_4_V21 = OracleLegacyEngineVersion.of('11.2.0.4.v21', '11.2');
/** Version "11.2.0.4.v22". */
OracleLegacyEngineVersion.VER_11_2_0_4_V22 = OracleLegacyEngineVersion.of('11.2.0.4.v22', '11.2');
/** Version "11.2.0.4.v23". */
OracleLegacyEngineVersion.VER_11_2_0_4_V23 = OracleLegacyEngineVersion.of('11.2.0.4.v23', '11.2');
/** Version "11.2.0.4.v24". */
OracleLegacyEngineVersion.VER_11_2_0_4_V24 = OracleLegacyEngineVersion.of('11.2.0.4.v24', '11.2');
/**
 * The versions for the Oracle instance engines
 * (those returned by {@link DatabaseInstanceEngine.oracleSe2} and
 * {@link DatabaseInstanceEngine.oracleEe}).
 */
class OracleEngineVersion {
    constructor(oracleFullVersion, oracleMajorVersion) {
        this.oracleFullVersion = oracleFullVersion;
        this.oracleMajorVersion = oracleMajorVersion;
    }
    /**
     * Creates a new OracleEngineVersion with an arbitrary version.
     *
     * @param oracleFullVersion the full version string,
     *   for example "19.0.0.0.ru-2019-10.rur-2019-10.r1"
     * @param oracleMajorVersion the major version of the engine,
     *   for example "19"
     */
    static of(oracleFullVersion, oracleMajorVersion) {
        return new OracleEngineVersion(oracleFullVersion, oracleMajorVersion);
    }
}
exports.OracleEngineVersion = OracleEngineVersion;
/** Version "12.1" (only a major version, without a specific minor version). */
OracleEngineVersion.VER_12_1 = OracleEngineVersion.of('12.1', '12.1');
/** Version "12.1.0.2.v1". */
OracleEngineVersion.VER_12_1_0_2_V1 = OracleEngineVersion.of('12.1.0.2.v1', '12.1');
/** Version "12.1.0.2.v2". */
OracleEngineVersion.VER_12_1_0_2_V2 = OracleEngineVersion.of('12.1.0.2.v2', '12.1');
/** Version "12.1.0.2.v3". */
OracleEngineVersion.VER_12_1_0_2_V3 = OracleEngineVersion.of('12.1.0.2.v3', '12.1');
/** Version "12.1.0.2.v4". */
OracleEngineVersion.VER_12_1_0_2_V4 = OracleEngineVersion.of('12.1.0.2.v4', '12.1');
/** Version "12.1.0.2.v5". */
OracleEngineVersion.VER_12_1_0_2_V5 = OracleEngineVersion.of('12.1.0.2.v5', '12.1');
/** Version "12.1.0.2.v6". */
OracleEngineVersion.VER_12_1_0_2_V6 = OracleEngineVersion.of('12.1.0.2.v6', '12.1');
/** Version "12.1.0.2.v7". */
OracleEngineVersion.VER_12_1_0_2_V7 = OracleEngineVersion.of('12.1.0.2.v7', '12.1');
/** Version "12.1.0.2.v8". */
OracleEngineVersion.VER_12_1_0_2_V8 = OracleEngineVersion.of('12.1.0.2.v8', '12.1');
/** Version "12.1.0.2.v9". */
OracleEngineVersion.VER_12_1_0_2_V9 = OracleEngineVersion.of('12.1.0.2.v9', '12.1');
/** Version "12.1.0.2.v10". */
OracleEngineVersion.VER_12_1_0_2_V10 = OracleEngineVersion.of('12.1.0.2.v10', '12.1');
/** Version "12.1.0.2.v11". */
OracleEngineVersion.VER_12_1_0_2_V11 = OracleEngineVersion.of('12.1.0.2.v11', '12.1');
/** Version "12.1.0.2.v12". */
OracleEngineVersion.VER_12_1_0_2_V12 = OracleEngineVersion.of('12.1.0.2.v12', '12.1');
/** Version "12.1.0.2.v13". */
OracleEngineVersion.VER_12_1_0_2_V13 = OracleEngineVersion.of('12.1.0.2.v13', '12.1');
/** Version "12.1.0.2.v14". */
OracleEngineVersion.VER_12_1_0_2_V14 = OracleEngineVersion.of('12.1.0.2.v14', '12.1');
/** Version "12.1.0.2.v15". */
OracleEngineVersion.VER_12_1_0_2_V15 = OracleEngineVersion.of('12.1.0.2.v15', '12.1');
/** Version "12.1.0.2.v16". */
OracleEngineVersion.VER_12_1_0_2_V16 = OracleEngineVersion.of('12.1.0.2.v16', '12.1');
/** Version "12.1.0.2.v17". */
OracleEngineVersion.VER_12_1_0_2_V17 = OracleEngineVersion.of('12.1.0.2.v17', '12.1');
/** Version "12.1.0.2.v18". */
OracleEngineVersion.VER_12_1_0_2_V18 = OracleEngineVersion.of('12.1.0.2.v18', '12.1');
/** Version "12.1.0.2.v19". */
OracleEngineVersion.VER_12_1_0_2_V19 = OracleEngineVersion.of('12.1.0.2.v19', '12.1');
/** Version "12.1.0.2.v20". */
OracleEngineVersion.VER_12_1_0_2_V20 = OracleEngineVersion.of('12.1.0.2.v20', '12.1');
/** Version "12.2" (only a major version, without a specific minor version). */
OracleEngineVersion.VER_12_2 = OracleEngineVersion.of('12.2', '12.2');
/** Version "12.2.0.1.ru-2018-10.rur-2018-10.r1". */
OracleEngineVersion.VER_12_2_0_1_2018_10_R1 = OracleEngineVersion.of('12.2.0.1.ru-2018-10.rur-2018-10.r1', '12.2');
/** Version "12.2.0.1.ru-2019-01.rur-2019-01.r1". */
OracleEngineVersion.VER_12_2_0_1_2019_01_R1 = OracleEngineVersion.of('12.2.0.1.ru-2019-01.rur-2019-01.r1', '12.2');
/** Version "12.2.0.1.ru-2019-04.rur-2019-04.r1". */
OracleEngineVersion.VER_12_2_0_1_2019_04_R1 = OracleEngineVersion.of('12.2.0.1.ru-2019-04.rur-2019-04.r1', '12.2');
/** Version "12.2.0.1.ru-2019-07.rur-2019-07.r1". */
OracleEngineVersion.VER_12_2_0_1_2019_07_R1 = OracleEngineVersion.of('12.2.0.1.ru-2019-07.rur-2019-07.r1', '12.2');
/** Version "12.2.0.1.ru-2019-10.rur-2019-10.r1". */
OracleEngineVersion.VER_12_2_0_1_2019_10_R1 = OracleEngineVersion.of('12.2.0.1.ru-2019-10.rur-2019-10.r1', '12.2');
/** Version "12.2.0.1.ru-2020-01.rur-2020-01.r1". */
OracleEngineVersion.VER_12_2_0_1_2020_01_R1 = OracleEngineVersion.of('12.2.0.1.ru-2020-01.rur-2020-01.r1', '12.2');
/** Version "12.2.0.1.ru-2020-04.rur-2020-04.r1". */
OracleEngineVersion.VER_12_2_0_1_2020_04_R1 = OracleEngineVersion.of('12.2.0.1.ru-2020-04.rur-2020-04.r1', '12.2');
/** Version "18" (only a major version, without a specific minor version). */
OracleEngineVersion.VER_18 = OracleEngineVersion.of('18', '18');
/** Version "18.0.0.0.ru-2019-07.rur-2019-07.r1". */
OracleEngineVersion.VER_18_0_0_0_2019_07_R1 = OracleEngineVersion.of('18.0.0.0.ru-2019-07.rur-2019-07.r1', '18');
/** Version "18.0.0.0.ru-2019-10.rur-2019-10.r1". */
OracleEngineVersion.VER_18_0_0_0_2019_10_R1 = OracleEngineVersion.of('18.0.0.0.ru-2019-10.rur-2019-10.r1', '18');
/** Version "18.0.0.0.ru-2020-01.rur-2020-01.r1". */
OracleEngineVersion.VER_18_0_0_0_2020_01_R1 = OracleEngineVersion.of('18.0.0.0.ru-2020-01.rur-2020-01.r1', '18');
/** Version "18.0.0.0.ru-2020-04.rur-2020-04.r1". */
OracleEngineVersion.VER_18_0_0_0_2020_04_R1 = OracleEngineVersion.of('18.0.0.0.ru-2020-04.rur-2020-04.r1', '18');
/** Version "19" (only a major version, without a specific minor version). */
OracleEngineVersion.VER_19 = OracleEngineVersion.of('19', '19');
/** Version "19.0.0.0.ru-2019-07.rur-2019-07.r1". */
OracleEngineVersion.VER_19_0_0_0_2019_07_R1 = OracleEngineVersion.of('19.0.0.0.ru-2019-07.rur-2019-07.r1', '19');
/** Version "19.0.0.0.ru-2019-10.rur-2019-10.r1". */
OracleEngineVersion.VER_19_0_0_0_2019_10_R1 = OracleEngineVersion.of('19.0.0.0.ru-2019-10.rur-2019-10.r1', '19');
/** Version "19.0.0.0.ru-2020-01.rur-2020-01.r1". */
OracleEngineVersion.VER_19_0_0_0_2020_01_R1 = OracleEngineVersion.of('19.0.0.0.ru-2020-01.rur-2020-01.r1', '19');
/** Version "19.0.0.0.ru-2020-04.rur-2020-04.r1". */
OracleEngineVersion.VER_19_0_0_0_2020_04_R1 = OracleEngineVersion.of('19.0.0.0.ru-2020-04.rur-2020-04.r1', '19');
class OracleInstanceEngineBase extends InstanceEngineBase {
    constructor(props) {
        super({
            ...props,
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.ORACLE_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.ORACLE_ROTATION_MULTI_USER,
            parameterGroupFamily: props.version ? `${props.engineType}-${props.version.majorVersion}` : undefined,
        });
    }
}
class OracleSeInstanceEngine extends OracleInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'oracle-se',
            version: version
                ? {
                    fullVersion: version.oracleLegacyFullVersion,
                    majorVersion: version.oracleLegacyMajorVersion,
                }
                : {
                    majorVersion: '11.2',
                },
        });
    }
}
class OracleSe1InstanceEngine extends OracleInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'oracle-se1',
            version: version
                ? {
                    fullVersion: version.oracleLegacyFullVersion,
                    majorVersion: version.oracleLegacyMajorVersion,
                }
                : {
                    majorVersion: '11.2',
                },
        });
    }
}
class OracleSe2InstanceEngine extends OracleInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'oracle-se2',
            version: version
                ? {
                    fullVersion: version.oracleFullVersion,
                    majorVersion: version.oracleMajorVersion,
                }
                : undefined,
        });
    }
}
class OracleEeInstanceEngine extends OracleInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'oracle-ee',
            version: version
                ? {
                    fullVersion: version.oracleFullVersion,
                    majorVersion: version.oracleMajorVersion,
                }
                : undefined,
        });
    }
}
/**
 * The versions for the SQL Server instance engines
 * (those returned by {@link DatabaseInstanceEngine.sqlServerSe},
 * {@link DatabaseInstanceEngine.sqlServerEx}, {@link DatabaseInstanceEngine.sqlServerWeb}
 * and {@link DatabaseInstanceEngine.sqlServerEe}).
 */
class SqlServerEngineVersion {
    constructor(sqlServerFullVersion, sqlServerMajorVersion) {
        this.sqlServerFullVersion = sqlServerFullVersion;
        this.sqlServerMajorVersion = sqlServerMajorVersion;
    }
    /**
     * Create a new SqlServerEngineVersion with an arbitrary version.
     *
     * @param sqlServerFullVersion the full version string,
     *   for example "15.00.3049.1.v1"
     * @param sqlServerMajorVersion the major version of the engine,
     *   for example "15.00"
     */
    static of(sqlServerFullVersion, sqlServerMajorVersion) {
        return new SqlServerEngineVersion(sqlServerFullVersion, sqlServerMajorVersion);
    }
}
exports.SqlServerEngineVersion = SqlServerEngineVersion;
/** Version "11.00" (only a major version, without a specific minor version). */
SqlServerEngineVersion.VER_11 = SqlServerEngineVersion.of('11.00', '11.00');
/** Version "11.00.5058.0.v1". */
SqlServerEngineVersion.VER_11_00_5058_0_V1 = SqlServerEngineVersion.of('11.00.5058.0.v1', '11.00');
/** Version "11.00.6020.0.v1". */
SqlServerEngineVersion.VER_11_00_6020_0_V1 = SqlServerEngineVersion.of('11.00.6020.0.v1', '11.00');
/** Version "11.00.6594.0.v1". */
SqlServerEngineVersion.VER_11_00_6594_0_V1 = SqlServerEngineVersion.of('11.00.6594.0.v1', '11.00');
/** Version "11.00.7462.6.v1". */
SqlServerEngineVersion.VER_11_00_7462_6_V1 = SqlServerEngineVersion.of('11.00.7462.6.v1', '11.00');
/** Version "11.00.7493.4.v1". */
SqlServerEngineVersion.VER_11_00_7493_4_V1 = SqlServerEngineVersion.of('11.00.7493.4.v1', '11.00');
/** Version "12.00" (only a major version, without a specific minor version). */
SqlServerEngineVersion.VER_12 = SqlServerEngineVersion.of('12.00', '12.00');
/** Version "12.00.5000.0.v1". */
SqlServerEngineVersion.VER_12_00_5000_0_V1 = SqlServerEngineVersion.of('12.00.5000.0.v1', '12.00');
/** Version "12.00.5546.0.v1". */
SqlServerEngineVersion.VER_12_00_5546_0_V1 = SqlServerEngineVersion.of('12.00.5546.0.v1', '12.00');
/** Version "12.00.5571.0.v1". */
SqlServerEngineVersion.VER_12_00_5571_0_V1 = SqlServerEngineVersion.of('12.00.5571.0.v1', '12.00');
/** Version "12.00.6293.0.v1". */
SqlServerEngineVersion.VER_12_00_6293_0_V1 = SqlServerEngineVersion.of('12.00.6293.0.v1', '12.00');
/** Version "12.00.6329.1.v1". */
SqlServerEngineVersion.VER_12_00_6329_1_V1 = SqlServerEngineVersion.of('12.00.6329.1.v1', '12.00');
/** Version "13.00" (only a major version, without a specific minor version). */
SqlServerEngineVersion.VER_13 = SqlServerEngineVersion.of('13.00', '13.00');
/** Version "13.00.2164.0.v1". */
SqlServerEngineVersion.VER_13_00_2164_0_V1 = SqlServerEngineVersion.of('13.00.2164.0.v1', '13.00');
/** Version "13.00.4422.0.v1". */
SqlServerEngineVersion.VER_13_00_4422_0_V1 = SqlServerEngineVersion.of('13.00.4422.0.v1', '13.00');
/** Version "13.00.4451.0.v1". */
SqlServerEngineVersion.VER_13_00_4451_0_V1 = SqlServerEngineVersion.of('13.00.4451.0.v1', '13.00');
/** Version "13.00.4466.4.v1". */
SqlServerEngineVersion.VER_13_00_4466_4_V1 = SqlServerEngineVersion.of('13.00.4466.4.v1', '13.00');
/** Version "13.00.4522.0.v1". */
SqlServerEngineVersion.VER_13_00_4522_0_V1 = SqlServerEngineVersion.of('13.00.4522.0.v1', '13.00');
/** Version "13.00.5216.0.v1". */
SqlServerEngineVersion.VER_13_00_5216_0_V1 = SqlServerEngineVersion.of('13.00.5216.0.v1', '13.00');
/** Version "13.00.5292.0.v1". */
SqlServerEngineVersion.VER_13_00_5292_0_V1 = SqlServerEngineVersion.of('13.00.5292.0.v1', '13.00');
/** Version "13.00.5366.0.v1". */
SqlServerEngineVersion.VER_13_00_5366_0_V1 = SqlServerEngineVersion.of('13.00.5366.0.v1', '13.00');
/** Version "13.00.5426.0.v1". */
SqlServerEngineVersion.VER_13_00_5426_0_V1 = SqlServerEngineVersion.of('13.00.5426.0.v1', '13.00');
/** Version "13.00.5598.27.v1". */
SqlServerEngineVersion.VER_13_00_5598_27_V1 = SqlServerEngineVersion.of('13.00.5598.27.v1', '13.00');
/** Version "14.00" (only a major version, without a specific minor version). */
SqlServerEngineVersion.VER_14 = SqlServerEngineVersion.of('14.00', '14.00');
/** Version "14.00.1000.169.v1". */
SqlServerEngineVersion.VER_14_00_1000_169_V1 = SqlServerEngineVersion.of('14.00.1000.169.v1', '14.00');
/** Version "14.00.3015.40.v1". */
SqlServerEngineVersion.VER_14_00_3015_40_V1 = SqlServerEngineVersion.of('14.00.3015.40.v1', '14.00');
/** Version "14.00.3035.2.v1". */
SqlServerEngineVersion.VER_14_00_3035_2_V1 = SqlServerEngineVersion.of('14.00.3035.2.v1', '14.00');
/** Version "14.00.3049.1.v1". */
SqlServerEngineVersion.VER_14_00_3049_1_V1 = SqlServerEngineVersion.of('14.00.3049.1.v1', '14.00');
/** Version "14.00.3192.2.v1". */
SqlServerEngineVersion.VER_14_00_3192_2_V1 = SqlServerEngineVersion.of('14.00.3192.2.v1', '14.00');
class SqlServerInstanceEngineBase extends InstanceEngineBase {
    constructor(props) {
        super({
            ...props,
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_MULTI_USER,
            version: props.version
                ? {
                    fullVersion: props.version.sqlServerFullVersion,
                    majorVersion: props.version.sqlServerMajorVersion,
                }
                : undefined,
            parameterGroupFamily: props.version
                // for some reason, even though SQL Server major versions usually end in '.00',
                // the ParameterGroup family has to end in '.0'
                ? `${props.engineType}-${props.version.sqlServerMajorVersion.endsWith('.00')
                    ? props.version.sqlServerMajorVersion.slice(0, -1)
                    : props.version.sqlServerMajorVersion}`
                : undefined,
        });
    }
    bindToInstance(_scope, _options) {
        return {};
    }
}
class SqlServerSeInstanceEngine extends SqlServerInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'sqlserver-se',
            version,
        });
    }
}
class SqlServerExInstanceEngine extends SqlServerInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'sqlserver-ex',
            version,
        });
    }
}
class SqlServerWebInstanceEngine extends SqlServerInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'sqlserver-web',
            version,
        });
    }
}
class SqlServerEeInstanceEngine extends SqlServerInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'sqlserver-ee',
            version,
        });
    }
}
/**
 * A database instance engine. Provides mapping to DatabaseEngine used for
 * secret rotation.
 */
class DatabaseInstanceEngine {
    /** Creates a new MariaDB instance engine. */
    static mariaDb(props) {
        return new MariaDbInstanceEngine(props.version);
    }
    /** Creates a new MySQL instance engine. */
    static mysql(props) {
        return new MySqlInstanceEngine(props.version);
    }
    /** Creates a new PostgreSQL instance engine. */
    static postgres(props) {
        return new PostgresInstanceEngine(props.version);
    }
    /** Creates a new Oracle Standard Edition instance engine. */
    static oracleSe(props) {
        return new OracleSeInstanceEngine(props.version);
    }
    /** Creates a new Oracle Standard Edition 1 instance engine. */
    static oracleSe1(props) {
        return new OracleSe1InstanceEngine(props.version);
    }
    /** Creates a new Oracle Standard Edition 1 instance engine. */
    static oracleSe2(props) {
        return new OracleSe2InstanceEngine(props.version);
    }
    /** Creates a new Oracle Enterprise Edition instance engine. */
    static oracleEe(props) {
        return new OracleEeInstanceEngine(props.version);
    }
    /** Creates a new SQL Server Standard Edition instance engine. */
    static sqlServerSe(props) {
        return new SqlServerSeInstanceEngine(props.version);
    }
    /** Creates a new SQL Server Express Edition instance engine. */
    static sqlServerEx(props) {
        return new SqlServerExInstanceEngine(props.version);
    }
    /** Creates a new SQL Server Web Edition instance engine. */
    static sqlServerWeb(props) {
        return new SqlServerWebInstanceEngine(props.version);
    }
    /** Creates a new SQL Server Enterprise Edition instance engine. */
    static sqlServerEe(props) {
        return new SqlServerEeInstanceEngine(props.version);
    }
}
exports.DatabaseInstanceEngine = DatabaseInstanceEngine;
/**
 * The unversioned 'mariadb' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link mariaDb()} method
 */
DatabaseInstanceEngine.MARIADB = new MariaDbInstanceEngine();
/**
 * The unversioned 'mysql' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link mysql()} method
 */
DatabaseInstanceEngine.MYSQL = new MySqlInstanceEngine();
/**
 * The unversioned 'oracle-ee' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link oracleEe()} method
 */
DatabaseInstanceEngine.ORACLE_EE = new OracleEeInstanceEngine();
/**
 * The unversioned 'oracle-se2' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link oracleSe2()} method
 */
DatabaseInstanceEngine.ORACLE_SE2 = new OracleSe2InstanceEngine();
/**
 * The unversioned 'oracle-se1' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link oracleSe1()} method
 */
DatabaseInstanceEngine.ORACLE_SE1 = new OracleSe1InstanceEngine();
/**
 * The unversioned 'oracle-se' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link oracleSe()} method
 */
DatabaseInstanceEngine.ORACLE_SE = new OracleSeInstanceEngine();
/**
 * The unversioned 'postgres' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link postgres()} method
 */
DatabaseInstanceEngine.POSTGRES = new PostgresInstanceEngine();
/**
 * The unversioned 'sqlserver-ee' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link sqlServerEe()} method
 */
DatabaseInstanceEngine.SQL_SERVER_EE = new SqlServerEeInstanceEngine();
/**
 * The unversioned 'sqlserver-se' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link sqlServerSe()} method
 */
DatabaseInstanceEngine.SQL_SERVER_SE = new SqlServerSeInstanceEngine();
/**
 * The unversioned 'sqlserver-ex' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link sqlServerEx()} method
 */
DatabaseInstanceEngine.SQL_SERVER_EX = new SqlServerExInstanceEngine();
/**
 * The unversioned 'sqlserver-web' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link sqlServerWeb()} method
 */
DatabaseInstanceEngine.SQL_SERVER_WEB = new SqlServerWebInstanceEngine();
//# sourceMappingURL=data:application/json;base64,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