"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SingletonFunction = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const function_1 = require("./function");
const function_base_1 = require("./function-base");
/**
 * A Lambda that will only ever be added to a stack once.
 *
 * This construct is a way to guarantee that the lambda function will be guaranteed to be part of the stack,
 * once and only once, irrespective of how many times the construct is declared to be part of the stack.
 * This is guaranteed as long as the `uuid` property and the optional `lambdaPurpose` property stay the same
 * whenever they're declared into the stack.
 *
 * @resource AWS::Lambda::Function
 */
class SingletonFunction extends function_base_1.FunctionBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.lambdaFunction = this.ensureLambda(props);
        this.permissionsNode = this.lambdaFunction.node;
        this.functionArn = this.lambdaFunction.functionArn;
        this.functionName = this.lambdaFunction.functionName;
        this.role = this.lambdaFunction.role;
        this.grantPrincipal = this.lambdaFunction.grantPrincipal;
        this.canCreatePermissions = true; // Doesn't matter, addPermission is overriden anyway
    }
    /**
     * Returns a `lambda.Version` which represents the current version of this
     * singleton Lambda function. A new version will be created every time the
     * function's configuration changes.
     *
     * You can specify options for this version using the `currentVersionOptions`
     * prop when initializing the `lambda.SingletonFunction`.
     */
    get currentVersion() {
        return this.lambdaFunction.currentVersion;
    }
    addPermission(name, permission) {
        return this.lambdaFunction.addPermission(name, permission);
    }
    /**
     * Using node.addDependency() does not work on this method as the underlying lambda function is modeled
     * as a singleton across the stack. Use this method instead to declare dependencies.
     */
    addDependency(...up) {
        this.lambdaFunction.node.addDependency(...up);
    }
    /**
     * The SingletonFunction construct cannot be added as a dependency of another construct using
     * node.addDependency(). Use this method instead to declare this as a dependency of another construct.
     */
    dependOn(down) {
        down.node.addDependency(this.lambdaFunction);
    }
    /** @internal */
    _checkEdgeCompatibility() {
        return this.lambdaFunction._checkEdgeCompatibility();
    }
    /**
     * Returns the construct tree node that corresponds to the lambda function.
     * @internal
     */
    _functionNode() {
        return this.lambdaFunction.node;
    }
    ensureLambda(props) {
        const constructName = (props.lambdaPurpose || 'SingletonLambda') + slugify(props.uuid);
        const existing = cdk.Stack.of(this).node.tryFindChild(constructName);
        if (existing) {
            // Just assume this is true
            return existing;
        }
        return new function_1.Function(cdk.Stack.of(this), constructName, props);
    }
}
exports.SingletonFunction = SingletonFunction;
function slugify(x) {
    return x.replace(/[^a-zA-Z0-9]/g, '');
}
//# sourceMappingURL=data:application/json;base64,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