import * as cloudwatch from "../../aws-cloudwatch";
import * as ec2 from "../../aws-ec2";
import * as iam from "../../aws-iam";
import { ConstructNode, IResource, Resource } from "../../core";
import { EventInvokeConfigOptions } from './event-invoke-config';
import { IEventSource } from './event-source';
import { EventSourceMapping, EventSourceMappingOptions } from './event-source-mapping';
import { IVersion } from './lambda-version';
import { Permission } from './permission';
export interface IFunction extends IResource, ec2.IConnectable, iam.IGrantable {
    /**
     * The name of the function.
     *
     * @attribute
     */
    readonly functionName: string;
    /**
     * The ARN fo the function.
     *
     * @attribute
     */
    readonly functionArn: string;
    /**
     * The IAM role associated with this function.
     */
    readonly role?: iam.IRole;
    /**
     * Whether or not this Lambda function was bound to a VPC
     *
     * If this is is `false`, trying to access the `connections` object will fail.
     */
    readonly isBoundToVpc: boolean;
    /**
     * The `$LATEST` version of this function.
     *
     * Note that this is reference to a non-specific AWS Lambda version, which
     * means the function this version refers to can return different results in
     * different invocations.
     *
     * To obtain a reference to an explicit version which references the current
     * function configuration, use `lambdaFunction.currentVersion` instead.
     */
    readonly latestVersion: IVersion;
    /**
     * The construct node where permissions are attached.
     */
    readonly permissionsNode: ConstructNode;
    /**
     * Adds an event source that maps to this AWS Lambda function.
     * @param id construct ID
     * @param options mapping options
     */
    addEventSourceMapping(id: string, options: EventSourceMappingOptions): EventSourceMapping;
    /**
     * Adds a permission to the Lambda resource policy.
     * @param id The id ƒor the permission construct
     * @param permission The permission to grant to this Lambda function. @see Permission for details.
     */
    addPermission(id: string, permission: Permission): void;
    /**
     * Adds a statement to the IAM role assumed by the instance.
     */
    addToRolePolicy(statement: iam.PolicyStatement): void;
    /**
     * Grant the given identity permissions to invoke this Lambda
     */
    grantInvoke(identity: iam.IGrantable): iam.Grant;
    /**
     * Return the given named metric for this Lambda
     */
    metric(metricName: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the Duration of this Lambda
     *
     * @default average over 5 minutes
     */
    metricDuration(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the number of invocations of this Lambda
     *
     * @default sum over 5 minutes
     */
    metricInvocations(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the number of throttled invocations of this Lambda
     *
     * @default sum over 5 minutes
     */
    metricThrottles(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    addEventSource(source: IEventSource): void;
    /**
     * Configures options for asynchronous invocation.
     */
    configureAsyncInvoke(options: EventInvokeConfigOptions): void;
}
/**
 * Represents a Lambda function defined outside of this stack.
 */
export interface FunctionAttributes {
    /**
     * The ARN of the Lambda function.
     *
     * Format: arn:<partition>:lambda:<region>:<account-id>:function:<function-name>
     */
    readonly functionArn: string;
    /**
     * The IAM execution role associated with this function.
     *
     * If the role is not specified, any role-related operations will no-op.
     */
    readonly role?: iam.IRole;
    /**
     * Id of the security group of this Lambda, if in a VPC.
     *
     * This needs to be given in order to support allowing connections
     * to this Lambda.
     *
     * @deprecated use `securityGroup` instead
     */
    readonly securityGroupId?: string;
    /**
     * The security group of this Lambda, if in a VPC.
     *
     * This needs to be given in order to support allowing connections
     * to this Lambda.
     */
    readonly securityGroup?: ec2.ISecurityGroup;
}
export declare abstract class FunctionBase extends Resource implements IFunction {
    /**
     * The principal this Lambda Function is running as
     */
    abstract readonly grantPrincipal: iam.IPrincipal;
    /**
     * The name of the function.
     */
    abstract readonly functionName: string;
    /**
     * The ARN fo the function.
     */
    abstract readonly functionArn: string;
    /**
     * The IAM role associated with this function.
     *
     * Undefined if the function was imported without a role.
     */
    abstract readonly role?: iam.IRole;
    /**
     * The construct node where permissions are attached.
     */
    abstract readonly permissionsNode: ConstructNode;
    /**
     * Whether the addPermission() call adds any permissions
     *
     * True for new Lambdas, false for imported Lambdas (they might live in different accounts).
     */
    protected abstract readonly canCreatePermissions: boolean;
    private _latestVersion?;
    /**
     * Adds a permission to the Lambda resource policy.
     * @param id The id ƒor the permission construct
     * @param permission The permission to grant to this Lambda function. @see Permission for details.
     */
    addPermission(id: string, permission: Permission): void;
    /**
     * Adds a statement to the IAM role assumed by the instance.
     */
    addToRolePolicy(statement: iam.PolicyStatement): void;
    /**
     * Access the Connections object
     *
     * Will fail if not a VPC-enabled Lambda Function
     */
    get connections(): ec2.Connections;
    get latestVersion(): IVersion;
    /**
     * Whether or not this Lambda function was bound to a VPC
     *
     * If this is is `false`, trying to access the `connections` object will fail.
     */
    get isBoundToVpc(): boolean;
    addEventSourceMapping(id: string, options: EventSourceMappingOptions): EventSourceMapping;
    /**
     * Grant the given identity permissions to invoke this Lambda
     */
    grantInvoke(grantee: iam.IGrantable): iam.Grant;
    /**
     * Adds an event source to this function.
     *
     * Event sources are implemented in the @aws-cdk/aws-lambda-event-sources module.
     *
     * The following example adds an SQS Queue as an event source:
     *
     *     import { SqsEventSource } from '@aws-cdk/aws-lambda-event-sources';
     *     myFunction.addEventSource(new SqsEventSource(myQueue));
     *
     * @param source The event source to bind to this function
     */
    addEventSource(source: IEventSource): void;
    configureAsyncInvoke(options: EventInvokeConfigOptions): void;
    /**
     * Translate IPrincipal to something we can pass to AWS::Lambda::Permissions
     *
     * Do some nasty things because `Permission` supports a subset of what the
     * full IAM principal language supports, and we may not be able to parse strings
     * outright because they may be tokens.
     *
     * Try to recognize some specific Principal classes first, then try a generic
     * fallback.
     */
    private parsePermissionPrincipal;
}
export declare abstract class QualifiedFunctionBase extends FunctionBase {
    abstract readonly lambda: IFunction;
    readonly permissionsNode: ConstructNode;
    /**
     * The qualifier of the version or alias of this function.
     * A qualifier is the identifier that's appended to a version or alias ARN.
     * @see https://docs.aws.amazon.com/lambda/latest/dg/API_GetFunctionConfiguration.html#API_GetFunctionConfiguration_RequestParameters
     */
    protected abstract readonly qualifier: string;
    get latestVersion(): IVersion;
    configureAsyncInvoke(options: EventInvokeConfigOptions): void;
}
