"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const fs = require("fs");
const path = require("path");
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const bundlers_1 = require("./bundlers");
const package_json_manager_1 = require("./package-json-manager");
const util_1 = require("./util");
/**
 * Bundling
 */
class Bundling {
    /**
     * Parcel bundled Lambda asset code
     */
    static parcel(options) {
        var _a, _b, _c, _d, _e;
        // Find project root
        const projectRoot = (_a = options.projectRoot) !== null && _a !== void 0 ? _a : util_1.findUp(`.git${path.sep}`);
        if (!projectRoot) {
            throw new Error('Cannot find project root. Please specify it with `projectRoot`.');
        }
        const relativeEntryPath = path.relative(projectRoot, path.resolve(options.entry));
        const packageJsonManager = new package_json_manager_1.PackageJsonManager(path.dirname(options.entry));
        // Collect external and install modules
        let includeNodeModules;
        let dependencies;
        const externalModules = (_b = options.externalModules) !== null && _b !== void 0 ? _b : ['aws-sdk'];
        if (externalModules || options.nodeModules) {
            const modules = [...externalModules, ...(_c = options.nodeModules) !== null && _c !== void 0 ? _c : []];
            includeNodeModules = {};
            for (const mod of modules) {
                includeNodeModules[mod] = false;
            }
            if (options.nodeModules) {
                dependencies = packageJsonManager.getVersions(options.nodeModules);
            }
        }
        let installer = bundlers_1.Installer.NPM;
        let lockFile;
        if (dependencies) {
            // Use npm unless we have a yarn.lock.
            if (fs.existsSync(path.join(projectRoot, bundlers_1.LockFile.YARN))) {
                installer = bundlers_1.Installer.YARN;
                lockFile = bundlers_1.LockFile.YARN;
            }
            else if (fs.existsSync(path.join(projectRoot, bundlers_1.LockFile.NPM))) {
                lockFile = bundlers_1.LockFile.NPM;
            }
        }
        // Configure target in package.json for Parcel
        packageJsonManager.update({
            targets: {
                'cdk-lambda': {
                    context: 'node',
                    includeNodeModules: includeNodeModules !== null && includeNodeModules !== void 0 ? includeNodeModules : true,
                    sourceMap: (_d = options.sourceMaps) !== null && _d !== void 0 ? _d : false,
                    minify: (_e = options.minify) !== null && _e !== void 0 ? _e : false,
                    engines: {
                        node: `>= ${runtimeVersion(options.runtime)}`,
                    },
                },
            },
        });
        // Local
        let localBundler;
        if (!options.forceDockerBundling) {
            localBundler = new bundlers_1.LocalBundler({
                projectRoot,
                relativeEntryPath,
                cacheDir: options.cacheDir,
                environment: options.parcelEnvironment,
                dependencies,
                installer,
                lockFile,
            });
        }
        // Docker
        const dockerBundler = new bundlers_1.DockerBundler({
            runtime: options.runtime,
            relativeEntryPath,
            cacheDir: options.cacheDir,
            environment: options.parcelEnvironment,
            buildImage: !bundlers_1.LocalBundler.runsLocally || options.forceDockerBundling,
            buildArgs: options.buildArgs,
            parcelVersion: options.parcelVersion,
            dependencies,
            installer,
            lockFile,
        });
        return lambda.Code.fromAsset(projectRoot, {
            assetHashType: cdk.AssetHashType.BUNDLE,
            bundling: {
                local: localBundler,
                ...dockerBundler.bundlingOptions,
            },
        });
    }
}
exports.Bundling = Bundling;
function runtimeVersion(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        throw new Error('Cannot extract version from runtime.');
    }
    return match[1];
}
//# sourceMappingURL=data:application/json;base64,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