"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.loadBalancerNameFromListenerArn = exports.TargetGroupBase = void 0;
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const elasticloadbalancingv2_generated_1 = require("../elasticloadbalancingv2.generated");
const enums_1 = require("./enums");
const util_1 = require("./util");
/**
 * Define the target of a load balancer
 */
class TargetGroupBase extends cdk.Construct {
    constructor(scope, id, baseProps, additionalProps) {
        super(scope, id);
        /**
         * Configurable dependable with all resources that lead to load balancer attachment
         */
        this.loadBalancerAttachedDependencies = new cdk.ConcreteDependable();
        /**
         * Attributes of this target group
         */
        this.attributes = {};
        /**
         * The JSON objects returned by the directly registered members of this target group
         */
        this.targetsJson = new Array();
        if (baseProps.deregistrationDelay !== undefined) {
            this.setAttribute('deregistration_delay.timeout_seconds', baseProps.deregistrationDelay.toSeconds().toString());
        }
        this.healthCheck = baseProps.healthCheck || {};
        this.vpc = baseProps.vpc;
        this.targetType = baseProps.targetType;
        this.resource = new elasticloadbalancingv2_generated_1.CfnTargetGroup(this, 'Resource', {
            name: baseProps.targetGroupName,
            targetGroupAttributes: cdk.Lazy.anyValue({ produce: () => util_1.renderAttributes(this.attributes) }, { omitEmptyArray: true }),
            targetType: cdk.Lazy.stringValue({ produce: () => this.targetType }),
            targets: cdk.Lazy.anyValue({ produce: () => this.targetsJson }, { omitEmptyArray: true }),
            vpcId: cdk.Lazy.stringValue({ produce: () => this.vpc && this.targetType !== enums_1.TargetType.LAMBDA ? this.vpc.vpcId : undefined }),
            // HEALTH CHECK
            healthCheckEnabled: cdk.Lazy.anyValue({ produce: () => this.healthCheck && this.healthCheck.enabled }),
            healthCheckIntervalSeconds: cdk.Lazy.numberValue({
                produce: () => this.healthCheck && this.healthCheck.interval && this.healthCheck.interval.toSeconds(),
            }),
            healthCheckPath: cdk.Lazy.stringValue({ produce: () => this.healthCheck && this.healthCheck.path }),
            healthCheckPort: cdk.Lazy.stringValue({ produce: () => this.healthCheck && this.healthCheck.port }),
            healthCheckProtocol: cdk.Lazy.stringValue({ produce: () => this.healthCheck && this.healthCheck.protocol }),
            healthCheckTimeoutSeconds: cdk.Lazy.numberValue({
                produce: () => this.healthCheck && this.healthCheck.timeout && this.healthCheck.timeout.toSeconds(),
            }),
            healthyThresholdCount: cdk.Lazy.numberValue({ produce: () => this.healthCheck && this.healthCheck.healthyThresholdCount }),
            unhealthyThresholdCount: cdk.Lazy.numberValue({ produce: () => this.healthCheck && this.healthCheck.unhealthyThresholdCount }),
            matcher: cdk.Lazy.anyValue({
                produce: () => this.healthCheck && this.healthCheck.healthyHttpCodes !== undefined ? {
                    httpCode: this.healthCheck.healthyHttpCodes,
                } : undefined,
            }),
            ...additionalProps,
        });
        this.targetGroupLoadBalancerArns = this.resource.attrLoadBalancerArns;
        this.targetGroupArn = this.resource.ref;
        this.targetGroupFullName = this.resource.attrTargetGroupFullName;
        this.loadBalancerArns = this.resource.attrLoadBalancerArns.toString();
        this.targetGroupName = this.resource.attrTargetGroupName;
        this.defaultPort = additionalProps.port;
    }
    /**
     * List of constructs that need to be depended on to ensure the TargetGroup is associated to a load balancer
     */
    get loadBalancerAttached() {
        return this.loadBalancerAttachedDependencies;
    }
    /**
     * Set/replace the target group's health check
     */
    configureHealthCheck(healthCheck) {
        this.healthCheck = healthCheck;
    }
    /**
     * Set a non-standard attribute on the target group
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-target-groups.html#target-group-attributes
     */
    setAttribute(key, value) {
        this.attributes[key] = value;
    }
    /**
     * Register the given load balancing target as part of this group
     */
    addLoadBalancerTarget(props) {
        if (this.targetType !== undefined && this.targetType !== props.targetType) {
            throw new Error(`Already have a of type '${this.targetType}', adding '${props.targetType}'; make all targets the same type.`);
        }
        this.targetType = props.targetType;
        if (this.targetType === enums_1.TargetType.LAMBDA && this.targetsJson.length >= 1) {
            throw new Error('TargetGroup can only contain one LAMBDA target. Create a new TargetGroup.');
        }
        if (props.targetJson) {
            this.targetsJson.push(props.targetJson);
        }
    }
    validate() {
        const ret = super.validate();
        if (this.targetType === undefined && this.targetsJson.length === 0) {
            this.node.addWarning("When creating an empty TargetGroup, you should specify a 'targetType' (this warning may become an error in the future).");
        }
        if (this.targetType !== enums_1.TargetType.LAMBDA && this.vpc === undefined) {
            ret.push("'vpc' is required for a non-Lambda TargetGroup");
        }
        return ret;
    }
}
exports.TargetGroupBase = TargetGroupBase;
/**
 * Extract the full load balancer name (used for metrics) from the listener ARN:
 *
 * Turns
 *
 *     arn:aws:elasticloadbalancing:us-west-2:123456789012:listener/app/my-load-balancer/50dc6c495c0c9188/f2f7dc8efc522ab2
 *
 * Into
 *
 *     app/my-load-balancer/50dc6c495c0c9188
 */
function loadBalancerNameFromListenerArn(listenerArn) {
    const arnParts = cdk.Fn.split('/', listenerArn);
    return `${cdk.Fn.select(1, arnParts)}/${cdk.Fn.select(2, arnParts)}/${cdk.Fn.select(3, arnParts)}`;
}
exports.loadBalancerNameFromListenerArn = loadBalancerNameFromListenerArn;
//# sourceMappingURL=data:application/json;base64,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