"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DockerImageAsset = void 0;
const fs = require("fs");
const path = require("path");
const assets = require("../../assets"); // Automatically re-written from '@aws-cdk/assets'
const ecr = require("../../aws-ecr"); // Automatically re-written from '@aws-cdk/aws-ecr'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const minimatch = require("minimatch");
/**
 * An asset that represents a Docker image.
 *
 * The image will be created in build time and uploaded to an ECR repository.
 */
class DockerImageAsset extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // none of the properties use tokens
        validateProps(props);
        // resolve full path
        const dir = path.resolve(props.directory);
        if (!fs.existsSync(dir)) {
            throw new Error(`Cannot find image directory at ${dir}`);
        }
        // validate the docker file exists
        const file = path.join(dir, props.file || 'Dockerfile');
        if (!fs.existsSync(file)) {
            throw new Error(`Cannot find file at ${file}`);
        }
        let exclude = props.exclude || [];
        const ignore = path.join(dir, '.dockerignore');
        if (fs.existsSync(ignore)) {
            exclude = [...exclude, ...fs.readFileSync(ignore).toString().split('\n').filter(e => !!e)];
        }
        // make sure the docker file and the dockerignore file end up in the staging area
        // see https://github.com/aws/aws-cdk/issues/6004
        exclude = exclude.filter(ignoreExpression => {
            return !(minimatch(file, ignoreExpression, { matchBase: true }) ||
                minimatch(ignore, ignoreExpression, { matchBase: true }));
        });
        if (props.repositoryName) {
            this.node.addWarning('DockerImageAsset.repositoryName is deprecated. Override "core.Stack.addDockerImageAsset" to control asset locations');
        }
        // include build context in "extra" so it will impact the hash
        const extraHash = {};
        if (props.extraHash) {
            extraHash.user = props.extraHash;
        }
        if (props.buildArgs) {
            extraHash.buildArgs = props.buildArgs;
        }
        if (props.target) {
            extraHash.target = props.target;
        }
        if (props.file) {
            extraHash.file = props.file;
        }
        if (props.repositoryName) {
            extraHash.repositoryName = props.repositoryName;
        }
        // add "salt" to the hash in order to invalidate the image in the upgrade to
        // 1.21.0 which removes the AdoptedRepository resource (and will cause the
        // deletion of the ECR repository the app used).
        extraHash.version = '1.21.0';
        const staging = new assets.Staging(this, 'Staging', {
            ...props,
            exclude,
            sourcePath: dir,
            extraHash: Object.keys(extraHash).length === 0
                ? undefined
                : JSON.stringify(extraHash),
        });
        this.sourceHash = staging.sourceHash;
        const stack = core_1.Stack.of(this);
        const location = stack.synthesizer.addDockerImageAsset({
            directoryName: staging.stagedPath,
            dockerBuildArgs: props.buildArgs,
            dockerBuildTarget: props.target,
            dockerFile: props.file,
            repositoryName: props.repositoryName,
            sourceHash: staging.sourceHash,
        });
        this.repository = ecr.Repository.fromRepositoryName(this, 'Repository', location.repositoryName);
        this.imageUri = location.imageUri;
    }
}
exports.DockerImageAsset = DockerImageAsset;
function validateProps(props) {
    for (const [key, value] of Object.entries(props)) {
        if (core_1.Token.isUnresolved(value)) {
            throw new Error(`Cannot use Token as value of '${key}': this value is used before deployment starts`);
        }
    }
    validateBuildArgs(props.buildArgs);
}
function validateBuildArgs(buildArgs) {
    for (const [key, value] of Object.entries(buildArgs || {})) {
        if (core_1.Token.isUnresolved(key) || core_1.Token.isUnresolved(value)) {
            throw new Error('Cannot use tokens in keys or values of "buildArgs" since they are needed before deployment');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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