"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InitConfig = exports.CloudFormationInit = void 0;
const crypto = require("crypto");
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const machine_image_1 = require("./machine-image");
const cfn_init_internal_1 = require("./private/cfn-init-internal");
/**
 * A CloudFormation-init configuration
 */
class CloudFormationInit {
    constructor(configSets, configs) {
        this._configSets = {};
        this._configs = {};
        Object.assign(this._configSets, configSets);
        Object.assign(this._configs, configs);
    }
    /**
     * Build a new config from a set of Init Elements
     */
    static fromElements(...elements) {
        return CloudFormationInit.fromConfig(new InitConfig(elements));
    }
    /**
     * Use an existing InitConfig object as the default and only config
     */
    static fromConfig(config) {
        return CloudFormationInit.fromConfigSets({
            configSets: {
                default: ['config'],
            },
            configs: { config },
        });
    }
    /**
     * Build a CloudFormationInit from config sets
     */
    static fromConfigSets(props) {
        return new CloudFormationInit(props.configSets, props.configs);
    }
    /**
     * Add a config with the given name to this CloudFormationInit object
     */
    addConfig(configName, config) {
        if (this._configs[configName]) {
            throw new Error(`CloudFormationInit already contains a config named '${configName}'`);
        }
        this._configs[configName] = config;
    }
    /**
     * Add a config set with the given name to this CloudFormationInit object
     *
     * The new configset will reference the given configs in the given order.
     */
    addConfigSet(configSetName, configNames = []) {
        if (this._configSets[configSetName]) {
            throw new Error(`CloudFormationInit already contains a configSet named '${configSetName}'`);
        }
        const unk = configNames.filter(c => !this._configs[c]);
        if (unk.length > 0) {
            throw new Error(`Unknown configs referenced in definition of '${configSetName}': ${unk}`);
        }
        this._configSets[configSetName] = [...configNames];
    }
    /**
     * Attach the CloudFormation Init config to the given resource
     *
     * This method does the following:
     *
     * - Renders the `AWS::CloudFormation::Init` object to the given resource's
     *   metadata, potentially adding a `AWS::CloudFormation::Authentication` object
     *   next to it if required.
     * - Updates the instance role policy to be able to call the APIs required for
     *   `cfn-init` and `cfn-signal` to work, and potentially add permissions to download
     *   referenced asset and bucket resources.
     * - Updates the given UserData with commands to execute the `cfn-init` script.
     *
     * As an app builder, use `instance.applyCloudFormationInit()` or
     * `autoScalingGroup.applyCloudFormationInit()` to trigger this method.
     *
     * @internal
     */
    _attach(attachedResource, attachOptions) {
        var _a, _b, _c;
        if (attachOptions.platform === machine_image_1.OperatingSystemType.UNKNOWN) {
            throw new Error('Cannot attach CloudFormationInit to an unknown OS type');
        }
        // Note: This will not reflect mutations made after attaching.
        const bindResult = this.bind(attachedResource.stack, attachOptions);
        attachedResource.addMetadata('AWS::CloudFormation::Init', bindResult.configData);
        const fingerprint = contentHash(JSON.stringify(bindResult.configData)).substr(0, 16);
        attachOptions.instanceRole.addToPolicy(new iam.PolicyStatement({
            actions: ['cloudformation:DescribeStackResource', 'cloudformation:SignalResource'],
            resources: [core_1.Aws.STACK_ID],
        }));
        if (bindResult.authData) {
            attachedResource.addMetadata('AWS::CloudFormation::Authentication', bindResult.authData);
        }
        // To identify the resources that have the metadata and where the signal
        // needs to be sent, we need { region, stackName, logicalId }
        const resourceLocator = `--region ${core_1.Aws.REGION} --stack ${core_1.Aws.STACK_NAME} --resource ${attachedResource.logicalId}`;
        const configSets = ((_a = attachOptions.configSets) !== null && _a !== void 0 ? _a : ['default']).join(',');
        const printLog = (_b = attachOptions.printLog) !== null && _b !== void 0 ? _b : true;
        if ((_c = attachOptions.embedFingerprint) !== null && _c !== void 0 ? _c : true) {
            // It just so happens that the comment char is '#' for both bash and PowerShell
            attachOptions.userData.addCommands(`# fingerprint: ${fingerprint}`);
        }
        if (attachOptions.platform === machine_image_1.OperatingSystemType.WINDOWS) {
            const errCode = attachOptions.ignoreFailures ? '0' : '$LASTEXITCODE';
            attachOptions.userData.addCommands(...[
                `cfn-init.exe -v ${resourceLocator} -c ${configSets}`,
                `cfn-signal.exe -e ${errCode} ${resourceLocator}`,
                ...printLog ? ['type C:\\cfn\\log\\cfn-init.log'] : [],
            ]);
        }
        else {
            const errCode = attachOptions.ignoreFailures ? '0' : '$?';
            attachOptions.userData.addCommands(...[
                // Run a subshell without 'errexit', so we can signal using the exit code of cfn-init
                '(',
                '  set +e',
                `  /opt/aws/bin/cfn-init -v ${resourceLocator} -c ${configSets}`,
                `  /opt/aws/bin/cfn-signal -e ${errCode} ${resourceLocator}`,
                ...printLog ? ['  cat /var/log/cfn-init.log >&2'] : [],
                ')',
            ]);
        }
    }
    bind(scope, options) {
        const nonEmptyConfigs = mapValues(this._configs, c => c.isEmpty() ? undefined : c);
        const configNameToBindResult = mapValues(nonEmptyConfigs, c => c._bind(scope, options));
        return {
            configData: {
                configSets: mapValues(this._configSets, configNames => configNames.filter(name => nonEmptyConfigs[name] !== undefined)),
                ...mapValues(configNameToBindResult, c => c.config),
            },
            authData: Object.values(configNameToBindResult).map(c => c.authentication).reduce(deepMerge, undefined),
        };
    }
}
exports.CloudFormationInit = CloudFormationInit;
/**
 * A collection of configuration elements
 */
class InitConfig {
    constructor(elements) {
        this.elements = new Array();
        this.add(...elements);
    }
    /**
     * Whether this configset has elements or not
     */
    isEmpty() {
        return this.elements.length === 0;
    }
    /**
     * Add one or more elements to the config
     */
    add(...elements) {
        this.elements.push(...elements);
    }
    /**
     * Called when the config is applied to an instance.
     * Creates the CloudFormation representation of the Init config and handles any permissions and assets.
     * @internal
     */
    _bind(scope, options) {
        const bindOptions = {
            instanceRole: options.instanceRole,
            platform: this.initPlatformFromOSType(options.platform),
            scope,
        };
        const packageConfig = this.bindForType(cfn_init_internal_1.InitElementType.PACKAGE, bindOptions);
        const groupsConfig = this.bindForType(cfn_init_internal_1.InitElementType.GROUP, bindOptions);
        const usersConfig = this.bindForType(cfn_init_internal_1.InitElementType.USER, bindOptions);
        const sourcesConfig = this.bindForType(cfn_init_internal_1.InitElementType.SOURCE, bindOptions);
        const filesConfig = this.bindForType(cfn_init_internal_1.InitElementType.FILE, bindOptions);
        const commandsConfig = this.bindForType(cfn_init_internal_1.InitElementType.COMMAND, bindOptions);
        // Must be last!
        const servicesConfig = this.bindForType(cfn_init_internal_1.InitElementType.SERVICE, bindOptions);
        const authentication = [packageConfig, groupsConfig, usersConfig, sourcesConfig, filesConfig, commandsConfig, servicesConfig]
            .map(c => c === null || c === void 0 ? void 0 : c.authentication)
            .reduce(deepMerge, undefined);
        return {
            config: {
                packages: packageConfig === null || packageConfig === void 0 ? void 0 : packageConfig.config,
                groups: groupsConfig === null || groupsConfig === void 0 ? void 0 : groupsConfig.config,
                users: usersConfig === null || usersConfig === void 0 ? void 0 : usersConfig.config,
                sources: sourcesConfig === null || sourcesConfig === void 0 ? void 0 : sourcesConfig.config,
                files: filesConfig === null || filesConfig === void 0 ? void 0 : filesConfig.config,
                commands: commandsConfig === null || commandsConfig === void 0 ? void 0 : commandsConfig.config,
                services: servicesConfig === null || servicesConfig === void 0 ? void 0 : servicesConfig.config,
            },
            authentication,
        };
    }
    bindForType(elementType, renderOptions) {
        var _a;
        const elements = this.elements.filter(elem => elem.elementType === elementType);
        if (elements.length === 0) {
            return undefined;
        }
        const bindResults = elements.map((e, index) => e._bind({ index, ...renderOptions }));
        return {
            config: (_a = bindResults.map(r => r.config).reduce(deepMerge, undefined)) !== null && _a !== void 0 ? _a : {},
            authentication: bindResults.map(r => r.authentication).reduce(deepMerge, undefined),
        };
    }
    initPlatformFromOSType(osType) {
        switch (osType) {
            case machine_image_1.OperatingSystemType.LINUX: {
                return cfn_init_internal_1.InitPlatform.LINUX;
            }
            case machine_image_1.OperatingSystemType.WINDOWS: {
                return cfn_init_internal_1.InitPlatform.WINDOWS;
            }
            default: {
                throw new Error('Cannot attach CloudFormationInit to an unknown OS type');
            }
        }
    }
}
exports.InitConfig = InitConfig;
/**
 * Deep-merge objects and arrays
 *
 * Treat arrays as sets, removing duplicates. This is acceptable for rendering
 * cfn-inits, not applicable elsewhere.
 */
function deepMerge(target, src) {
    var _a, _b;
    if (target == null) {
        return src;
    }
    if (src == null) {
        return target;
    }
    for (const [key, value] of Object.entries(src)) {
        if (Array.isArray(value)) {
            if (target[key] && !Array.isArray(target[key])) {
                throw new Error(`Trying to merge array [${value}] into a non-array '${target[key]}'`);
            }
            target[key] = Array.from(new Set([
                ...(_a = target[key]) !== null && _a !== void 0 ? _a : [],
                ...value,
            ]));
            continue;
        }
        if (typeof value === 'object' && value) {
            target[key] = deepMerge((_b = target[key]) !== null && _b !== void 0 ? _b : {}, value);
            continue;
        }
        if (value !== undefined) {
            target[key] = value;
        }
    }
    return target;
}
/**
 * Map a function over values of an object
 *
 * If the mapping function returns undefined, remove the key
 */
function mapValues(xs, fn) {
    const ret = {};
    for (const [k, v] of Object.entries(xs)) {
        const mapped = fn(v);
        if (mapped !== undefined) {
            ret[k] = mapped;
        }
    }
    return ret;
}
function contentHash(content) {
    return crypto.createHash('sha256').update(content).digest('hex');
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2ZuLWluaXQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJjZm4taW5pdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxpQ0FBaUM7QUFDakMscUNBQXFDLENBQUMsbURBQW1EO0FBQ3pGLHFDQUF5RCxDQUFDLGdEQUFnRDtBQUUxRyxtREFBc0Q7QUFDdEQsbUVBQW1JO0FBQ25JOztHQUVHO0FBQ0gsTUFBYSxrQkFBa0I7SUEwQjNCLFlBQW9CLFVBQW9DLEVBQUUsT0FBbUM7UUFGNUUsZ0JBQVcsR0FBNkIsRUFBRSxDQUFDO1FBQzNDLGFBQVEsR0FBK0IsRUFBRSxDQUFDO1FBRXZELE1BQU0sQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxVQUFVLENBQUMsQ0FBQztRQUM1QyxNQUFNLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFDMUMsQ0FBQztJQTVCRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxZQUFZLENBQUMsR0FBRyxRQUF1QjtRQUNqRCxPQUFPLGtCQUFrQixDQUFDLFVBQVUsQ0FBQyxJQUFJLFVBQVUsQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDO0lBQ25FLENBQUM7SUFDRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxVQUFVLENBQUMsTUFBa0I7UUFDdkMsT0FBTyxrQkFBa0IsQ0FBQyxjQUFjLENBQUM7WUFDckMsVUFBVSxFQUFFO2dCQUNSLE9BQU8sRUFBRSxDQUFDLFFBQVEsQ0FBQzthQUN0QjtZQUNELE9BQU8sRUFBRSxFQUFFLE1BQU0sRUFBRTtTQUN0QixDQUFDLENBQUM7SUFDUCxDQUFDO0lBQ0Q7O09BRUc7SUFDSSxNQUFNLENBQUMsY0FBYyxDQUFDLEtBQXFCO1FBQzlDLE9BQU8sSUFBSSxrQkFBa0IsQ0FBQyxLQUFLLENBQUMsVUFBVSxFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQztJQUNuRSxDQUFDO0lBT0Q7O09BRUc7SUFDSSxTQUFTLENBQUMsVUFBa0IsRUFBRSxNQUFrQjtRQUNuRCxJQUFJLElBQUksQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLEVBQUU7WUFDM0IsTUFBTSxJQUFJLEtBQUssQ0FBQyx1REFBdUQsVUFBVSxHQUFHLENBQUMsQ0FBQztTQUN6RjtRQUNELElBQUksQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLEdBQUcsTUFBTSxDQUFDO0lBQ3ZDLENBQUM7SUFDRDs7OztPQUlHO0lBQ0ksWUFBWSxDQUFDLGFBQXFCLEVBQUUsY0FBd0IsRUFBRTtRQUNqRSxJQUFJLElBQUksQ0FBQyxXQUFXLENBQUMsYUFBYSxDQUFDLEVBQUU7WUFDakMsTUFBTSxJQUFJLEtBQUssQ0FBQywwREFBMEQsYUFBYSxHQUFHLENBQUMsQ0FBQztTQUMvRjtRQUNELE1BQU0sR0FBRyxHQUFHLFdBQVcsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUN2RCxJQUFJLEdBQUcsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQ2hCLE1BQU0sSUFBSSxLQUFLLENBQUMsZ0RBQWdELGFBQWEsTUFBTSxHQUFHLEVBQUUsQ0FBQyxDQUFDO1NBQzdGO1FBQ0QsSUFBSSxDQUFDLFdBQVcsQ0FBQyxhQUFhLENBQUMsR0FBRyxDQUFDLEdBQUcsV0FBVyxDQUFDLENBQUM7SUFDdkQsQ0FBQztJQUNEOzs7Ozs7Ozs7Ozs7Ozs7OztPQWlCRztJQUNJLE9BQU8sQ0FBQyxnQkFBNkIsRUFBRSxhQUFnQzs7UUFDMUUsSUFBSSxhQUFhLENBQUMsUUFBUSxLQUFLLG1DQUFtQixDQUFDLE9BQU8sRUFBRTtZQUN4RCxNQUFNLElBQUksS0FBSyxDQUFDLHdEQUF3RCxDQUFDLENBQUM7U0FDN0U7UUFDRCw4REFBOEQ7UUFDOUQsTUFBTSxVQUFVLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLEVBQUUsYUFBYSxDQUFDLENBQUM7UUFDcEUsZ0JBQWdCLENBQUMsV0FBVyxDQUFDLDJCQUEyQixFQUFFLFVBQVUsQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUNqRixNQUFNLFdBQVcsR0FBRyxXQUFXLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQ3JGLGFBQWEsQ0FBQyxZQUFZLENBQUMsV0FBVyxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUMzRCxPQUFPLEVBQUUsQ0FBQyxzQ0FBc0MsRUFBRSwrQkFBK0IsQ0FBQztZQUNsRixTQUFTLEVBQUUsQ0FBQyxVQUFHLENBQUMsUUFBUSxDQUFDO1NBQzVCLENBQUMsQ0FBQyxDQUFDO1FBQ0osSUFBSSxVQUFVLENBQUMsUUFBUSxFQUFFO1lBQ3JCLGdCQUFnQixDQUFDLFdBQVcsQ0FBQyxxQ0FBcUMsRUFBRSxVQUFVLENBQUMsUUFBUSxDQUFDLENBQUM7U0FDNUY7UUFDRCx3RUFBd0U7UUFDeEUsNkRBQTZEO1FBQzdELE1BQU0sZUFBZSxHQUFHLFlBQVksVUFBRyxDQUFDLE1BQU0sWUFBWSxVQUFHLENBQUMsVUFBVSxlQUFlLGdCQUFnQixDQUFDLFNBQVMsRUFBRSxDQUFDO1FBQ3BILE1BQU0sVUFBVSxHQUFHLE9BQUMsYUFBYSxDQUFDLFVBQVUsbUNBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUN2RSxNQUFNLFFBQVEsU0FBRyxhQUFhLENBQUMsUUFBUSxtQ0FBSSxJQUFJLENBQUM7UUFDaEQsVUFBSSxhQUFhLENBQUMsZ0JBQWdCLG1DQUFJLElBQUksRUFBRTtZQUN4QywrRUFBK0U7WUFDL0UsYUFBYSxDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUMsa0JBQWtCLFdBQVcsRUFBRSxDQUFDLENBQUM7U0FDdkU7UUFDRCxJQUFJLGFBQWEsQ0FBQyxRQUFRLEtBQUssbUNBQW1CLENBQUMsT0FBTyxFQUFFO1lBQ3hELE1BQU0sT0FBTyxHQUFHLGFBQWEsQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsZUFBZSxDQUFDO1lBQ3JFLGFBQWEsQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDLEdBQUc7Z0JBQ2xDLG1CQUFtQixlQUFlLE9BQU8sVUFBVSxFQUFFO2dCQUNyRCxxQkFBcUIsT0FBTyxJQUFJLGVBQWUsRUFBRTtnQkFDakQsR0FBRyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsaUNBQWlDLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRTthQUN6RCxDQUFDLENBQUM7U0FDTjthQUNJO1lBQ0QsTUFBTSxPQUFPLEdBQUcsYUFBYSxDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUM7WUFDMUQsYUFBYSxDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUMsR0FBRztnQkFDbEMscUZBQXFGO2dCQUNyRixHQUFHO2dCQUNILFVBQVU7Z0JBQ1YsOEJBQThCLGVBQWUsT0FBTyxVQUFVLEVBQUU7Z0JBQ2hFLGdDQUFnQyxPQUFPLElBQUksZUFBZSxFQUFFO2dCQUM1RCxHQUFHLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxpQ0FBaUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFO2dCQUN0RCxHQUFHO2FBQ04sQ0FBQyxDQUFDO1NBQ047SUFDTCxDQUFDO0lBQ08sSUFBSSxDQUFDLEtBQWdCLEVBQUUsT0FBMEI7UUFJckQsTUFBTSxlQUFlLEdBQUcsU0FBUyxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDbkYsTUFBTSxzQkFBc0IsR0FBRyxTQUFTLENBQUMsZUFBZSxFQUFFLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsT0FBTyxDQUFDLENBQUMsQ0FBQztRQUN4RixPQUFPO1lBQ0gsVUFBVSxFQUFFO2dCQUNSLFVBQVUsRUFBRSxTQUFTLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxXQUFXLENBQUMsRUFBRSxDQUFDLFdBQVcsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDLEtBQUssU0FBUyxDQUFDLENBQUM7Z0JBQ3ZILEdBQUcsU0FBUyxDQUFDLHNCQUFzQixFQUFFLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQzthQUN0RDtZQUNELFFBQVEsRUFBRSxNQUFNLENBQUMsTUFBTSxDQUFDLHNCQUFzQixDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLGNBQWMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxTQUFTLEVBQUUsU0FBUyxDQUFDO1NBQzFHLENBQUM7SUFDTixDQUFDO0NBQ0o7QUFuSUQsZ0RBbUlDO0FBQ0Q7O0dBRUc7QUFDSCxNQUFhLFVBQVU7SUFFbkIsWUFBWSxRQUF1QjtRQURsQixhQUFRLEdBQUcsSUFBSSxLQUFLLEVBQWUsQ0FBQztRQUVqRCxJQUFJLENBQUMsR0FBRyxDQUFDLEdBQUcsUUFBUSxDQUFDLENBQUM7SUFDMUIsQ0FBQztJQUNEOztPQUVHO0lBQ0ksT0FBTztRQUNWLE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQyxNQUFNLEtBQUssQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFDRDs7T0FFRztJQUNJLEdBQUcsQ0FBQyxHQUFHLFFBQXVCO1FBQ2pDLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLEdBQUcsUUFBUSxDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUNEOzs7O09BSUc7SUFDSSxLQUFLLENBQUMsS0FBZ0IsRUFBRSxPQUEwQjtRQUNyRCxNQUFNLFdBQVcsR0FBRztZQUNoQixZQUFZLEVBQUUsT0FBTyxDQUFDLFlBQVk7WUFDbEMsUUFBUSxFQUFFLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDO1lBQ3ZELEtBQUs7U0FDUixDQUFDO1FBQ0YsTUFBTSxhQUFhLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxtQ0FBZSxDQUFDLE9BQU8sRUFBRSxXQUFXLENBQUMsQ0FBQztRQUM3RSxNQUFNLFlBQVksR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLG1DQUFlLENBQUMsS0FBSyxFQUFFLFdBQVcsQ0FBQyxDQUFDO1FBQzFFLE1BQU0sV0FBVyxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsbUNBQWUsQ0FBQyxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFDeEUsTUFBTSxhQUFhLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxtQ0FBZSxDQUFDLE1BQU0sRUFBRSxXQUFXLENBQUMsQ0FBQztRQUM1RSxNQUFNLFdBQVcsR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLG1DQUFlLENBQUMsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUFDO1FBQ3hFLE1BQU0sY0FBYyxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsbUNBQWUsQ0FBQyxPQUFPLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFDOUUsZ0JBQWdCO1FBQ2hCLE1BQU0sY0FBYyxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsbUNBQWUsQ0FBQyxPQUFPLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFDOUUsTUFBTSxjQUFjLEdBQUcsQ0FBQyxhQUFhLEVBQUUsWUFBWSxFQUFFLFdBQVcsRUFBRSxhQUFhLEVBQUUsV0FBVyxFQUFFLGNBQWMsRUFBRSxjQUFjLENBQUM7YUFDeEgsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxhQUFELENBQUMsdUJBQUQsQ0FBQyxDQUFFLGNBQWMsQ0FBQzthQUMzQixNQUFNLENBQUMsU0FBUyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQ2xDLE9BQU87WUFDSCxNQUFNLEVBQUU7Z0JBQ0osUUFBUSxFQUFFLGFBQWEsYUFBYixhQUFhLHVCQUFiLGFBQWEsQ0FBRSxNQUFNO2dCQUMvQixNQUFNLEVBQUUsWUFBWSxhQUFaLFlBQVksdUJBQVosWUFBWSxDQUFFLE1BQU07Z0JBQzVCLEtBQUssRUFBRSxXQUFXLGFBQVgsV0FBVyx1QkFBWCxXQUFXLENBQUUsTUFBTTtnQkFDMUIsT0FBTyxFQUFFLGFBQWEsYUFBYixhQUFhLHVCQUFiLGFBQWEsQ0FBRSxNQUFNO2dCQUM5QixLQUFLLEVBQUUsV0FBVyxhQUFYLFdBQVcsdUJBQVgsV0FBVyxDQUFFLE1BQU07Z0JBQzFCLFFBQVEsRUFBRSxjQUFjLGFBQWQsY0FBYyx1QkFBZCxjQUFjLENBQUUsTUFBTTtnQkFDaEMsUUFBUSxFQUFFLGNBQWMsYUFBZCxjQUFjLHVCQUFkLGNBQWMsQ0FBRSxNQUFNO2FBQ25DO1lBQ0QsY0FBYztTQUNqQixDQUFDO0lBQ04sQ0FBQztJQUNPLFdBQVcsQ0FBQyxXQUE0QixFQUFFLGFBQTZDOztRQUMzRixNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxXQUFXLEtBQUssV0FBVyxDQUFDLENBQUM7UUFDaEYsSUFBSSxRQUFRLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUN2QixPQUFPLFNBQVMsQ0FBQztTQUNwQjtRQUNELE1BQU0sV0FBVyxHQUFHLFFBQVEsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLEVBQUUsS0FBSyxFQUFFLEdBQUcsYUFBYSxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBQ3JGLE9BQU87WUFDSCxNQUFNLFFBQUUsV0FBVyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxNQUFNLENBQUMsU0FBUyxFQUFFLFNBQVMsQ0FBQyxtQ0FBSSxFQUFFO1lBQ3pFLGNBQWMsRUFBRSxXQUFXLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLGNBQWMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxTQUFTLEVBQUUsU0FBUyxDQUFDO1NBQ3RGLENBQUM7SUFDTixDQUFDO0lBQ08sc0JBQXNCLENBQUMsTUFBMkI7UUFDdEQsUUFBUSxNQUFNLEVBQUU7WUFDWixLQUFLLG1DQUFtQixDQUFDLEtBQUssQ0FBQyxDQUFDO2dCQUM1QixPQUFPLGdDQUFZLENBQUMsS0FBSyxDQUFDO2FBQzdCO1lBQ0QsS0FBSyxtQ0FBbUIsQ0FBQyxPQUFPLENBQUMsQ0FBQztnQkFDOUIsT0FBTyxnQ0FBWSxDQUFDLE9BQU8sQ0FBQzthQUMvQjtZQUNELE9BQU8sQ0FBQyxDQUFDO2dCQUNMLE1BQU0sSUFBSSxLQUFLLENBQUMsd0RBQXdELENBQUMsQ0FBQzthQUM3RTtTQUNKO0lBQ0wsQ0FBQztDQUNKO0FBNUVELGdDQTRFQztBQWNEOzs7OztHQUtHO0FBQ0gsU0FBUyxTQUFTLENBQUMsTUFBNEIsRUFBRSxHQUF5Qjs7SUFDdEUsSUFBSSxNQUFNLElBQUksSUFBSSxFQUFFO1FBQ2hCLE9BQU8sR0FBRyxDQUFDO0tBQ2Q7SUFDRCxJQUFJLEdBQUcsSUFBSSxJQUFJLEVBQUU7UUFDYixPQUFPLE1BQU0sQ0FBQztLQUNqQjtJQUNELEtBQUssTUFBTSxDQUFDLEdBQUcsRUFBRSxLQUFLLENBQUMsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxFQUFFO1FBQzVDLElBQUksS0FBSyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsRUFBRTtZQUN0QixJQUFJLE1BQU0sQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLEVBQUU7Z0JBQzVDLE1BQU0sSUFBSSxLQUFLLENBQUMsMEJBQTBCLEtBQUssdUJBQXVCLE1BQU0sQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLENBQUM7YUFDekY7WUFDRCxNQUFNLENBQUMsR0FBRyxDQUFDLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLEdBQUcsQ0FBQztnQkFDN0IsU0FBRyxNQUFNLENBQUMsR0FBRyxDQUFDLG1DQUFJLEVBQUU7Z0JBQ3BCLEdBQUcsS0FBSzthQUNYLENBQUMsQ0FBQyxDQUFDO1lBQ0osU0FBUztTQUNaO1FBQ0QsSUFBSSxPQUFPLEtBQUssS0FBSyxRQUFRLElBQUksS0FBSyxFQUFFO1lBQ3BDLE1BQU0sQ0FBQyxHQUFHLENBQUMsR0FBRyxTQUFTLE9BQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxtQ0FBSSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7WUFDbEQsU0FBUztTQUNaO1FBQ0QsSUFBSSxLQUFLLEtBQUssU0FBUyxFQUFFO1lBQ3JCLE1BQU0sQ0FBQyxHQUFHLENBQUMsR0FBRyxLQUFLLENBQUM7U0FDdkI7S0FDSjtJQUNELE9BQU8sTUFBTSxDQUFDO0FBQ2xCLENBQUM7QUFDRDs7OztHQUlHO0FBQ0gsU0FBUyxTQUFTLENBQU8sRUFBcUIsRUFBRSxFQUEyQjtJQUN2RSxNQUFNLEdBQUcsR0FBc0IsRUFBRSxDQUFDO0lBQ2xDLEtBQUssTUFBTSxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxFQUFFO1FBQ3JDLE1BQU0sTUFBTSxHQUFHLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUNyQixJQUFJLE1BQU0sS0FBSyxTQUFTLEVBQUU7WUFDdEIsR0FBRyxDQUFDLENBQUMsQ0FBQyxHQUFHLE1BQU0sQ0FBQztTQUNuQjtLQUNKO0lBQ0QsT0FBTyxHQUFHLENBQUM7QUFDZixDQUFDO0FBQ0QsU0FBUyxXQUFXLENBQUMsT0FBZTtJQUNoQyxPQUFPLE1BQU0sQ0FBQyxVQUFVLENBQUMsUUFBUSxDQUFDLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQztBQUNyRSxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgY3J5cHRvIGZyb20gJ2NyeXB0byc7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSBcIi4uLy4uL2F3cy1pYW1cIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nXG5pbXBvcnQgeyBBd3MsIENmblJlc291cmNlLCBDb25zdHJ1Y3QgfSBmcm9tIFwiLi4vLi4vY29yZVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvY29yZSdcbmltcG9ydCB7IEluaXRFbGVtZW50IH0gZnJvbSAnLi9jZm4taW5pdC1lbGVtZW50cyc7XG5pbXBvcnQgeyBPcGVyYXRpbmdTeXN0ZW1UeXBlIH0gZnJvbSAnLi9tYWNoaW5lLWltYWdlJztcbmltcG9ydCB7IEF0dGFjaEluaXRPcHRpb25zLCBJbml0QmluZE9wdGlvbnMsIEluaXRFbGVtZW50Q29uZmlnLCBJbml0RWxlbWVudFR5cGUsIEluaXRQbGF0Zm9ybSB9IGZyb20gJy4vcHJpdmF0ZS9jZm4taW5pdC1pbnRlcm5hbCc7XG4vKipcbiAqIEEgQ2xvdWRGb3JtYXRpb24taW5pdCBjb25maWd1cmF0aW9uXG4gKi9cbmV4cG9ydCBjbGFzcyBDbG91ZEZvcm1hdGlvbkluaXQge1xuICAgIC8qKlxuICAgICAqIEJ1aWxkIGEgbmV3IGNvbmZpZyBmcm9tIGEgc2V0IG9mIEluaXQgRWxlbWVudHNcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGZyb21FbGVtZW50cyguLi5lbGVtZW50czogSW5pdEVsZW1lbnRbXSk6IENsb3VkRm9ybWF0aW9uSW5pdCB7XG4gICAgICAgIHJldHVybiBDbG91ZEZvcm1hdGlvbkluaXQuZnJvbUNvbmZpZyhuZXcgSW5pdENvbmZpZyhlbGVtZW50cykpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBVc2UgYW4gZXhpc3RpbmcgSW5pdENvbmZpZyBvYmplY3QgYXMgdGhlIGRlZmF1bHQgYW5kIG9ubHkgY29uZmlnXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyBmcm9tQ29uZmlnKGNvbmZpZzogSW5pdENvbmZpZyk6IENsb3VkRm9ybWF0aW9uSW5pdCB7XG4gICAgICAgIHJldHVybiBDbG91ZEZvcm1hdGlvbkluaXQuZnJvbUNvbmZpZ1NldHMoe1xuICAgICAgICAgICAgY29uZmlnU2V0czoge1xuICAgICAgICAgICAgICAgIGRlZmF1bHQ6IFsnY29uZmlnJ10sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgY29uZmlnczogeyBjb25maWcgfSxcbiAgICAgICAgfSk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEJ1aWxkIGEgQ2xvdWRGb3JtYXRpb25Jbml0IGZyb20gY29uZmlnIHNldHNcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGZyb21Db25maWdTZXRzKHByb3BzOiBDb25maWdTZXRQcm9wcyk6IENsb3VkRm9ybWF0aW9uSW5pdCB7XG4gICAgICAgIHJldHVybiBuZXcgQ2xvdWRGb3JtYXRpb25Jbml0KHByb3BzLmNvbmZpZ1NldHMsIHByb3BzLmNvbmZpZ3MpO1xuICAgIH1cbiAgICBwcml2YXRlIHJlYWRvbmx5IF9jb25maWdTZXRzOiBSZWNvcmQ8c3RyaW5nLCBzdHJpbmdbXT4gPSB7fTtcbiAgICBwcml2YXRlIHJlYWRvbmx5IF9jb25maWdzOiBSZWNvcmQ8c3RyaW5nLCBJbml0Q29uZmlnPiA9IHt9O1xuICAgIHByaXZhdGUgY29uc3RydWN0b3IoY29uZmlnU2V0czogUmVjb3JkPHN0cmluZywgc3RyaW5nW10+LCBjb25maWdzOiBSZWNvcmQ8c3RyaW5nLCBJbml0Q29uZmlnPikge1xuICAgICAgICBPYmplY3QuYXNzaWduKHRoaXMuX2NvbmZpZ1NldHMsIGNvbmZpZ1NldHMpO1xuICAgICAgICBPYmplY3QuYXNzaWduKHRoaXMuX2NvbmZpZ3MsIGNvbmZpZ3MpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBBZGQgYSBjb25maWcgd2l0aCB0aGUgZ2l2ZW4gbmFtZSB0byB0aGlzIENsb3VkRm9ybWF0aW9uSW5pdCBvYmplY3RcbiAgICAgKi9cbiAgICBwdWJsaWMgYWRkQ29uZmlnKGNvbmZpZ05hbWU6IHN0cmluZywgY29uZmlnOiBJbml0Q29uZmlnKSB7XG4gICAgICAgIGlmICh0aGlzLl9jb25maWdzW2NvbmZpZ05hbWVdKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYENsb3VkRm9ybWF0aW9uSW5pdCBhbHJlYWR5IGNvbnRhaW5zIGEgY29uZmlnIG5hbWVkICcke2NvbmZpZ05hbWV9J2ApO1xuICAgICAgICB9XG4gICAgICAgIHRoaXMuX2NvbmZpZ3NbY29uZmlnTmFtZV0gPSBjb25maWc7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEFkZCBhIGNvbmZpZyBzZXQgd2l0aCB0aGUgZ2l2ZW4gbmFtZSB0byB0aGlzIENsb3VkRm9ybWF0aW9uSW5pdCBvYmplY3RcbiAgICAgKlxuICAgICAqIFRoZSBuZXcgY29uZmlnc2V0IHdpbGwgcmVmZXJlbmNlIHRoZSBnaXZlbiBjb25maWdzIGluIHRoZSBnaXZlbiBvcmRlci5cbiAgICAgKi9cbiAgICBwdWJsaWMgYWRkQ29uZmlnU2V0KGNvbmZpZ1NldE5hbWU6IHN0cmluZywgY29uZmlnTmFtZXM6IHN0cmluZ1tdID0gW10pIHtcbiAgICAgICAgaWYgKHRoaXMuX2NvbmZpZ1NldHNbY29uZmlnU2V0TmFtZV0pIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgQ2xvdWRGb3JtYXRpb25Jbml0IGFscmVhZHkgY29udGFpbnMgYSBjb25maWdTZXQgbmFtZWQgJyR7Y29uZmlnU2V0TmFtZX0nYCk7XG4gICAgICAgIH1cbiAgICAgICAgY29uc3QgdW5rID0gY29uZmlnTmFtZXMuZmlsdGVyKGMgPT4gIXRoaXMuX2NvbmZpZ3NbY10pO1xuICAgICAgICBpZiAodW5rLmxlbmd0aCA+IDApIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgVW5rbm93biBjb25maWdzIHJlZmVyZW5jZWQgaW4gZGVmaW5pdGlvbiBvZiAnJHtjb25maWdTZXROYW1lfSc6ICR7dW5rfWApO1xuICAgICAgICB9XG4gICAgICAgIHRoaXMuX2NvbmZpZ1NldHNbY29uZmlnU2V0TmFtZV0gPSBbLi4uY29uZmlnTmFtZXNdO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBBdHRhY2ggdGhlIENsb3VkRm9ybWF0aW9uIEluaXQgY29uZmlnIHRvIHRoZSBnaXZlbiByZXNvdXJjZVxuICAgICAqXG4gICAgICogVGhpcyBtZXRob2QgZG9lcyB0aGUgZm9sbG93aW5nOlxuICAgICAqXG4gICAgICogLSBSZW5kZXJzIHRoZSBgQVdTOjpDbG91ZEZvcm1hdGlvbjo6SW5pdGAgb2JqZWN0IHRvIHRoZSBnaXZlbiByZXNvdXJjZSdzXG4gICAgICogICBtZXRhZGF0YSwgcG90ZW50aWFsbHkgYWRkaW5nIGEgYEFXUzo6Q2xvdWRGb3JtYXRpb246OkF1dGhlbnRpY2F0aW9uYCBvYmplY3RcbiAgICAgKiAgIG5leHQgdG8gaXQgaWYgcmVxdWlyZWQuXG4gICAgICogLSBVcGRhdGVzIHRoZSBpbnN0YW5jZSByb2xlIHBvbGljeSB0byBiZSBhYmxlIHRvIGNhbGwgdGhlIEFQSXMgcmVxdWlyZWQgZm9yXG4gICAgICogICBgY2ZuLWluaXRgIGFuZCBgY2ZuLXNpZ25hbGAgdG8gd29yaywgYW5kIHBvdGVudGlhbGx5IGFkZCBwZXJtaXNzaW9ucyB0byBkb3dubG9hZFxuICAgICAqICAgcmVmZXJlbmNlZCBhc3NldCBhbmQgYnVja2V0IHJlc291cmNlcy5cbiAgICAgKiAtIFVwZGF0ZXMgdGhlIGdpdmVuIFVzZXJEYXRhIHdpdGggY29tbWFuZHMgdG8gZXhlY3V0ZSB0aGUgYGNmbi1pbml0YCBzY3JpcHQuXG4gICAgICpcbiAgICAgKiBBcyBhbiBhcHAgYnVpbGRlciwgdXNlIGBpbnN0YW5jZS5hcHBseUNsb3VkRm9ybWF0aW9uSW5pdCgpYCBvclxuICAgICAqIGBhdXRvU2NhbGluZ0dyb3VwLmFwcGx5Q2xvdWRGb3JtYXRpb25Jbml0KClgIHRvIHRyaWdnZXIgdGhpcyBtZXRob2QuXG4gICAgICpcbiAgICAgKiBAaW50ZXJuYWxcbiAgICAgKi9cbiAgICBwdWJsaWMgX2F0dGFjaChhdHRhY2hlZFJlc291cmNlOiBDZm5SZXNvdXJjZSwgYXR0YWNoT3B0aW9uczogQXR0YWNoSW5pdE9wdGlvbnMpIHtcbiAgICAgICAgaWYgKGF0dGFjaE9wdGlvbnMucGxhdGZvcm0gPT09IE9wZXJhdGluZ1N5c3RlbVR5cGUuVU5LTk9XTikge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdDYW5ub3QgYXR0YWNoIENsb3VkRm9ybWF0aW9uSW5pdCB0byBhbiB1bmtub3duIE9TIHR5cGUnKTtcbiAgICAgICAgfVxuICAgICAgICAvLyBOb3RlOiBUaGlzIHdpbGwgbm90IHJlZmxlY3QgbXV0YXRpb25zIG1hZGUgYWZ0ZXIgYXR0YWNoaW5nLlxuICAgICAgICBjb25zdCBiaW5kUmVzdWx0ID0gdGhpcy5iaW5kKGF0dGFjaGVkUmVzb3VyY2Uuc3RhY2ssIGF0dGFjaE9wdGlvbnMpO1xuICAgICAgICBhdHRhY2hlZFJlc291cmNlLmFkZE1ldGFkYXRhKCdBV1M6OkNsb3VkRm9ybWF0aW9uOjpJbml0JywgYmluZFJlc3VsdC5jb25maWdEYXRhKTtcbiAgICAgICAgY29uc3QgZmluZ2VycHJpbnQgPSBjb250ZW50SGFzaChKU09OLnN0cmluZ2lmeShiaW5kUmVzdWx0LmNvbmZpZ0RhdGEpKS5zdWJzdHIoMCwgMTYpO1xuICAgICAgICBhdHRhY2hPcHRpb25zLmluc3RhbmNlUm9sZS5hZGRUb1BvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICBhY3Rpb25zOiBbJ2Nsb3VkZm9ybWF0aW9uOkRlc2NyaWJlU3RhY2tSZXNvdXJjZScsICdjbG91ZGZvcm1hdGlvbjpTaWduYWxSZXNvdXJjZSddLFxuICAgICAgICAgICAgcmVzb3VyY2VzOiBbQXdzLlNUQUNLX0lEXSxcbiAgICAgICAgfSkpO1xuICAgICAgICBpZiAoYmluZFJlc3VsdC5hdXRoRGF0YSkge1xuICAgICAgICAgICAgYXR0YWNoZWRSZXNvdXJjZS5hZGRNZXRhZGF0YSgnQVdTOjpDbG91ZEZvcm1hdGlvbjo6QXV0aGVudGljYXRpb24nLCBiaW5kUmVzdWx0LmF1dGhEYXRhKTtcbiAgICAgICAgfVxuICAgICAgICAvLyBUbyBpZGVudGlmeSB0aGUgcmVzb3VyY2VzIHRoYXQgaGF2ZSB0aGUgbWV0YWRhdGEgYW5kIHdoZXJlIHRoZSBzaWduYWxcbiAgICAgICAgLy8gbmVlZHMgdG8gYmUgc2VudCwgd2UgbmVlZCB7IHJlZ2lvbiwgc3RhY2tOYW1lLCBsb2dpY2FsSWQgfVxuICAgICAgICBjb25zdCByZXNvdXJjZUxvY2F0b3IgPSBgLS1yZWdpb24gJHtBd3MuUkVHSU9OfSAtLXN0YWNrICR7QXdzLlNUQUNLX05BTUV9IC0tcmVzb3VyY2UgJHthdHRhY2hlZFJlc291cmNlLmxvZ2ljYWxJZH1gO1xuICAgICAgICBjb25zdCBjb25maWdTZXRzID0gKGF0dGFjaE9wdGlvbnMuY29uZmlnU2V0cyA/PyBbJ2RlZmF1bHQnXSkuam9pbignLCcpO1xuICAgICAgICBjb25zdCBwcmludExvZyA9IGF0dGFjaE9wdGlvbnMucHJpbnRMb2cgPz8gdHJ1ZTtcbiAgICAgICAgaWYgKGF0dGFjaE9wdGlvbnMuZW1iZWRGaW5nZXJwcmludCA/PyB0cnVlKSB7XG4gICAgICAgICAgICAvLyBJdCBqdXN0IHNvIGhhcHBlbnMgdGhhdCB0aGUgY29tbWVudCBjaGFyIGlzICcjJyBmb3IgYm90aCBiYXNoIGFuZCBQb3dlclNoZWxsXG4gICAgICAgICAgICBhdHRhY2hPcHRpb25zLnVzZXJEYXRhLmFkZENvbW1hbmRzKGAjIGZpbmdlcnByaW50OiAke2ZpbmdlcnByaW50fWApO1xuICAgICAgICB9XG4gICAgICAgIGlmIChhdHRhY2hPcHRpb25zLnBsYXRmb3JtID09PSBPcGVyYXRpbmdTeXN0ZW1UeXBlLldJTkRPV1MpIHtcbiAgICAgICAgICAgIGNvbnN0IGVyckNvZGUgPSBhdHRhY2hPcHRpb25zLmlnbm9yZUZhaWx1cmVzID8gJzAnIDogJyRMQVNURVhJVENPREUnO1xuICAgICAgICAgICAgYXR0YWNoT3B0aW9ucy51c2VyRGF0YS5hZGRDb21tYW5kcyguLi5bXG4gICAgICAgICAgICAgICAgYGNmbi1pbml0LmV4ZSAtdiAke3Jlc291cmNlTG9jYXRvcn0gLWMgJHtjb25maWdTZXRzfWAsXG4gICAgICAgICAgICAgICAgYGNmbi1zaWduYWwuZXhlIC1lICR7ZXJyQ29kZX0gJHtyZXNvdXJjZUxvY2F0b3J9YCxcbiAgICAgICAgICAgICAgICAuLi5wcmludExvZyA/IFsndHlwZSBDOlxcXFxjZm5cXFxcbG9nXFxcXGNmbi1pbml0LmxvZyddIDogW10sXG4gICAgICAgICAgICBdKTtcbiAgICAgICAgfVxuICAgICAgICBlbHNlIHtcbiAgICAgICAgICAgIGNvbnN0IGVyckNvZGUgPSBhdHRhY2hPcHRpb25zLmlnbm9yZUZhaWx1cmVzID8gJzAnIDogJyQ/JztcbiAgICAgICAgICAgIGF0dGFjaE9wdGlvbnMudXNlckRhdGEuYWRkQ29tbWFuZHMoLi4uW1xuICAgICAgICAgICAgICAgIC8vIFJ1biBhIHN1YnNoZWxsIHdpdGhvdXQgJ2VycmV4aXQnLCBzbyB3ZSBjYW4gc2lnbmFsIHVzaW5nIHRoZSBleGl0IGNvZGUgb2YgY2ZuLWluaXRcbiAgICAgICAgICAgICAgICAnKCcsXG4gICAgICAgICAgICAgICAgJyAgc2V0ICtlJyxcbiAgICAgICAgICAgICAgICBgICAvb3B0L2F3cy9iaW4vY2ZuLWluaXQgLXYgJHtyZXNvdXJjZUxvY2F0b3J9IC1jICR7Y29uZmlnU2V0c31gLFxuICAgICAgICAgICAgICAgIGAgIC9vcHQvYXdzL2Jpbi9jZm4tc2lnbmFsIC1lICR7ZXJyQ29kZX0gJHtyZXNvdXJjZUxvY2F0b3J9YCxcbiAgICAgICAgICAgICAgICAuLi5wcmludExvZyA/IFsnICBjYXQgL3Zhci9sb2cvY2ZuLWluaXQubG9nID4mMiddIDogW10sXG4gICAgICAgICAgICAgICAgJyknLFxuICAgICAgICAgICAgXSk7XG4gICAgICAgIH1cbiAgICB9XG4gICAgcHJpdmF0ZSBiaW5kKHNjb3BlOiBDb25zdHJ1Y3QsIG9wdGlvbnM6IEF0dGFjaEluaXRPcHRpb25zKToge1xuICAgICAgICBjb25maWdEYXRhOiBhbnk7XG4gICAgICAgIGF1dGhEYXRhOiBhbnk7XG4gICAgfSB7XG4gICAgICAgIGNvbnN0IG5vbkVtcHR5Q29uZmlncyA9IG1hcFZhbHVlcyh0aGlzLl9jb25maWdzLCBjID0+IGMuaXNFbXB0eSgpID8gdW5kZWZpbmVkIDogYyk7XG4gICAgICAgIGNvbnN0IGNvbmZpZ05hbWVUb0JpbmRSZXN1bHQgPSBtYXBWYWx1ZXMobm9uRW1wdHlDb25maWdzLCBjID0+IGMuX2JpbmQoc2NvcGUsIG9wdGlvbnMpKTtcbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgIGNvbmZpZ0RhdGE6IHtcbiAgICAgICAgICAgICAgICBjb25maWdTZXRzOiBtYXBWYWx1ZXModGhpcy5fY29uZmlnU2V0cywgY29uZmlnTmFtZXMgPT4gY29uZmlnTmFtZXMuZmlsdGVyKG5hbWUgPT4gbm9uRW1wdHlDb25maWdzW25hbWVdICE9PSB1bmRlZmluZWQpKSxcbiAgICAgICAgICAgICAgICAuLi5tYXBWYWx1ZXMoY29uZmlnTmFtZVRvQmluZFJlc3VsdCwgYyA9PiBjLmNvbmZpZyksXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgYXV0aERhdGE6IE9iamVjdC52YWx1ZXMoY29uZmlnTmFtZVRvQmluZFJlc3VsdCkubWFwKGMgPT4gYy5hdXRoZW50aWNhdGlvbikucmVkdWNlKGRlZXBNZXJnZSwgdW5kZWZpbmVkKSxcbiAgICAgICAgfTtcbiAgICB9XG59XG4vKipcbiAqIEEgY29sbGVjdGlvbiBvZiBjb25maWd1cmF0aW9uIGVsZW1lbnRzXG4gKi9cbmV4cG9ydCBjbGFzcyBJbml0Q29uZmlnIHtcbiAgICBwcml2YXRlIHJlYWRvbmx5IGVsZW1lbnRzID0gbmV3IEFycmF5PEluaXRFbGVtZW50PigpO1xuICAgIGNvbnN0cnVjdG9yKGVsZW1lbnRzOiBJbml0RWxlbWVudFtdKSB7XG4gICAgICAgIHRoaXMuYWRkKC4uLmVsZW1lbnRzKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogV2hldGhlciB0aGlzIGNvbmZpZ3NldCBoYXMgZWxlbWVudHMgb3Igbm90XG4gICAgICovXG4gICAgcHVibGljIGlzRW1wdHkoKSB7XG4gICAgICAgIHJldHVybiB0aGlzLmVsZW1lbnRzLmxlbmd0aCA9PT0gMDtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQWRkIG9uZSBvciBtb3JlIGVsZW1lbnRzIHRvIHRoZSBjb25maWdcbiAgICAgKi9cbiAgICBwdWJsaWMgYWRkKC4uLmVsZW1lbnRzOiBJbml0RWxlbWVudFtdKSB7XG4gICAgICAgIHRoaXMuZWxlbWVudHMucHVzaCguLi5lbGVtZW50cyk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIENhbGxlZCB3aGVuIHRoZSBjb25maWcgaXMgYXBwbGllZCB0byBhbiBpbnN0YW5jZS5cbiAgICAgKiBDcmVhdGVzIHRoZSBDbG91ZEZvcm1hdGlvbiByZXByZXNlbnRhdGlvbiBvZiB0aGUgSW5pdCBjb25maWcgYW5kIGhhbmRsZXMgYW55IHBlcm1pc3Npb25zIGFuZCBhc3NldHMuXG4gICAgICogQGludGVybmFsXG4gICAgICovXG4gICAgcHVibGljIF9iaW5kKHNjb3BlOiBDb25zdHJ1Y3QsIG9wdGlvbnM6IEF0dGFjaEluaXRPcHRpb25zKTogSW5pdEVsZW1lbnRDb25maWcge1xuICAgICAgICBjb25zdCBiaW5kT3B0aW9ucyA9IHtcbiAgICAgICAgICAgIGluc3RhbmNlUm9sZTogb3B0aW9ucy5pbnN0YW5jZVJvbGUsXG4gICAgICAgICAgICBwbGF0Zm9ybTogdGhpcy5pbml0UGxhdGZvcm1Gcm9tT1NUeXBlKG9wdGlvbnMucGxhdGZvcm0pLFxuICAgICAgICAgICAgc2NvcGUsXG4gICAgICAgIH07XG4gICAgICAgIGNvbnN0IHBhY2thZ2VDb25maWcgPSB0aGlzLmJpbmRGb3JUeXBlKEluaXRFbGVtZW50VHlwZS5QQUNLQUdFLCBiaW5kT3B0aW9ucyk7XG4gICAgICAgIGNvbnN0IGdyb3Vwc0NvbmZpZyA9IHRoaXMuYmluZEZvclR5cGUoSW5pdEVsZW1lbnRUeXBlLkdST1VQLCBiaW5kT3B0aW9ucyk7XG4gICAgICAgIGNvbnN0IHVzZXJzQ29uZmlnID0gdGhpcy5iaW5kRm9yVHlwZShJbml0RWxlbWVudFR5cGUuVVNFUiwgYmluZE9wdGlvbnMpO1xuICAgICAgICBjb25zdCBzb3VyY2VzQ29uZmlnID0gdGhpcy5iaW5kRm9yVHlwZShJbml0RWxlbWVudFR5cGUuU09VUkNFLCBiaW5kT3B0aW9ucyk7XG4gICAgICAgIGNvbnN0IGZpbGVzQ29uZmlnID0gdGhpcy5iaW5kRm9yVHlwZShJbml0RWxlbWVudFR5cGUuRklMRSwgYmluZE9wdGlvbnMpO1xuICAgICAgICBjb25zdCBjb21tYW5kc0NvbmZpZyA9IHRoaXMuYmluZEZvclR5cGUoSW5pdEVsZW1lbnRUeXBlLkNPTU1BTkQsIGJpbmRPcHRpb25zKTtcbiAgICAgICAgLy8gTXVzdCBiZSBsYXN0IVxuICAgICAgICBjb25zdCBzZXJ2aWNlc0NvbmZpZyA9IHRoaXMuYmluZEZvclR5cGUoSW5pdEVsZW1lbnRUeXBlLlNFUlZJQ0UsIGJpbmRPcHRpb25zKTtcbiAgICAgICAgY29uc3QgYXV0aGVudGljYXRpb24gPSBbcGFja2FnZUNvbmZpZywgZ3JvdXBzQ29uZmlnLCB1c2Vyc0NvbmZpZywgc291cmNlc0NvbmZpZywgZmlsZXNDb25maWcsIGNvbW1hbmRzQ29uZmlnLCBzZXJ2aWNlc0NvbmZpZ11cbiAgICAgICAgICAgIC5tYXAoYyA9PiBjPy5hdXRoZW50aWNhdGlvbilcbiAgICAgICAgICAgIC5yZWR1Y2UoZGVlcE1lcmdlLCB1bmRlZmluZWQpO1xuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgICAgY29uZmlnOiB7XG4gICAgICAgICAgICAgICAgcGFja2FnZXM6IHBhY2thZ2VDb25maWc/LmNvbmZpZyxcbiAgICAgICAgICAgICAgICBncm91cHM6IGdyb3Vwc0NvbmZpZz8uY29uZmlnLFxuICAgICAgICAgICAgICAgIHVzZXJzOiB1c2Vyc0NvbmZpZz8uY29uZmlnLFxuICAgICAgICAgICAgICAgIHNvdXJjZXM6IHNvdXJjZXNDb25maWc/LmNvbmZpZyxcbiAgICAgICAgICAgICAgICBmaWxlczogZmlsZXNDb25maWc/LmNvbmZpZyxcbiAgICAgICAgICAgICAgICBjb21tYW5kczogY29tbWFuZHNDb25maWc/LmNvbmZpZyxcbiAgICAgICAgICAgICAgICBzZXJ2aWNlczogc2VydmljZXNDb25maWc/LmNvbmZpZyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBhdXRoZW50aWNhdGlvbixcbiAgICAgICAgfTtcbiAgICB9XG4gICAgcHJpdmF0ZSBiaW5kRm9yVHlwZShlbGVtZW50VHlwZTogSW5pdEVsZW1lbnRUeXBlLCByZW5kZXJPcHRpb25zOiBPbWl0PEluaXRCaW5kT3B0aW9ucywgJ2luZGV4Jz4pOiBJbml0RWxlbWVudENvbmZpZyB8IHVuZGVmaW5lZCB7XG4gICAgICAgIGNvbnN0IGVsZW1lbnRzID0gdGhpcy5lbGVtZW50cy5maWx0ZXIoZWxlbSA9PiBlbGVtLmVsZW1lbnRUeXBlID09PSBlbGVtZW50VHlwZSk7XG4gICAgICAgIGlmIChlbGVtZW50cy5sZW5ndGggPT09IDApIHtcbiAgICAgICAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgICAgIH1cbiAgICAgICAgY29uc3QgYmluZFJlc3VsdHMgPSBlbGVtZW50cy5tYXAoKGUsIGluZGV4KSA9PiBlLl9iaW5kKHsgaW5kZXgsIC4uLnJlbmRlck9wdGlvbnMgfSkpO1xuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgICAgY29uZmlnOiBiaW5kUmVzdWx0cy5tYXAociA9PiByLmNvbmZpZykucmVkdWNlKGRlZXBNZXJnZSwgdW5kZWZpbmVkKSA/PyB7fSxcbiAgICAgICAgICAgIGF1dGhlbnRpY2F0aW9uOiBiaW5kUmVzdWx0cy5tYXAociA9PiByLmF1dGhlbnRpY2F0aW9uKS5yZWR1Y2UoZGVlcE1lcmdlLCB1bmRlZmluZWQpLFxuICAgICAgICB9O1xuICAgIH1cbiAgICBwcml2YXRlIGluaXRQbGF0Zm9ybUZyb21PU1R5cGUob3NUeXBlOiBPcGVyYXRpbmdTeXN0ZW1UeXBlKTogSW5pdFBsYXRmb3JtIHtcbiAgICAgICAgc3dpdGNoIChvc1R5cGUpIHtcbiAgICAgICAgICAgIGNhc2UgT3BlcmF0aW5nU3lzdGVtVHlwZS5MSU5VWDoge1xuICAgICAgICAgICAgICAgIHJldHVybiBJbml0UGxhdGZvcm0uTElOVVg7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBjYXNlIE9wZXJhdGluZ1N5c3RlbVR5cGUuV0lORE9XUzoge1xuICAgICAgICAgICAgICAgIHJldHVybiBJbml0UGxhdGZvcm0uV0lORE9XUztcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIGRlZmF1bHQ6IHtcbiAgICAgICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0Nhbm5vdCBhdHRhY2ggQ2xvdWRGb3JtYXRpb25Jbml0IHRvIGFuIHVua25vd24gT1MgdHlwZScpO1xuICAgICAgICAgICAgfVxuICAgICAgICB9XG4gICAgfVxufVxuLyoqXG4gKiBPcHRpb25zIGZvciBDbG91ZEZvcm1hdGlvbkluaXQud2l0aENvbmZpZ1NldHNcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDb25maWdTZXRQcm9wcyB7XG4gICAgLyoqXG4gICAgICogVGhlIGRlZmluaXRpb25zIG9mIGVhY2ggY29uZmlnIHNldFxuICAgICAqL1xuICAgIHJlYWRvbmx5IGNvbmZpZ1NldHM6IFJlY29yZDxzdHJpbmcsIHN0cmluZ1tdPjtcbiAgICAvKipcbiAgICAgKiBUaGUgc2V0cyBvZiBjb25maWdzIHRvIHBpY2sgZnJvbVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGNvbmZpZ3M6IFJlY29yZDxzdHJpbmcsIEluaXRDb25maWc+O1xufVxuLyoqXG4gKiBEZWVwLW1lcmdlIG9iamVjdHMgYW5kIGFycmF5c1xuICpcbiAqIFRyZWF0IGFycmF5cyBhcyBzZXRzLCByZW1vdmluZyBkdXBsaWNhdGVzLiBUaGlzIGlzIGFjY2VwdGFibGUgZm9yIHJlbmRlcmluZ1xuICogY2ZuLWluaXRzLCBub3QgYXBwbGljYWJsZSBlbHNld2hlcmUuXG4gKi9cbmZ1bmN0aW9uIGRlZXBNZXJnZSh0YXJnZXQ/OiBSZWNvcmQ8c3RyaW5nLCBhbnk+LCBzcmM/OiBSZWNvcmQ8c3RyaW5nLCBhbnk+KSB7XG4gICAgaWYgKHRhcmdldCA9PSBudWxsKSB7XG4gICAgICAgIHJldHVybiBzcmM7XG4gICAgfVxuICAgIGlmIChzcmMgPT0gbnVsbCkge1xuICAgICAgICByZXR1cm4gdGFyZ2V0O1xuICAgIH1cbiAgICBmb3IgKGNvbnN0IFtrZXksIHZhbHVlXSBvZiBPYmplY3QuZW50cmllcyhzcmMpKSB7XG4gICAgICAgIGlmIChBcnJheS5pc0FycmF5KHZhbHVlKSkge1xuICAgICAgICAgICAgaWYgKHRhcmdldFtrZXldICYmICFBcnJheS5pc0FycmF5KHRhcmdldFtrZXldKSkge1xuICAgICAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgVHJ5aW5nIHRvIG1lcmdlIGFycmF5IFske3ZhbHVlfV0gaW50byBhIG5vbi1hcnJheSAnJHt0YXJnZXRba2V5XX0nYCk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICB0YXJnZXRba2V5XSA9IEFycmF5LmZyb20obmV3IFNldChbXG4gICAgICAgICAgICAgICAgLi4udGFyZ2V0W2tleV0gPz8gW10sXG4gICAgICAgICAgICAgICAgLi4udmFsdWUsXG4gICAgICAgICAgICBdKSk7XG4gICAgICAgICAgICBjb250aW51ZTtcbiAgICAgICAgfVxuICAgICAgICBpZiAodHlwZW9mIHZhbHVlID09PSAnb2JqZWN0JyAmJiB2YWx1ZSkge1xuICAgICAgICAgICAgdGFyZ2V0W2tleV0gPSBkZWVwTWVyZ2UodGFyZ2V0W2tleV0gPz8ge30sIHZhbHVlKTtcbiAgICAgICAgICAgIGNvbnRpbnVlO1xuICAgICAgICB9XG4gICAgICAgIGlmICh2YWx1ZSAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgICB0YXJnZXRba2V5XSA9IHZhbHVlO1xuICAgICAgICB9XG4gICAgfVxuICAgIHJldHVybiB0YXJnZXQ7XG59XG4vKipcbiAqIE1hcCBhIGZ1bmN0aW9uIG92ZXIgdmFsdWVzIG9mIGFuIG9iamVjdFxuICpcbiAqIElmIHRoZSBtYXBwaW5nIGZ1bmN0aW9uIHJldHVybnMgdW5kZWZpbmVkLCByZW1vdmUgdGhlIGtleVxuICovXG5mdW5jdGlvbiBtYXBWYWx1ZXM8QSwgQj4oeHM6IFJlY29yZDxzdHJpbmcsIEE+LCBmbjogKHg6IEEpID0+IEIgfCB1bmRlZmluZWQpOiBSZWNvcmQ8c3RyaW5nLCBCPiB7XG4gICAgY29uc3QgcmV0OiBSZWNvcmQ8c3RyaW5nLCBCPiA9IHt9O1xuICAgIGZvciAoY29uc3QgW2ssIHZdIG9mIE9iamVjdC5lbnRyaWVzKHhzKSkge1xuICAgICAgICBjb25zdCBtYXBwZWQgPSBmbih2KTtcbiAgICAgICAgaWYgKG1hcHBlZCAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgICByZXRba10gPSBtYXBwZWQ7XG4gICAgICAgIH1cbiAgICB9XG4gICAgcmV0dXJuIHJldDtcbn1cbmZ1bmN0aW9uIGNvbnRlbnRIYXNoKGNvbnRlbnQ6IHN0cmluZykge1xuICAgIHJldHVybiBjcnlwdG8uY3JlYXRlSGFzaCgnc2hhMjU2JykudXBkYXRlKGNvbnRlbnQpLmRpZ2VzdCgnaGV4Jyk7XG59XG4iXX0=