"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.jenkinsArtifactsBounds = exports.JenkinsProvider = exports.BaseJenkinsProvider = void 0;
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const custom_action_registration_1 = require("../custom-action-registration");
class BaseJenkinsProvider extends cdk.Construct {
    constructor(scope, id, version) {
        super(scope, id);
        this.version = version || '1';
    }
}
exports.BaseJenkinsProvider = BaseJenkinsProvider;
/**
 * A class representing Jenkins providers.
 *
 * @see #import
 */
class JenkinsProvider extends BaseJenkinsProvider {
    constructor(scope, id, props) {
        super(scope, id, props.version);
        this.buildIncluded = false;
        this.testIncluded = false;
        this.providerName = props.providerName;
        this.serverUrl = props.serverUrl;
        if (props.forBuild === true) {
            this._registerBuildProvider();
        }
        if (props.forTest === true) {
            this._registerTestProvider();
        }
    }
    /**
     * Import a Jenkins provider registered either outside the CDK,
     * or in a different CDK Stack.
     *
     * @param scope the parent Construct for the new provider
     * @param id the identifier of the new provider Construct
     * @param attrs the properties used to identify the existing provider
     * @returns a new Construct representing a reference to an existing Jenkins provider
     */
    static fromJenkinsProviderAttributes(scope, id, attrs) {
        return new ImportedJenkinsProvider(scope, id, attrs);
    }
    /**
     * @internal
     */
    _registerBuildProvider() {
        if (this.buildIncluded) {
            return;
        }
        this.buildIncluded = true;
        this.registerJenkinsCustomAction('JenkinsBuildProviderResource', codepipeline.ActionCategory.BUILD);
    }
    /**
     * @internal
     */
    _registerTestProvider() {
        if (this.testIncluded) {
            return;
        }
        this.testIncluded = true;
        this.registerJenkinsCustomAction('JenkinsTestProviderResource', codepipeline.ActionCategory.TEST);
    }
    registerJenkinsCustomAction(id, category) {
        new custom_action_registration_1.CustomActionRegistration(this, id, {
            category,
            artifactBounds: exports.jenkinsArtifactsBounds,
            provider: this.providerName,
            version: this.version,
            entityUrl: appendToUrl(this.serverUrl, 'job/{Config:ProjectName}'),
            executionUrl: appendToUrl(this.serverUrl, 'job/{Config:ProjectName}/{ExternalExecutionId}'),
            actionProperties: [
                {
                    name: 'ProjectName',
                    required: true,
                    key: true,
                    queryable: true,
                },
            ],
        });
    }
}
exports.JenkinsProvider = JenkinsProvider;
class ImportedJenkinsProvider extends BaseJenkinsProvider {
    constructor(scope, id, props) {
        super(scope, id, props.version);
        this.providerName = props.providerName;
        this.serverUrl = props.serverUrl;
    }
    _registerBuildProvider() {
        // do nothing
    }
    _registerTestProvider() {
        // do nothing
    }
}
function appendToUrl(baseUrl, path) {
    return baseUrl.endsWith('/') ? baseUrl + path : `${baseUrl}/${path}`;
}
exports.jenkinsArtifactsBounds = {
    minInputs: 0,
    maxInputs: 5,
    minOutputs: 0,
    maxOutputs: 5,
};
//# sourceMappingURL=data:application/json;base64,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